package com.sonicsw.mf.jmx.client;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.rmi.dgc.VMID;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.management.Notification;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;

import com.sonicsw.mf.comm.INotificationListener;
import com.sonicsw.mf.comm.jms.ConnectorClient;
import com.sonicsw.mf.comm.jms.DurableConnector;
import com.sonicsw.mf.common.IConsumer;
import com.sonicsw.mf.common.runtime.INotification;

public final class NotificationListenerDelegate
implements INotificationListener, NotificationListener, Serializable
{
    // for NotificationListenerDelegate serialization
    private static final long serialVersionUID = 8484344976480223320L;
    private static final short m_serialVersion = 0;

    // the underlying JMS connector
    private ConnectorClient m_connector;
    private NotificationListener m_listener;
    private IConsumer m_consumer;
    public String m_destination; // this is only set when the object is unserialized
    public int m_listenerHash; // this is only set when the object is unserialized

    private long m_notificationSubscriptionTimeout = DurableConnector.DURABLE_SUBSCRIPTION_TTL;

    private HashMap m_LocalToRemoteHandbacks = new HashMap();
    private HashMap m_filterToRemoteHandbacks = new HashMap();

    // for serialization use
    public NotificationListenerDelegate() { }

    NotificationListenerDelegate(ConnectorClient connector, NotificationListener listener)
    {
        m_connector = connector;
        m_listener = listener;

        // create the subscription
        m_consumer = m_connector.addDirectedNotificationListener(this);
    }

    String addSubscription(NotificationFilter filter, Object handback)
    {
        String remoteHandback = null;

        synchronized(m_filterToRemoteHandbacks)
        {
            remoteHandback = (String)m_LocalToRemoteHandbacks.get(handback);
            if (remoteHandback == null)
            {
                remoteHandback = new VMID().toString();
                m_LocalToRemoteHandbacks.put(handback, remoteHandback);
            }

            ArrayList remoteHandbacks = (ArrayList)m_filterToRemoteHandbacks.get(filter);
            if (remoteHandbacks == null)
            {
                remoteHandbacks = new ArrayList();
                m_filterToRemoteHandbacks.put(filter, remoteHandbacks);
            }

            if (!remoteHandbacks.contains(remoteHandback))
            {
                remoteHandbacks.add(remoteHandback);
            }
        }

        return remoteHandback;
    }

    String removeSubscription(NotificationFilter filter, Object handback)
    {
        String remoteHandback = null;

        synchronized(m_filterToRemoteHandbacks)
        {
            remoteHandback = (String)m_LocalToRemoteHandbacks.get(handback);
            if (remoteHandback == null)
             {
                return null; // its already been removed
            }

            // first count the handback usage (at least determine if the usage is > 1
            int handbackUsageCount = 0;
            Iterator iterator = m_filterToRemoteHandbacks.values().iterator();
            while (iterator.hasNext())
            {
                Iterator remoteHandbacks = ((ArrayList)iterator.next()).iterator();
                while (remoteHandbacks.hasNext())
                {
                    if (remoteHandbacks.next().equals(remoteHandback))
                    {
                        handbackUsageCount++;
                    }
                    if (handbackUsageCount > 1)
                    {
                        break;
                    }
                }
                if (handbackUsageCount > 1)
                {
                    break;
                }
            }

            // now remove the filter/handback combination
            ArrayList remoteHandbacks = (ArrayList)m_filterToRemoteHandbacks.get(filter);
            if (remoteHandbacks == null)
             {
                return null; // in the unlikely event its already been removed
            }
            remoteHandbacks.remove(remoteHandback);

            // if the list is now empty, remove from the table
            if (remoteHandbacks.isEmpty())
            {
                m_filterToRemoteHandbacks.remove(filter);
            }

            // if the handback usage is 1, then remove the handback from the local to remote mapping table
            if (handbackUsageCount == 1)
            {
                m_LocalToRemoteHandbacks.remove(handback);
            }
        }

        return remoteHandback;
    }

    ArrayList getSubscriptionRenewalPairs()
    {
        ArrayList subscriptionRenewalPairs = new ArrayList();

        synchronized(m_filterToRemoteHandbacks)
        {
            Iterator iterator = m_filterToRemoteHandbacks.entrySet().iterator();
            while (iterator.hasNext())
            {
                Map.Entry entry = (Map.Entry)iterator.next();
                NotificationFilter filter = (NotificationFilter)entry.getKey();
                Iterator remoteHandbacks = ((ArrayList)entry.getValue()).iterator();
                while (remoteHandbacks.hasNext())
                {
                    subscriptionRenewalPairs.add(new Object[] { filter, remoteHandbacks.next() });
                }
            }
        }

        return subscriptionRenewalPairs;
    }

    public void setNotificationSubscriptionTimeout(long timeout)
    {
        m_notificationSubscriptionTimeout = timeout;
    }

    public long getNotificationSubscriptionTimeout()
    {
        return m_notificationSubscriptionTimeout;
    }


    // simply forward the notification
    @Override
    public void handleNotification(Notification notification, Object handback)
    {
        Object remoteHandback = null;
        Object localHandback = null;

        // [Sonic00020469] if the given handback is a compund handback ID (see JMSConnectorClient.buildCompoundHandbackID())
        // then the notification was sent from a v2.0 container and we need to strip off the actual handback
        if (handback instanceof String && ((String)handback).charAt(0) == JMSConnectorClient.COMPOUND_HANDBACK_ID_DELIMITER && ((String)handback).charAt(1) == JMSConnectorClient.COMPOUND_HANDBACK_ID_DELIMITER)
        {
            char[] chars = ((String)handback).toCharArray();
            if (chars.length > 4 && chars[chars.length - 1] != JMSConnectorClient.COMPOUND_HANDBACK_ID_DELIMITER)
            {
                int i = 3;
                for (;i < chars.length;i++)
                {
                    if (chars[i] == JMSConnectorClient.COMPOUND_HANDBACK_ID_DELIMITER)
                    {
                        break;
                    }
                }
                if (chars[i] == JMSConnectorClient.COMPOUND_HANDBACK_ID_DELIMITER)
                {
                    handback = String.valueOf(chars, ++i, chars.length - i);
                }
            }
        }

        synchronized(m_filterToRemoteHandbacks)
        {
            Iterator iterator = m_LocalToRemoteHandbacks.entrySet().iterator();
            while (iterator.hasNext())
            {
                Map.Entry entry = (Map.Entry)iterator.next();
                if (entry.getValue().equals(handback))
                {
                    remoteHandback = entry.getValue();
                    localHandback = entry.getKey();
                    break;
                }
            }
        }

        // if the handback is still known, then forward the notification
        if (remoteHandback != null)
        {
            m_listener.handleNotification(notification, localHandback);
        }
    }

    void close()
    {
        if (m_filterToRemoteHandbacks != null)
        {
            synchronized (m_filterToRemoteHandbacks)
            {
                m_LocalToRemoteHandbacks.clear();
                m_filterToRemoteHandbacks.clear();
            }
        }
        m_consumer.close();
    }

    //
    // INotificationListener implementations
    //

    @Override
    public String getListenerType() { return JMSConstants.JMX_COMMS_TYPE; }

    @Override
    public void handleNotification(INotification notification, Object handback)
    {
        // map the MF notification to a JMX notification and push back up to the
        // listener
        // NOTE: At the moment the underlying notification implementation will be a
        // JMX notification so we can just cast the notification
        handleNotification((Notification)notification, handback);
    }

    //
    // Serializable interface
    //

    private void writeObject(ObjectOutputStream stream)
    throws IOException
    {
        // write the [fixed] number of name/value pairs in the stream
        stream.writeInt(4);  // Sonic00022226 - interpret this as "number of name/value pairs"

        // NotificationListenerDelegate specific fields

        stream.writeUTF("serialVersion");
        stream.writeObject(new Short(m_serialVersion));

        stream.writeUTF("destination");
        if (m_destination != null)
        {
            stream.writeObject(m_destination);
        }
        else
        {
            stream.writeObject(m_connector.getReplyTo());
        }

        stream.writeUTF("listenerHash");
        stream.writeObject(new Integer(m_listenerHash == 0 ? this.hashCode() : m_listenerHash));

        stream.writeUTF("notificationSubscriptionTimeout");
        stream.writeObject(new Long(this.m_notificationSubscriptionTimeout));
    }

    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException
    {
        // get the number of name/value pairs in the stream
        int count = stream.readInt();

        // put the name/value pairs in a hash map
        HashMap map = new HashMap(count);
        for (int i = 0; i < count; i++)
        {
            String key = stream.readUTF();
            Object value = stream.readObject();
            map.put(key, value);
        }

        // now map them back into the local variables based on the version
        // Note: As a general rule need to catch exceptions and either do something to set a default
        //       value or ignore
        short serialVersion = ( (Short) map.get("serialVersion")).shortValue();
        switch (serialVersion)
        {
            // case olderVersion<n> ...
            // case olderVersion<n> ...
            default: // the current version or newer versions
            {
                // NotificationListenerDelegate fields
                try { m_destination = (String) map.get("destination"); } catch (Exception e) {}
                try { m_listenerHash = ( (Integer) map.get("listenerHash")).intValue(); } catch (Exception e) {}
                try { m_notificationSubscriptionTimeout = ( (Long) map.get("notificationSubscriptionTimeout")).longValue(); } catch (Exception e) {}
                break;
            }
        }
    }
}

