/**
 * Copyright (c) 2003 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.mf.mgmtapi.config.impl;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.PrintWriter;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import com.sonicsw.ma.mgmtapi.config.IMgmtBase;
import com.sonicsw.ma.mgmtapi.config.IMgmtBeanBase;
import com.sonicsw.ma.mgmtapi.config.IMgmtMapBase;
import com.sonicsw.ma.mgmtapi.config.MgmtBeanFactory;
import com.sonicsw.ma.mgmtapi.config.MgmtException;
import com.sonicsw.ma.mgmtapi.util.ValidName;

import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.security.IManagementPermission;
import com.sonicsw.mf.common.security.IManagementSecurityConfigurator;
import com.sonicsw.mf.common.util.Container;
import com.sonicsw.mf.mgmtapi.config.IAgentManagerBean;
import com.sonicsw.mf.mgmtapi.config.IAuthenticationDomainBean;
import com.sonicsw.mf.mgmtapi.config.IAuthenticationGroupBean;
import com.sonicsw.mf.mgmtapi.config.IAuthenticationUserBean;
import com.sonicsw.mf.mgmtapi.config.IBackupAgentManagerBean;
import com.sonicsw.mf.mgmtapi.config.IBackupDirectoryServiceBean;
import com.sonicsw.mf.mgmtapi.config.IComponentCollectionBean;
import com.sonicsw.mf.mgmtapi.config.IContainerBean;
import com.sonicsw.mf.mgmtapi.config.IContainerCollectionBean;
import com.sonicsw.mf.mgmtapi.config.IDirectoryServiceBean;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.config.gen.impl.AbstractContainerBean;


public class ContainerBean
extends AbstractContainerBean implements IContainerBean
{
    private static final String AM_RUNTIME_ID = "AGENT MANAGER";
    private static final String DS_RUNTIME_ID = "DIRECTORY SERVICE";
    private static boolean m_bEnableFT        = false;
    private static String EXCLUDE_TEMPLATE_TYPE = "EXCLUDE_TEMPLATE_TYPE";
    private static volatile String CURRENT_FT_ROLE = "";

    /**
     * Construct a bean
     */
    public ContainerBean(MgmtBeanFactory factory)
    {
        super(factory);
    }

    @Override
    public void createBean(String viewName, String type, String cVersion, String pVersion)
    throws MgmtException
    {
        super.createBean(viewName, type, cVersion, pVersion);

        setContainerName(getConfigBeanNameTail());

        // This required attribute needs to get set to the same domain name
        // as used in the domain connection.
        //
        super.setDomainName(getMgmtBeanFactory().getDomain());
    }

    //Need to override load, to keep  CURRENT_FT_ROLE in sync
    @Override
    public void loadBean(String viewPath, String type)
    throws MgmtException
    {
        super.loadBean(viewPath, type);

        if (isFaultTolerant())
        {
            CURRENT_FT_ROLE = getFaultToleranceParameters().getFaultToleranceRole();
        }
        else
        {
            CURRENT_FT_ROLE = IContainerConstants.FAULT_TOLERANCE_ROLE_DEFAULT;
        }
    }

    /**
     * Overridden to prevent changing the domain name to something other than
     * that used to create the connection to the domain.
     *
     * @param value  The domain name.
     * @throws MgmtException  Always throws an exception.
     */
    @Override
    public void setDomainName(String value) throws MgmtException
    {
        throw new MgmtException("Can't set Domain Name - implied through Domain Connection");
    }

    protected void viewNameChange(String viewName)
    throws MgmtException
    {
        setContainerName(viewName.substring(viewName.lastIndexOf('/') + 1));
    }

     /**
     * Generate the data for a Container bootfile
     * @return the boot file data
     * @throws MgmtException Thrown if there is a problem generating the bootfile.
     * @deprecated 8.0 release onwards should use generateSetupFile instead.
     */
    @Override
    public String generateBootFile() throws MgmtException
    {
        try
        {
            return getMgmtBeanFactory().exportContainerBootConfiguration(this, getDomainName());
        }
        catch(Exception e)
        {
            throw new MgmtException("Failed to generate boot file", e);
        }
    }

    /**
     * Generates setup file which is used to generate the startup scripts for the Container.
      *
     * @param targetFolder  The path where the setup file would be generated and saved.
     * @param windowsServiceName  The name for windows service. This value is applicable only for Windows OS. 
     * @throws MgmtException Thrown if there is a problem generating the setupfile.
     */
     @Override
    public void generateSetupFile(String targetFolder, String windowsServiceName) throws MgmtException {
         try {
             validatePath(targetFolder);
             String elementPath = getConfigBean().getName();
             IDirElement containerConfig = getMgmtBeanFactory().getConfigServerUtility().getDirectoryService().getFSElement(elementPath, false);

             Properties properties = new Properties();
             properties.setProperty("WINDOWS_SERVICE_NAME", windowsServiceName != null ? windowsServiceName : "");
             properties.setProperty("CONTAINER_PATH", elementPath);

             Container.bootConfigurationToSetupProp(containerConfig, properties);
             targetFolder = targetFolder + File.separator + getConfigBeanNameTail() + ".setup";
             File setupINIFile = new File(targetFolder);

             PrintWriter writer = new PrintWriter(new BufferedWriter(new FileWriter(setupINIFile, false)), true);


             Set keys= properties.keySet();
             Iterator iterator = keys.iterator();
             while(iterator.hasNext()){
                 String key = (String) iterator.next();
                 String value = properties.getProperty(key);
                 writer.println(key + "=" + value);
             }
             writer.close();
         }
         catch (Exception e) {
             throw new MgmtException("Failed to generate setup file", e);
         }
     }
   
    private void validatePath(String targetPath) throws MgmtException{
        File folder = new File(targetPath);
        if(!folder.exists() || !folder.isDirectory()){
            throw new MgmtException("The target file path '" + targetPath + "' is not a valid directory path");
        }
    }

    @Override
    public IContainerBean createBackupContainer(String viewName)
    throws MgmtException
    {
        if(isNewBean())
        {
            throw new MgmtException("Container " + getContainerName() + " should be saved first.");
        }

        //first verify if this container is hosting only legal FT components
        verifyComponentsList();

        IContainerBean.IFaultToleranceType ftParams = null;
        boolean isNew = false;

        try
        {
            ftParams = getFaultToleranceParameters(); //attribute is optonal
        }
        catch (Exception e)
        {
            ftParams = createFaultToleranceParameters();
            isNew = true;
        }

        //Check if backup container already referenced
        IMgmtBeanBase backup = ftParams.getFaultTolerancePeerRef(null);

        if (backup != null)
        {
            throw new MgmtException("Can not create backup container - already exists");
        }
        try
        {
            m_bEnableFT = true; //enables restricted ft operation

            ContainerBean bean = (ContainerBean)clone(viewName);

            if(bean != null)
            {
                //set ft attributes for backup
                IContainerBean.IFaultToleranceType backupFTParams = null;
                boolean isbNew = false;
                try
                {
                    backupFTParams = bean.getFaultToleranceParameters();
                }
                catch (Exception e)
                {
                    backupFTParams = createFaultToleranceParameters();
                    isbNew = true;
                }

                //we need to supress templatable for primary container
                //that SMC can't use any of "Paste as .." action for
                //primary
                HashMap metaAttr = getConfigBean().getMetaAttributes();
                metaAttr.put(EXCLUDE_TEMPLATE_TYPE, Boolean.TRUE);
                getConfigBean().setMetaAttributes(metaAttr);

                //set ft attributes for backup
                backupFTParams.setFaultToleranceRole(IContainerConstants.FAULT_TOLERANCE_ROLE_BACKUP);
                backupFTParams.setFaultTolerancePeerRef(this);

                if(isbNew)
                {
                    bean.setFaultToleranceParameters(backupFTParams);
                }

                //set ft attributes for primary
                ftParams.setFaultToleranceRole(IContainerConstants.FAULT_TOLERANCE_ROLE_PRIMARY);
                ftParams.setFaultTolerancePeerRef(bean);

                if (isNew)
                {
                    this.setFaultToleranceParameters(ftParams);
                }

                m_bEnableFT = false; //resticts ft operations for setting ft role and ft peer-ref

                return bean;
            }
        }
        catch(Exception e)
        {
            m_bEnableFT = false;
            throw new MgmtException(e.getMessage());
        }
        return null;
    }

    @Override
    protected void deleteBean() throws Exception
    {
        IContainerBean.IFaultToleranceType ftParams = null;

        try
        {
            ftParams = getFaultToleranceParameters();
            MgmtBeanFactory factory = getMgmtBeanFactory();

            if (ftParams.getFaultToleranceRole().equals(IContainerConstants.FAULT_TOLERANCE_ROLE_DEFAULT))
            {
                super.deleteBean();
            }

            if (ftParams.getFaultToleranceRole().equals(IContainerConstants.FAULT_TOLERANCE_ROLE_PRIMARY))
            {
                factory.deleteBean(ftParams.getFaultTolerancePeerRef());
                super.deleteBean();
            }

            if (ftParams.getFaultToleranceRole().equals(IContainerConstants.FAULT_TOLERANCE_ROLE_BACKUP))
            {
                IContainerBean peerBean = (IContainerBean)ftParams.getFaultTolerancePeerRef();
                ((ContainerBean)peerBean).resetFT();

                //remove restiction on templatable
                HashMap metaAttr = peerBean.getConfigBean().getMetaAttributes();
                metaAttr.remove(EXCLUDE_TEMPLATE_TYPE);
                peerBean.getConfigBean().setMetaAttributes(metaAttr);

                //delete this backup
                super.deleteBean();
            }

            getMgmtBeanFactory().commit();

        }
        catch(Exception e)//FT parameters  not set so we can just delete
        {
            super.deleteBean();
        }
    }

    private void resetFT()
    {
        IContainerBean.IFaultToleranceType ftParams = null;
        try
        {
            ftParams = getFaultToleranceParameters();
            if (ftParams.getFaultToleranceRole().equals(IContainerConstants.FAULT_TOLERANCE_ROLE_PRIMARY))
            {
                m_bEnableFT = true;
                ftParams.setFaultToleranceRole(IContainerConstants.FAULT_TOLERANCE_ROLE_DEFAULT);
                ftParams.removeAttribute(IContainerConstants.FAULT_TOLERANCE_PEER_REF_ATTR);
                m_bEnableFT = false;
            }
        }
        catch(Exception e){ m_bEnableFT = false; } //nothing to reset
    }

    private void verifyComponentsList()
    throws MgmtException
    {
        String errorMessage = "Can't create backup broker. This container hosting ";
        IContainerBean.IComponentsType components = getComponents();
        List names = components.getKeyNames();

        if (names.isEmpty())
        {
            return;
        }

        if (names.contains(DS_RUNTIME_ID))
        {
            throw new MgmtException(errorMessage + DS_RUNTIME_ID);
        }

        if (names.contains(AM_RUNTIME_ID))
        {
            throw new MgmtException(errorMessage + AM_RUNTIME_ID);
        }

        for(Iterator i = names.iterator(); i.hasNext();)
        {
            IContainerBean.IStartupParams params = components.getEntry((String)i.next());
            IMgmtBeanBase refBean =  params.getConfigRef();
            String type = refBean.getConfigBean().getConfigType().getName();
            if (type.equals("MQ_BROKER"))
            {
                throw new MgmtException(errorMessage + "broker component.");
            }
            if (type.equals("MQ_BACKUPBROKER"))
            {
                throw new MgmtException(errorMessage + "backup broker component.");
            }
        }

    }

    @Override
    public boolean isFaultTolerant()
    {
        try
        {
            if (getFaultToleranceParameters().getFaultToleranceRole().equals(IContainerConstants.FAULT_TOLERANCE_ROLE_DEFAULT))
            {
                return false;
            }

            return true;

        }
        catch(Exception e){ return false; }
    }

    /*
    * Restricts cloning if container is configured as primary or backup
    */
    @Override
    public IMgmtBeanBase clone(String viewName)
    throws MgmtException
    {
        IContainerBean.IFaultToleranceType ftParams = null;

        try
        {
            ftParams = getFaultToleranceParameters();

            if (ftParams.getFaultToleranceRole().equals(IContainerConstants.FAULT_TOLERANCE_ROLE_PRIMARY) ||
                    ftParams.getFaultToleranceRole().equals(IContainerConstants.FAULT_TOLERANCE_ROLE_BACKUP))
            {
                throw new MgmtException("Illegal operation.Primary or backup container can't be clone.");
            }

        }
        catch(Exception e)// ftParams not set, so container is "Not Fault Tolerant"
        {
            //ignore it
        }

        ContainerBean cloneBean = (ContainerBean)super.clone(viewName);
        //Set CONTAINER_NAME attribute
        cloneBean.setContainerName(cloneBean.getConfigBeanNameTail());

        return cloneBean;
    }

    /*
    * Inner classes
    */
    public static class MonitoringType
    extends AbstractMonitoringType implements IMonitoringType
    {
        public MonitoringType(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }
    }

    public static class CacheType
    extends AbstractCacheType implements ICacheType
    {
        public CacheType(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }
    }

    public static class ComponentsType
    extends AbstractComponentsType implements IComponentsType
    {
        public ComponentsType(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }

        /*
        *  Overwrite MgmtMapBase.add(String name, Object val) in order to prevent adding illegal components to the FT container.
        */
        @Override
        public void add(String name, Object val)
        throws MgmtException
        {
             if (val instanceof IContainerBean.IStartupParams)
             {
                 if (!CURRENT_FT_ROLE.equals(IContainerConstants.FAULT_TOLERANCE_ROLE_DEFAULT))
                {
                    validateComponentType(((IContainerBean.IStartupParams)val).getConfigRef().getConfigBean().getConfigType().getName());
                }

                 validateComponentName(name, ((IContainerBean.IStartupParams)val).getConfigRef());
                 super.add(name, val);
             }
            else
            {
                throw new MgmtException("Invalid object type. Only IContainerBean.IStartupParams can be added to the components list");
            }
        }


        @Override
        public void addEntry(String name, IContainerBean.IStartupParams value) throws MgmtException
        {
            add(name, value);
        }

        @Override
        public void deleteEntry(String name) throws MgmtException {
            IDirectoryFileSystemService ds = super.getMgmtBeanFactory().getConfigServerUtility().getDirectoryService();
            String componentPath = getConfigBean().getName()+":ID="+name;
            try {
                ((IManagementSecurityConfigurator)ds).removeManagementPermissions(new String[]{componentPath}, IManagementPermission.MANAGE_TYPE);
            } catch (DirectoryServiceException e) {
                throw new MgmtException("You are not authorized to (un)set manage permissions already defined " +
                        "for this component, you are therefore not allowed to remove this component from its container",e);
            }
            super.deleteEntry(name);
        }

        private void validateComponentName(String name, IMgmtBeanBase component) throws MgmtException
        {
           try
           {
               ValidName.validate(name, ValidName.COMPONENT_TYPE);
           }
           catch (Exception e)
           {
               throw new MgmtException("Invalid format name.\nAllowed alpha-numeric, '-' and space characters.", e);
           }

            if ( component instanceof IAgentManagerBean || component instanceof IBackupAgentManagerBean)
            {
                if (!name.equals(AM_RUNTIME_ID))
                {
                    throw new MgmtException("Invalid name. Valid name is - \"AGENT MANAGER\".");
                }
            }
            else if ( component instanceof IDirectoryServiceBean || component instanceof IBackupDirectoryServiceBean )
            {
                if (!name.equals(DS_RUNTIME_ID))
                {
                    throw new MgmtException("Invalid name. Valid name is - \"DIRECTORY SERVICE\".");
                }
            }
            else
            {
                if ( (name.equals(AM_RUNTIME_ID)) || (name.equals(DS_RUNTIME_ID)) )
                {
                    throw new MgmtException(name + " is reserved name.");
                }
            }
        }

        private void validateComponentType(String type)
        throws MgmtException
        {
            if ( type.equals("MQ_BROKER")
                 || type.equals("MQ_BACKUPBROKER")
                 || type.equals(AgentManagerBean.DS_TYPE)
                 || type.equals(DirectoryServiceBean.DS_TYPE))
            {
                throw new MgmtException("Illegal component");
            }

        }

    }

    public static class MetricsCollectionType
    extends AbstractMetricsCollectionType implements IMetricsCollectionType
    {
        public MetricsCollectionType(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }
    }

    public static class SystemPropertiesType
    extends AbstractSystemPropertiesType implements ISystemPropertiesType
    {
        public SystemPropertiesType(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }
    }

    public static class EnabledMetricsType
    extends AbstractEnabledMetricsType implements IEnabledMetricsType
    {
        public EnabledMetricsType(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }
    }

    public static class EnabledAlertsType
    extends AbstractEnabledAlertsType implements IEnabledAlertsType
    {
        public EnabledAlertsType(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }

        @Override
        public void addEntry(String name, IEnabledAlertParams value) throws MgmtException {
            if(name.endsWith(".*"))
            {
                throw new MgmtException("Cannot set alerts for a metric pattern containing wild cards '" + name + "'" );
            }
            super.addEntry(name, value); 
        }
    }

    public static class EnabledAlertParams
    extends AbstractEnabledAlertParams implements IEnabledAlertParams
    {
        public EnabledAlertParams(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }
    }


    public static class CentralConnectionType
    extends AbstractCentralConnectionType implements ICentralConnectionType
    {
        public CentralConnectionType(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }
    }


    public static class ExtensionsType
    extends AbstractExtensionsType implements IExtensionsType
    {
        public ExtensionsType(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }
    }

    public static class FaultToleranceType
    extends AbstractFaultToleranceType implements IFaultToleranceType
    {
        public FaultToleranceType(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }

        @Override
        public void setFaultToleranceRole(String value) throws MgmtException
        {
           if(!m_bEnableFT)
        {
            throw new MgmtException("Illegal operation");
        }

           CURRENT_FT_ROLE = value;
           super.setFaultToleranceRole(value);

        }

        @Override
        public void setFaultTolerancePeerRef(IMgmtBeanBase value) throws MgmtException
        {
        	if(!m_bEnableFT)
            {
                throw new MgmtException("Illegal operation");
            }

            super.setFaultTolerancePeerRef(value);
        }

        @Override
        public void setFaultDetectionInterval(int value) throws MgmtException {
        	if (!((ContainerBean) super.m_parent).isFaultTolerant())
            {
                throw new MgmtException("Illegal operation");
            }

			if (isPrimaryFaultRole())
            {
                super.setFaultDetectionInterval(value);
            }
            else
            {
                throw new MgmtException("Cannot set FaultDetectionInterval for Backup Container");
            }
        }

        @Override
        public void setFaultDetectionTimeout(int value) throws MgmtException {
        	if (!((ContainerBean) super.m_parent).isFaultTolerant())
            {
                throw new MgmtException("Illegal operation");
            }

			if (isPrimaryFaultRole())
            {
                super.setFaultDetectionTimeout(value);
            }
            else
            {
                throw new MgmtException("Cannot set FaultDetectionTimeout for Backup Container");
            }
        }

        @Override
        public void setFailureDetectionConnectionurls(String value) throws MgmtException {
        	if (!((ContainerBean) super.m_parent).isFaultTolerant())
            {
                throw new MgmtException("Illegal operation");
            }

			if (isPrimaryFaultRole())
            {
                super.setFailureDetectionConnectionurls(value);
            }
            else
            {
                throw new MgmtException("Cannot set Failure Detection ConnectionURLs for Backup Container");
            }
        }

        @Override
        public void setFailureDetectionDefaultUser(String value) throws MgmtException {
        	if (!((ContainerBean) super.m_parent).isFaultTolerant())
            {
                throw new MgmtException("Illegal operation");
            }

			if (isPrimaryFaultRole())
            {
                super.setFailureDetectionDefaultUser(value);
            }
            else
            {
                throw new MgmtException("Cannot set Failure Detection DefaultUser for Backup Container");
            }
        }

        @Override
        public void setFailureDetectionDefaultPassword(String value) throws MgmtException {
        	if (!((ContainerBean) super.m_parent).isFaultTolerant())
            {
                throw new MgmtException("Illegal operation");
            }

			if (isPrimaryFaultRole())
            {
                super.setFailureDetectionDefaultPassword(value);
            }
            else
            {
                throw new MgmtException("Cannot set Failure Detection DefaultPassword for Backup Container");
            }
        }

        @Override
        public void setFailureDetectionConnectTimeout(int value) throws MgmtException {
        	if (!((ContainerBean) super.m_parent).isFaultTolerant())
            {
                throw new MgmtException("Illegal operation");
            }

			if (isPrimaryFaultRole())
            {
                super.setFailureDetectionConnectTimeout(value);
            }
            else
            {
                throw new MgmtException("Cannot set Failure Detection ConnectTimeout for Backup Container");
            }
        }

        @Override
        public void setFailureDetectionSocketConnectTimeout(int value) throws MgmtException {
        	if (!((ContainerBean) super.m_parent).isFaultTolerant())
            {
                throw new MgmtException("Illegal operation");
            }

			if (isPrimaryFaultRole())
            {
                super.setFailureDetectionSocketConnectTimeout(value);
            }
            else
            {
                throw new MgmtException("Cannot set Failure Detection SocketConnectTimeout for Backup Container");
            }
        }

        private boolean isPrimaryFaultRole() throws MgmtException {
        	return IContainerConstants.FAULT_TOLERANCE_ROLE_PRIMARY.equals(getFaultToleranceRole());
        }
    }

    public static class ConnectionType
    extends AbstractConnectionType implements IConnectionType
    {
        public ConnectionType(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }
    }

    public static class StartupParams
    extends AbstractStartupParams implements IStartupParams
    {
        public StartupParams(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }

        @Override
        public void setConfigRef(IMgmtBeanBase value) throws MgmtException
        {
            if (!validateComponentType(value))
            {
                throw new MgmtException("Invalid component type.");
            }
            checkDuplicates(value);
            super.setReferenceAttribute(CONFIG_REF_ATTR, value);
        }

        private void checkDuplicates(IMgmtBeanBase component) throws MgmtException
        {
            String duplicate = null;

            if ( component instanceof IAgentManagerBean || component instanceof IBackupAgentManagerBean )
            {
                duplicate = AM_RUNTIME_ID;
            }
            if ( component instanceof IDirectoryServiceBean || component instanceof IBackupDirectoryServiceBean )
            {
                duplicate = DS_RUNTIME_ID;
            }

            if (duplicate == null)
            {
                return;
            }

            List cNames = ((IMgmtMapBase)getParent()).getKeyNames();
            if (cNames.contains(duplicate))
            {
                String type = (duplicate.equals(AM_RUNTIME_ID)) ? "MF_AGENT_MANAGER" : "MF_DIRECTORY_SERVICE";
                throw new MgmtException( type + " can't be deployed twice to the container");
            }
        }

        private boolean validateComponentType(IMgmtBeanBase component)
        {
            if( (component instanceof IContainerBean) ||
                (component instanceof IComponentCollectionBean) ||
                (component instanceof IContainerCollectionBean) ||
                (component instanceof IAuthenticationDomainBean)||
                (component instanceof IAuthenticationUserBean)  ||
                (component instanceof IAuthenticationGroupBean))
            {
                return false;
            }

            return true;
        }

    }

    public static class ExtensionType
    extends AbstractExtensionType implements IExtensionType
    {
        public ExtensionType(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }
    }

    public static class LibrariesType
    extends AbstractLibrariesType implements ILibrariesType
    {
        public LibrariesType(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }
    }

    public static class DeploymentParametersSet
    extends AbstractDeploymentParametersSet implements IDeploymentParametersSet
    {
        public DeploymentParametersSet(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }
    }

    public static class LibraryParams
    extends AbstractLibraryParams implements ILibraryParams
    {
        public LibraryParams(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }
    }

    public static class DeploymentParameterType
    extends AbstractDeploymentParameterType implements IDeploymentParameterType
    {
        public DeploymentParameterType(IMgmtBase parent, String name, boolean isNew)
        throws MgmtException
        {
            super(parent, name, isNew);
        }
    }
}
