// Copyright (c) 2009 Progress Software Corporation. All Rights Reserved.
package com.sonicsw.mf.mgmtapi.runtime;

import java.util.Properties;

import com.sonicsw.mf.common.runtime.IFileDescriptor;
import com.sonicsw.mf.common.runtime.IRemoteCallResult;
import com.sonicsw.mf.common.runtime.IRemoteExecResult;

/**
 * Provides remote access to the Host Manager. Used, in particular, for Centralized Install related management.
 *
 * The following is an example of connecting to 'AutoTestContainer' (a root container); Setting up the new container '/Containers/AutoTestContainer'; and launching it:
 *
 * <pre>
 * Hashtable env = new Hashtable();
 * env.put(IContainerConstants.CONNECTIONURLS_ATTR, "tcp://localhost:2506");
 * JMSConnectorAddress address = new JMSConnectorAddress(env);
 * JMSConnectorClient connector = new JMSConnectorClient();
 * connector.connect(address);
 * System.out.println("Connected...");
 *
 * //Create a proxy to the Host Manager in the AutoTestContainer container
 * IHostManagerProxy hostManager = MFProxyFactory.createHostManagerProxy(connector, new ObjectName("Domain1.AutoTestContainer:ID=HOST MANAGER"));
 *
 * //Setup the SonicRemote container
 * Properties containerProps = new Properties();
 * containerProps.put("WINDOWS_SERVICE_NAME", "AutoTestContainer");
 * containerProps.put("DOMAIN_NAME", "Domain1");
 * containerProps.put("CREATE_IF_DOES_NOT_EXIST", "true");
 * containerProps.put("CONTAINER_PATH", "/Containers/AutoTestContainer");
 * containerProps.put("ConnectionURLs", "tcp://localhost:2506");
 * containerProps.put("DefaultPassword", "Administrator");
 * containerProps.put("DefaultUser", "Administrator");
 * System.out.println("Setup result: " + hostManager.setupContainer(containerProps, null));
 *
 * //Launch the SonicRemote container from a script
 * System.out.println("Launch result: " + hostManager.launchContainer("AutoTestContainer", false));
 * </pre>
 */
public interface IHostManagerProxy
{
   /**
    * Returns the system properties of the container.
    *
    * @return Returns the system properties of the container
    *
    * @exception com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException thrown due to communication failure that prevented the remote execution or the reception of the execution result
    */
    public Properties getJVMProperties();

   /**
    * Download a file from the Directory Service into the container's file-system.
    *
    * @param dsFilePath File path in the DS
    * @param destinationFilePath destination path on the container side
    * @param rewriteIfExists fails if false and the file already exists
    * @param expandZip if 'true' expand the zip file after downloading it
    *
    * @return success/failure result
    *
    * @exception com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException thrown due to communication failure that prevented the remote execution or the reception of the execution result
    */
    public IRemoteCallResult downloadFileFromDS(String dsFilePath, String destinationFilePath, boolean rewriteIfExists, boolean expandZip);


   /**
    * Copy a file or a directory from the client's file system to the container's file-system. This is intended for small to medium size files
    * since the file is fully read into memory.
    *
    * @param sourceFilePath Client side path
    * @param destinationFilePath destination path on the container side
    * @param rewriteIfExists fails if false and the file already exists
    *
    * @return success/failure result
    *
    * @exception com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException thrown due to communication failure that prevented the remote execution or the reception of the execution result
    */
    public IRemoteCallResult putFiles(String sourceFilePath, String destinationFilePath, boolean rewriteIfExists);

   /**
    * Return the list of files and directories under directoryPath
    *
    * @param directoryPath an absolute path or a path relative to the working directory of the container
    * @return list of sub-directories and files
    *
    * @exception com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException thrown due to communication failure that prevented the remote execution or the reception of the execution result
    */
    public IFileDescriptor[] listDirectory(String directoryPath);

   /**
    * Install the launcher that launched the container that hosts this HM under targetSonicHome
    *
    * @param targetSonicHome install the launcher under this directory
    *
    * @return success/failure result
    *
    * @exception com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException thrown due to communication failure that prevented the remote execution or the reception of the execution result
    *
    *
    */
    public IRemoteCallResult installLauncher(String targetSonicHome);

   /**
    * Setup a new container by generating the launch and Windows Service scripts and the container.ini file. The generated container scripts will use the same launcher version
    * used to launch the remote container.
    *
    * @param properties the container setup properties
    * @param INIEncryptionPassword an optional password for INI file encryption
    *
    * @return success/failure result
    *
    * @exception com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException thrown due to communication failure that prevented the remote execution or the reception of the execution result
    *
    * Example
    *   Properties containerProps = new Properties();
    *   containerProps.put("WINDOWS_SERVICE_NAME", "SonicRemote");
    *   containerProps.put("DOMAIN_NAME", "Domain1");
    *   containerProps.put("CREATE_IF_DOES_NOT_EXIST", "true");
    *   containerProps.put("CONTAINER_PATH", "/Containers/SonicRemote");
    *   containerProps.put("ConnectionURLs", "tcp://localhost:2506");
    *   containerProps.put("DefaultPassword", "Administrator");
    *  containerProps.put("DefaultUser", "Administrator");
    *
    *
    */
    public IRemoteCallResult setupContainer(Properties properties, String INIEncryptionPassword);

   /**
    * Setup a new container under the specified soniv home. NOTE: This method should be used only in special cases. The setupContainer methods that implictly use the
    * same sonic home used by the current container should be typically used.
    *
    * @param sonicHome the sonic home used by the new container
    * @param launcherVersion the launcher version used by the new container. If 'null' will use the same launcher version used to launch the remote container.
    * @param properties the container setup properties
    * @param INIEncryptionPassword an optional password for INI file encryption
    *
    * @return success/failure result
    *
    * @exception com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException thrown due to communication failure that prevented the remote execution or the reception of the execution result
    *
    * Example
    *   Properties containerProps = new Properties();
    *   containerProps.put("WINDOWS_SERVICE_NAME", "SonicRemote");
    *   containerProps.put("DOMAIN_NAME", "Domain1");
    *   containerProps.put("CREATE_IF_DOES_NOT_EXIST", "true");
    *   containerProps.put("CONTAINER_PATH", "/Containers/SonicRemote");
    *   containerProps.put("ConnectionURLs", "tcp://localhost:2506");
    *   containerProps.put("DefaultPassword", "Administrator");
    *  containerProps.put("DefaultUser", "Administrator");
    *
    *
    */
    public IRemoteCallResult setupContainer(String sonicHome, String launcherVersion, Properties properties, String INIEncryptionPassword);

   /**
    * Setup a new container by generating the launch and Windows Service scripts and the container.ini file. The generated container scripts will use the same launcher version
    * used to launch the remote container. Only non DS configured properties
    * (see example below) should be passed in the 'unConfiguredProps' parameter. The configured properties are retrieved from the container configuration in the DS.
    *
    * @param containerDSPath the path of the container configuration in the DS
    * @param unConfiguredProps the container unconfigured setup properties
    * @param INIEncryptionPassword an optional password for INI file encryption
    *
    * @return success/failure result
    *
    * @exception com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException thrown due to communication failure that prevented the remote execution or the reception of the execution result
    *
    * Properties Example
    *   Properties unConfiguredProps = new Properties();
    *   unConfiguredProps.put("WINDOWS_SERVICE_NAME", "SonicRemote");
    *
    *
    *
    */
    public IRemoteCallResult setupContainer(String containerDSPath, Properties unConfiguredProps, String INIEncryptionPassword);

   /**
    * Launches a container that was previously setup. If launchAsWindowsService is true and the container is launched
    * on Windows then the method would launch the container as Windows Service. The method starts an asynchronous process to start the container
    * and doesn't wait for it to actually start up.
    *
    * @param containerName Container name
    * @param launchAsWindowsService if true and on Windows will launch the container as Windows Service
    *
    * @return success/failure result; the successful start of the container startup process doesn't guarantee an unltimate successful startup.
    *
    * @exception com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException thrown due to communication failure that prevented the remote execution or the reception of the execution result
    */
    public IRemoteCallResult launchContainer(String containerName, boolean launchAsWindowsService);

   /**
    * Launches a container that was previously setup in a different home. If launchAsWindowsService is true and the container is launched
    * on Windows then the method would launch the container as Windows Service. The method starts an asynchronous process to start the container
    * and doesn't wait for it to actually start up.
    *
    * @param sonicHome the home of this container
    * @param containerName Container name
    * @param launchAsWindowsService if true and on Windows will launch the container as Windows Service
    *
    * @return success/failure result; the successful start of the container startup process doesn't guarantee an unltimate successful startup.
    *
    * @exception com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException thrown due to communication failure that prevented the remote execution or the reception of the execution result
    */
    public IRemoteCallResult launchContainer(String sonicHome, String containerName, boolean launchAsWindowsService);

   /**
    * Uninstall the Windows Service (if installed) and remove the files from the working directory. The container must be must shutdown
    * prior to calling removeContainer.
    *
    ** @param containerName Container name
    *
    * @return success/failure result
    *
    * @exception com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException thrown due to communication failure that prevented the remote execution or the reception of the execution result
    */
    public IRemoteCallResult removeContainer(String containerName);

   /**
    * The remoteExec call executes java.lang.Runtime.exec on the target container. The standard output and standard error of the call are returned in
    * IRemoteExecResult as byte arrays - that imposes some practical limit on the size of the output.
    *
    * @param cmdarray a specified system command.
    * @param   envp      array of strings, each element of which
    *                    has environment variable settings in the format
    *                    <i>name</i>=<i>value</i>, or
    *                    null if the subprocess should inherit
    *                    the environment of the current process.
    * @param workDirPath the working directory of the subprocess, or
    *                    null if the subprocess should inherit
    *                    the working directory of the current process.
    * @param inputBytes        Standrd input for the process (or null if there is no input)
    *
    *
    * @return success/failure result and the standard output and error generated by the remote execution
    *
    * @exception com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException thrown due to communication failure that prevented the remote execution or the reception of the execution result
    */
    public IRemoteExecResult remoteExec(String[] cmdarray, String[] envp, String workDirPath, byte[] inputBytes);

   /**
    * Copy a file from the container's file-system to the client's file system. This is intended for small to medium size files
    * since the file is fully read into memory.
    *
    * @param srcFilePath The file path on the container side
    * @param destinationFilePath The file path on the client side
    * @param rewriteIfExists fails if false and the file already exists
    * @return success/failure result
    *
    * @exception com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException thrown due to communication failure that prevented the remote execution or the reception of the execution result
    */
    public IRemoteCallResult getFiles(String srcFilePath, String destinationFilePath, boolean rewriteIfExists);

   /**
    * Delete the file or directory. If 'deleteContent' is false and the directory is not empty the operation will fail; if true, the content
    * of the directory will be first deleted
    *
    * @param pathOfdirectoryOrFile the path of a directory or a file
    * @param deleteContent if true and 'pathOfdirectoryOrFile' is a directory, deletes the directory's content
    *
    * @return success/failure result
    *
    * @exception com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException thrown due to communication failure that prevented the remote execution or the reception of the execution result
    */
    public IRemoteCallResult deleteFiles(String pathOfdirectoryOrFile, boolean deleteContent);
    
    /**
     * Gets all known hostnames and ip addresses of this machine. This will
     * (on request) include the loopback interface (localhost) as well as all ip
     * address/hostnames of all network interfaces.
     * 
     * <p>
     * Hostnames are added as strings to the set as <b>lower case</b> values.
     * 
     * <p>
     * <b>Warning</b>: this method can be slow, since for each known address of
     * each network interface we do reverse dns lookup <b>twice</b>.
     * 
     * @param includeLoopback
     *            if the loopback address and hostnames should be included or
     *            not
     * 
     * @return array of all known ip addresses and hostnames both domain name
     *         qualified and unqualified.
     * @throws com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException 
     */
    public String[] getAllHostnamesAndIPs(boolean includeLoopback);

    /**
     * Invokes the reload operation.
     * <p>
     * Asynchronously reload the component and restart the component if it was previously online.
     * <p>
     * Impact: ACTION
     */
    public void reload();

    /**
     * Invokes the start operation.
     * <p>
     * Start providing service.
     * <p>
     * Impact: ACTION
     * @see #stop()
     */
    public void start();

    /**
     * Invokes the stop operation.
     * <p>
     * Stop providing service.
     * <p>
     * Impact: ACTION
     * @see #start()
     */
    public void stop();
}
