package com.sonicsw.mf.runtime.tools;

import java.io.File;
import java.io.FileWriter;
import java.io.PrintWriter;
import java.io.Writer;
import java.text.CollationKey;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.StringTokenizer;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanInfo;
import javax.management.MBeanNotificationInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanParameterInfo;
import javax.management.ObjectName;

import com.sonicsw.mx.util.ICollationKeyProvider;
import com.sonicsw.mx.util.IEmptyArray;
import com.sonicsw.mx.util.Sorter;

import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.metrics.MetricsFactory;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.jmx.client.JMSConnectorAddress;
import com.sonicsw.mf.jmx.client.JMSConnectorClient;
import com.sonicsw.mf.jmx.client.MFNotification;

public final class GenerateProxy
implements ICollationKeyProvider
{
    private static final String[] GET_ELEMENT_SIGNATURE = new String[]
    {
        String.class.getName(),
        Boolean.class.getName()
    };

    private static final String DEFAULT_URL = "localhost";
    private static final String DEFAULT_USER = "Administrator";
    private static final String DEFAULT_PWD = "Administrator";
    private static final String MONITORING_DOC_DIR = "mgmt_monitoring";

    private String m_url       = DEFAULT_URL;
    private String m_user      = DEFAULT_USER;
    private String m_password  = DEFAULT_PWD;

    private String m_component  = null;
    private String m_name       = null;
    private String m_srcOutput  = null;
    private String m_docOutput  = null;
    private String m_metricsSeedDir = null;

    private boolean m_jmx = false;

    private PrintWriter m_writer;

    private MBeanInfo m_mBeanInfo = null;
    private HashMap m_attributes = new HashMap();
    private boolean m_hasMetrics = false;
    private IMetricInfo[] m_metricInfos = null;
    private String[] m_packageTokens = new String[3];
    private ObjectName m_objectName = null;
    // Optional. Used if DS is in separate container from component of interest
    private String m_dsContainerName = null; 

    public GenerateProxy(String[] args)
    throws Exception
    {
        try
        {
            parseArgs(args);
        }
        catch(Exception e)
        {
            printUsage();

            throw e;
        }

        System.out.println("Generate Runtime Proxy for....");
        System.out.println("    JMX Name      = " + m_component);
        System.out.println("    Name          = " + m_name);
        System.out.println("    Output        = " + m_srcOutput);
        System.out.println("    Doc Output    = " + m_docOutput);

        Hashtable env = new Hashtable();
        env.put("ConnectionURLs", m_url);
        env.put("DefaultUser", m_user);
        env.put("DefaultPassword", m_user);
        JMSConnectorAddress address = new JMSConnectorAddress(env);
        JMSConnectorClient connector = new JMSConnectorClient();
        connector.connect(address);
        m_objectName = new ObjectName(m_component);

        // get the info we need to generate the interface and impl
        m_mBeanInfo = connector.getMBeanInfo(m_objectName);
        MBeanAttributeInfo[] attributeInfos = m_mBeanInfo.getAttributes();
        String[] attributeNames = new String[attributeInfos.length];
        for (int i = 0; i < attributeInfos.length; i++)
        {
            attributeNames[i] = attributeInfos[i].getName();
        }
        AttributeList attributes = connector.getAttributes(m_objectName, attributeNames);
        for (int i = attributes.size() - 1; i >= 0; i--)
        {
            Attribute attribute = (Attribute)attributes.get(i);
            m_attributes.put(attribute.getName(), attribute.getValue());
        }
        // are there metrics ?
        MBeanOperationInfo[] infos = m_mBeanInfo.getOperations();
        for (int i = 0; i < infos.length; i++)
        {
            if (infos[i].getName().equals("enableMetrics"))
            {
                m_hasMetrics = true;
                m_metricInfos = (IMetricInfo[])connector.invoke(m_objectName, "getMetricsInfo", new Object[0], new String[0]);
                Sorter.sort(m_metricInfos, this, m_metricInfos.length);
                break;
            }
        }

        // determine tha package tokens that will be used to generate the full output path of java source code
        // and the package statement
        StringTokenizer st = new StringTokenizer((String)m_attributes.get("Classname"), ".");
        m_packageTokens[0] = st.nextToken();
        m_packageTokens[1] = st.nextToken();
        m_packageTokens[2] = st.nextToken();

        String filename = null;
        Writer writer = null;

        System.out.println("Generating...");

        //
        // Proxy interface
        //
        if (m_srcOutput != null)
        {
            filename = m_srcOutput + File.separator + m_packageTokens[0] + File.separator + m_packageTokens[1] + File.separator + m_packageTokens[2] + File.separator + "mgmtapi" + File.separator + "runtime" + File.separator + "I" + m_name + "Proxy.java";
            System.out.println(" - " + filename + " ...");
            writer = new FileWriter(filename);
            m_writer = new PrintWriter(writer);
            generateInterface();
            writer.close();

            //
            // Proxy impl
            //
            filename = m_srcOutput + File.separator + m_packageTokens[0] + File.separator + m_packageTokens[1] + File.separator + m_packageTokens[2] + File.separator + "mgmtapi" + File.separator + "runtime" + File.separator + "impl" + File.separator
                       + m_name + "Proxy.java";
            System.out.println(" - " + filename + " ...");
            writer = new FileWriter(filename);
            m_writer = new PrintWriter(writer);
            generateImpl();
            writer.close();
        }

        //
        // Metrics seed and html
        //
        if (m_hasMetrics)
        {
            if (m_metricsSeedDir != null)
            {
                String releaseVersion = (String)m_attributes.get("ReleaseVersion");
                filename = m_metricsSeedDir + File.separator + m_name + "MetricsMetadata" + ".seed";
                System.out.println(" - " + filename + " ...");
                writer = new FileWriter(filename);
                m_writer = new PrintWriter(writer);
                generateMetricsMetadataSeed(connector, releaseVersion);
                writer.close();
            }

            if (m_docOutput != null)
            {
                filename = m_docOutput + File.separator + m_name + "Metrics.html";
                System.out.println(" - " + filename + " ...");
                writer = new FileWriter(filename);
                m_writer = new PrintWriter(writer);
                generateMetricsDoc();
                writer.close();
            }
        }

        //
        // Notifications html
        //
        if (m_docOutput != null && m_mBeanInfo.getNotifications().length > 0)
        {
            filename = m_docOutput + File.separator + m_name + "Notifications.html";
            System.out.println(" - " + filename + " ...");
            writer = new FileWriter(filename);
            m_writer = new PrintWriter(writer);
            generateNotificationsDoc();
            writer.close();
        }

        System.out.println("... Done");
    }

    @Override
    public CollationKey getCollationKey(Object object)
    {
        if (object instanceof IMetricInfo)
        {
            return Sorter.COLLATOR.getCollationKey(((IMetricInfo)object).getMetricIdentity().getName());
        }
        if (object instanceof MBeanAttributeInfo)
        {
            return Sorter.COLLATOR.getCollationKey(((MBeanAttributeInfo)object).getName());
        }
        if (object instanceof MBeanOperationInfo)
        {
            return Sorter.COLLATOR.getCollationKey(((MBeanOperationInfo)object).getName());
        }
        if (object instanceof MBeanNotificationInfo)
        {
            return Sorter.COLLATOR.getCollationKey(MFNotification.getType(((MBeanNotificationInfo)object).getNotifTypes()));
        }

        throw new IllegalArgumentException("Unexpected type: " + object.getClass().getName());
    }

    private void generateInterface()
    throws Exception
    {
        println(m_writer, "// This file has been generated for component class " + m_attributes.get("Classname") + " (" + m_attributes.get("ReleaseVersion") + ")");
        println(m_writer, "// Copyright (c) 2002-2007 Progress Software Corporation. All Rights Reserved.");
        println(m_writer);

        String interfacePackage = m_packageTokens[0] + '.' + m_packageTokens[1] + '.' + m_packageTokens[2] + ".mgmtapi.runtime";

        // package definition
        println(m_writer, "package " + interfacePackage + ";");
        println(m_writer);

        println(m_writer, "/**");
        println(m_writer, " * Remote interface to a runtime instance of " + (startsWithVowel(m_name) ? "an " : "a ") + m_name + ".");
        println(m_writer, " * <p>");
        println(m_writer, " * Management attributes and operations are documented in this interface.");
        println(m_writer, " * <p>");

        String proxyFactoryClass = m_packageTokens[2].toUpperCase() + "ProxyFactory";
        String proxyFactoryLink = "<a href=\"./" + proxyFactoryClass + ".html#create" + m_name + "Proxy(com.sonicsw.mf.jmx.client.JMSConnectorClient, javax.management.ObjectName)\">" + proxyFactoryClass + "</a>";
        println(m_writer, " * An implementation of this interface can be created using the " + proxyFactoryLink + " class.");
        String connectorLink = "<a href=\"../../../../../com/sonicsw/mf/jmx/client/JMSConnectorClient.html\">JMS/JMX Connector</a>";
        String objectNameLink = "<a href=\"../../../../../javax/management/ObjectName.html\">ObjectName</a>";
        println(m_writer, " * Each implementation instance is valid for the life of the provided " + connectorLink + " and MBean " + objectNameLink + ".");
        println(m_writer, " * The implementation is built using remote JMX MBeanServer calls.");
        println(m_writer, " * <p>");
        println(m_writer, " * For each JMX MBean attribute, this interface provides an associated <b>get</b> method (for readable attributes) and/or a <b>set</b> method (for writable attributes).");
        String mBeanOpInfoLink = "<a href=\"../../../../../javax/management/MBeanOperationInfo.html\">operation meta-data</a>";
        println(m_writer, " * <p>");
        println(m_writer, " * For each JMX MBean operation, this interface provides an associated method with a signature as described by the MBean's " + mBeanOpInfoLink + ".");
        if (m_jmx)
        {
            println(m_writer, " * Each JMX MBean operation has an associated impact:");
            println(m_writer, " * <ul>");
            println(m_writer, " * <li>" + convertImpactTypeToImpactString(MBeanOperationInfo.ACTION) + " - Indicates that the operation is a write-like, and would modify the MBean in some way, typically by writing some value or changing a configuration.");
            println(m_writer, " * <li>" + convertImpactTypeToImpactString(MBeanOperationInfo.INFO) + " - Indicates that the operation is read-like, it basically returns information.");
            println(m_writer, " * <li>" + convertImpactTypeToImpactString(MBeanOperationInfo.ACTION_INFO) + " - Indicates that the operation is both read-like and write-like.");
            println(m_writer, " * <li>" + convertImpactTypeToImpactString(MBeanOperationInfo.UNKNOWN) + " - Indicates that the operation has an \"unknown\" nature.");
            println(m_writer, " * </ul>");
        }
        println(m_writer, " * <p>");
        String proxyExceptionLink = "<a href=\"../../../../../com/sonicsw/mf/mgmtapi/runtime/ProxyRuntimeException.html\">ProxyRuntimeException</a>";
        println(m_writer, " * All exceptions emitted by implementations are wrapped as a " + proxyExceptionLink + ".");

        if (m_name.equals("AgentManager"))
        {
            println(m_writer, " * <p>");
            println(m_writer, " * <b>Note:</b> The Agent Manager receives and forwards all system state notifications published by all deployed components.");
            println(m_writer, " * Subscriptions to such notifications are allowed even though the Agent Manager does not describe those notifications in it's MBeanInfo.");
        }

        if (m_mBeanInfo.getNotifications().length > 0 || m_hasMetrics)
        {
            println(m_writer, " * <dl>");
            println(m_writer, " *   <dt><b>See Also:</b></dt>");
            println(m_writer, " *   <dd>");
            if (m_mBeanInfo.getNotifications().length > 0)
            {
                println(m_writer, " *     <a href=\"../../../../../../" + MONITORING_DOC_DIR + "/" + m_name + "Notifications.html\">" + m_name + " Notifications</a>,");
            }
            if (m_hasMetrics)
            {
                println(m_writer, " *     <a href=\"../../../../../../" + MONITORING_DOC_DIR + "/" + m_name + "Metrics.html\">" + m_name + " Metrics</a>");
            }
            println(m_writer, " *   </dd>");
            println(m_writer, " * </dl>");
        }

        println(m_writer, " */");
        println(m_writer, "public interface I" + m_name + "Proxy");
        println(m_writer, "{");

        generateInterfaceForConstants();
        generateInterfaceForAttributes();
        generateInterfaceForOperations();
        if (m_mBeanInfo.getNotifications().length > 0)
        {
            generateInterfaceForNotifications();
        }

        println(m_writer, "}");
    }

    private void generateInterfaceForConstants()
    {
        int indent = 1;

        if (m_name.equals("AgentManager"))
        {
            println(m_writer, indent, "//");
            println(m_writer, indent, "// Useful constants");
            println(m_writer, indent, "//");

            println(m_writer);
            println(m_writer, indent, "/**");
            println(m_writer, indent, " * The well known global component identity for this singleton component.");
            println(m_writer, indent, " */");
            println(m_writer, indent, "public static final String GLOBAL_ID = \"AGENT MANAGER\";");
            println(m_writer);
        }

        if (m_name.equals("DirectoryService"))
        {
            println(m_writer, indent, "//");
            println(m_writer, indent, "// Useful constants");
            println(m_writer, indent, "//");

            println(m_writer);
            println(m_writer, indent, "/**");
            println(m_writer, indent, " * The well known global component identity for this singleton component.");
            println(m_writer, indent, " */");
            println(m_writer, indent, "public static final String GLOBAL_ID = \"DIRECTORY SERVICE\";");
            println(m_writer);
        }

        if (m_name.equals("Agent"))
        {
            println(m_writer, indent, "//");
            println(m_writer, indent, "// Useful constants");
            println(m_writer, indent, "//");

            println(m_writer);
            println(m_writer, indent, "/**");
            println(m_writer, indent, " * The well known component identity for this component.");
            println(m_writer, indent, " */");
            println(m_writer, indent, "public static final String ID = \"AGENT\";");
            println(m_writer);
        }

        MBeanNotificationInfo[] infos = m_mBeanInfo.getNotifications();
        if (infos.length > 0)
        {
            infos = (MBeanNotificationInfo[])Sorter.sort(infos, this, infos.length);

            println(m_writer, indent, "//");
            println(m_writer, indent, "// Notification type constants");
            println(m_writer, indent, "//");

            println(m_writer);

            for (int i = 0;i < infos.length;i++)
            {
                println(m_writer, indent, "/**");
                println(m_writer, indent, " * Notification type constant for the notification " + MFNotification.getType(infos[i].getNotifTypes()) + '.');
                println(m_writer, indent, " * <p>");
                println(m_writer, indent, " * Description: " + infos[i].getDescription());
                println(m_writer, indent, " * <dl>");
                println(m_writer, indent, " *  <dt><b>See Also:</b></dt>");
                println(m_writer, indent, " *   <dd>");
                String notificationsHTML = "../../../../../../" + MONITORING_DOC_DIR + "/" + m_name + "Notifications.html";
                String notificationLink = "<a href=\"" + notificationsHTML + "#" + MFNotification.getType(infos[i].getNotifTypes()) + "\"><code>" + MFNotification.getType(infos[i].getNotifTypes()) + "</code></a>";
                println(m_writer, indent, " *    <li>" + notificationLink);
                println(m_writer, indent, " *   </dd>");
                println(m_writer, indent, " * </dl>");
                println(m_writer, indent, " */");

                StringBuffer sb = new StringBuffer("public static final String ");
                String[] nameComponents = infos[i].getNotifTypes();
                for (int j = 0; j < nameComponents.length; j++)
                {
                    sb.append(nameComponents[j].toUpperCase().replace('.', '_').replace(' ', '_')).append('_');
                }
                sb.append("NOTIFICATION_ID");
                sb.append(" = \"");
                sb.append(MFNotification.getType(infos[i].getNotifTypes()));
                sb.append("\";");
                println(m_writer, indent, sb.toString());
                println(m_writer);
            }
        }

        if (m_metricInfos != null)
        {
            println(m_writer, indent, "//");
            println(m_writer, indent, "// Metric indentity constants");
            println(m_writer, indent, "//");

            println(m_writer);
            for (int i = 0; i < m_metricInfos.length; i++)
            {
                println(m_writer, indent, "/**");
                if (m_metricInfos[i].isInstanceMetric())
                {
                    println(m_writer, indent, " * Parent metric identity constant for the instance metric " + m_metricInfos[i].getMetricIdentity().getName() + '.');
                }
                else
                {
                    println(m_writer, indent, " * Metric identity constant for the metric " + m_metricInfos[i].getMetricIdentity().getName() + '.');
                }
                println(m_writer, indent, " * <p>");
                println(m_writer, indent, " * Description: " + m_metricInfos[i].getDescription());
                println(m_writer, indent, " * <dl>");
                println(m_writer, indent, " *  <dt><b>See Also:</b></dt>");
                println(m_writer, indent, " *   <dd>");
                String metricsHTML = "../../../../../../" + MONITORING_DOC_DIR + "/" + m_name + "Metrics.html";
                String metricLink = "<a href=\"" + metricsHTML + "#" + m_metricInfos[i].getMetricIdentity().getName() + "\"><code>" + m_metricInfos[i].getMetricIdentity().getName() + "</code></a>";
                println(m_writer, indent, " *    <li>" + metricLink);
                println(m_writer, indent, " *   </dd>");
                println(m_writer, indent, " * </dl>");
                println(m_writer, indent, " */");

                StringBuffer sb = new StringBuffer("public static final com.sonicsw.mf.common.metrics.IMetricIdentity ");
                String[] nameComponents = m_metricInfos[i].getMetricIdentity().getNameComponents();
                for (int j = 0; j < nameComponents.length; j++)
                {
                    sb.append(nameComponents[j].toUpperCase()).append('_');
                }
                sb.append("METRIC_ID");
                sb.append(" = com.sonicsw.mf.common.metrics.MetricsFactory.createMetricIdentity(new String[]");
                println(m_writer, indent, sb.toString());
                println(m_writer, indent, "{");
                for (int j = 0; j < nameComponents.length; j++)
                {
                    println(m_writer, indent + 1, '"' + nameComponents[j] + '"' + (j < nameComponents.length - 1 ? "," : ""));
                }
                println(m_writer, indent, "});");
                println(m_writer);
            }
        }
    }

    private void generateInterfaceForAttributes()
    {
        MBeanAttributeInfo[] infos = m_mBeanInfo.getAttributes();
        infos = (MBeanAttributeInfo[])Sorter.sort(infos, this, infos.length);

        int indent = 1;

        println(m_writer, indent, "//");
        println(m_writer, indent, "// MBean attributes");
        println(m_writer, indent, "//");

        // bulk getter
        println(m_writer);
        println(m_writer, indent, "/**");
        println(m_writer, indent, " * Gets the values of several attributes of " + (startsWithVowel(m_name) ? "an " : "a ") + m_name + " component.");
        println(m_writer, indent, " * <p>");
        println(m_writer, indent, " * Getting multiple attribute values at once is typically more efficient than getting individual attribute values.");
        println(m_writer, indent, " * <p>");
        println(m_writer, indent, " * Readable attributes (and their datatypes) exposed by " + (startsWithVowel(m_name) ? "an " : "a ") + m_name + " are:");
        println(m_writer, indent, " * <ul>");
        for (int i = 0; i < infos.length; i++)
        {
            if (infos[i].isReadable())
            {
                String getterLink = "<a href=\"#get" + infos[i].getName() + "()\"><code>" + infos[i].getName() + "</code></a>";
                println(m_writer, indent, " * <li>" + getterLink + " (" + convertJavaTypeToJavaCode(infos[i].getType()) + ")");
            }
        }
        println(m_writer, indent, " * </ul>");
        println(m_writer, indent, " * @param attributeNames A list of the attribute names of attributes to be retrieved.");
        println(m_writer, indent, " *");
        println(m_writer, indent, " * @return The list of the retrieved attributes.");
        println(m_writer, indent, " *");
        println(m_writer, indent, " * @see #setAttributes(javax.management.AttributeList)");
        println(m_writer, indent, " */");
        println(m_writer, indent, "public javax.management.AttributeList getAttributes(String[] attributeNames);");

        // bulk setter
        println(m_writer);
        println(m_writer, indent, "/**");
        println(m_writer, indent, " * Sets the values of several attributes of " + (startsWithVowel(m_name) ? "an " : "a ") + m_name + " component.");
        println(m_writer, indent, " * <p>");
        println(m_writer, indent, " * Setting multiple attribute values at once is typically more efficient than setting individual attribute values.");
        println(m_writer, indent, " * <p>");
        println(m_writer, indent, " * Settable attributes (and their datatypes) exposed by " + (startsWithVowel(m_name) ? "an " : "a ") + m_name + " are:");
        println(m_writer, indent, " * <ul>");
        for (int i = 0; i < infos.length; i++)
        {
            if (infos[i].isWritable())
            {
                String setterLink = "<a href=\"#set" + infos[i].getName() + "(" + convertJavaTypeToJavaCode(infos[i].getType()) + ")\"><code>" + infos[i].getName() + "</code></a>";
                println(m_writer, indent, " * <li>" + setterLink + " (" + convertJavaTypeToJavaCode(infos[i].getType()) + ")");
            }
        }
        println(m_writer, indent, " * </ul>");
        println(m_writer, indent, " * @param attributes A list of attributes: The identification of the attributes to be set and the values they are to be set to.");
        println(m_writer, indent, " *");
        println(m_writer, indent, " * @return The list of attributes that were set, with their new values.");
        println(m_writer, indent, " *");
        println(m_writer, indent, " * @see #getAttributes(java.lang.String[])");
        println(m_writer, indent, " */");
        println(m_writer, indent, "public javax.management.AttributeList setAttributes(javax.management.AttributeList attributes);");

        for (int i = 0; i < infos.length; i++)
        {
            generateInterfaceForAttribute(infos[i]);
        }
    }

    private void generateInterfaceForAttribute(MBeanAttributeInfo info)
    {
        int indent = 1;

        if (info.isReadable())
        {
            println(m_writer);
            println(m_writer, indent, "/**");
            println(m_writer, indent, " * Gets the value of the " + info.getName() + " attribute.");
            if (info.getName().equals("TraceMask"))
            {
                generateJavadocForTraceMaskValues((String)m_attributes.get("TraceMaskValues"));
            }
            println(m_writer, indent, " *");
            println(m_writer, indent, " * @return " + info.getDescription());
            if (info.getName().equals("State"))
            {
                println(m_writer, indent, " *");
                println(m_writer, indent, " * @see #getStateString()");
                println(m_writer, indent, " * @see com.sonicsw.mf.common.runtime.IComponentState");
            }
            if (info.getName().equals("StateString"))
            {
                println(m_writer, indent, " *");
                println(m_writer, indent, " * @see #getState()");
                println(m_writer, indent, " * @see com.sonicsw.mf.common.runtime.IComponentState#STATE_TEXT");
            }
            if (info.getName().equals("TraceMaskValues"))
            {
                println(m_writer, indent, " *");
                println(m_writer, indent, " * @see #getTraceMask()");
                println(m_writer, indent, " * @see #setTraceMask(java.lang.Integer)");
            }
            if (info.isWritable())
            {
                println(m_writer, indent, " *");
                println(m_writer, indent, " * @see #set" + info.getName() + "(" + convertJavaTypeToJavaCode(info.getType()) + ")");
                // other specific "@see"
                if (info.getName().equals("TraceMask"))
                {
                    println(m_writer, indent, " * @see #getTraceMaskValues()");
                }
                if (info.getName().equals("LastErrorLevel"))
                {
                    println(m_writer, indent, " * @see #getLastErrorLevelString()");
                    println(m_writer, indent, " * @see #clearError()");
                    println(m_writer, indent, " * @see com.sonicsw.mf.common.runtime.Level");
                }
                if (info.getName().equals("LastErrorLevelString"))
                {
                    println(m_writer, indent, " * @see #getLastErrorLevel()");
                }
                if (info.getName().equals("State"))
                {
                    println(m_writer, indent, " * @see #getStateString()");
                }
                if (info.getName().equals("StateString"))
                {
                    println(m_writer, indent, " * @see #getState()");
                }
            }
            println(m_writer, indent, " */");
            println(m_writer, indent, "public " + convertJavaTypeToJavaCode(info.getType()) + " get" + info.getName() + "();");
        }

        if (info.isWritable())
        {
            println(m_writer);
            println(m_writer, indent, "/**");
            println(m_writer, indent, " * Sets the value of the " + info.getName() + " attribute.");
            if (info.getName().equals("TraceMask"))
            {
                generateJavadocForTraceMaskValues((String)m_attributes.get("TraceMaskValues"));
            }
            println(m_writer, indent, " *");
            println(m_writer, indent, " * @param value " + info.getDescription());
            if (info.isReadable())
            {
                println(m_writer, indent, " *");
                println(m_writer, indent, " * @see #get" + info.getName() + "()");
                // other specific "@see"
                if (info.getName().equals("TraceMask"))
                {
                    println(m_writer, indent, " * @see #getTraceMaskValues()");
                }
            }
            println(m_writer, indent, " */");
            println(m_writer, indent, "public void set" + info.getName() + "(" + convertJavaTypeToJavaCode(info.getType()) + " value);");
        }
    }

    private void generateInterfaceForOperations()
    {
        MBeanOperationInfo[] infos = m_mBeanInfo.getOperations();
        infos = (MBeanOperationInfo[])Sorter.sort(infos, this, infos.length);

        println(m_writer);
        println(m_writer, 1, "//");
        println(m_writer, 1, "// MBean operations");
        println(m_writer, 1, "//");

        for (int i = 0; i < infos.length; i++)
        {
            generateInterfaceForOperation(infos, i);
        }
    }

    private void generateInterfaceForOperation(MBeanOperationInfo[] infos, int index)
    {
        MBeanOperationInfo info = infos[index];

        int indent = 1;

        println(m_writer);
        println(m_writer, indent, "/**");
        println(m_writer, indent, " * Invokes the " + info.getName() + " operation.");
        println(m_writer, indent, " * <p>");
        println(m_writer, indent, " * " + info.getDescription());

        if (m_jmx)
        {
            println(m_writer, indent, " * <p>");
            println(m_writer, indent, " * Impact: " + convertImpactTypeToImpactString(info.getImpact()));
        }

        if (info.getName().equals("enableMetrics") || info.getName().equals("disableMetrics"))
        {
            println(m_writer, indent, " * <p>");
            println(m_writer, indent, " * Dynamic metrics are:");
            println(m_writer, indent, " * <ul>");
            String metricsHTML = "../../../../../../" + MONITORING_DOC_DIR + "/" + m_name + "Metrics.html";
            for (int i = 0; i < m_metricInfos.length; i++)
            {
                String metricLink = "<a href=\"" + metricsHTML + "#" + m_metricInfos[i].getMetricIdentity().getName() + "\"><code>" + m_metricInfos[i].getMetricIdentity().getName() + "</code></a>";
                println(m_writer, indent, " * <li>" + metricLink);
            }
            println(m_writer, indent, " * </ul>");
        }

        if (info.getName().equals("enableAlerts") || info.getName().equals("disableAlerts"))
        {
            println(m_writer, indent, " * <p>");
            println(m_writer, indent, " * Metrics that allow alert thresholds to be specified are:");
            println(m_writer, indent, " * <ul>");
            String metricsHTML = "../../../../../../" + MONITORING_DOC_DIR + "/" + m_name + "Metrics.html";
            for (int i = 0; i < m_metricInfos.length; i++)
            {
                if (m_metricInfos[i].supportsHighThresholdAlerts() || m_metricInfos[i].supportsLowThresholdAlerts())
                {
                    String metricLink = "<a href=\"" + metricsHTML + "#" + m_metricInfos[i].getMetricIdentity().getName() + "\"><code>" + m_metricInfos[i].getMetricIdentity().getName() + "</code></a>";
                    println(m_writer, indent, " * <li>" + metricLink);
                }
            }
            println(m_writer, indent, " * </ul>");
        }

        MBeanParameterInfo[] paramInfos = info.getSignature();
        if (paramInfos.length > 0)
        {
            println(m_writer, indent, " *");
            for (int i = 0; i < paramInfos.length; i++)
            {
                if (m_jmx)
                {
                    println(m_writer, indent, " * @param " + paramInfos[i].getName() + " " + paramInfos[i].getDescription());
                }
                else
                {
                    println(m_writer, indent, " * @param param" + i + " ???");
                }
            }
        }

        if (!info.getReturnType().equals("void") && !m_jmx)
        {
            println(m_writer, indent, " *");
            println(m_writer, indent, " * @return ???");
        }

        // general "@see"
        if (info.getName().indexOf("Metrics") > -1)
        {
            generateSeeAlso(infos, index, "Metrics");
        }
        if (info.getName().indexOf("Alerts") > -1)
        {
            generateSeeAlso(infos, index, "Alerts");
        }
        // specific "@see"
        if (info.getName().equals("start"))
        {
            println(m_writer, indent, " * @see #stop()");
        }
        if (info.getName().equals("stop"))
        {
            println(m_writer, indent, " * @see #start()");
        }
        if (info.getName().equals("clearError"))
        {
            println(m_writer, indent, " * @see #getLastErrorLevel()");
        }

        println(m_writer, indent, " */");
        println(m_writer, indent, "public " + convertJavaTypeToJavaCode(info.getReturnType()) + " " + info.getName() + "(" + generateParamsForOperation(info) + ");");
    }

    private void generateInterfaceForNotifications()
    {
        MBeanNotificationInfo[] infos = m_mBeanInfo.getNotifications();
        infos = (MBeanNotificationInfo[])Sorter.sort(infos, this, infos.length);

        int indent = 1;

        println(m_writer);
        println(m_writer, indent, "//");
        println(m_writer, indent, "// MBean notifications");
        println(m_writer, indent, "//");

        println(m_writer);
        println(m_writer, indent, "/**");
        println(m_writer, indent, " * Adds a notification listener.");
        println(m_writer, indent, " * <p>");
        println(m_writer, indent, " * " + (startsWithVowel(m_name) ? "An " : "A ") + m_name + " component publishes the following notifications:");
        println(m_writer, indent, " * <ul>");
        String notificationsHTML = "../../../../../../" + MONITORING_DOC_DIR + "/" + m_name + "Notifications.html";
        for (int i = 0; i < infos.length; i++)
        {
            String notificationLink = "<a href=\"" + notificationsHTML + "#" + MFNotification.getType(infos[i].getNotifTypes()) + "\"><code>" + MFNotification.getType(infos[i].getNotifTypes()) + "</code></a>";
            println(m_writer, indent, " * <li>" + notificationLink);
        }
        println(m_writer, indent, " * </ul>");
        println(m_writer, indent, " * @param listener The listener object which will handle the notifications emitted component.");
        println(m_writer, indent, " * @param filter The filter object. If filter is null, no filtering will be performed before handling notifications.");
        println(m_writer, indent, " * @param handback The context to be sent to the listener when a notification is emitted.");
        println(m_writer, indent, " *");
        println(m_writer, indent, " * @see #removeNotificationListener(javax.management.NotificationListener)");
        println(m_writer, indent, " */");
        println(m_writer, indent, "public void addNotificationListener(javax.management.NotificationListener listener, javax.management.NotificationFilter filter, Object handback);");
        println(m_writer);
        println(m_writer, indent, "/**");
        println(m_writer, indent, " * Removes a notification listener.");
        println(m_writer, indent, " * <p>");
        println(m_writer, indent, " * This method will remove all the information related to the given listener.");
        println(m_writer, indent, " * @param listener The listener object which will handle the notifications emitted by the registered MBean.");
        println(m_writer, indent, " *");
        println(m_writer, indent, " * @see #addNotificationListener(javax.management.NotificationListener, javax.management.NotificationFilter, Object)");
        println(m_writer, indent, " */");
        println(m_writer, indent, "public void removeNotificationListener(javax.management.NotificationListener listener);");
    }

    private void generateImpl()
    throws Exception
    {
        println(m_writer, "// This file has been generated for component class " + m_attributes.get("Classname") + " (" + m_attributes.get("ReleaseVersion") + ")");
        println(m_writer, "// Copyright (c) 2002-2007 Progress Software Corporation. All Rights Reserved.");
        println(m_writer);

        String clientPackage = m_packageTokens[0] + '.' + m_packageTokens[1] + '.' + m_packageTokens[2] + ".mgmtapi.runtime";

        // package definition
        println(m_writer, "package " + clientPackage + ".impl;");
        println(m_writer);

        println(m_writer, "public final class " + m_name + "Proxy");
        println(m_writer, "extends com.sonicsw.mf.mgmtapi.runtime.impl.AbstractProxy");
        println(m_writer, "implements " + clientPackage + ".I" + m_name + "Proxy");
        println(m_writer, "{");

        // constructors
        println(m_writer, 1, "//");
        println(m_writer, 1, "// Constructors");
        println(m_writer, 1, "//");
        println(m_writer);

        println(m_writer, 1, "public " + m_name + "Proxy(com.sonicsw.mf.jmx.client.JMSConnectorClient connector, javax.management.ObjectName objectName)");
        println(m_writer, 1, "{");
        println(m_writer, 2, "super(connector, objectName);");
        println(m_writer, 1, "}");
        println(m_writer);

        generateImplForAttributes();
        generateImplForOperations();
        if (m_mBeanInfo.getNotifications().length > 0)
        {
            generateImplForNotifications();
        }

        println(m_writer, "}");
    }

    private void generateImplForAttributes()
    {
        MBeanAttributeInfo[] infos = m_mBeanInfo.getAttributes();
        infos = (MBeanAttributeInfo[])Sorter.sort(infos, this, infos.length);

        println(m_writer, 1, "//");
        println(m_writer, 1, "// MBean attributes");
        println(m_writer, 1, "//");

        // bulk getter
        println(m_writer);
        println(m_writer, 1, "public javax.management.AttributeList getAttributes(String[] attributeNames)");
        println(m_writer, 1, "{");
        println(m_writer, 2, "try");
        println(m_writer, 2, "{");
        println(m_writer, 3, "return super.m_connector.getAttributes(super.m_objectName, attributeNames);");
        println(m_writer, 2, "}");
        println(m_writer, 2, "catch(Exception e) { throw new com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException(e); }");
        println(m_writer, 1, "}");

        // bulk setter
        println(m_writer);
        println(m_writer, 1, "public javax.management.AttributeList setAttributes(javax.management.AttributeList attributes)");
        println(m_writer, 1, "{");
        println(m_writer, 2, "try");
        println(m_writer, 2, "{");
        println(m_writer, 3, "return super.m_connector.setAttributes(super.m_objectName, attributes);");
        println(m_writer, 2, "}");
        println(m_writer, 2, "catch(Exception e) { throw new com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException(e); }");
        println(m_writer, 1, "}");

        for (int i = 0; i < infos.length; i++)
        {
            generateImplForAttribute(infos[i]);
        }
    }

    private void generateImplForAttribute(MBeanAttributeInfo info)
    {
        if (info.isReadable())
        {
            println(m_writer);
            println(m_writer, 1, "public " + convertJavaTypeToJavaCode(info.getType()) + " get" + info.getName() + "()");
            println(m_writer, 1, "{");
            println(m_writer, 2, "try");
            println(m_writer, 2, "{");
            println(m_writer, 3, "return (" + convertJavaTypeToJavaCode(info.getType()) + ")super.m_connector.getAttribute(super.m_objectName, \"" + info.getName() + "\");");
            println(m_writer, 2, "}");
            println(m_writer, 2, "catch(Exception e) { throw new com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException(e); }");
            println(m_writer, 1, "}");
        }

        if (info.isWritable())
        {
            println(m_writer);
            println(m_writer, 1, "public void set" + info.getName() + "(" + convertJavaTypeToJavaCode(info.getType()) + " value)");
            println(m_writer, 1, "{");
            println(m_writer, 2, "try");
            println(m_writer, 2, "{");
            println(m_writer, 3, "javax.management.Attribute attribute = new javax.management.Attribute(\"" + info.getName() + "\", value);");
            println(m_writer, 3, "super.m_connector.setAttribute(super.m_objectName, attribute);");
            println(m_writer, 2, "}");
            println(m_writer, 2, "catch(Exception e) { throw new com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException(e); }");
            println(m_writer, 1, "}");
        }
    }

    private void generateImplForOperations()
    {
        MBeanOperationInfo[] infos = m_mBeanInfo.getOperations();
        infos = (MBeanOperationInfo[])Sorter.sort(infos, this, infos.length);

        println(m_writer);
        println(m_writer, 1, "//");
        println(m_writer, 1, "// MBean operations");
        println(m_writer, 1, "//");

        for (int i = 0; i < infos.length; i++)
        {
            generateImplForOperation(infos[i]);
        }
    }

    private void generateImplForOperation(MBeanOperationInfo info)
    {
        int indent = 1;

        MBeanParameterInfo[] params = info.getSignature();

        println(m_writer);
        println(m_writer, indent, "public " + convertJavaTypeToJavaCode(info.getReturnType()) + " " + info.getName() + "(" + generateParamsForOperation(info) + ")");
        println(m_writer, 1, "{");
        println(m_writer, 2, "try");
        println(m_writer, 2, "{");
        println(m_writer, 3, "Object[] params = new Object[]");
        println(m_writer, 3, "{");

        for (int i = 0; i < params.length; i++)
        {
            println(m_writer, 4, (m_jmx ? params[i].getName() : ("param" + i)) + (i < params.length - 1 ? "," : ""));
        }

        println(m_writer, 3, "};");
        println(m_writer, 3, "String[] signature = new String[]");
        println(m_writer, 3, "{");

        for (int i = 0; i < params.length; i++)
        {
            println(m_writer, 4, '"' + params[i].getType() + '"' + (i < params.length - 1 ? "," : ""));
        }

        println(m_writer, 3, "};");

        String returnStmt = convertJavaTypeToJavaCode(info.getReturnType()).equals("void") ? "" : "return (" + convertJavaTypeToJavaCode(info.getReturnType()) + ")";

        println(m_writer, 3, returnStmt + "super.m_connector.invoke(super.m_objectName, \"" + info.getName() + "\", params, signature);");
        println(m_writer, 2, "}");
        println(m_writer, 2, "catch(Exception e) { throw new com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException(e); }");
        println(m_writer, 1, "}");
    }

    private void generateImplForNotifications()
    {
        int indent = 1;

        println(m_writer);
        println(m_writer, indent, "//");
        println(m_writer, indent, "// MBean notifications");
        println(m_writer, indent, "//");

        println(m_writer);
        println(m_writer, indent, "public void addNotificationListener(javax.management.NotificationListener listener, javax.management.NotificationFilter filter, Object handback)");
        println(m_writer, 1, "{");
        println(m_writer, 2, "try");
        println(m_writer, 2, "{");
        println(m_writer, 3, "super.m_connector.addNotificationListener(super.m_objectName, listener, filter, handback);");
        println(m_writer, 2, "}");
        println(m_writer, 2, "catch(Exception e) { throw new com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException(e); }");
        println(m_writer, 1, "}");
        println(m_writer);
        println(m_writer, indent, "public void removeNotificationListener(javax.management.NotificationListener listener)");
        println(m_writer, 1, "{");
        println(m_writer, 2, "try");
        println(m_writer, 2, "{");
        println(m_writer, 3, "super.m_connector.removeNotificationListener(super.m_objectName, listener);");
        println(m_writer, 2, "}");
        println(m_writer, 2, "catch(Exception e) { throw new com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException(e); }");
        println(m_writer, 1, "}");
    }

    private void generateMetricsMetadataSeed(JMSConnectorClient connector, String releaseVersion)
    throws Exception
    {
        String metricsVersion = "1";
        String componentObjectDomain = m_objectName.getDomain();
        String dsObjectDomain;
        String mfDomainName = componentObjectDomain.substring(0, componentObjectDomain.indexOf('.'));
        if (m_dsContainerName == null)
        {
            dsObjectDomain = componentObjectDomain;
        }
        else
        {
            dsObjectDomain = mfDomainName + "." + m_dsContainerName;
        }

        String componentName = m_objectName.getKeyProperty("ID");
        String configID = (String)m_attributes.get("ConfigID");
        ObjectName dsObjName = new ObjectName(dsObjectDomain, "ID", "DIRECTORY SERVICE");
        IDirElement element = (IDirElement)connector.invoke(dsObjName, "getElement", new Object[]
        { configID, new Boolean(false) }, GET_ELEMENT_SIGNATURE);
        String componentType = element.getIdentity().getType();
        String componentVersion = element.getIdentity().getReleaseVersion();

        if (componentType == null)
        {
            throw new IllegalArgumentException("Unknown component: " + componentName + (configID == null ? "" : " ConfigID: " + configID));
        }

        println(m_writer, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
        println(m_writer, "<Domain xmlns=\"http://www.sonicsw.com/mf\" xsi:schemaLocation=\"http://www.sonicsw.com/mf MFConfigurationElements.xsd\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" name=\"@DOMAIN_NAME@\">");
        println(m_writer, "  <Directory name=\"/\">");
        println(m_writer, "    <Directory name=\"/meta-inf\">");
        println(m_writer, "      <Directory name=\"/meta-inf/" + componentType + "\">");
        println(m_writer, "        <Directory name=\"/meta-inf/" + componentType + "/" + releaseVersion + "\">");
        println(m_writer, "          <ConfigElement>");
        println(m_writer, "            <ElementID name=\"/meta-inf/" + componentType + "/" + releaseVersion + "/metrics\" type=\"MF_METRICS_METADATA\" version=\"" + metricsVersion + "\" releaseVersion=\"" + releaseVersion + "\"/>");
        println(m_writer, "             <AttributeSet>");
        println(m_writer, "               <AttributeName name=\"METRICS\"/>");

        for (int i = 0; i < m_metricInfos.length; i++)
        {
            generateMetricsMetadata(m_metricInfos[i]);
        }

        println(m_writer, "             </AttributeSet>");
        println(m_writer, "          </ConfigElement>");
        println(m_writer, "        </Directory>");
        println(m_writer, "      </Directory>");
        println(m_writer, "    </Directory>");
        println(m_writer, "  </Directory>");
        println(m_writer, "</Domain>");

    }

    private void generateMetricsMetadata(IMetricInfo metricInfo)
    {
        println(m_writer, "                 <AttributeSet>");
        println(m_writer, "                   <AttributeName name=\"" + metricInfo.getMetricIdentity().getName() + "\"/>");
        println(m_writer, "                   <Attribute name=\"VALUE_TYPE\" type=\"integer\" value=\"" + metricInfo.getValueType() + "\" />");
        println(m_writer, "                   <Attribute name=\"DESCRIPTION\" type=\"string\" value=\"" + metricInfo.getDescription() + "\" />");
        println(m_writer, "                   <Attribute name=\"EXTENDED_DATA\" type=\"string\" value=\"" + (metricInfo.getExtendedData() == null ? "" : metricInfo.getExtendedData()) + "\" />");
        println(m_writer, "                   <Attribute name=\"INSTANCE_METRIC\" type=\"boolean\" value=\"" + metricInfo.isInstanceMetric() + "\" />");
        println(m_writer, "                   <Attribute name=\"DYNAMIC\" type=\"boolean\" value=\"" + metricInfo.isDynamic() + "\" />");
        println(m_writer, "                   <Attribute name=\"SUPPORTS_HIGH_THRESHOLDS\" type=\"boolean\" value=\"" + metricInfo.supportsHighThresholdAlerts() + "\" />");
        println(m_writer, "                   <Attribute name=\"SUPPORTS_LOW_THRESHOLDS\" type=\"boolean\" value=\"" + metricInfo.supportsLowThresholdAlerts() + "\" />");
        println(m_writer, "                   <Attribute name=\"UNITS\" type=\"string\" value=\"" + (metricInfo.getUnits() == null ? "" : metricInfo.getUnits()) + "\" />");
        println(m_writer, "                 </AttributeSet>");

    }
    
    private void generateMetricsDoc()
    throws Exception
    {
        println(m_writer, "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\" \"http://www.w3.org/TR/REC-html40/loose.dtd\">");
        println(m_writer, "<html>");
        println(m_writer, "  <!-- This file was initially generated for component class " + m_attributes.get("Classname") + " (" + m_attributes.get("ReleaseVersion") + ") -->");
        println(m_writer, "  <!-- Copyright (c) 2002-2007 Progress Software Corporation. All Rights Reserved. -->");
        println(m_writer);
        println(m_writer, "  <head>");
        println(m_writer, "    <title>" + m_name + " Metrics</title>");
        println(m_writer, "    <link rel=\"stylesheet\" type=\"text/css\" href=\"stylesheet.css\" title=\"Style\">");
        println(m_writer, "    <script> function asd() { parent.document.title=\"" + m_name + " Metrics\"; } </script>");
        println(m_writer, "  </head>");
        println(m_writer);
        println(m_writer, "  <body bgcolor=\"white\" onload=\"asd();\">");
        println(m_writer);
        println(m_writer, "    <a name=\"top\"/>");
        println(m_writer);

        generateMetricsDocNavbar();

        println(m_writer);
        println(m_writer, "    <hr>");
        println(m_writer);
        println(m_writer, "    <!-- ======== START OF METRICS ======== -->");
        println(m_writer, "    <h2>" + m_name + " Metrics</h2>");
        println(m_writer);
        println(m_writer, "    <hr>");
        println(m_writer);
        println(m_writer, "    <dl>");
        println(m_writer, "      <dt><b>Component:</b> " + m_name + " (classname=" +  m_attributes.get("Classname") + ", version=" + m_attributes.get("ReleaseVersion") + ") </dt>");
        println(m_writer, "    </dl>");
        println(m_writer, "    <p>This documentation describes the metrics and alert capabilities that are exposed by a " + m_name + " component running within the Sonic Management environment.</p>");
        println(m_writer, "    <p>A general description of metrics and metric alerts is provided in the <a href=\"./MetricsOverview.html\">overview</a>.</p>");
        println(m_writer, "    <dl>");
        println(m_writer, "      <dt><b>See Also:</b></dt>");
        println(m_writer, "      <dd>");
        println(m_writer, "        <a href=\"./" + m_name + "Notifications.html\"><code>" + m_name + " Notifications</code></a>,");
        println(m_writer, "        <a href=\"../mgmt_api/" + m_packageTokens[0] + '/' + m_packageTokens[1] + '/' + m_packageTokens[2] + "/mgmtapi/runtime/I" + m_name + "Proxy.html\"><code>" + m_name + " Operations/Attributes</code></a>");
        println(m_writer, "      </dd>");
        println(m_writer, "    </dl>");
        println(m_writer);
        println(m_writer, "    <hr>");
        println(m_writer);
        println(m_writer, "    <a name=\"summary\"/>");
        println(m_writer);
        println(m_writer, "    <!-- ========== START METRICS SUMMARY =========== -->");
        println(m_writer, "    <p>");
        println(m_writer, "      <table border=\"1\" cellpadding=\"3\" cellspacing=\"0\" width=\"100%\">");
        println(m_writer, "        <tbody>");
        println(m_writer, "          <tr bgcolor=\"#ccccff\" class=\"TableHeadingColor\">");
        println(m_writer, "            <td colspan=\"1\" rowspan=\"1\"><font size=\"+2\"><b>Metrics Summary</b></font><br></td>");
        println(m_writer, "          </tr>");

        for (int i = 0; i < m_metricInfos.length; i++)
        {
            generateMetricsDocForMetricSummary(m_metricInfos[i]);
        }

        println(m_writer, "        </tbody>");
        println(m_writer, "      </table>");
        println(m_writer, "    </p>");
        println(m_writer, "    <!-- ========== END METRICS SUMMARY =========== -->");
        println(m_writer);
        println(m_writer, "    <a name=\"detail\"/>");
        println(m_writer);
        println(m_writer, "    <!-- ============ START METRICS DETAIL ========== -->");
        println(m_writer, "    <p>");
        println(m_writer, "      <table border=\"1\" cellpadding=\"3\" cellspacing=\"0\" width=\"100%\">");
        println(m_writer, "        <tbody>");
        println(m_writer, "          <tr bgcolor=\"#ccccff\" class=\"TableHeadingColor\">");
        println(m_writer, "            <td colspan=\"1\"><font size=\"+2\"><b>Metrics Detail</b></font></td>");
        println(m_writer, "          </tr>");
        println(m_writer, "        </tbody>");
        println(m_writer, "      </table>");
        println(m_writer, "    </p>");

        for (int i = 0; i < m_metricInfos.length; i++)
        {
            if (i > 0)
            {
                println(m_writer);
                println(m_writer, "    <hr>");
            }
            generateMetricsDocForMetricDetail(m_metricInfos, i);
        }

        println(m_writer);
        println(m_writer, "    <!-- ============ END METRICS DETAIL ========== -->");
        println(m_writer);
        println(m_writer, "    <hr>");
        println(m_writer);

        generateMetricsDocNavbar();

        println(m_writer);
        println(m_writer, "    <hr>");
        println(m_writer);
        println(m_writer, "    <a name=\"bottom\"/><font size=\"-1\">Copyright &copy; 2002-2007 <a href=\"http://www.progress.com\" target=\"_top\">Progress Software Corporation</a>.  All Rights Reserved.</font>");
        println(m_writer);
        println(m_writer, "  </body>");
        println(m_writer);
        println(m_writer, "</html>");
    }

    private void generateMetricsDocNavbar()
    {
        println(m_writer, "    <!-- ========== START OF NAVBAR ========== -->");
        println(m_writer, "    <table border=\"0\" width=\"100%\" cellpadding=\"1\" cellspacing=\"0\">");
        println(m_writer, "      <tbody>");
        println(m_writer, "        <tr>");
        println(m_writer, "          <td colspan=\"3\" bgcolor=\"#eeeeff\" class=\"NavBarCell1\" width=\"80%\">");
        println(m_writer, "            <a name=\"navbar_top\"/>");
        println(m_writer, "            <table border=\"0\" cellpadding=\"0\" cellspacing=\"3\">");
        println(m_writer, "              <tbody>");
        println(m_writer, "                <tr align=\"center\" valign=\"top\">");
        println(m_writer, "                  <td bgcolor=\"#eeeeff\" class=\"NavBarCell1\">");
        println(m_writer, "                    <a href=\"./MetricsOverview.html\"><font class=\"NavBarFont1\"><b>Overview</b></font></a>&nbsp;");
        println(m_writer, "                  </td>");
        println(m_writer, "                  <td bgcolor=\"#eeeeff\" class=\"NavBarCell1\">");
        println(m_writer, "                    <a href=\"./index.html\"><font class=\"NavBarFont1\"><b>Index</b></font></a>&nbsp;");
        println(m_writer, "                  </td>");
        println(m_writer, "                  <td bgcolor=\"#eeeeff\" class=\"NavBarCell1\">");
        println(m_writer, "                    <a href=\"./Help.html\"><font class=\"NavBarFont1\"><b>Help</b></font></a>&nbsp;");
        println(m_writer, "                  </td>");
        println(m_writer, "                </tr>");
        println(m_writer, "              </tbody>");
        println(m_writer, "            </table>");
        println(m_writer, "          </td>");
        println(m_writer, "          <td align=\"right\" valign=\"top\" rowspan=\"3\" nowrap>");
        println(m_writer, "            <em>Sonic Management API</em>");
        println(m_writer, "          </td>");
        println(m_writer, "        </tr>");
        println(m_writer, "        <tr>");
        println(m_writer, "          <td valign=\"top\" class=\"NavBarCell3\"><font size=\"-2\">SUMMARY:&nbsp;<a href=\"../mgmt_api/" + m_packageTokens[0] + '/' + m_packageTokens[1] + '/' + m_packageTokens[2] + "/mgmtapi/runtime/I" + m_name + "Proxy.html#method_summary\">OPERATIONS/ATTRIBUTES</a>&nbsp;|&nbsp;<a href=\"./" + m_name + "Notifications.html#summary\">NOTIFICATIONS</a>&nbsp;|&nbsp;<a href=\"#summary\">METRICS</a></font></td>");
        println(m_writer, "          <td valign=\"top\" class=\"NavBarCell3\"><font size=\"-2\">DETAIL:&nbsp;<a href=\"../mgmt_api/" + m_packageTokens[0] + '/' + m_packageTokens[1] + '/' + m_packageTokens[2] + "/mgmtapi/runtime/I" + m_name + "Proxy.html#method_detail\">OPERATIONS/ATTRIBUTES</a>&nbsp;|&nbsp;<a href=\"./" + m_name + "Notifications.html#detail\">NOTIFICATIONS</a>&nbsp;|&nbsp;<a href=\"#detail\">METRICS</a></font></td>");
        println(m_writer, "        </tr>");
        println(m_writer, "      </tbody>");
        println(m_writer, "    </table>");
        println(m_writer, "    <!-- =========== END OF NAVBAR =========== -->");
    }

    private void generateMetricsDocForMetricSummary(IMetricInfo info)
    throws Exception
    {
        println(m_writer, "          <tr bgcolor=\"white\" class=\"TableRowColor\">");
        println(m_writer, "            <td rowspan=\"1\" colspan=\"1\">");
        println(m_writer, "              <code><b><a href=\"#" + info.getMetricIdentity() + "\">" + info.getMetricIdentity() + "</a></b></code>");
        println(m_writer, "              <br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;" + info.getDescription());
        println(m_writer, "            </td>");
        println(m_writer, "          </tr>");
    }

    private void generateMetricsDocForMetricDetail(IMetricInfo[] infos, int index)
    throws Exception
    {
        IMetricInfo info = infos[index];

        println(m_writer);
        println(m_writer, "    <a name=\"" + info.getMetricIdentity() + "\"/>");
        println(m_writer, "    <h3>" + info.getMetricIdentity() + "</h3>");
        println(m_writer, "    <dl>");
        println(m_writer, "      <dd>" + info.getDescription() + "<br>");

        // units comment
        String units = info.getUnits();
        if (units != null && units.length() > 0)
        {
            String unitsComment = "This metric's value is captured in terms of " + units + ".";
            println(m_writer, "      <br>" + unitsComment + "<br>");
        }

        // instance comment
        if (info.isInstanceMetric())
        {
            String instanceType = info.getMetricIdentity().getNameComponents()[0];
            String instanceComment = "This metric may be enabled for individual "
                                   + instanceType
                                   + "s or multiple "
                                   + instanceType
                                   + "s using <a href=\"./MetricsOverview.html#patterns\">patterns</a>.";
            println(m_writer, "      <br>" + instanceComment + "<br>");
        }

        // threshold alerts comment
        if (info.supportsHighThresholdAlerts() || info.supportsHighThresholdAlerts())
        {
            String[] nameComponents = info.getMetricIdentity().getNameComponents();
            String notificationName = (nameComponents[0].equals("system") ? "system" : "application") + ".alert";
            for (int i = 0; i < nameComponents.length; i++)
            {
                notificationName += '.' + nameComponents[i];
            }
            String alertsComment = "When alert thresholds are specified for this metric and are subsequently broken at runtime, a "
                                 + "<a href=\"./" + m_name + "Notifications.html#" + notificationName + "\">" + notificationName + "</a>"
                                 + " notification will be published.";
            println(m_writer, "      <br>" + alertsComment + "<br>");
        }

        println(m_writer, "      </dd>");
        println(m_writer, "    </dl>");
        println(m_writer, "    <blockquote>");
        println(m_writer, "      <dl>");

        // instance metric ?
        println(m_writer, "        <dt><b>Metric Type:</b></dt>");
        println(m_writer, "        <dd>" + (info.isInstanceMetric() ? "<a href=\"./MetricsOverview.html#instances\">Instance</a>" : "Normal") + "<br></dd>");

        // can be enabled/disabled at runtime
        println(m_writer, "        <dt><b>Dynamic:</b></dt>");
        String dynamic = (info.isDynamic() ? "Can be enabled/disabled" : "Cannot be disabled/reenabled") + " at runtime.";
        println(m_writer, "        <dd>" + dynamic + "<br></dd>");

        // are there thresholds ?
        String thresholdAlertSupport = info.supportsHighThresholdAlerts() ? "<a href=\"./MetricsOverview.html#thresholds\">High</a>" : null;
        String lowThresholdAlertSupport = info.supportsLowThresholdAlerts() ? "<a href=\"./MetricsOverview.html#thresholds\">Low</a>" : null;
        if (lowThresholdAlertSupport != null)
        {
            if (thresholdAlertSupport == null)
            {
                thresholdAlertSupport = lowThresholdAlertSupport;
            }
            else
            {
                thresholdAlertSupport += ", " + lowThresholdAlertSupport;
            }
        }
        if (thresholdAlertSupport == null)
        {
            thresholdAlertSupport = "None";
        }
        println(m_writer, "        <dt><b>Threshold Alert Support:</b></dt>");
        println(m_writer, "        <dd>" + thresholdAlertSupport + "<br></dd>");

        // any siblings to report as see also
        String[] nameComponents = info.getMetricIdentity().getNameComponents();
        String[] parentNameComponents = new String[nameComponents.length - 1];
        System.arraycopy(nameComponents, 0, parentNameComponents, 0, nameComponents.length - 1);
        IMetricIdentity parentId = MetricsFactory.createMetricIdentity(parentNameComponents);
        ArrayList siblings = new ArrayList();
        for (int i = 0; i < infos.length; i++)
        {
            if (i != index && infos[i].getMetricIdentity().isInstanceOf(parentId))
            {
                siblings.add(infos[i]);
            }
        }
        if (!siblings.isEmpty())
        {
            println(m_writer, "        <dt><b>See Also:</b></dt>");
            println(m_writer, "        <dd>");
            for (int i = 0; i < siblings.size(); i++)
            {
                IMetricInfo sibling = (IMetricInfo)siblings.get(i);
                String delimit = (siblings.size() > 0 && i < siblings.size() - 1) ? ", " : "";
                println(m_writer, "          <a href=\"#" + sibling.getMetricIdentity() + "\"><code>" + sibling.getMetricIdentity() + "</code></a>" + delimit);
            }
            println(m_writer, "        </dd>");
        }

        println(m_writer, "      </dl>");
        println(m_writer, "    </blockquote>");
    }

    private void generateNotificationsDoc()
    throws Exception
    {
        MBeanNotificationInfo[] notificationInfos = m_mBeanInfo.getNotifications();
        notificationInfos = (MBeanNotificationInfo[])Sorter.sort(notificationInfos, this, notificationInfos.length);

        println(m_writer, "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\" \"http://www.w3.org/TR/REC-html40/loose.dtd\">");
        println(m_writer, "<html>");
        println(m_writer, "  <!-- This file was initially generated for component class " + m_attributes.get("Classname") + " (" + m_attributes.get("ReleaseVersion") + ") -->");
        println(m_writer, "  <!-- Copyright (c) 2002-2007 Progress Software Corporation. All Rights Reserved. -->");
        println(m_writer);
        println(m_writer, "  <head>");
        println(m_writer, "    <title>" + m_name + " Notifications</title>");
        println(m_writer, "    <link rel=\"stylesheet\" type=\"text/css\" href=\"stylesheet.css\" title=\"Style\">");
        println(m_writer, "    <script> function asd() { parent.document.title=\"" + m_name + " Notifications\"; } </script>");
        println(m_writer, "  </head>");
        println(m_writer);
        println(m_writer, "  <body bgcolor=\"white\" onload=\"asd();\">");
        println(m_writer);
        println(m_writer, "    <a name=\"top\"/>");
        println(m_writer);

        generateNotificationsDocNavbar();

        println(m_writer);
        println(m_writer, "    <hr>");
        println(m_writer);
        println(m_writer, "    <!-- ======== START OF NOTIFICATIONS ======== -->");
        println(m_writer, "    <h2>" + m_name + " Notifications</h2>");
        println(m_writer);
        println(m_writer, "    <hr>");
        println(m_writer);
        println(m_writer, "    <dl>");
        println(m_writer, "      <dt><b>Component:</b> " + m_name + " (classname=" +  m_attributes.get("Classname") + ", version=" + m_attributes.get("ReleaseVersion") + ") </dt>");
        println(m_writer, "    </dl>");
        println(m_writer, "    <p>This documentation describes the notifications that are exposed by a " + m_name + " component running within the Sonic Management environment.</p>");
        println(m_writer, "    <p>A general description of notifications is provided in the <a href=\"./NotificationsOverview.html\">overview</a>.</p>");
        if (m_name.equals("AgentManager"))
        {
            println(m_writer, "    <p><b>Note:</b> The Agent Manager receives and forwards all system state notifications published by all deployed components. Subscriptions to such notifications are allowed even though those notifications are not described here.");
        }
        println(m_writer, "    <dl>");
        println(m_writer, "      <dt><b>See Also:</b></dt>");
        println(m_writer, "      <dd>");
        if (m_hasMetrics)
        {
            println(m_writer, "        <a href=\"./" + m_name + "Metrics.html\"><code>" + m_name + " Metrics</code></a>,");
        }
        println(m_writer, "        <a href=\"../mgmt_api/" + m_packageTokens[0] + '/' + m_packageTokens[1] + '/' + m_packageTokens[2] + "/mgmtapi/runtime/I" + m_name + "Proxy.html\"><code>" + m_name + " Operations/Attributes</code></a>");
        println(m_writer, "      </dd>");
        println(m_writer, "    </dl>");
        println(m_writer);
        println(m_writer, "    <hr>");
        println(m_writer);
        println(m_writer, "    <a name=\"summary\"/>");
        println(m_writer);
        println(m_writer, "    <!-- ========== START NOTIFICATIONS SUMMARY =========== -->");
        println(m_writer, "    <p>");
        println(m_writer, "      <table border=\"1\" cellpadding=\"3\" cellspacing=\"0\" width=\"100%\">");
        println(m_writer, "        <tbody>");
        println(m_writer, "          <tr bgcolor=\"#ccccff\" class=\"TableHeadingColor\">");
        println(m_writer, "            <td colspan=\"1\" rowspan=\"1\"><font size=\"+2\"><b>Notifications Summary</b></font><br></td>");
        println(m_writer, "          </tr>");

        for (int i = 0; i < notificationInfos.length; i++)
        {
            generateNotificationsDocForNotificationSummary(notificationInfos[i]);
        }

        println(m_writer, "        </tbody>");
        println(m_writer, "      </table>");
        println(m_writer, "    </p>");
        println(m_writer, "    <!-- ========== END NOTIFICATIONS SUMMARY =========== -->");
        println(m_writer);
        println(m_writer, "    <a name=\"detail\"/>");
        println(m_writer);
        println(m_writer, "    <!-- ============ START NOTIFICATIONS DETAIL ========== -->");
        println(m_writer, "    <p>");
        println(m_writer, "      <table border=\"1\" cellpadding=\"3\" cellspacing=\"0\" width=\"100%\">");
        println(m_writer, "        <tbody>");
        println(m_writer, "          <tr bgcolor=\"#ccccff\" class=\"TableHeadingColor\">");
        println(m_writer, "            <td colspan=\"1\"><font size=\"+2\"><b>Notifications Detail</b></font></td>");
        println(m_writer, "          </tr>");
        println(m_writer, "        </tbody>");
        println(m_writer, "      </table>");
        println(m_writer, "    </p>");

        for (int i = 0; i < notificationInfos.length; i++)
        {
            if (i > 0)
            {
                println(m_writer);
                println(m_writer, "    <hr>");
            }
            generateNotificationsDocForNotificationDetail(notificationInfos, i);
        }

        println(m_writer);
        println(m_writer, "    <!-- ============ END NOTIFICATIONS DETAIL ========== -->");
        println(m_writer);
        println(m_writer, "    <hr>");
        println(m_writer);

        generateNotificationsDocNavbar();

        println(m_writer);
        println(m_writer, "    <hr>");
        println(m_writer);
        println(m_writer, "    <a name=\"bottom\"/><font size=\"-1\">Copyright &copy; 2002-2007 <a href=\"http://www.progress.com\" target=\"_top\">Progress Software Corporation</a>.  All Rights Reserved.</font>");
        println(m_writer);
        println(m_writer, "  </body>");
        println(m_writer);
        println(m_writer, "</html>");
    }

    private void generateNotificationsDocNavbar()
    {
        println(m_writer, "    <!-- ========== START OF NAVBAR ========== -->");
        println(m_writer, "    <table border=\"0\" width=\"100%\" cellpadding=\"1\" cellspacing=\"0\">");
        println(m_writer, "      <tbody>");
        println(m_writer, "        <tr>");
        println(m_writer, "          <td colspan=\"3\" bgcolor=\"#eeeeff\" class=\"NavBarCell1\" width=\"80%\">");
        println(m_writer, "            <a name=\"navbar_top\"/>");
        println(m_writer, "            <table border=\"0\" cellpadding=\"0\" cellspacing=\"3\">");
        println(m_writer, "              <tbody>");
        println(m_writer, "                <tr align=\"center\" valign=\"top\">");
        println(m_writer, "                  <td bgcolor=\"#eeeeff\" class=\"NavBarCell1\">");
        println(m_writer, "                    <a href=\"./NotificationsOverview.html\"><font class=\"NavBarFont1\"><b>Overview</b></font></a>&nbsp;");
        println(m_writer, "                  </td>");
        println(m_writer, "                  <td bgcolor=\"#eeeeff\" class=\"NavBarCell1\">");
        println(m_writer, "                    <a href=\"./index.html\"><font class=\"NavBarFont1\"><b>Index</b></font></a>&nbsp;");
        println(m_writer, "                  </td>");
        println(m_writer, "                  <td bgcolor=\"#eeeeff\" class=\"NavBarCell1\">");
        println(m_writer, "                    <a href=\"./Help.html\"><font class=\"NavBarFont1\"><b>Help</b></font></a>&nbsp;");
        println(m_writer, "                  </td>");
        println(m_writer, "                </tr>");
        println(m_writer, "              </tbody>");
        println(m_writer, "            </table>");
        println(m_writer, "          </td>");
        println(m_writer, "          <td align=\"right\" valign=\"top\" rowspan=\"3\" nowrap>");
        println(m_writer, "            <em>Sonic Management API</em>");
        println(m_writer, "          </td>");
        println(m_writer, "        </tr>");
        println(m_writer, "        <tr>");
        println(m_writer, "          <td valign=\"top\" class=\"NavBarCell3\"><font size=\"-2\">SUMMARY:&nbsp;<a href=\"../mgmt_api/" + m_packageTokens[0] + '/' + m_packageTokens[1] + '/' + m_packageTokens[2] + "/mgmtapi/runtime/I" + m_name + "Proxy.html#method_summary\">OPERATIONS/ATTRIBUTES</a>&nbsp;|&nbsp;<a href=\"./" + m_name + "Notifications.html#summary\">NOTIFICATIONS</a>&nbsp;|&nbsp;<a href=\"#summary\">METRICS</a></font></td>");
        println(m_writer, "          <td valign=\"top\" class=\"NavBarCell3\"><font size=\"-2\">DETAIL:&nbsp;<a href=\"../mgmt_api/" + m_packageTokens[0] + '/' + m_packageTokens[1] + '/' + m_packageTokens[2] + "/mgmtapi/runtime/I" + m_name + "Proxy.html#method_detail\">OPERATIONS/ATTRIBUTES</a>&nbsp;|&nbsp;<a href=\"./" + m_name + "Notifications.html#detail\">NOTIFICATIONS</a>&nbsp;|&nbsp;<a href=\"#detail\">METRICS</a></font></td>");
        println(m_writer, "        </tr>");
        println(m_writer, "      </tbody>");
        println(m_writer, "    </table>");
        println(m_writer, "    <!-- =========== END OF NAVBAR =========== -->");
    }

    private void generateNotificationsDocForNotificationSummary(MBeanNotificationInfo info)
    throws Exception
    {
        println(m_writer, "          <tr bgcolor=\"white\" class=\"TableRowColor\">");
        println(m_writer, "            <td rowspan=\"1\" colspan=\"1\">");
        println(m_writer, "              <code><b><a href=\"#" + MFNotification.getType(info.getNotifTypes()) + "\">" + MFNotification.getType(info.getNotifTypes()) + "</a></b></code>");
        println(m_writer, "              <br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;" + info.getDescription());
        println(m_writer, "            </td>");
        println(m_writer, "          </tr>");
    }

    private void generateNotificationsDocForNotificationDetail(MBeanNotificationInfo[] infos, int index)
    throws Exception
    {
        MBeanNotificationInfo info = infos[index];

        println(m_writer);
        println(m_writer, "    <a name=\"" + MFNotification.getType(info.getNotifTypes()) + "\"/>");
        println(m_writer, "    <h3>" + MFNotification.getType(info.getNotifTypes()) + "</h3>");
        println(m_writer, "    <dl>");
        println(m_writer, "      <dd>" + info.getDescription() + "<br>");

        // alerts comment
        String[] nameComponents = info.getNotifTypes();
        if (nameComponents[1].equals(INotification.SUBCATEGORY_TEXT[INotification.ALERT_SUBCATEGORY]))
        {
            String metricName = nameComponents[2];
            if (nameComponents.length > 3)
            {
                for (int i = 3; i < nameComponents.length; i++)
                {
                    metricName += '.' + nameComponents[i];
                }
            }
            String metricsComment = "This notification is published when configured alert thresholds on the metric "
                                  + "<a href=\"./" + m_name + "Metrics.html#" + metricName + "\">" + metricName + "</a>"
                                  + " are broken.";
            println(m_writer, "      <br>" + metricsComment + "<br>");
        }

        println(m_writer, "      </dd>");
        println(m_writer, "    </dl>");
        println(m_writer, "    <blockquote>");
        println(m_writer, "      <dl>");

        // severity (must be manually updated)
        println(m_writer, "        <dt><b>Severity:</b></dt>");
        println(m_writer, "        <dd><a href=\"../mgmt_api/com/sonicsw/mf/common/runtime/Level.html#???\"><code>Level.???</code></a></dd>");

        // attributes (must be manually updated)
        println(m_writer, "        <dt><b>Attributes:</b></dt>");
        println(m_writer, "        <dd><code>???attribute name???</code>&nbsp;-&nbsp;???attribute description</dd>");
        println(m_writer, "        <dd><code>???attribute name???</code>&nbsp;-&nbsp;???attribute description</dd>");

        // any siblings to report as see also
        nameComponents = info.getNotifTypes();
        String[] parentNameComponents = new String[nameComponents.length - 1];
        System.arraycopy(nameComponents, 0, parentNameComponents, 0, nameComponents.length - 1);
        ArrayList siblings = new ArrayList();
        for (int i = 0; i < infos.length; i++)
        {
            if (i != index)
            {
                String[] siblingNameComponents = infos[i].getNotifTypes();
                for (int j = 0; j < parentNameComponents.length; j++)
                {
                    if (!parentNameComponents[j].equals(siblingNameComponents[j]))
                    {
                        break;
                    }
                    if (j == parentNameComponents.length - 1)
                    {
                        siblings.add(infos[i]);
                    }
                }
            }
        }
        if (!siblings.isEmpty())
        {
            println(m_writer, "        <dt><b>See Also:</b></dt>");
            println(m_writer, "        <dd>");
            for (int i = 0; i < siblings.size(); i++)
            {
                MBeanNotificationInfo sibling = (MBeanNotificationInfo)siblings.get(i);
                String delimit = (siblings.size() > 0 && i < siblings.size() - 1) ? ", " : "";
                println(m_writer, "          <a href=\"#" + MFNotification.getType(sibling.getNotifTypes()) + "\"><code>" + MFNotification.getType(sibling.getNotifTypes()) + "</code></a>" + delimit);
            }
            println(m_writer, "        </dd>");
        }

        println(m_writer, "      </dl>");
        println(m_writer, "    </blockquote>");
    }

    private String generateParamsForOperation(MBeanOperationInfo info)
    {
        MBeanParameterInfo[] params = info.getSignature();

        if (params.length == 0)
         {
            return ""; // no params
        }

        String paramsForOperation = "";
        for (int i = 0; i < params.length; i++)
        {
            paramsForOperation += (i > 0 ? ", " : "") + convertJavaTypeToJavaCode(params[i].getType()) + " ";
            paramsForOperation += (m_jmx ? params[i].getName() : ("param" + i));
        }

        return paramsForOperation;
    }

    private void generateSeeAlso(MBeanOperationInfo[] infos, int index, String relatedName)
    {
        int count = 0;
        for (int i = 0; i < infos.length; i++)
        {
            if (i != index && infos[i].getName().indexOf(relatedName) > -1)
            {
                if (count++ < 1)
                {
                    println(m_writer, 1, " *");
                }
                println(m_writer, 1, " * @see #" + infos[i].getName() + "(" + generateJavadocParamsForOperation(infos[i]) + ")");
            }
        }
    }

    private String generateJavadocParamsForOperation(MBeanOperationInfo info)
    {
        MBeanParameterInfo[] params = info.getSignature();

        if (params.length == 0)
         {
            return ""; // no params
        }

        String paramsForOperation = "";
        for (int i = 0; i < params.length; i++)
        {
            paramsForOperation += (i > 0 ? ", " : "") + convertJavaTypeToJavaCode(params[i].getType());
        }

        return paramsForOperation;
    }

    private void generateJavadocForTraceMaskValues(String traceMaskValues)
    {
        int indent = 1;

        println(m_writer, indent, " *");
        println(m_writer, indent, " * The trace mask is a bit mask composed from the following possible values:");
        println(m_writer, indent, " * <ul>");
        StringTokenizer st = new StringTokenizer(traceMaskValues, ",=");
        while (st.hasMoreTokens())
        {
            println(m_writer, indent, " * <li>" + st.nextToken() + " - " + st.nextToken());
        }
        println(m_writer, indent, " * </ul>");
    }

    private void println(PrintWriter writer)
    {
        writer.println();
    }

    private void println(PrintWriter writer, String text)
    {
        writer.println(text);
    }

    private void println(PrintWriter writer, int indent, String text)
    {
        for(int i = 0; i < indent; i++)
        {
            writer.print("    ");
        }

        writer.println(text);
    }

    // remove any class names from the import list that are implicitly included by
    // the given package
    private void fixupImportList(HashSet importList, String excludePackage)
    {
        String[] classnames = (String[])importList.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
        for (int i = 0; i < classnames.length; i++)
        {
            if (!classnames[i].startsWith(excludePackage))
            {
                continue;
            }
            if (!(classnames[i].length() > excludePackage.length() + 1))
            {
                continue;
            }
            if (classnames[i].substring(excludePackage.length() + 1).indexOf('.') == -1)
            {
                importList.remove(classnames[i]);
            }
        }
    }

    private String convertJavaTypeToJavaCode(String javaType)
    {
        if (javaType.equals(Void.class.getName()))
        {
            return "void";
        }

        return javaType.startsWith("[L") ? javaType.substring(2, javaType.length() - 1) + "[]" : javaType;
    }

    private boolean startsWithVowel(String name)
    {
        return ("AEIOUaeiou".indexOf(name.substring(0, 1)) > -1);
    }

    private String convertImpactTypeToImpactString(int impact)
    {
        switch (impact)
        {
            case MBeanOperationInfo.ACTION:
                return "ACTION";
            case MBeanOperationInfo.ACTION_INFO:
                return "ACTION/INFO";
            case MBeanOperationInfo.INFO:
                return "INFO";
            default:
                return "UNKNOWN";
        }
    }

    private void parseArgs(String[] args) throws Exception
    {
        int i = 0;

        while(i < args.length)
        {
            if(args[i].equals("-url"))
            {
                m_url = args[++i];
            }
            else if(args[i].equals("-user"))
            {
                m_user = args[++i];
            }
            else if(args[i].equals("-password"))
            {
                m_password = args[++i];
            }
            else if(args[i].equals("-objectname"))
            {
                m_component = args[++i];
            }
            else if(args[i].equals("-src"))
            {
                m_srcOutput = args[++i];
            }
            else if(args[i].equals("-doc"))
            {
                m_docOutput = args[++i];
            }
            else if(args[i].equals("-seed"))
            {
                m_metricsSeedDir = args[++i];
            }
            else if(args[i].equals("-name"))
            {
                m_name = args[++i];
            }
            else if(args[i].equals("-dscontainername"))
            {
                m_dsContainerName = args[++i];
            }
            else if(args[i].equals("-jmx"))
            {
                m_jmx = true;
            }
            else
            {
                throw new Exception("Unsupported argument " + args[i]);
            }

            i++;
        }

        if (m_name == null)
        {
            throw new Exception("Arg: -name is not specified");
        }

        if (m_component == null)
        {
            throw new Exception("Arg: -objectname is not specified");
        }

        // Allow generation of metrics seed file alone, so as not 
        // to overwrite manually modified src and doc output. 
        if (m_metricsSeedDir == null)
        {
            if (m_srcOutput == null)
            {
                throw new Exception("Arg: -src is not specified");
            }

            if (m_docOutput == null)
            {
                throw new Exception("Arg: -doc is not specified");
            }
        }
    }

    private static void printUsage()
    {
        System.out.println(GenerateProxy.class.getName());
        System.out.println("    [-url <DS URL>]                     (Default = " + DEFAULT_URL + ")");
        System.out.println("    [-user <DS user name>]              (Default = " + DEFAULT_USER + ")");
        System.out.println("    [-password <DS password>]           (Default = " + DEFAULT_PWD + ")");
        System.out.println("    [-objectname <JMX object name>]     (Default = None)");
        System.out.println("    [-dscontainername                   (Default = None, Use only if DS is in separate container from compoenent of interest)");
        System.out.println("    [-name <generated name>]            (Default = None)");
        System.out.println("    [-src <generated src root dir>]     (Default = None)");
        System.out.println("    [-doc <generated html dir>]         (Default = None)");
        System.out.println("    [-seed <generated metrics.seed dir>](Default = None, Not required)");
        System.out.println("    [-jmx]                              (Default = false)");
        System.out.println("          Use when component implements getMBeanInfo()");
    }

    public static void main(String[] args)
    {
        try
        {
            GenerateProxy bean = new GenerateProxy(args);
        }
        catch(Exception e)
        {
            e.printStackTrace();
        }
        System.exit(0);
    }
}
