package com.sonicsw.mf.runtime.tools;

import java.io.File;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.PrintWriter;
import java.io.Writer;
import java.text.CollationKey;
import java.util.ArrayList;
import java.util.StringTokenizer;

import com.sonicsw.mx.util.ICollationKeyProvider;
import com.sonicsw.mx.util.IEmptyArray;

// works on the assumption that packages are of the form com.<something>.<something>.mgmtapi.runtime
public final class GenerateProxyFactory
implements ICollationKeyProvider, FilenameFilter
{
    private String m_proxySrcDir  = null;

    private String[] m_packageTokens;
    private String m_clientPackage = "";
    private String[] m_proxyNames;

    private PrintWriter m_writer;

    public GenerateProxyFactory(String[] args)
    throws Exception
    {
        try
        {
            parseArgs(args);
        }
        catch(Exception e)
        {
            printUsage();

            throw e;
        }

        System.out.println("Generate Runtime Proxy Factory for....");
        System.out.println("    Proxy Src Dir    = " + m_proxySrcDir);

        String filename = null;
        Writer writer = null;

        System.out.println("Generating...");

        File dir = new File(m_proxySrcDir);
        if (!dir.isDirectory())
        {
            throw new IllegalArgumentException("Not a directory: " + m_proxySrcDir);
        }
        StringTokenizer st = new StringTokenizer(dir.getAbsolutePath(), "/\\");
        ArrayList packageTokens = new ArrayList();
        while (st.hasMoreTokens())
        {
            String token = st.nextToken();
            if (token.equals("com"))
            {
                packageTokens.add(token);
                break;
            }
        }
        while (st.hasMoreTokens())
        {
            packageTokens.add(st.nextToken());
        }
        m_packageTokens = (String[])packageTokens.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
        for (int i = 0; i < m_packageTokens.length; i++)
        {
            m_clientPackage += (i == 0 ? "" : ".") + m_packageTokens[i];
        }

        // get the proxy interface names
        m_proxyNames = dir.list((FilenameFilter)this);
        for (int i = 0; i < m_proxyNames.length; i++)
        {
            m_proxyNames[i] = m_proxyNames[i].substring(1, m_proxyNames[i].length() - "Proxy.java".length());
        }

        //
        // Proxy interface
        //
        filename = m_proxySrcDir + File.separator
                 + m_packageTokens[2].toUpperCase() + "ProxyFactory.java";
        System.out.println(" - " + filename + " ...");
        writer = new FileWriter(filename);
        m_writer = new PrintWriter(writer);
        generateFactory();
        writer.close();
        System.out.println("... Done");
    }

    @Override
    public CollationKey getCollationKey(Object object)
    {
        /*
        if (object instanceof IMetricInfo)
            return Sorter.COLLATOR.getCollationKey(((IMetricInfo)object).getMetricIdentity().getName());
            */

        throw new IllegalArgumentException("Unexpected type: " + object.getClass().getName());
    }

    @Override
    public boolean accept(File dir, String name)
    {
        if (!name.startsWith("I"))
        {
            return false;
        }
        if (!name.endsWith("Proxy.java"))
        {
            return false;
        }
        return true;
    }

    private void generateFactory()
    throws Exception
    {
        println(m_writer, "// This file has been generated for proxy interfaces found in the " + m_clientPackage + " package.");
        println(m_writer, "// Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.");
        println(m_writer);

        // package definition
        println(m_writer, "package " + m_clientPackage + ";");
        println(m_writer);

        println(m_writer, "/**");
        println(m_writer, " * Provides factory methods to create a runtime proxies that remotely interact with the management API of Sonic MBeans.");
        println(m_writer, " */");
        println(m_writer, "public final class " + m_packageTokens[2].toUpperCase() + "ProxyFactory");
        println(m_writer, "{");
        println(m_writer, 1, "private " + m_packageTokens[2].toUpperCase() + "ProxyFactory() { }");

        generateFactoryCreates();

        println(m_writer, "}");
    }

    private void generateFactoryCreates()
    {
        println(m_writer, 1, "//");
        println(m_writer, 1, "// Proxy factory methods");
        println(m_writer, 1, "//");

        for (int i = 0; i < m_proxyNames.length; i++)
        {
            generateFactoryCreateProxy(m_proxyNames[i]);
        }
    }

    private void generateFactoryCreateProxy(String proxyName)
    {
        println(m_writer);
        println(m_writer, 1, "/**");
        println(m_writer, 1, " * Creates a runtime proxy to remotely interact with the management API of " + (startsWithVowel(proxyName) ? "an " : "a ") + proxyName + " MBean.");
        println(m_writer, 1, " *");
        println(m_writer, 1, " * @param connector The JMS/JMX connector to be used for communications.");
        println(m_writer, 1, " * @param objectName The JMX ObjectName of the " + proxyName + " MBean.");
        println(m_writer, 1, " *");
        println(m_writer, 1, " * @return " + (startsWithVowel(proxyName) ? "An " : "A ") + proxyName + " proxy.");
        println(m_writer, 1, " */");
        println(m_writer, 1, "public static I" + proxyName + "Proxy create" + proxyName + "Proxy(com.sonicsw.mf.jmx.client.JMSConnectorClient connector, javax.management.ObjectName objectName)");;
        println(m_writer, 1, "{");
        if (proxyName.equals("Agent") || proxyName.equals("AgentManager") || proxyName.equals("DirectoryService"))
        {
            println(m_writer, 2, "com.sonicsw.mf.common.runtime.impl.CanonicalName canonicalName = new com.sonicsw.mf.common.runtime.impl.CanonicalName(objectName.getCanonicalName());");
            if (proxyName.equals("Agent"))
            {
                println(m_writer, 2, "if (!canonicalName.getComponentName().equals(IAgentProxy.ID))");
                println(m_writer, 3, "throw new IllegalArgumentException(\"ID must equal \\\"\" + IAgentProxy.ID + \"\\\"\");");
            }
            if (proxyName.equals("AgentManager"))
            {
                println(m_writer, 2, "if (!canonicalName.getComponentName().equals(IAgentManagerProxy.GLOBAL_ID))");
                println(m_writer, 3, "throw new IllegalArgumentException(\"ID must equal \\\"\" + IAgentManagerProxy.GLOBAL_ID + \"\\\"\");");
            }
            if (proxyName.equals("DirectoryService"))
            {
                println(m_writer, 2, "if (!canonicalName.getComponentName().equals(IDirectoryServiceProxy.GLOBAL_ID))");
                println(m_writer, 3, "throw new IllegalArgumentException(\"ID must equal \\\"\" + IDirectoryServiceProxy.GLOBAL_ID + \"\\\"\");");
            }
        }
        println(m_writer, 2, "return new " + m_clientPackage + ".impl." + proxyName + "Proxy(connector, objectName);");
        println(m_writer, 1, "}");
    }

    private void println(PrintWriter writer)
    {
        writer.println();
    }

    private void println(PrintWriter writer, String text)
    {
        writer.println(text);
    }

    private void println(PrintWriter writer, int indent, String text)
    {
        for(int i = 0; i < indent; i++)
        {
            writer.print("    ");
        }

        writer.println(text);
    }

    private boolean startsWithVowel(String name)
    {
        return ("AEIOUaeiou".indexOf(name.substring(0, 1)) > -1);
    }

    private void parseArgs(String[] args) throws Exception
    {
        int i = 0;

        while(i < args.length)
        {
            if(args[i].equals("-proxySrcDir"))
            {
                m_proxySrcDir = args[++i];
            }
            else
            {
                throw new Exception("Unsupported argument " + args[i]);
            }

            i++;
        }

        if (m_proxySrcDir == null)
        {
            throw new Exception("Arg: -proxySrcDir is not specified");
        }
    }

    private static void printUsage()
    {
        System.out.println(GenerateProxyFactory.class.getName());
        System.out.println("    [-proxySrcDir <proxy src dir>]        (Default = None)");
    }

    public static void main(String[] args)
    {
        try
        {
            GenerateProxyFactory bean = new GenerateProxyFactory(args);
        }
        catch(Exception e)
        {
            e.printStackTrace();
        }
        System.exit(0);
    }
}
