/*
 * Copyright (c) 1999-2000 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Sonic Software Corporation. ("Confidential Information"). You
 * shall not disclose such Confidential Information and shall use it
 * only in accordance with the terms of the license agreement you
 * entered into with Sonic Software.
 *
 * SONIC SOFTWARE MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY
 * OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. PROGRESS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 *
 */
package com.sonicsw.mx.config;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.StringBufferInputStream;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.management.ObjectName;

import com.sonicsw.mx.config.impl.ConfigBeanDocument;
import com.sonicsw.mx.config.impl.ConfigBeanDocument.ConfigBeanIdentity;
import com.sonicsw.mx.config.impl.ConfigBeanFile;
import com.sonicsw.mx.config.impl.ConfigServer;
import com.sonicsw.mx.config.impl.ConfigTypeDocument;
import com.sonicsw.mx.config.impl.ConfigTypeImpl;
import com.sonicsw.mx.config.impl.InitialValuesImpl;
import com.sonicsw.mx.config.impl.Util;
import com.sonicsw.mx.config.util.ConfigHelper;
import com.sonicsw.mx.config.util.SonicFSFileSystem;

import com.sonicsw.mf.comm.jms.ConnectorClient;
import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.config.query.FromElementType;
import com.sonicsw.mf.common.config.query.Query;
import com.sonicsw.mf.common.config.query.Select;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDeltaDirElement;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.framework.agent.ContainerUtil;
import com.sonicsw.mf.framework.directory.DirectoryServiceFactory;
import com.sonicsw.mf.framework.directory.IDirectoryService;
import com.sonicsw.mf.framework.directory.IFSStorage;
import com.sonicsw.mf.jmx.client.DirectoryServiceProxy;
import com.sonicsw.mf.jmx.client.IRemoteMBeanServer;
import com.sonicsw.mf.jmx.client.JMSConnectorAddress;
import com.sonicsw.mf.jmx.client.JMSConnectorClient;
import com.sonicsw.mf.mgmtapi.config.constants.IBackupDirectoryServiceConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IDirectoryServiceConstants;


public class ConfigServerUtility
{

    public static final String EXCLUDE_CONTAINERS_PROP = "exclude_containers";
    public static final String NEW_SEARCH_PATH_PROP = "new_search_path";
    public static final String CONTAINER_LIST_PROP = "container_list";

    public static final String TYPE                  = "TYPE";
    public static final String CONFIG_VERSION        = "CONFIG_VERSION";
    public static final String PRODUCT_VERSION       = "PRODUCT_VERSION";
    public static final String TEMPLATE_TYPE         = "TEMPLATE_TYPE";
    public static final String EXCLUDE_TEMPLATE_TYPE = "EXCLUDE_TEMPLATE_TYPE";

    public static final String DEFAULT_SUFFIX           = "_Default";
    public static final String DOMAIN_DESCRIPTOR_SUFFIX = "_MFDomainDescriptor";
    public static final String POLICY_DESCRIPTOR_SUFFIX = "_MQPolicyDescriptor";

    public static final String TEMPLATE_TYPE_PROTOTYPE = "prototype";
    public static final String TEMPLATE_TYPE_INSTANCE  = "instance";

    protected IConfigServer               m_configServer = null;
    protected IRemoteMBeanServer          m_server       = null;
    protected IDirectoryFileSystemService m_dfs          = null;
    protected boolean                     m_isLocalDS    = false;
    protected boolean                     m_isOwnedDS    = false;

    protected String  m_user                = null;
    protected long    m_connectTimeout      = -1;
    protected long    m_requestTimeout      = -1;
    protected boolean m_loadBalancing       = true;
    protected boolean m_enableCompression	= false;
    protected boolean m_useDRA              = false;
    protected String  m_managementNode      = null;
    protected String  m_requiredVersion     = null;

    private int m_socketConnectTimeout;

    public ConfigServerUtility()
    {
    }

    //-------------------------------------------------------------------------
    // Methods to set connection parameters. Called prior to a connect()
    //

    /**
     * Gets the timeout for connection attempts to the domain. A value of -1
     * indicates that the underlying default value will be used.
     *
     * @return Current connect timeout in seconds
     */
    public long getConnectTimeout() { return m_connectTimeout; }

    /**
     * Sets the timeout for connection attempts to the domain. A value of -1
     * indicates that the underlying default value will be user.
     *
     * @param seconds Timeout value in seconds.
     */
    public void setConnectTimeout(long seconds) { m_connectTimeout = seconds; }

    /**
     * Gets the timeout for responses from remote requests. A value of
     * -1 indicates that the default value is to be used.
     *
     * @return Current request timeout value in seconds
     */
    public long getRequestTimeout() { return m_requestTimeout; }

    /**
     * Sets the timeout for responses from remote requests in SECONDS. A value
     * of -1 indicates that the default value will be used.
     *
     * NOTE: This method must be called before connect is called otherwise the
     *       new timeout will never take effect.
     *
     * @param seconds Timeout value in seconds
     */
    public void setRequestTimeout(long seconds) { m_requestTimeout = seconds; }

    /**
     * Is Load Balancing enabled on the underlying connection to the domain.
     * Default value is true.
     *
     * @return True if load balancing is enabled.
     */
    public boolean isLoadBalancing() { return m_loadBalancing; }

    /**
     * Enable / disable Load Balancing on the underlying connection to the domain.
     *
     * @param loadBalancing True to enable load balancing.
     */
    public void setLoadBalancing(boolean loadBalancing) { m_loadBalancing = loadBalancing; }
    
    /**
     * Is Compression enabled on the underlying connection to the domain.
     * Default value is false.
     *
     * @return True if Compression is enabled.
     */
    public boolean isEnableCompression() { return m_enableCompression; }

    /**
     * Enable / disable Compression on the underlying connection to the domain.
     *
     * @param enableCompression True to enable Compression.
     */
    public void setEnableCompression(boolean enableCompression) { m_enableCompression = enableCompression; }

    /**
     * Is use of DRA enabled for connection to the domain. If isUseDRA() == true
     * and a value node is specified in getManagementNode() then this routing
     * node will be used for connecting to the domain.
     *
     * @return True if the use of DRA is enabled.
     */
    public boolean isUseDRA() { return m_useDRA; }

    /**
     * Enable / disable the use of DRA for connection to the domain.
     *
     * @param useDRA True to enable DRA.
     */
    public void setUseDRA(boolean useDRA) { m_useDRA = useDRA; }

    /**
     * Get the routing node through which we will communicate with the domain.
     * Default value == null indicating that the local routing node will be
     * used.
     *
     * @return The name of the routing node.
     */
    public String getManagementNode() { return m_managementNode; }

    /**
     * Set the routing node through which we will communicate with the domain.
     * This routing node will only be used if isUseDRA() == true.
     *
     * @param managementNode The name of the routing node.
     */
    public void setManagementNode(String managementNode) { m_managementNode = managementNode; }

    /**
     * Sets the release version of the DS that we require for connection. By default
     * m_requiredVersion == null which means no version checking will be performed
     */
    public void setRequiredVersion(String version) { m_requiredVersion = version; }

    /**
     * Get the required version of DS
     */
    public String getRequiredVersion() { return m_requiredVersion; }

    //-------------------------------------------------------------------------

    /**
     * Is the DirectoryService running local or remote.
     * @return Returns false if the directory service is a DirectoryServiceProxy
     * returns true if the directory service is a local implementations
     */
    public boolean isLocal()
    {
        return m_isLocalDS;
    }

    /**
     * Connect to the ConfigServer and Directory Service using the specified
     * domain, url, user and password. If the specified URL is a ds.xml
     * reference then we will connect in local mode. When connecting in local mode,
     * if the password is not null then we will try to decrypt the DS boot file.
     * If the url field is not the path to a ds.xml file, we will create
     * a JMSConnectorClient.
     * In all cases applications should call disconnect() when done with the
     * ConfigServerUtility.
     * @param domain
     * @param url
     * @param user
     * @param password
     * @param forUpdate
     * @throws ConfigServiceException
     */
    public final void connect(String domain, String url, String user, String password, boolean forUpdate)
    throws ConfigServiceException
    {
        m_user = user;

        try
        {
            if(url.endsWith(".xml"))
            {
                connect(createDSConnection(url, password), forUpdate);
            }
            else
            {
                m_server = createMBeanServerConnection(url, user, password);

                connect(createDSConnection(m_server, domain), forUpdate);
            }
            m_isOwnedDS = true;
        }
        catch(ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            ConfigServiceException cse = new ConfigServiceException("csu-connect-failed-error", e);
            cse.setErrorArgs(new String[] { e.getMessage() } );
            throw cse;
        }
    }

    /**
     * Connect the ConfigServerUtility to the specified DS. This method will
     * create an instance of a ConfigServer using the specified
     * IDirectoryFileSystemServer
     * In all cases applications should call disconnect() when done with the
     * ConfigServerUtility.
     * @param ds
     * @param forUpdate
     * @throws ConfigServiceException
     */
    public void connect(IDirectoryFileSystemService ds, boolean forUpdate)
    throws ConfigServiceException
    {
        try
        {
            pingDirectoryService(ds);

            checkForRequiredVersion(ds);

            m_dfs          = ds;
            m_configServer = createConfigServer(ds, forUpdate);
            m_isLocalDS = !(ds instanceof com.sonicsw.mf.jmx.client.DirectoryServiceProxy);
            m_isOwnedDS = false;

            ((ConfigServer)m_configServer).setFilterDefaults(true);
        }
        catch(ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            ConfigServiceException cse = new ConfigServiceException("csu-connect-failed-error", e);
            cse.setErrorArgs(new String[] { e.getMessage() } );
            throw cse;
        }
    }

    private void checkForRequiredVersion(IDirectoryFileSystemService ds)
    throws ConfigServiceException
    {
        if(m_requiredVersion == null)
        {
            return;
        }

        String version = null;

        try
        {
            version = ((IDirectoryAdminService)ds).getDirectoryServiceReleaseVersion();
        }
        catch(Exception e)
        {
        }

        if(version == null || !(version.startsWith(m_requiredVersion)))
        {
            throw new ConfigServiceException("csu-invalid-ds-version");
        }
    }

    /**
     * Create the ConfigServer. The TxnConfigServerUtility overrides this
     * method to create a transactioned config server rather than the standard
     * non-transactioned one created here.
     * @param ds
     * @param forUpdate
     * @return
     * @throws ConfigServiceException
     */
    protected IConfigServer createConfigServer(IDirectoryFileSystemService ds, boolean forUpdate)
    throws ConfigServiceException
    {
        return ConfigServerFactory.connectConfigServer(ds, forUpdate);
    }

    /**
     * Create a JMSConnectorClient with the specified parameters. Additional
     * attributes on the Connector e.g. loadBalancing are set prior to this
     * call.
     * @param url
     * @param user
     * @param password
     * @return
     * @throws java.lang.Exception
     */
    public IRemoteMBeanServer createMBeanServerConnection(String url, String user, String password)
    throws Exception
    {
        JMSConnectorClient server = new JMSConnectorClient();

        if (getRequestTimeout() != -1)
        {
            server.setRequestTimeout(getRequestTimeout() * 1000);
        }


        if(getSocketConnectTimeout() != -1){
            long socketConnectTimeout = getSocketConnectTimeout() * 1000;
            server.setSocketConnectTimeout(socketConnectTimeout);
        }


        long connectTimeout = ConnectorClient.CONNECT_TIMEOUT_DEFAULT;
        long initialConnectTimeout = connectTimeout;

        if (getConnectTimeout() != -1)
        {
            connectTimeout = getConnectTimeout() * 1000;
            server.setConnectTimeout(connectTimeout);
        }

        server.setFailWhenDisconnected(true);

        Hashtable env = new Hashtable();

        env.put("ConnectionURLs", url);

        if (user != null)
        {
            env.put("DefaultUser", user);
        }

        if (password != null)
        {
            env.put("DefaultPassword", password);
        }

        env.put("LoadBalancingBoolean", new Boolean(isLoadBalancing()));
        env.put("EnableCompressionBoolean", new Boolean(isEnableCompression()));
        
        JMSConnectorAddress address = new JMSConnectorAddress(env);

        String managementNode = getManagementNode();

        if (isUseDRA() && (managementNode != null) && (managementNode.length() > 0))
        {
            address.setManagementNode(managementNode);
        }

        server.connect(address, initialConnectTimeout);

        return server;
    }

    /**
     * Create a DirectoryServiceProxy using the specified IRemoteMBeanServer.
     * @param server
     * @param domain
     * @return
     * @throws java.lang.Exception
     */
    private IDirectoryFileSystemService createDSConnection(IRemoteMBeanServer server, String domain)
    throws Exception
    {
        m_isLocalDS = false;

        return new DirectoryServiceProxy(server, new ObjectName(domain + ".DIRECTORY SERVICE:ID=DIRECTORY SERVICE"));
    }

    /**
     * Create a local DirectoryServer using information contained in the ds.xml
     * file specified.
     * @param filename path to ds.xml
     * @return a local implementation
     * @throws java.lang.Exception
     */
    private IDirectoryFileSystemService createDSConnection(String filename, String decryptPassword) throws Exception
    {
        if (decryptPassword != null && decryptPassword.length() == 0)
        {
            decryptPassword = null;
        }

        IDirElement   dsConfig = ContainerUtil.importConfiguration(new File(filename), decryptPassword, IDirectoryServiceConstants.DS_TYPE);//  ElementFactory.importFromXML(readFile(filename), null);

        if (dsConfig == null)
        {
            dsConfig = ContainerUtil.importConfiguration(new File(filename), decryptPassword, IBackupDirectoryServiceConstants.DS_TYPE);
        }

        IAttributeSet dsAttributes = dsConfig.getAttributes();

        String hostDir    = (String) dsAttributes.getAttribute(IDirectoryServiceConstants.HOST_DIRECTORY_ATTR);
        String domainName = (String) dsAttributes.getAttribute(IDirectoryServiceConstants.DOMAIN_NAME_ATTR);
        Object tmp        = dsAttributes.getAttribute(IDirectoryServiceConstants.FILE_SYSTEM_STORAGE_ATTR);

        if (domainName == null)
        {
            domainName = IDirectoryServiceConstants.DOMAIN_NAME_DEFAULT;
        }
        if (tmp == null)
        {
            throw new Exception("Bad Directory Service Configuration - must contain DOMAIN_NAME and FILE_SYSTEM_STORAGE.");
        }

        if (!(tmp instanceof IAttributeSet))
        {
            throw new Exception("Bad Directory Service Configuration - FILE_SYSTEM_STORAGE must be an attribute set.");
        }

        IAttributeSet fsStorage = (IAttributeSet)tmp;

        String hostDirDepricated = (String) fsStorage.getAttribute(IDirectoryServiceConstants.HOST_DIRECTORY_ATTR);
        String password = (String) fsStorage.getAttribute(IDirectoryServiceConstants.PASSWORD_ATTR);

        // Creates the directory service object
        Hashtable directoryEnv = new Hashtable();
        directoryEnv.put(IDirectoryService.STORAGE_TYPE_ATTRIBUTE, IDirectoryService.PSE_STORAGE);
        if (hostDirDepricated != null)
        {
            hostDirDepricated = convertToAbsolute(hostDirDepricated, filename);
            directoryEnv.put(IFSStorage.FS_HOST_DIRECTORY_ATTRIBUTE, hostDirDepricated);
        }
        if (hostDir != null)
        {
            hostDir = convertToAbsolute(hostDir, filename);
            directoryEnv.put(IFSStorage.HOST_DIRECTORY_ATTRIBUTE, hostDir);
        }
        if ((password != null) && (password.length() != 0))
        {
            directoryEnv.put(IFSStorage.PASSWORD_ATTRIBUTE, password);
        }

        m_isLocalDS = true;

        return new DirectoryServiceFactory(directoryEnv).createDirectoryService(domainName);
    }

    /**
     * Converts a file name to an absolute file name based on a reference file.
     * The parent directory of the reference is used to resolve relative paths.
     *
     * @param hostDir
     *            the path retrieved from parsing the ds.xml
     * @param referenceFile
     *            typically the path to the ds.xml
     * @return an absolute path
     * @throws IOException
     */
    private static String convertToAbsolute(final String hostDir,
            final String referenceFile) throws IOException {
        final File hostDirFile = new File(hostDir);
        if (!hostDirFile.isAbsolute()) {
            final File ref = new File(referenceFile);
            return new File(ref.getAbsoluteFile().getParentFile(), hostDir)
                    .getCanonicalPath();
        }
        return hostDir;
    }

    /**
     * Check that the DirectoryService is responding. Returns the current DS
     * version number.
     * @param ds
     * @return
     * @throws ConfigServiceException
     */
    private int pingDirectoryService(IDirectoryFileSystemService ds)
    throws ConfigServiceException
    {
        try
        {
            return ds.getDirectoryServiceVersion();
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("csu-ping-failed", new Object[] { ds.getDomain() }, e);
        }
    }

    /**
     * Disconnect the ConfigServerUtility from the ConfigServer, DS and any
     * JMSConnectorClient
     * @throws ConfigServiceException
     */
    public final void disconnect()
    throws ConfigServiceException
    {
        if(m_dfs != null)
        {
            try
            {
                if (m_isLocalDS && m_isOwnedDS)
                {
                    ((IDirectoryService) m_dfs).close();
                }
            }
            catch(Exception e)
            {
                throw new ConfigServiceException("csu-disconnect-failed", e);
            }
        }

        m_dfs = null;

        if (m_configServer != null)
        {
            m_configServer.close();

            m_configServer = null;
        }

        if (m_server != null)
        {
            m_server.disconnect();
            m_server = null;
        }
    }

    /**
     * Return a reference to the wrapped IConfigServer
     * @return
     */
    public IConfigServer getConfigServer()
    {
        return m_configServer;
    }

    /**
     * Return a reference to the internally created IRemoteMBeanServer.
     * @return
     */
    public IRemoteMBeanServer getMBeanServer()
    {
        return m_server;
    }

    /**
     * Return the domain name for the underlying DS
     * @return
     * @throws com.sonicsw.mf.common.security.ManagePermissionDeniedException
     */
    public String getDomain()
    {
        return m_dfs.getDomain();
    }

    /**
     * Return a reference to the wrapped IDirectoryFileSystemServer
     * @return
     */
    public IDirectoryFileSystemService getDirectoryService()
    {
        return m_dfs;
    }

    /**
     * Iterate through all configurations in the DS looking for broken references
     * and attempt to fix them.
     * We have hardcoded exclusions to skip over certain directories
     * @throws ConfigServiceException
     */
    public final void repairDanglingBeanReferences()
    throws ConfigServiceException
    {
        checkConnected();

        try
        {
            String[] exclusions =
            {
                "/mx",
                "/_MF", // Will get _MFUsers, _MFGroups, _MFRuntime, _MFSystem etc
                "/_MQQoPs",
                "/_MQACLs",
            };
            m_dfs.repairReferences(exclusions);
        }

        catch(Exception e)
        {
            throw new ConfigServiceException("csu-repair-references-failed", e);
        }
    }

    /**
     * Import the xml file specified and create a IConfigBean from it.
     * @param beanURL
     * @param overwrite
     * @return
     * @throws ConfigServiceException
     */
    public IConfigBean importBean(String beanURL, boolean overwrite)
    throws ConfigServiceException
    {
        checkConnected();

        try
        {
        	String url = Util.name2Url(beanURL);

            ConfigBeanDocument.ConfigBeanIdentity id = ConfigBeanDocument.parseIdentity(url);

            IConfigBean bean = _importBean(url, id, false, overwrite);

            commitTransaction();

            return bean;
        }
        catch(ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("csu-import-bean-failed", new Object[] { beanURL } , e);
        }
    }

    /**
     * Use the specified xml string as the definition for a IConfigBean. The
     * xml is parsed and a bean created (if the xml is valid)
     * @param content
     * @param overwrite
     * @return
     * @throws ConfigServiceException
     */
    public IConfigBean importBeanContent(String content, boolean overwrite)
    throws ConfigServiceException
    {
        checkConnected();

        try
        {
            // Parse the xml to create a bean
            ConfigBeanDocument.ConfigBeanIdentity id = ConfigBeanDocument.parseIdentityContent(content);

            IConfigBean bean = _importBean(content, id, true, overwrite);

            commitTransaction();

            return bean;
        }
        catch(ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("csu-import-bean-content-failed", e);
        }
    }

    private final IConfigBean _importBean(String urlOrContent,
                                          ConfigBeanDocument.ConfigBeanIdentity id,
                                          boolean content,
                                          boolean overwrite)
    throws Exception
    {
        String name    = id.m_name;
        String type    = id.m_typeName;
        String version = id.m_version;

        boolean folderElement    = false;
        boolean complexType      = false;
        boolean complexTypeChild = false;

        if ( (type.startsWith("MF_AUTHENTICATION") && !type.equals("MF_AUTHENTICATION_SPI"))||
           	type.startsWith("MQ_AUTHORIZATION")  ||
           	type.startsWith("PIM_PARTNER"))
        {
            complexType = true;
        }
        else
        // Check to see if the name ends with _Default and it isn't a
        // MF_FILE then we have a folder element.
        if (name.endsWith(DEFAULT_SUFFIX) && !type.equals(ConfigBeanFile.CONFIG_TYPE))
        {
            folderElement = true;
        }

        if (type.equals("MF_AUTHENTICATION_GROUP") ||
            type.equals("MF_AUTHENTICATION_USER")  ||
           	type.equals("MQ_AUTHORIZATION_ACL")    ||
   			type.equals("MQ_AUTHORIZATION_QOP")    ||
   			type.equals("PIM_PARTNER"))
        {
            complexTypeChild = true;
        }

        // Find the parent folder
        String parentFolder     = name.substring(0, name.lastIndexOf('/'));
        if (parentFolder.length() == 0)
        {
            parentFolder = "/";
        }
        String realParentFolder = new String(parentFolder);

        if(complexType)
        {
            parentFolder = parentFolder.substring(0, parentFolder.lastIndexOf('/'));
        }

        // If the parent folder doesn't already exist - create it as part
        // of this transaction
        if ((!complexType || (complexType && !complexTypeChild)) &&
            !m_configServer.pathExists(parentFolder))
        {
            if(parentFolder != null && parentFolder.length() > 0)
            {
                m_configServer.createPath(parentFolder, true);
            }
        }

        // Work out the meta attributes to add to the view
        HashMap map = new HashMap();
        map.put(TYPE, type);
        map.put(CONFIG_VERSION, version);
        if(id.m_productVersion != null)
        {
            map.put(PRODUCT_VERSION, id.m_productVersion);
        }

        IConfigBean bean = null;

        if(content)
        {
            bean = ConfigBeanDocument.parseContent(urlOrContent, m_configServer, overwrite);
        }
        else
        {
            bean = ConfigBeanDocument.parse(urlOrContent, m_configServer, overwrite);
        }

        if (!complexType)
        {
            bean.setMetaAttributes(map);
        }

        m_configServer.storeConfigElement(bean);

        // if it is a folder element then we also have to add the same
        // set of meta attributes to the folder
        if ((complexType && !complexTypeChild) || folderElement)
        {
            m_configServer.setMetaAttributes(realParentFolder, map);
        }

        return bean;
    }

    /**
     * Recursively import all .xml files found in the specified path and create
     * IConfigBeans for each one.
     * @param path
     * @param overwrite
     * @throws ConfigServiceException
     */
    public final void importBeanPath(String path, boolean overwrite)
    throws ConfigServiceException
    {
        checkConnected();

        try
        {
            List list = new ArrayList();

            _getBeanPathIdentities(path, list);

            for(int i = 0; i < list.size(); i++)
            {
                Object[] item = (Object[])list.get(i);

                _importBean((String)item[1], (ConfigBeanDocument.ConfigBeanIdentity)item[0], false, overwrite);
            }

            commitTransaction();
        }
        catch(ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("csu-import-bean-failed", new Object[] { path } , e);
        }
    }

    private void _getBeanPathIdentities(String path, List result)
    throws Exception
    {
        File file = new File(path);

        if (!file.exists())
        {
            throw new ConfigServiceException("csu-import-bean-file-not-found", new Object[] { path });
        }

        // Generate a list of files in the directory config folder.
        File[] list = file.isDirectory() ? file.listFiles() : new File[] { file };

        if (list != null)
        {
            // Find all files ending with .xml after retrieving the contents
            // of the directory
            for (int i = 0; i < list.length; i++)
            {
                if (list[i].isFile() && list[i].toString().endsWith(".xml"))
                {
                    String url = Util.name2Url(list[i].toString());

                    try
                    {
                        ConfigBeanIdentity id = ConfigBeanDocument.parseIdentity(url);

                        result.add(new Object[] { id, url } );
                    }
                    catch(Exception e)
                    {
                    }
                }
            }

            for (int i = 0; i < list.length; i++)
            {
                if (list[i].isDirectory())
                {
                    _getBeanPathIdentities(list[i].toString(), result);
                }
            }
        }
    }

    //------------------------------------------------------------------------
    //
    // Export
    //
    //------------------------------------------------------------------------

    // AHJ Left in so as not to break any code...even though I couldn't
    //     find anything that referenced it.
    //
    /**
     * Export a single bean given its bean name. The generated XML data is
     * returned as a String
     * @param beanName the name of the bean
     * @return the XML for the bean
     * @throws ConfigServiceException
     */
    public String exportBean(String beanName)
    throws ConfigServiceException
    {
        checkConnected();

        Util.validateConfigBeanName(beanName);

        // Try to load the config bean first - this will check whether it is
        // a valid config bean or something else in the DS that we don't know
        // about
        IConfigBean bean = null;

        try
        {
            IConfigElement element = m_configServer.loadConfigElement(beanName);

            if(element instanceof IConfigBean)
            {
                bean = (IConfigBean)element;
            }
        }
        catch(Exception e)
        {
            // Eat it.
        }

        if(bean == null || (bean instanceof IConfigBeanFile))
        {
            return null;
        }

        return m_configServer.exportConfigBean(bean);
    }

    /**
     * Sets the socket connect timeout in seconds.
     * This limits the connection time spent per-URL when multiple connection URLs are provided.
     * 
     * @param seconds Timeout value in seconds.
     */
    public void setSocketConnectTimeout(int seconds)
    {
        m_socketConnectTimeout = seconds;
    }

    /**
     * Gets the socket connect timeout in seconds.
     * 
     * @return Timeout value in seconds 
     */
    public int getSocketConnectTimeout() {
        return m_socketConnectTimeout;
    }

    // Control structure used by the export code
    private class ExportControl
    {
        boolean doConfig = true;
        boolean doFile = false;
        boolean replaceAll = false;
        boolean cancel = false;
        boolean single = false;  // Set to indicate that we're only exporting a single item
    }

    /**
     * Export a single bean given its bean name to the specified filename.
     * @param beanName the name of the bean
     * @param filename the file to use for the output XML
     * @return the XML for the bean
     * @throws ConfigServiceException
     */
    public String exportBean(String beanName, String filename)
    throws ConfigServiceException
    {
        ExportControl control = new ExportControl();
        control.doFile = false;
        control.doConfig = true;

        return exportBean(beanName, filename, null, control);
    }

    private String exportBean(String beanName, String filename, IFileExistHandler handler, ExportControl control)
    throws ConfigServiceException
    {
        String xmlRes = null;

        if ((filename == null) || (filename.length() == 0))
        {
            throw new ConfigServiceException("csu-export-filename-invalid", new Object[] { beanName, filename });
        }

        Util.validateConfigBeanName(beanName);
        InputStream is = null;

        try
        {
            IConfigElement element = m_configServer.loadConfigElement(beanName);

            if (control.doFile && (element instanceof IConfigBeanFile))
            {
                IConfigBeanFile beanFile = (IConfigBeanFile)element;
                is = beanFile.getInputStream();
                if (is == null)
                 {
                    is = new ByteArrayInputStream(new byte[0]);  // Empty stream will still cause file to be created
                }
            }
            else
            if (control.doConfig && (element instanceof IConfigBean))
            {
                filename += ".xml";

                IConfigBean beanConfig = (IConfigBean)element;
                xmlRes = m_configServer.exportConfigBean(beanConfig);

                if (xmlRes != null)
                {
                    is = new StringBufferInputStream(xmlRes);
                }
            }
        }
        catch(Exception e)
        {
            // Eat it.
        }

        if (is == null)
        {
            return null;
        }

        try
        {
            // If the file already exists then handle it...
            File file = new File(filename);
            if (file.exists() && (handler != null) && !control.replaceAll)
            {
                int res = handler.fileExists(file, !control.single);

                switch (res)
                {
                    case IFileExistHandler.CANCEL :
                    {
                        control.cancel = true;
                        return null;
                    }
                    case IFileExistHandler.NO :
                    {
                        return null; // Return immediately - no file save
                    }
                    case IFileExistHandler.YES :
                    {
                        break; // Do nothing
                    }
                    case IFileExistHandler.YES_ALL :
                    {
                        control.replaceAll = true;
                        break;
                    }
                }
            }

            // ensure the directory already exists for the file
            File parent = file.getParentFile();
            if(!parent.exists())
            {
                parent.mkdirs();
            }

            writeToFile(is, file);
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("csu-export-cant-write-file", new Object[] { beanName, filename }, e);
        }

        return xmlRes;
    }

    /**
     * Given a path in the DS, find all config beans and export them.  If what
     * you find is another folder, then recurse the tree, finding all config
     * beans under the specified path.
     * @param path Root of the search in the DS
     * @param outputDir Directory to use for generated output files
     * @throws ConfigServiceException
     */
    public final void exportBeanPath(String path, String outputDir)
    throws ConfigServiceException
    {
        exportBeanPath(path, outputDir, null, true, false);
    }

    public void exportBeanPath(String path, String outputDir, IFileExistHandler handler, boolean doConfig, boolean doFile)
    throws ConfigServiceException
    {
        ExportControl control = new ExportControl();
        control.doConfig = doConfig;
        control.doFile = doFile;

        exportBeanPath(new String[] { path }, outputDir, handler, doConfig, doFile);
    }

    public void exportBeanPath(String[] path, String outputDir)
    throws ConfigServiceException
    {
        exportBeanPath(path, outputDir, null, true, false);
    }

    public void exportBeanPath(String[] path, String outputDir, IFileExistHandler handler, boolean doConfig, boolean doFile)
    throws ConfigServiceException
    {
        checkConnected();

        ExportControl control = new ExportControl();
        control.doConfig = doConfig;
        control.doFile = doFile;
        control.single = (path.length == 1) && !isFolder(path[0]);  // Determines that if we have a single entry thats
                                                                    // NOT a (config) folder

        exportBeanPath(path, outputDir, handler, control);
    }

    private void exportBeanPath(String[] path, String outputDir, IFileExistHandler handler, ExportControl control)
    throws ConfigServiceException
    {
        for (int i = 0; i < path.length; i++)
        {
            if (control.cancel)
            {
                return;
            }

        	Map metaMap = m_configServer.getMetaAttributes(path[i]);

        	if (metaMap != null && metaMap.get(ConfigServer.ELEMENT_IDENTITY) != null)
            {
                exportBean(path[i], outputDir + path[i], handler, control);
            }
            else
            {
                _exportBeanPath(path[i], outputDir, handler, control);
            }
        }
    }

    /**
     * This method assumes that 'path' is a folder, NOT an element.
     *
     * @param path
     * @param outputDir
     * @throws ConfigServiceException
     */
    private void _exportBeanPath(String path, String outputDir, IFileExistHandler handler, ExportControl control)
    throws ConfigServiceException
    {
        try
        {
            String[] list = listConfigElements(path);

            for (int i = 0; i < list.length; i++)
            {
                if (control.cancel)
                {
                    return;
                }

                if (list[i].endsWith("/"))
                {
                    exportBeanPath(new String[] { list[i] }, outputDir, handler, control);
                }
                else
                {
                    exportBean(list[i], outputDir + list[i], handler, control);
                }
            }
        }
        catch(ConfigServiceException e)
        {
            throw e;
        }
        catch(Exception e)
        {
            throw new ConfigServiceException("csu-export-cant-write-file", new Object[] { path, outputDir }, e);
        }
    }

    private boolean isFolder(String logicalPath)
    throws ConfigServiceException
    {
        checkConnected();

        if (logicalPath.equals("/"))
        {
            return true;
        }

        String parentPath = logicalPath.substring(0, logicalPath.lastIndexOf("/") + 1);
        Iterator i = m_configServer.listFolders(parentPath).iterator();

        while (i.hasNext())
        {
            String folder = (String)i.next();

            if (folder.equals(logicalPath))
            {
                return true;
            }
        }

        return false;
    }

    private void writeToFile(InputStream is, File file)
    throws ConfigServiceException
    {
        FileOutputStream os = null;

        try
        {
            os = new FileOutputStream(file);
        }
        catch (FileNotFoundException e)
        {
            throw new ConfigServiceException("failed to create: " + file);
        }

        try
        {
            byte[] buffer = new byte[4096];

            while (true)
            {
                int length = is.read(buffer);

                if (length == -1)
                {
                    break;
                }

                os.write(buffer, 0, length);
            }
        }
        catch (IOException e)
        {
            throw new ConfigServiceException("failed during write of export file: " + file);
        }
        finally
        {
            try
            {
                if (os != null)
                {
                    os.flush();
                    os.close();
                }
            }
            catch (Exception e)
            {
                // Ignore Exception due to resource closing
            }
        }
    }

    //------------------------------------------------------------------------

    /**
     * Delete all beans and folders in the specified path in the DS
     * @param path
     * @throws ConfigServiceException
     */
    public void deleteBeanPath(String path)
    throws ConfigServiceException
    {
        checkConnected();

        deleteBeanPath(m_configServer, path);

        commitTransaction();
    }

    /**
     * Delete all beans and folders in the specified path in the DS using the
     * specified IConfigServer
     * @param server
     * @param path
     * @throws ConfigServiceException
     */
    public void deleteBeanPath(IConfigServer server, String path)
    throws ConfigServiceException
    {
        try
        {
            _deleteBeanPath(server, path);
        }
        catch(ConfigServiceException e)
        {
            throw e;
        }
        catch(Exception e)
        {
            throw new ConfigServiceException("csu-delete-bean-path-failed", new Object[] { path }, e);
        }
    }

    private final void _deleteBeanPath(IConfigServer server, String path)
    throws Exception
    {
        Set set = server.listConfigElements(path);
        String names[] = (String[])set.toArray(new String[0]);
        server.removeConfigElements(names);

        set = server.listFolders(path);
        for(Iterator iter = set.iterator(); iter.hasNext(); )
        {
            _deleteBeanPath(server, iter.next().toString());
        }

        server.deleteFolder(path);
    }

    /**
     * Generate the Container Boot file xml for the specified container
     * @param containerConfigElement
     * @param domainName
     * @return
     * @throws ConfigServiceException
     */
    public String exportContainerBootConfiguration(IConfigElement containerConfigElement, String domainName)
    throws ConfigServiceException
    {
        checkConnected();

        return m_configServer.exportContainerBootConfiguration(containerConfigElement, domainName);
    }
    /**
     * Generate the Container Boot file xml for the specified container
     * @param containerConfigElement
     * @return
     * @throws ConfigServiceException
     */
    public String exportDSBootConfiguration(IConfigElement configElement)
    throws ConfigServiceException
    {
        checkConnected();

        return m_configServer.exportDSBootConfiguration(configElement);
    }

    /**
     * Delete the specified bean from the DS
     * @param beanName
     * @throws ConfigServiceException
     */
    public void deleteBean(String beanName)
    throws ConfigServiceException
    {
        checkConnected();

        Util.validateConfigBeanName(beanName);

        m_configServer.removeConfigElement(beanName);

        commitTransaction();
    }

    /**
     * Import the specified .xsd file into the ConfigLayer and create a
     * IConfigType from its contents
     * @param typeURL
     * @param namespace
     * @param overwrite
     * @return
     * @throws ConfigServiceException
     */
    public IConfigType[] importType(String typeURL, String namespace, boolean overwrite)
    throws ConfigServiceException
    {
        checkConnected();

        try
        {
            Collection cConfigTypes = ConfigTypeDocument.parse(namespace, typeURL, (ConfigServer)m_configServer, null);
            Collection cConfigTypeNames = new ArrayList();

            Iterator i = cConfigTypes.iterator();

            while (i.hasNext())
            {
                IConfigType configType = (IConfigType)i.next();
                String typeName = configType.getName();
                String version = configType.getVersion();

                cConfigTypeNames.add(overwrite ? ConfigTypeImpl.convertTypeName2DSName(typeName, version) : null);
            }

            IConfigType[] configTypes     = (IConfigType[])cConfigTypes.toArray(new IConfigType[0]);
            String[]      configTypeNames = (String[])cConfigTypeNames.toArray(new String[0]);

            ((ConfigServer)m_configServer).storeConfigTypes(configTypes, configTypeNames);

            return configTypes;
        }
        catch(ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("csu-import-type-failed", new Object[] { typeURL }, e);
        }
    }


    /**
        */
   public void importTypeProperties(String typeURL, String namespace, String path)
   throws ConfigServiceException
   {
       checkConnected();

        try
        {
            Iterator configTypes = ConfigTypeDocument.parse(namespace, typeURL, (ConfigServer)m_configServer, path).iterator();

            while (configTypes.hasNext())
            {
                IConfigType configType = (IConfigType) configTypes.next();

                String typeName = configType.getName();
                String version = configType.getVersion();

                ((ConfigServer)m_configServer).storeConfigTypes(new IConfigType[] { configType },
                                              new String[] { ConfigTypeImpl.convertTypeName2DSName(typeName, version) } );
            }
        }
        catch(ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("csu-import-type-failed", new Object[] { typeURL }, e);
        }

   }

   public IConfigType importInitialValues(String url)
       throws ConfigServiceException
   {
       IConfigType configType = null;

       checkConnected();

       try
       {
           configType = InitialValuesImpl.attach(m_configServer, url);

           String typeName    = configType.getName();
           String typeVersion = configType.getVersion();

           ((ConfigServer)m_configServer).storeConfigType(configType);
       }
       catch(ConfigServiceException e)
       {
           throw e;
       }
       catch (Exception e)
       {
           throw new ConfigServiceException("csu-import-type-failed", new Object[] { url }, e);
       }

       return configType;
   }

   public IInitialValues createInitialValues(String type, String version)
       throws Exception
   {
       return new InitialValuesImpl(type, version, getConfigServer());
   }

   public IInitialValues loadInitialValues(File file)
       throws Exception
   {
       return new InitialValuesImpl(file, getConfigServer());
   }

    /**
     * Delete the specified config type
     * @param typeName
     * @param version
     * @throws ConfigServiceException
     */
    public void deleteType(String typeName, String version)
    throws ConfigServiceException
    {
        checkConnected();

        Util.validateConfigTypeName(typeName);

        ((ConfigServer)m_configServer).removeConfigType(typeName, version);
    }

    /**
     * Get the names of all config elements in a particular DS directory. This
     * method will return all config elements regardless of whether they are
     * configElements, configBeans, configTypes etc.
     * A calling application will either have to know the type of a returned
     * element or should attempt to load as a bean / type to check.
     * @param path The Directory Service path e.g. /containers
     * @return A String array containing the names of all elements
     * @throws ConfigServiceException
     */
    public String[] listConfigElements(String path)
    throws ConfigServiceException
    {
        checkConnected();

        Util.validateConfigBeanName(path);

        ArrayList result = new ArrayList();

        Set set = m_configServer.listFolders(path);
        for(Iterator iter = set.iterator(); iter.hasNext(); )
        {
            result.add(iter.next().toString() + "/");
        }

        set = m_configServer.listConfigElements(path);
        for(Iterator iter = set.iterator(); iter.hasNext(); )
        {
            result.add(iter.next().toString());
        }

        return (String[])result.toArray(new String[result.size()]);
    }

    public void importFile(String path, String filename, boolean overwrite)
        throws ConfigServiceException
    {
        ConfigHelper.importFile(m_configServer, path, filename, m_user, overwrite);

        commitTransaction();
    }

    /**
     * Obtain a dump the contents of the DS (in DS format XML) starting at
     * the specified location in the DS.
     * @param path
     * @return
     * @throws ConfigServiceException
     */
    public String dumpDirectoryService(String path)
    throws ConfigServiceException
    {
        String dump = null;

        try
        {
            try
            {
                dump = ((IDirectoryAdminService)m_dfs).exportDirectoryToXML(path);
            }
            catch (Exception e)
            {
                dump = ((IDirectoryAdminService)m_dfs).exportElementToXML(path);
            }
        }
        catch(Exception e)
        {
            throw new ConfigServiceException("csu-dump-ds-failed", new Object[] { path }, e);
        }
        return dump;
    }

    /**
     * Dump the containers of the DS (in DS format XML) to the specified File.
     * @param path
     * @param file
     * @throws ConfigServiceException
     */
    public void dumpDirectoryService(String path, String file)
    throws ConfigServiceException
    {
        String dump = dumpDirectoryService(path);

        try
        {
            Writer writer = null;

            if (file != null)
            {
                writer = new FileWriter(file);
            }
            else
            {
                writer = new PrintWriter(System.out);
            }

            writer.write(dump);
            writer.flush();
            writer.close();
        }
        catch(Exception e)
        {
            throw new ConfigServiceException("csu-dump-ds-failed", new Object[] { path }, e);
        }
    }

    /**
     * Read the contents of the specified file and seed it into the Directory
     * Service
     * @param file
     * @throws ConfigServiceException
     */
    public void seedDirectoryService(String file)
    throws ConfigServiceException
    {
        try
        {
            ((IDirectoryAdminService)m_dfs).importFromXML(readFile(file));
        }
        catch(Exception e)
        {
            throw new ConfigServiceException("csu-seed-ds-failed", new Object[] { file }, e);
        }
    }

    /**
     * Deep copy of files from the file-system 'fromFSPath' folder into the DS 'toDSPath' folder. DS folders in the path are created is missing. Files are
     * over-written if alerady exist.
     *
     * @param fromFSPath the file-system source directory
     * @param toDSPath the DS destination folder
     *
     * @ConfigServiceException ConfigServiceException
     *                         if there is a problem accessing the underlying storage or the source is missing or not a folder.
     *
     */
    public void importFiles(String fromFSPath, String toDSPath)
    throws ConfigServiceException
    {
        try
        {
            new SonicFSFileSystem(m_dfs, m_user).importFiles(fromFSPath, toDSPath);
        }
        catch(Exception e)
        {
            throw new ConfigServiceException("import-files failed", new Object[] {fromFSPath, toDSPath}, e);
        }
    }

    /**
     * Copy DS files to the cache of a given list of containers
     *
     * @param dsPath A directory or file - copy all the files under the directory and sub-directories
     *
     * @param allContainers
     *            The list should include all the containers
     *
     * @param excludeContainers
     *            use the list as an exclusion list
     *
     * @param containerListFilePath
     *            the file that contains container configuration paths
     *
     * @exception ConfigServiceException
     *                if there is a problem accessing the underlying storage
     *
     */
    public void copyDSFilesToContainerCaches(String dsPath, boolean allContainers, boolean excludeContainers, String containerListFilePath)
    throws ConfigServiceException
    {
        try
        {
            String[] fileList = new SonicFSFileSystem(m_dfs, m_user).list(dsPath, true);
            String[] actionList = getContainerList(allContainers, excludeContainers, getFileListedContainers(containerListFilePath));

            String domaniName = getDomain();
            for (int i = 0; i < actionList.length; i++)
            {
                IDirElement containerConfigElement = null;
                try
                {
                    containerConfigElement = m_dfs.getFSElement(actionList[i], false);
                }
                catch (Exception e)
                {

                }

                if (containerConfigElement == null)
                {
                    System.out.println("No configuration was found for \"" + actionList[i] + "\"");
                    continue;
                }



                String containerName = new File(actionList[i]).getName();
                ObjectName containerObjectName = new ObjectName(domaniName + '.' + containerName + ":ID=AGENT");


                for (int j = 0; j < fileList.length; j++)
                {
                    if (!fileList[j].endsWith(  "/"))
                    {
                        try
                        {
                            System.out.println("Copying \"" + fileList[j] + "\" into the cache of container \"" + actionList[i] + "\"");
                            downloadFile(fileList[j], containerObjectName);
                        }
                        catch (Exception e)
                        {
                            System.out.println("Failed to copy file(s) into the cache of container \"" + actionList[i] + "\": " + e.toString());
                            break;
                        }
                    }
                }
            }

        }
        catch(Exception e)
        {
            throw new ConfigServiceException("copy-ds-files-to-container-caches failed", new Object[] {dsPath}, e);
        }
    }

    private void downloadFile(String dsFilePath, ObjectName containerObjectName) throws Exception
    {

        Object[] params = new Object[]
        {
            dsFilePath
        };
        String[] signature = new String[]
        {
            "java.lang.String"
        };
        ((JMSConnectorClient)m_server).invoke(containerObjectName, "downloadFile", params, signature);

    }

    /**
     * Copy the 'fromPath' file or folder into 'toPath'. Deep copy is performed if 'fromPath' is a folder. If 'toPath' is an existing
     * folder then 'fromPath' is copied into it. If 'toPath' does not exit then the content of 'fromPath' is copied into its parent.
     * An exception is thrown if the parent of 'toPath' does not exist.
     *
     * @param fromPath a file or a folder source
     * @param toPath the target path
     *
     * @ConfigServiceException DirectoryServiceException
     *                         if there is a problem accessing the underlying storage or the parent folder of 'toPath' doesn't exit
     *
     */
    public void copyFiles(String fromPath, String toPath)
    throws ConfigServiceException
    {
        try
        {
            m_dfs.copyFiles(fromPath, toPath);
        }
        catch(Exception e)
        {
            throw new ConfigServiceException("copy-files failed", new Object[] {fromPath, toPath}, e);
        }
    }

    /**
     * Create a new folder. The parent folder must exist.
     *
     * @param folderPath
     *            the name of the new folder
     *
     * @exception ConfigServiceException
     *                if there is a problem accessing the underlying storage or the parent folder doesn't exist
     *
     */
    public void createFolder(String folderPath0)
    throws ConfigServiceException
    {
        try
        {
            String folderPath = getCanonicalPath(folderPath0);
            m_dfs.createFolder(folderPath);
        }
        catch(Exception e)
        {
            throw new ConfigServiceException("create-folder failed", new Object[] {folderPath0}, e);
        }
    }

    /**
     * Deletes a folder. The folder must be empty.
     *
     * @param folderPath
     *            the name of the folder
     *
     * @exception ConfigServiceException
     *                if there is a problem accessing the underlying storage or the folder does not exist or not empty
     *
     */
    public void deleteFolder(String folderPath0)
    throws ConfigServiceException
    {
        try
        {
            String folderPath = getCanonicalPath(folderPath0);
            m_dfs.deleteFolder(folderPath);
        }
        catch(Exception e)
        {
            throw new ConfigServiceException("delete-folder failed", new Object[] {folderPath0}, e);
        }
    }

    /**
     * Modify the container archives search path attribure given a list of containers
     *
     * @param newSearchPath
     *
     * @param allContainers
     *            The list should include all the containers
     *
     * @param excludeContainers
     *            use the list as an exclusion list
     *
     * @param containerListFilePath
     *            the file that contains container configuration paths
     *
     * @exception ConfigServiceException
     *                if there is a problem accessing the underlying storage or the list of containers is incorrect
     *
     */
    public void modifyArchivesSearchPath(String newSearchPath, boolean allContainers, boolean excludeContainers, String containerListFilePath)
    throws ConfigServiceException
    {

        try
        {
            String[] actionList = getContainerList(allContainers, excludeContainers, getFileListedContainers(containerListFilePath));

             for (int i = 0; i < actionList.length; i++)
             {
                 IDirElement containerConfigElement = m_dfs.getFSElement(actionList[i], true);
                 IAttributeSet containerConfigAtts = containerConfigElement.getAttributes();
                 String oldSearchPath = (String)containerConfigAtts.getAttribute(IContainerConstants.ARCHIVE_SEARCH_PATH_ATTR);
                 if (!oldSearchPath.equals(newSearchPath))
                 {
                     containerConfigAtts.setStringAttribute(IContainerConstants.ARCHIVE_SEARCH_PATH_ATTR, newSearchPath);
                     m_dfs.updateFSElement((IDeltaDirElement)containerConfigElement.doneUpdate());
                 }
             }

        }
        catch(Exception e)
        {
            if (e instanceof ConfigServiceException)
            {
                throw (ConfigServiceException)e;
            }
            else
            {
                throw new ConfigServiceException("modify-archive-search-path failed", new Object[] {newSearchPath,new Boolean(allContainers),new Boolean(excludeContainers),containerListFilePath}, e);
            }
        }
    }

    private ArrayList getFileListedContainers(String containerListFilePath)
    throws Exception
    {

        ArrayList resultList = new ArrayList();
        if (containerListFilePath != null && containerListFilePath.length() != 0)
        {
            File listFile = new File(containerListFilePath);
            if (!listFile.exists())
            {
                throw new ConfigServiceException("\"" + containerListFilePath + "\" does not exist");
            }

            BufferedReader reader = new BufferedReader (new InputStreamReader(new FileInputStream(listFile)));

            while (true)
            {
                String srcLine = reader.readLine();
                if (srcLine == null)
                {
                    break;
                }
                String containerPath = srcLine.trim();

                if (containerPath.length() == 0)
                {
                    continue;
                }

                resultList.add(containerPath);
            }

         }

         return resultList;
    }

    private String[] getContainerList(boolean modifyAll, boolean excludeContainers, ArrayList specifiedList) throws Exception
    {
        if (!excludeContainers && !modifyAll)
        {
            return (String[])specifiedList.toArray(new String[0]);
        }

        if (excludeContainers && modifyAll)
        {
            return new String[0];
        }

        Query findContainersQuery = new Query();
        findContainersQuery.setFrom(new FromElementType(IContainerConstants.DS_TYPE));
        findContainersQuery.setSelect(new Select(new AttributeName[0]));
        IDirElement[] containerConfigs = m_dfs.getFSElements(findContainersQuery, false);

        HashSet allContainers = new HashSet();
        for (int i = 0; i < containerConfigs.length; i++)
        {
            allContainers.add(containerConfigs[i].getIdentity().getName());
        }

        if (excludeContainers)
        {
            for (int i = 0; i < specifiedList.size(); i++)
            {
                allContainers.remove(specifiedList.get(i));
            }
        }

        String[] result = new String[allContainers.size()];
        Iterator iterator = allContainers.iterator();
        for (int i = 0; iterator.hasNext(); i++)
        {
            result[i] = (String)iterator.next();
        }

        return result;
    }

    /**
     * Deletes a file or a folder with all its content.
     *
     * @param path
     *            the name of the file or folder
     *
     * @exception ConfigServiceException
     *                if there is a problem accessing the underlying storage or the path does not exist
     *
     */
    public void deleteFiles(String path)
    throws ConfigServiceException
    {
        try
        {
            new SonicFSFileSystem(m_dfs, m_user).deleteFiles(path);
        }
        catch(Exception e)
        {
            throw new ConfigServiceException("delete-files failed", new Object[] {path}, e);
        }
    }

    /**
     * Rename a folder.
     *
     * @param oldPath
     *            of the folder
     * @param newPath
     *            of the folder
     *
     * @exception Exception
     *                if there is a problem accessing the underlying storage or oldPath does not exist or is not a folder or there is no
     *                parent folder for the new path or oldPath and newPath are identical
     *
     */
    public void renameFolder(String oldPath, String newPath) throws Exception
    {
        rename(oldPath, newPath, true);
    }

    /**
     * Rename a file.
     *
     * @param oldPath
     *            of the file
     * @param newPath
     *            of the file
     *
     * @exception Exception
     *                if there is a problem accessing the underlying storage or oldPath does not exist or is not a file or there is no
     *                parent folder for the new path or oldPath and newPath are identical
     *
     */
    public void renameFile(String oldPath, String newPath) throws Exception
    {
        rename(oldPath, newPath, false);
    }


    private  void rename(String oldPath0, String newPath0, boolean folder) throws Exception
    {
        if (oldPath0 == null || oldPath0.length() == 0)
        {
            throw new ConfigServiceException("rename failed: Cannot rename( >>> blank <<<, " + newPath0 + ")");
        }

        if (newPath0 == null|| newPath0.length() == 0)
        {
            throw new ConfigServiceException("rename failed: Cannot rename(" + oldPath0 + ", >>> blank <<<)");
        }

        try
        {
            String oldPath = getCanonicalPath(oldPath0);
            String newPath = getCanonicalPath(newPath0);

        if (oldPath.equals("/"))
        {
            throw new Exception("Cannot rename the root folder");
        }

        if (oldPath.equals(newPath))
        {
            throw new Exception("Cannot rename : oldPath and newPath are identical");
        }

            if (folder)
            {
                m_dfs.renameFolder(oldPath, newPath);
            }
            else
            {
                m_dfs.renameFile(oldPath, newPath);
            }
        }
        catch(Exception e)
        {
            throw new ConfigServiceException("rename failed ", new Object[] {oldPath0, newPath0}, e);
        }
    }


    private String getCanonicalPath(String path) throws Exception
    {
        if (!path.startsWith("/"))
        {
            throw new Exception("\"" + path + "\" does not start with /");
        }

        StringBuffer sb = new StringBuffer();

        int pathLength = path.length();
        boolean prevWasSeparator = false;
        for (int i = 0; i < pathLength; i++)
        {
            char c = path.charAt(i);
            if (prevWasSeparator && c == '/')
            {
                continue;
            }

            sb.append(c);

            prevWasSeparator = c == '/';

        }

        String canonical = sb.toString();

        int canonicalLength = canonical.length();

        //Drop the last '/' if exists and is not the root directory
        if (canonicalLength > 1 && canonical.charAt(canonicalLength - 1) == '/')
        {
            return canonical.substring(0, canonicalLength - 1);
        }
        else
        {
            return canonical;
        }


    }


    private String readFile(String file) throws Exception
    {
        BufferedReader in  = null;
        StringWriter   out = null;

        try
        {
            in  = new BufferedReader(new FileReader(file));
            out = new StringWriter();

            char[] buffer = new char[1024];
            int read = in.read(buffer, 0, 1024);

            while(read != -1)
            {
                out.write(buffer, 0, read);
                read = in.read(buffer, 0, 1024);
            }
            return out.toString();
        }
        catch(Exception e)
        {
            throw new Exception ("Error reading file: " + file + e.getMessage());
        }
        finally
        {
            try
            {
                if (in != null) in.close();
                if (out != null) out.close();
            }
            catch (Exception e) { }
        }
    }

    public void setDSHandlerWorkspace(String path)
    throws ConfigServiceException
    {
        try
        {
            IDirElement handlerMap = ElementFactory.createElement("/workspace/mapping/EclipseMap", "Eclipse_Map", "1.0");
            IAttributeSet topSet = handlerMap.getAttributes();

            topSet.setStringAttribute("ECLIPSE_WORKSPACE_DISK_LOCATION", path);

            m_dfs.createFSElement(handlerMap);
        }
        catch(Exception e)
        {
            throw new ConfigServiceException("csu-set-workspace-failed", new Object[] { path }, e);
        }
    }

    /**
     * Add a plugin definition into the DS _MFLibrary
     * @param name
     * @param config
     * @param runtime
     * @param classpath
     */
    public void addPlugin(String name, String config, String runtime, String classpath)
    throws ConfigServiceException
    {
        try
        {
            String path = "/_MFLibrary/plugins/" + name;

            IDirElement element = ElementFactory.createElement(path, "MF_PLUGIN", "1.0", false);

            IAttributeSet attributes = element.getAttributes();
            if(config != null)
            {
                attributes.setStringAttribute("CONFIG_PLUGIN_FACTORY_CLASS", config);
            }
            if(runtime != null)
            {
                attributes.setStringAttribute("RUNTIME_PLUGIN_FACTORY_CLASS", runtime);
            }
            if(classpath != null)
            {
                attributes.setStringAttribute("PLUGIN_FACTORY_CLASSPATH", classpath);
            }

            ((IDirectoryAdminService)m_dfs).
                setElements(new IBasicElement[] { element.doneUpdate() },
                            new String[] { path },
                            null);
        }
        catch(Exception e)
        {
            throw new ConfigServiceException("csu-add-plugin-failed", new Object[] { name }, e);
        }
    }

    /**
     * Delete a plugin definition from the DS _MFLibrary
     * @param name
     */
    public void deletePlugin(String name)
    throws ConfigServiceException
    {
        try
        {
            String path = "/_MFLibrary/plugins/" + name;

            ((IDirectoryAdminService)m_dfs).deleteElement(path, null);
        }
        catch(Exception e)
        {
            throw new ConfigServiceException("csu-delete-plugin-failed", new Object[] { name }, e);
        }
    }

    private void checkConnected()
    throws ConfigServiceException
    {
        if(m_configServer == null)
        {
            throw new ConfigServiceException("csu-not-connected");
        }
    }

    @Override
    protected void finalize() throws Throwable
    {
        disconnect();
    }

    protected void commitTransaction()
    throws ConfigServiceException
    {
    }
}
