/*
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.mx.config;

import java.io.InputStream;
import java.util.HashMap;

/**
 * <p>Title: IConfigElement</p>
 * <p>Description: Interface definition of a configuration element.</p>
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: Sonic Software</p>
 * @author not attributable
 * @version 1.0
 */
public interface IConfigElement extends IAttributeMap, IAnnotationExtension
{
    /**
     * Indicates whether some other object is "equal to" this one.
     *
     * @param obj  the reference object with which to compare.
     * @return     <code>true</code> if this object is the same as the obj
     *             argument; <code>false</code> otherwise.
     */
    @Override
    public boolean equals(Object obj);

    /**
     * Determines whether or not the element has been removed or deleted.
     *
     * removed - the element is about to be deleted, i.e. not committed to DS
     * deleted - the element has been deleted in the DS
     *
     * @return <code>true</code> if the element has been removed;
     *         <code>false</code> otherwise.
     */
    public boolean isRemoved();

    /**
     * Determines whether or not the element has been modified.
     *
     * An element is classified as modified if the element is itself new, or
     * it is an existing element and attributes or meta-attributes have been
     * changed.
     *
     * @return <code>true</code> if the element has been modified;
     *         <code>false</code> otherwise.
     */
    public boolean isModified();

    /**
     * Indicates whether or not the element is new. An element can only be
     * 'new' if it has not yet been stored in the Directory Service.
     *
     * @return <code>true</code> if the element is new; <code>false</code> otherwise.
     */
    public boolean isNew();

    /**
     * The name (path) that uniquely identifying this config element.
     *
     * The name is a fully qualified path that defines the location and name
     * of this config element in the DS file system.
     *
     * @return  The name of the config element.
     */
    public String getName();

    /**
     * Provides a way of changing the name of the config element.
     * This essentially is performing a rename.
     *
     * There are restrictions on what constitutes a valid name:
     *  - maximum size of name is limited to 128 characters
     *  - name can't start with the / character
     *  - name can't contain the following character sequences  $ \ :: .*. .#. < > &
     *
     * @param name                     The new name for the config element.
     * @throws ConfigServiceException  An exception if the name fails validation
     */
    public void setName(String name)
        throws ConfigServiceException;

    /**
     * Gets the ConfigServer under which this config element was loaded.
     *
     * Note: that config elements are tied to a particular config server.
     * A config element with the same path (getName) but loaded using different
     * config servers will NOT be 'equal' in the eyes of the Config Layer.
     *
     * @return  The <code>ConfigServer</code> that loaded this config element.
     */
    public IConfigServer getConfigServer();

    /**
     * Calling this method causes this config element to be re-loaded from the
     * Directory Service.
     *
     * @throws ConfigServiceException  If the reload and subsequent refresh
     *                                 fails for whatever reason. Any underlying
     *                                 DS exception will be nested inside this
     *                                 exception.
     */
    public void refresh()
        throws ConfigServiceException;
    //-------------------------------------------------------------------------
    //
    // Blob support
    //
    //-------------------------------------------------------------------------

    /**
     * Set an InputStream for this config element. The contents of the
     * InputStream will be stored to the DS when the configElement is stored.
     *
     * @param stream                   The InputStream from which the blob
     *                                 contents will be stored to the DS.
     * @throws ConfigServiceException  An exception if the stream fails to
     *                                 attach to the config element.
     */
    public void setInputStream(InputStream stream)
        throws ConfigServiceException;

    /**
     * Get the InputStream for the blob attached to this config element.
     *
     * @return                         An InputStream from which the blob can
     *                                 be read/extracted.
     * @throws ConfigServiceException  Exception is thrown if it fails to get
     *                                 the blob as an InputStream, e.g. there
     *                                 is no blob.
     */
    public InputStream getInputStream()
        throws ConfigServiceException;

    //-------------------------------------------------------------------------
    //
    // Back Referencing support
    //
    //-------------------------------------------------------------------------

    /**
     *
     * Returns the list of attributes that reference this config element.
     *
     * Note: each ConfigPath returned contains both the referencing element
     * and the path of the attribute inside the referencing element.
     * Use IConfigPath::getFirstComponent to get the referencing element path.
     *
     * Also, this feature has to be enabled on the DS...and operates on a
     * per-config type basis. e.g. back reference support might be turned on
     * for Broker configurations but off for everything else.
     *
     * @return                         A list of ConfigPath's that reference
     *                                 this config element. If there are no
     *                                 references then an empty array is returned.
     * @throws ConfigServiceException  If the method fails to get the references
     *                                 for whatever reason.
     */
    public IConfigPath[] getReferences()
        throws ConfigServiceException;

    //-------------------------------------------------------------------------
    //
    // Meta-Attribute support
    //
    //-------------------------------------------------------------------------

    /**
     * Indicates whether or not the meta attributes for this config element
     * have been modified (locally).
     *
     * @return  <code>true</code> if the meta-attributes have changed; otherwise
     *          <code>false</code>.
     */
    public boolean isMetaAttributesModified();

    /**
     * Fetches the meta attributes for this config element.
     * The map will always contain at least 1 entry - keyed off_ELEMENT_IDENTITY.
     *
     * @return                         A map of the meta attributes.
     * @throws ConfigServiceException  An exception if the meta attributes
     *                                 could not be obtained.
     */
    public HashMap getMetaAttributes()
        throws ConfigServiceException;

    /**
     * Sets a new map of meta attributes for this config element. The new map
     * will override any previous meta attributes set on this element.
     *
     * @param attributes  A <code>HashMap</code> of meta attributes to be
     *                    set on this config element.
     */
    public void setMetaAttributes(HashMap attributes);

    //-------------------------------------------------------------------------
    //
    // Template / Instance support
    //
    //-------------------------------------------------------------------------

    /**
     * Determines if this config element is an instance of a prototype (template).
     *
     * @return  Returns <code>true</code> if it is an instance; otherwise
     *          <code>false</code>.
     */
    public boolean isPrototypeInstance();

    /**
     * If this config element is an instance then this method will return the
     * prototype (template) from which it was derived.
     *
     * @return  Returns the prototype (<code>ConfigBean</code>) from which this
     *          instance config element was derived; or <code>null</code> if
     *          this config element is not an instance and has no prototype.
     */
    public IConfigPrototype getPrototype();

    /**
     * If this config element is an instance (of a prototype) then this method
     * will create a clone of this config element, but with the name specified
     * by the parameter.
     *
     * @param name  The name (fully qualified path) that the new instance
     *              config element is to be called.
     * @return      The cloned (copy) instance config element which should be
     *              identical to this config element in that it will contain
     *              the same attributes and have the same prototype config
     *              element.
     */
    public Object clonePrototypeInstance(String name);

    /**
     *
     * Removes the binding between the prototype config element and this
     * config element. Only works if this config element is an instance of a
     * prototype (template).
     *
     * @throws ConfigServiceException  If the config element is not an instance
     *                                 or if (for some reason) the config
     *                                 element has already been unshared.
     */
    public void unshare()
        throws ConfigServiceException;

    /**
     * Creates a new configuration object marked as a template (prototype) from
     * which instances can be created.
     *
     * @param name                     The name (logical path) of the new
     *                                 prototype configuration.
     * @return                         The new prototype (template) - an exact
     *                                 copy of this element.
     * @throws ConfigServiceException  An exception if it failed to create a
     *                                 prototype.
     */
    public IConfigPrototype createPrototype(String name)
        throws ConfigServiceException;
    
    /**
     * Gets the date/time when this file element was created.
     *
     * @return  The <code>long</code> value representing the creation date/time.
     */
    public long getCreationTime();
}
