/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.mx.config;

import java.util.Iterator;

/**
 * <p>Title: IConfigPath</p>
 * <p>Description: The IConfigPath represents the path through a series of nested
 * AttributeMap's to a particular point in either a IConfigBean or IConfigType
 * structure.</p>
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: Sonic Software</p>
 * @author not attributable
 * @version 1.0
 */
public interface IConfigPath extends Cloneable
{
    /**
     * Returns a clone of this configuration path.
     *
     * @return  Copy of this configuration path.
     */
    public Object clone();

    /**
     * Append the supplied array of components to the end of this configuration path.
     *
     * @param   path Array of components to appended to this path.
     * @return  This configuration path with <code>path</code> appended to it.
     */
    public IConfigPath append(String[] path);

    /**
     * Append the supplied string of components to the end of this configuration
     * path. Components in the string are separated by '.' character.
     *
     * @param   path String of components to appended to this path.
     * @return  This configuration path with <code>path</code> appended to it.
     */
    public IConfigPath append(String path);

    /**
     * Append the supplied path to the end of this configuration path.
     *
     * @param   path Configuration path to append to this path
     * @return  This configuration path with <code>path</code> appended to it.
     */
    public IConfigPath append(IConfigPath path);

    /**
     * Append a placeholder component to the end of this configuration path.
     * Placeholder components are special markers placed in a configuration
     * path to indicate where subsequent substitutions will occur.
     *
     * @return  This configuration path with a placeholder component appended to it.
     * @see     com.sonicsw.mx.config.IConfigPath#substitute
     */
    public IConfigPath appendPlaceholder();

    /**
     * Insert the supplied string of components into this configuration path at the specified inded.
     *
     * @param index The index at which to insert <code>path</code>.
     * @param path  The string of components to insert into this path
     * @return      This configuration path with <code>path</code> appended to it.
     */
    public IConfigPath insert(int index, String path);

    /**
     * Insert the supplied array of components into this configuration path at the specified inded.
     *
     * @param index  The index at which to insert <code>path</code>.
     * @param path   The array of components to insert into this path
     * @return       This configuration path with <code>path</code> appended to it.
     */
    public IConfigPath insert(int index, String[] path);

    /**
     * Insert the supplied path into this configuration path at the specified inded.
     *
     * @param index  The index at which to insert <code>path</code>.
     * @param path   The configuration path to insert into this path
     * @return       This configuration path with <code>path</code> appended to it.
     */
    public IConfigPath insert(int index, IConfigPath path);

    /**
     * Returns the first component in this configuration path.
     *
     * @return  The first component in path.
     */
    public String getFirstComponent();

    /**
     * Returns the component at the specified index in this configuration path.
     *
     * @param index  The index of requested component.
     * @return       The component at the specified index.
     */
    public String getComponent(int index);

    /**
     * Set the supplied component at the specified index in this configuration path.
     * The previous component at that index is removed and discarded.
     *
     * @param index      The index in path where the component is set.
     * @param component  The component set in path.
     */
    public void setComponent(int index, String component);

    /**
     * Inserts the supplied component at the specified index in configuration path.
     * Each component in this path with an index greater
     * or equal to the specified index is shifted upward to have an
     * index one greater than the value it had previously.
     *
     * @param index      The index in path where the component is inserted.
     * @param component  The component inserted into path.
     */
    public void insertComponent(int index, String component);

    /**
     * Removes the component at the specified index in configuration path.
     * Each component in this path with an index greater
     * or equal to the specified index is shifted downward to have an
     * index one less than the value it had previously.
     *
     * @param index   The index of the component removed.
     */
    public void removeComponent(int index);

    /**
     * Returns the last component in path.
     *
     * @return  The last component in path.
     */
    public String getLastComponent();

    /**
     * Creates a copy of this configuration path with the
     * first placeholder component substituted with the
     * the supplied component
     *
     * @param component  Component to be substituted.
     * @return           A configuration path with <code>component</code> substituted.
     */
    public IConfigPath substitute(String component);

    /**
     * Creates a new IConfigPath based on the current instance
     * but substitutes the  elements with the
     * supplied new elements.
     *
     * @param element  An array of components to be substituted.
     * @return         A configuration path with <code>components</code> substituted.
     */
    public IConfigPath substitute(String[] components);

    /**
     * Returns a new configuration path that is a sub-path of this path.
     * The subpath begins with the component at the specified index and extends
     * to the end of this path.
     *
     * @param beginIndex  The beginning index, inclusive.
     * @return            The specified sub-path.
     */
    public IConfigPath subPath(int beginIndex);

    /**
     * Returns a new configuration path that is a sub-path of this path.
     * The sub-path begins at the specified beginIndex and extends to the
     * componenet at index endIndex - 1. Thus the length of the sub-path is
     * endIndex-beginIndex.
     *
     * @param beginIndex  The beginning index, inclusive.
     * @param endIndex    The ending index, exclusive.
     * @return            The specified sub-path.
     */
    public IConfigPath subPath(int beginIndex, int endIndex);

    /**
     * Returns an iterator over the components in this configuration path in proper sequence.
     *
     * @return  An iterator over the componets in this configuration path in proper sequence.
     */
    public Iterator iterator();

    /**
     * Returns the number of components in this configuration path.
     *
     * @return  The number of components in this configuration path.
     */
    public int size();

   /**
    * Returns a string representation of the object.
    *
    * @return  The string representation of the object.
    */
    @Override
    public String toString();

}

