/*
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.mx.config;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

import com.sonicsw.mf.common.config.query.Query;

public interface IConfigServer
{
    public final static String FOLDER_NAME      = "_FOLDER_NAME";
    public final static String ELEMENT_IDENTITY = "_ELEMENT_IDENTITY";
    public final static String IS_COMPLEX       = "_IS_COMPLEX";

    /**
     * Returns true if this server is writeable.
     *
     * A writeable server allows configurations to be changed and stored back
     * to the DS.
     * All server's are writeable by default...when creating a new server it
     * is possible to make it read-only.
     *
     * @return  true if this server is writeable.
     * @see     ConfigFactory#connectConfigServer
     */
    public boolean isWriteable();

    /**
     * Returns true if this server is transacted.
     *
     * A transacted config server stores all changes at the client until a
     * <code>commit</code> is called and then sends all the modifications
     * back to the DS in one go.
     *
     * Not only is this more efficient but the modifications are made atomically
     * on the DS meaning that if a single item in the transaction fails to
     * execute correctly on the DS then the entire transaction is rolled back.
     *
     * @return  true if this server is transacted.
     * @see     ConfigFactory#connectTransactedConfigServer
     */
    public boolean isTransacted();

    /**
     * Stores an element back to the Directory Service.
     *
     * @param configElement            The element to be stored.
     * @throws ConfigServiceException  A wrapped exception if a failure occurs
     *                                 while storing the element.
     */
    public void storeConfigElement(IConfigElement configElement)
        throws ConfigServiceException;

    /**
     * This method stores the provided array of elements back to the Directory
     * Service AND deletes a separate grouping of elements named in the string
     * array.
     *
     * The method performs the store and delete as an atomic operation.
     *
     * @param configElements           An array of elements to be stored.
     * @param deleteElements           An array of logical paths delimiting the
     *                                 elements that are to be deleted.
     * @throws ConfigServiceException  A wrapped exception if failure occurs
     *                                 while performing the operation.
     */
    public void storeConfigElements(IConfigElement[] configElements, String[] deleteElements)
        throws ConfigServiceException;
    
    /**
     * Method fetches the configuration element specified by the logical path
     * 'configElementName' into the Config Layer.
     *
     * @param configElementName        The path to the element.
     * @return                         If the element is typed then a <code>IConfigBean</code>
     *                                 is returned otherwise it will be a
     *                                 <code>IConfigElement</code>.
     * @throws ConfigServiceException  A wrapped exception describing any problem
     *                                 that occurs while fetching the element -
     *                                 the most common being that the element
     *                                 could not be found.
     */
    public IConfigElement loadConfigElement(String configElementName)
        throws ConfigServiceException;

    /**
     *
     * @param configElementName
     * @return
     * @throws ConfigServiceException
     */
    public IConfigElement loadLocalConfigElement(String configElementName)
        throws ConfigServiceException;

    /**
     * Performs a query in the Directory Service that might result in a match
     * on a number of configuration elements. Elements that match the query
     * are loaded and returned in a set.
     *
     * @param query                    A query construct ({@link com.sonicsw.mf.common.config.query.Query}).
     * @return                         A set of <code>IConfigElement</code>'s
     *                                 Typed elements are returned as the subclass
     *                                 <code>IConfigBean</code>.
     * @throws ConfigServiceException  A wrapped exception if a failure occurs.
     */
    public Set loadConfigElements(Query query)
        throws ConfigServiceException;

    /**
     * Load all configuration elements in the specified logical folder
     * @param path 
     * @return A set of configurations
     * @throws ConfigServiceException
     */
    public Set loadConfigElements(String path)
        throws ConfigServiceException;

    /**
     * Returns a list of all elements that match the criteria specified by the
     * query object, see {@link com.sonicsw.mf.common.config.query.Query Query}.
     *
     * @param query                    The query object upon which matching
     *                                 elements are to be returned.
     * @return                         A set of all elements that matched the
     *                                 query.
     * @throws ConfigServiceException  An exception if the operation fails.
     */
    public Set listConfigElements(Query query)
        throws ConfigServiceException;

    /**
     * Method fetches the configuration type element specified by the logical
     * path 'configBeanTypeName' into the Config Layer.
     *
     * @param configBeanTypeName       The path to the type element.
     * @param version                  Specifies which version to load. This
     *                                 string parameter should be in the format
     *                                 "n.n<.n>".
     * @return                         If the load was successful then a
     *                                 <IConfigType> is returned.
     * @throws ConfigServiceException  A wrapped exception describing any problem
     *                                 that occurs while fetching the typed element.
     */
    public IConfigType loadConfigType(String configBeanTypeName, String version)
        throws ConfigServiceException;

    /**
     * @deprecated  Never used
     */
    public Set loadConfigTypes(Query query)
       throws ConfigServiceException;

    /**
     * Removes the element described by the path from the Directory Service.
     *
     * @param configElementName        The logical path to the element.
     * @throws ConfigServiceException  A wrapped exception if the element was
     *                                 not deleted.
     */
    public void removeConfigElement(String configElementName)
        throws ConfigServiceException;

    /**
     * Removes a number of elements from the Directory Service.
     *
     * The operation is atomic - if just one of the paths does not point
     * to a valid element then the entire operation will fail.
     *
     * @param configElements           An array of logical paths to elements
     *                                 earmarked for deletion.
     * @throws ConfigServiceException  A wrapped exception if the remove operation
     *                                 fails.
     */
    public void removeConfigElements(String[] configElements)
        throws ConfigServiceException;

    /**
     * Creates a new config element with the given path (and name) in the
     * Directory Service.
     *
     * @param name                     The fully qualified path that will be
     *                                 the logical location of the new element
     *                                 in the Directory Service.
     * @return                         The newly created <code>IConfigElement</code>.
     * @throws ConfigServiceException  An exception if the operation failed.
     */
    public IConfigElement createConfigElement(String name)
        throws ConfigServiceException;

    /**
     * Creates a new config bean, i.e. a typed config element with the given
     * path (name).
     *
     * Note: a config bean has to be typed and versioned.
     *
     * @param name                     The fully qualified path that will be
     *                                 the logical location of the new element
     *                                 in the Directory Service.
     * @param type                     The type of element (<code>IConfigBean</code>)
     *                                 being created.
     * @param version                  Types are versioned so we must specify
     *                                 which particular configuration version
     *                                 of the type is to be created.
     * @return                         The newly created <code>IConfigBean</code>.
     * @throws ConfigServiceException  An exception if the operation failed.
     * @deprecated                     Use other createconfigBean method.
     */
    public IConfigBean createConfigBean(String name, String type, String version)
        throws ConfigServiceException;

    /**
     * Creates a new config bean, i.e. a typed config element with the given
     * path (name).
     *
     * Note: a config bean has to be typed and versioned.
     *
     * @param name                     The fully qualified path that will be
     *                                 the logical location of the new element
     *                                 in the Directory Service.
     * @param type                     The type of element (<code>IConfigBean</code>)
     *                                 being created.
     * @param version                  Types are versioned so we must specify
     *                                 which particular configuration version
     *                                 of the type is to be created.
     * @param isTemplate               Determines whether or not the bean will
     *                                 be marked as a template or not.
     * @return                         The newly created <code>IConfigBean</code>.
     * @throws ConfigServiceException  An exception if the operation failed.
     */
    public IConfigBean createConfigBean(String name, String type, String version, boolean isTemplate)
        throws ConfigServiceException;

    /**
     * Creates a new, empty configuration bean of type MF_FILE. This special
     * configuration element allows the storage of binary content as a blob.
     *
     * @param name                     The logical location in the DS where the
     *                                 new file element is to be created.
     * @param user                     Provides an optional 'createdBy' user
     *                                 meta attribute for identification
     *                                 purposes.
     * @return                         A <code>IConfigBeanFile</code> representing
     *                                 the newly created element.
     * @throws ConfigServiceException  An exception if the operation failed.
     */
    public IConfigBeanFile createConfigBeanFile(String name, String user)
        throws ConfigServiceException;

    /**
     * @deprecated Never Used
     */
    public Collection parseConfigTypes(String namespaceURI, String schemaDocURL)
        throws ConfigServiceException;

    /**
     * @deprecated Never Used
     */
    public IConfigBean parseConfigBean(String beanDocumentURL)
        throws ConfigServiceException;

    /**
     * Method generates a valid XML string for a bean located under the provided
     * path.
     *
     * @param path                     The path to a configuration bean for export.
     * @return                         A valid XML string representation of the bean.
     * @throws ConfigServiceException  An exception if the operation failed.
     */
    public String exportConfigBean(String name)
        throws ConfigServiceException;

    /**
     * Method generates a valid XML string for a bean.
     * @param bean                     The bean from which XML text is generated.
     * @return                         The generated XML text.
     * @throws ConfigServiceException  An exception if the operation failed.
     */
    public String exportConfigBean(IConfigBean bean)
        throws ConfigServiceException;

    /**
     * Generates a custom XML string that acts as an MF Container's boot file.
     * This method differs from the basic <code>exportConfigBean</code> in that
     * it generates a different type of XML document...the domain name parameter
     * is used to tie the boot file to a specific domain.
     *
     * @param path                     The logical path to a MF Container configuration.
     * @param domainName               The domain name that the generated boot
     *                                 file is to be targeted for.
     * @return                         The generated XML text.
     * @throws ConfigServiceException  An exception if the operation failed.
     */
    public String exportContainerBootConfiguration(IConfigElement containerConfigElement, String domainName)
        throws ConfigServiceException;

    /**
     * Generates a custom XML string that acts as an Directoy Service boot file.
     * 
     *
     * @param path                     The logical path to a MF Container configuration. 
     * @return                         The generated XML text.
     * @throws ConfigServiceException  An exception if the operation failed.
     */
    public String exportDSBootConfiguration(IConfigElement name)
        throws ConfigServiceException;
    /**
     * Method provides a way of checking for a valid path.
     *
     * A valid path is defined as either:
     *
     * 1. a path representing a folder structure where all parts of the
     *    path map directly onto a folder in the Directory Service structure.
     * 2. a path representing an element for which each folder part of the path
     *    maps directly onto a folder and the final part of the path maps
     *    directly onto a configuration element.
     *
     * @param path  A path to a folder or an element.
     * @return      returns true if the path "exists", i.e. maps onto a folder
     *              or element, otherwise returns false.
     */
    public boolean pathExists(String path);

    /**
     * Creates the path (series of folders) in the Directory Service.
     *
     * The behavior is slightly unusual if one or more of the (sub) folders
     * specified in the path already exist - the last component of the path
     * must NOT currently exist in the DS before the request is made otherwise
     * the operation will fail.
     *
     * Basically, this method is useful when it must be guaranteed that the
     * last folder in the path (deepest sub-folder) does not already exist
     * and is then created.
     *
     * @param path                     The path being created.
     * @throws ConfigServiceException  An exception if the operation fails.
     * @deprecated                     Use <code>createFolder</code> instead.
     */
    public void createPath(String path)
        throws ConfigServiceException;

    /**
     * This method is the same as {@link createPath(String path) createPath}
     * except there is an option of overriding the check on the last component
     * / folder of the path.
     *
     * @param path                     The path being created.
     * @param existingOK               If true then no exception will occur
     *                                 even if the last component of the path
     *                                 already exists, i.e. a folder exists
     *                                 in the Directory Service structure.
     * @throws ConfigServiceException  An exception if the operation fails.
     * @deprecated                     Use <code>createFolder</code> instead.
     */
    public void createPath(String path, boolean existingOK)
        throws ConfigServiceException;

    /**
     * Creates the path (series of folders) in the Directory Service.
     *
     * If the path specifies the creation of a folder nested inside other
     * folders then if any of those folders do NOT exist, the operation will
     * fail. In other words, this method doesn't create any intermediary folder
     * structure.
     *
     * If the intermediary folder structure exists then the method attempts to
     * create the inner-most folder - this folder must NOT already exist.
     *
     * Basically, this method is useful when it must be guaranteed that the
     * last folder in the path (deepest sub-folder) does not already exist
     * and is then created.
     *
     * @param path                     The path being created.
     * @throws ConfigServiceException  An exception if the operation fails.
     */
    public void createFolder(String path)
        throws ConfigServiceException;

    /**
     * Atomic operation that creates a path (one or more nested folders) in the
     * Directory Service along with meta attributes.
     *
     * This is useful because often listeners to change notifications attempt
     * to fetch meta attributes on a create folder notification. Doing both
     * the create and meta attribute set in the same operation means that the
     * meta attributes are immediately available to any listener.
     *
     * @param path                     The path (of folders) being created.
     * @param metaAttributes           A group of key-value mappings that are
     *                                 to be assosiated (set) on this path, i.e.
     *                                 the last component (folder) in the path.
     * @param createParentDirs         If true, then any missing intermediary
     *                                 folders will be automatically created.
     * @throws ConfigServiceException  An exception if the operation fails.
     */
    public void createFolder(String path, Map metaAttributes, boolean createParentDirs)
        throws ConfigServiceException;

    /**
     * Deletes a folder.
     *
     * Only empty folders can be deleted, i.e. if a folder contains sub-folders
     * or elements then the delete will fail. Must first delete the sub-folders
     * and elements.
     *
     * Config Configuration, e.g. Domain Authorizations and Policy Authentications,
     * can be successfully deleted when non-empty.
     *
     * @param path                     Path specifying folder to be deleted.
     * @throws ConfigServiceException  An exception if the operation fails.
     */
    public void deleteFolder(String path)
        throws ConfigServiceException;

    /**
     * Renames a folder or element.
     *
     * Note: that this operation will fail if the rename is applied to a folder
     *       or element that is part of a complex configuration.
     *
     * @param oldName                  Path pointing to the old folder/element.
     * @param newName                  Path pointing to new folder/element.
     * @throws ConfigServiceException  An exception if the operation fails.
     */
    public void rename(String oldName, String newName)
        throws ConfigServiceException;


    /**
     * Gets any meta attributes currently set on the Directory Service structure
     * specified by the path.
     *
     * @param path                     A path pointing to the structure (folder
     *                                 or element) from which any meta attributes
     *                                 are fetched.
     * @return                         A <code>Map</code> of key / value string
     *                                 pairs representing the meta attributes.
     * @throws ConfigServiceException  An exception if the operation fails.
     */
    public Map getMetaAttributes(String path)
        throws ConfigServiceException;

    /**
     * Sets a group of key/value meta attributes on the structure specified
     * by the path.
     * Note: that the supplied <code>metaAttributes</code> override any existing
     *       meta attributes stored on the structure.
     *
     * @param path                     The path underwhich the meta attributes
     *                                 are to be stored.
     * @param metaAttributes           The meta attributes.
     * @throws ConfigServiceException  An exception if the set operation fails.
     */
    public void setMetaAttributes(String path, Map metaAttributes)
        throws ConfigServiceException;

    /**
     * Lists all folders and elements found under the structure specified by path.
     *
     * @param path                     Logical path pointing to a DS structure.
     * @return                         A set of <code>Map</code>'s where each
     *                                 map is the meta-attributes for a folder
     *                                 or element. Checking each map's keys
     *                                 for either <code>FOLDER_NAME</code> or
     *                                 <code>ELEMENT_IDENTITY</code> will
     *                                 determine if the map represents a folder
     *                                 or element.
     * @throws ConfigServiceException  An exception if the operation fails.
     */
    public Set list(String path)
        throws ConfigServiceException;

    /**
     * Fetches all the folders that live underneath a particular parent folder.
     *
     * @param path                     The parent folder
     * @return                         A set of logical path strings to all
     *                                 the folders found under the parent.
     * @throws ConfigServiceException  An exception if the operation fails.
     */
    public Set listFolders(String path)
        throws ConfigServiceException;

    /**
     * Fetches all the elements that live underneath a particular parent folder.
     *
     * @param path                     The parent folder.
     * @return                         A set of logical path strings to all the
     *                                 elements found under the parent folder.
     * @throws ConfigServiceException  An exception if the operation fails.
     */
    public Set listConfigElements(String path)
        throws ConfigServiceException;

    //-------------------------------------------------------------------------
    //
    // Transaction Support
    //
    //-------------------------------------------------------------------------

    /**
     * Commits (persists) multiple changes made in the server as part of a
     * single transaction.
     *
     * @throws ConfigServiceException  An exception if the commit fails.
     */
    public void commit()
        throws ConfigServiceException;

    /**
     * Rolls back (undoes) any changes made in the server to local and new
     * configurations.
     *
     * @throws ConfigServiceException  An exception if the rollback could not
     *                                 complete successfully.
     */
    public void rollback()
        throws ConfigServiceException;

    /**
     * Flushing results in all changes being thrown away.
     *
     * @throws ConfigServiceException  An exception if the operation fails.
     */
    public void flush()
        throws ConfigServiceException;

    /**
     * Creates a new config server that is a sub transaction - the server is
     * parented to another server. When the sub transacted config server is
     * committed, changes are merged back into the parent server.
     *
     * Only when the parent server is also committed will the changes be set
     * to the DS.
     *
     * @return                         A new server that acts as a sub-transaction
     *                                 of the parent config server.
     * @throws ConfigServiceException  A wrapped exception if the server is unable
     *                                 to create a sub-transacted version of itself.
     */
    public IConfigServer subtransaction()
        throws ConfigServiceException;

    //-------------------------------------------------------------------------

    /**
     * Creates a new server but with the type cached off the parent server.
     * Results in slightly optimized behavior that reduces type loading.
     *
     * @return                         A new config server.
     * @throws ConfigServiceException  If the method failed to create a new
     *                                 server then a wrapped exception is returned.
     */
    public IConfigServer connect()
        throws ConfigServiceException;

    /**
     * Creates a new transacted server but with the type cached off the parent
     * server.
     * Results in slightly optimized behavior that reduces type loading.
     *
     * @return                         A new transacted config server.
     * @throws ConfigServiceException  If the method failed to create a new
     *                                 server then a wrapped exception is returned.
     */
    public IConfigServer connectTransacted()
        throws ConfigServiceException;

    //-------------------------------------------------------------------------

    /**
     * Registers a listener to recieve change notifications. If the filter and
     * handback are non-null then
     * @param configElementName  The name (logical path) of the element on
     *                           which change notifications are to be forwarded.
     * @param listener           The listener implementor that will receive
     *                           the change notifications.
     * @param filter             An interface that can be used to filter out
     *                           unwanted changes.
     * @param handback           A listener specific handback object that can
     *                           be used to uniquely identify a particular
     *                           listener registration. See {@link ConfigChange}
     */
    public void addConfigChangeListener(String configElementName,
                                        IConfigChangeListener listener,
                                        IConfigChangeFilter filter,
                                        Object handback);

    /**
     * Unregisters / unsubscribes the listener from receiving change notification
     * events for the specified configuration (<code>configElementName</code>).
     * If a specific filter and handback object are provided then that specific
     * subscription is removed. Otherwise, all subscriptions for that specific
     * listener will be removed.
     *
     * @param configElementName        The name (logical path) of the element on
     *                                 which notifications are to be unregistered.
     * @param listener                 The listener that was receiving notifications.
     * @param filter                   The filter (if any).
     * @param handback                 The listener-specific unique handback object.
     * @throws ConfigServiceException  An exception if the removal did not
     *                                 complete successfully.
     */
    public void removeConfigChangeListener(String configElementName,
                                           IConfigChangeListener listener,
                                           IConfigChangeFilter filter,
                                           Object handback)
        throws ConfigServiceException;

    /**
     * Unregisters / unsubscribes the listener from receiving change notification
     * events for the specified configuration (<code>configElementName</code>).
     * All subscriptions for that specific listener will be removed.
     *
     * @param configElementName        The name (logical path) of the element on
     *                                 which notifications are to be unregistered.
     * @param listener                 The listener that was receiving notifications.
     * @throws ConfigServiceException  An exception if the removal did not
     *                                 complete successfully.
     */
    public void removeConfigChangeListener(String configElementName,
                                           IConfigChangeListener listener)
        throws ConfigServiceException;

    /**
     * Converts a storage (physical DS) path into a logical path.
     *
     * @param name                     The storage path being converted.
     * @return                         The logical path - if a mapping exists.
     * @throws ConfigServiceException  A wrapped exception if the convertion
     *                                 fails. Reason for failure might be that
     *                                 a storage-to-logical mapping doesn't exist.
     */
    public String storageToLogical(String name)
        throws ConfigServiceException;

    /**
     * Converts a logical path into a storage (physical DS) path.
     *
     * @param name                     The logical path being converted.
     * @return                         The storage path - if a mapping exists.
     * @throws ConfigServiceException  A wrapped exception is thrown if the
     *                                 conversion fails.
     */
    public String logicalToStorage(String name)
        throws ConfigServiceException;

    /**
     * Method finds all places in the DS where the specified element pointed
     * to by the path parameter is referenced.
     *
     * NOTE: that this method is NOT guaranteed to return a complete list because
     *       it is dependent on the feature being enabled in the DS (on a per
     *       configuration type basis).
     *
     * @param path                     The path to the element that we want to
     *                                 find references too.
     * @return                         An array of fully-qualified <code>IConfigPath</code>'s
     *                                 that point to all the elements that
     *                                 reference the element specified in the path.
     * @throws ConfigServiceException  A wrapped exception if the DS / Config Layer
     *                                 is unable to complete the operation.
     */
    public IConfigPath[] getReferences(String configElementName)
        throws ConfigServiceException;

    /**
     * Call this method when you want to close and cleanup the server.
     *
     * The implementation performs cleanup and should unregister any subscriptions
     * and/or notifications.
     *
     * @throws ConfigServiceException  An exception is thrown to wrap any failures
     *                                 that occur during cleanup.
     */
    public void close()
        throws ConfigServiceException;
}
