/*
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.mx.config.impl;

import java.math.BigDecimal;
import java.util.Date;
import java.util.ListIterator;
import java.util.Set;

import org.apache.xerces.impl.dv.InvalidDatatypeFacetException;
import org.apache.xerces.impl.dv.InvalidDatatypeValueException;
import org.apache.xerces.impl.dv.SchemaDVFactory;
import org.apache.xerces.impl.dv.ValidatedInfo;
import org.apache.xerces.impl.dv.ValidationContext;
import org.apache.xerces.impl.dv.XSFacets;
import org.apache.xerces.impl.dv.XSSimpleType;
import org.apache.xerces.impl.dv.util.HexBin;
import org.apache.xerces.impl.validation.ValidationState;

import com.sonicsw.mx.config.ConfigAttributeException;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.ConfigServiceRuntimeException;
import com.sonicsw.mx.config.ConfigTypeException;
import com.sonicsw.mx.config.IAttributeDescription;
import com.sonicsw.mx.config.IAttributeList;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IConfigElement;
import com.sonicsw.mx.config.IConfigPath;
import com.sonicsw.mx.config.IConfigType;

import com.sonicsw.mf.common.config.IAttributeSet;

public class AttributeDescriptionImpl
implements IAttributeDescription
{
    /** The config type that contains and owns this attribute description.
     *  An attribute description can reside within only one config type
     *  at a time.
     */
    protected IConfigType m_owner = null;

    /** Text Description of Attribute.
     */
    protected String m_description = null;

    /** The Java Class Type of the Attribute.
     */
    protected Class  m_type = null;

    /** The Properties of the Attribute.
     */
    protected PropertyMapImpl m_properties = null;

    /** The Attribute Descriptions of Sub Attributes for composite type
     *  (IAttributeMap and IAttributeList).
     */
    protected AttributeDescriptionMapImpl m_attributeDescriptions = null;

    /** A Xerces Simple Type used to validate runtime objects and convert string
     *  values into runtime objects.
     */
    protected XSSimpleType m_xsSimpleType = null;

    /** The Config Server this Attribute Description is associated with.
     */
    protected ConfigServer m_configServer = null;

    /*  Attribute Description Attribute Name.
        Name of the attribute in the attribute description's underlying DS Set
        storing the attribute description's attribute descriptions. */
    protected static final String ATTRIBUTE_DESCRIPTIONS_ATTRIBUTE = "attributeDescriptions";

    /*  Type Attribute Name.
        Name of the attribute in the attribute description's underlying DS Set
        storing the attribute description's type attribute. */
    protected static final String TYPE_ATTRIBUTE = "type";

    /*  Description Attribute Name.
        Name of the attribute in the attribute description's underlying DS Set
        storing the attribute description's description attribute. */
    protected static final String DESCRIPTION_ATTRIBUTE = "description";

    /*  Properties Attribute Name.
        Name of the attribute in the attribute description's underlying DS Set
        storing the cattribute description's propeties */
    protected static final String PROPERTIES_ATTRIBUTE = "properties";
    
    protected AttributeDescriptionImpl()
    {
    }

    protected AttributeDescriptionImpl(ConfigServer configServer)
    {
        m_configServer = configServer;
        m_attributeDescriptions = new AttributeDescriptionMapImpl(m_configServer);
        m_properties = new PropertyMapImpl(m_configServer);
    }

    protected AttributeDescriptionImpl(IAttributeSet description, ConfigServer configServer)
    {
        init(description, configServer);
    }

    protected final void
    init(IAttributeSet dsAttributeSet, ConfigServer configServer)
    {
        try
        {
            /* Initialize reference to config server. */
            m_configServer = configServer;

            /* Initialize Attribute Descriptions. */
            IAttributeSet attributeDescrs = (IAttributeSet)dsAttributeSet.getAttribute(ATTRIBUTE_DESCRIPTIONS_ATTRIBUTE);
            if (attributeDescrs == null)
            {
                attributeDescrs = dsAttributeSet.createAttributeSet(ATTRIBUTE_DESCRIPTIONS_ATTRIBUTE);
            }
            m_attributeDescriptions = new AttributeDescriptionMapImpl(attributeDescrs, m_configServer);

            /* Initialize Properties. */
            IAttributeSet properties = (IAttributeSet)dsAttributeSet.getAttribute(PROPERTIES_ATTRIBUTE);
            if (properties == null)
            {
                properties = dsAttributeSet.createAttributeSet(PROPERTIES_ATTRIBUTE);
            }
            m_properties = new PropertyMapImpl(properties, m_configServer);

            /* Initialize Description Attribute. */
            m_description = (String)dsAttributeSet.getAttribute(DESCRIPTION_ATTRIBUTE);

            /* Initialize Attribute Type if specified. */
            String typeValue = (String)dsAttributeSet.getAttribute(TYPE_ATTRIBUTE);
            if (typeValue != null)
            {
                m_type = Class.forName(typeValue);
            }
        }
        catch (Exception e)
        {
            throw new ConfigServiceRuntimeException("ad-init-failed", e);
        }
    }

    @Override
    public String
    toString()
    {
        return toString(0);
    }


    public String
    toString(int indent)
    {
        StringBuffer buffer = new StringBuffer();
        StringBuffer indentBuf = new StringBuffer();
        for (int i = 0; i < indent; i++)
        {
            indentBuf.append("    ");
        }
        String indentStr = indentBuf.toString();

        String className = getType().getName();
        String typeName  = className.substring(className.lastIndexOf(".") + 1);

        buffer.append("[" + typeName + "] =\n");
        buffer.append(indentStr).append("{\n");

        if (getDescription() != null)
        {
            buffer.append(indentStr).append("    ").append("description = " + getDescription() + "\n");
        }

        if (!m_properties.isEmpty())
        {
            buffer.append(indentStr).append("    ").append("properties =\n");
            buffer.append(indentStr).append("    ").append("{\n");
            buffer.append(((PropertyMapImpl)m_properties).toString(indent + 2));
            buffer.append(indentStr).append("    ").append("}\n");
        }

        if (getAttributeDescriptions().size() > 0)
        {
            buffer.append(indentStr).append("    ").append("attributes =\n");
            buffer.append(indentStr).append("    ").append("{\n");
            buffer.append(((AttributeDescriptionMapImpl)getAttributeDescriptions()).toString(indent + 2));
            buffer.append(indentStr).append("    ").append("}\n");
        }

        buffer.append(indentStr).append("}\n");

        return buffer.toString();
    }


    @Override
    public Object
    clone()
    {
        try
        {
            /*  First make shallow copy of description */
            AttributeDescriptionImpl copy = (AttributeDescriptionImpl) super.clone();

            /*  Clones are not initially owned by any config type. */
            copy.m_owner = null;

            /*  Deep copy properties    */
            copy.m_properties = (PropertyMapImpl) m_properties.clone();

            /* Deep copy attribute descriptions */
            copy.m_attributeDescriptions = (AttributeDescriptionMapImpl) m_attributeDescriptions.clone();

            /*  Clone must reinitialize its xerces simple type validator */
            m_xsSimpleType = null;

            return copy;
        }
        catch (CloneNotSupportedException e)
        {
            throw new ConfigServiceRuntimeException("ad-clone-failed", e);
        }
    }

    @Override
    public String
    getDescription()
    {
        return m_description;
    }

    public void
    setDescription(String description)
    {
        m_description = description;
    }

    @Override
    public Class
    getType()
    {
        return m_type;
    }

    public void
    setType(Class type)
    {
        if (type == null)
        {
            throw new ConfigServiceRuntimeException("ad-set-type-is-null");
        }

        m_type = type;
    }

    @Override
    public Object
    getProperty(String propertyName)
    {
        if (m_properties != null)
        {
            return m_properties.getProperty(propertyName);
        }
        return null;
    }

    public void
    setProperty(String propertyName, Object propertyValue)
    throws ConfigServiceException
    {
        if (m_properties != null)
        {
            m_properties.setProperty(propertyName, propertyValue);
            if (m_owner != null)
            {
                ((ConfigTypeImpl)m_owner).setModified();
            }
        }
    }

    public Object
    removeProperty(String propertyName)
    throws ConfigServiceException
    {
        if (m_properties != null)
        {
            Object obj = m_properties.removeProperty(propertyName);
            if (m_owner != null && obj != null)
            {
                ((ConfigTypeImpl)m_owner).setModified();
            }
            return obj;
        }
        return null;
    }

    @Override
    public Set
    getPropertyNames()
    {
        return m_properties.getPropertyNames();
    }

    public AttributeDescriptionMapImpl
    getAttributeDescriptions()
    {
        return m_attributeDescriptions;
    }

    @Override
    public IAttributeDescription
    getAttributeDescription(String attributeName)
    {
        return m_attributeDescriptions.getAttributeDescription(attributeName);
    }

    public void
    setAttributeDescription(String attributeName, IAttributeDescription attributeValue)
    throws ConfigServiceException
    {
        ((AttributeDescriptionMapImpl)m_attributeDescriptions).setAttributeDescription(attributeName, attributeValue);
    }

    public IAttributeDescription
    removeAttributeDescription(String attributeName)
    throws ConfigServiceException
    {
        return ((AttributeDescriptionMapImpl)m_attributeDescriptions).removeAttributeDescription(attributeName);
    }

    @Override
    public Set
    getAttributeDescriptionNames()
    {
        return m_attributeDescriptions.getAttributeDescriptionNames();
    }

    @Override
    public IAttributeDescription
    getAttributeDescription(IConfigPath path)
    {
        return m_attributeDescriptions.getAttributeDescription(path);
    }

    public void
    setAttributeDescription(IConfigPath path, IAttributeDescription attributeValue)
    throws ConfigServiceException
    {
        ((AttributeDescriptionMapImpl)m_attributeDescriptions).setAttributeDescription(path, attributeValue);
    }

    public IAttributeDescription
    removeAttributeDescription(IConfigPath path)
    throws ConfigServiceException
    {
        return ((AttributeDescriptionMapImpl)m_attributeDescriptions).removeAttributeDescription(path);
    }

    @Override
    public void
    validate(Object value)
    throws ConfigServiceException
    {
        if (m_xsSimpleType == null)
        {
            m_xsSimpleType = buildSimpleType();
        }

        if (value == null)
        {
            throw new ConfigAttributeException(null, "config-attr-val-is-null");
        }

        if (m_type == IAttributeMap.class || m_type == IAttributeList.class)
        {
            throw new ConfigAttributeException(null, "config-attr-val-can-not-validate", new Object []{ value, m_type.getName() });
        }

        if (value instanceof ConfigReference)
        {
            if (m_type != IConfigElement.class)
            {
                throw new ConfigAttributeException(null, "config-attr-val-incorrect-type", new Object []{ value, IConfigElement.class.getName(), m_type.getName() });
            }
        }
        else
        if (!m_type.isInstance(value))
        {
            throw new ConfigAttributeException(null, "config-attr-val-incorrect-type", new Object []{ value, value.getClass().getName(), m_type.getName() });
        }

        try
        {
            ValidatedInfo validatedInfo = new ValidatedInfo();
            m_xsSimpleType.validate(Util.toString(value), new ValidationState(), validatedInfo);
        }
        catch (InvalidDatatypeValueException e)
        {
            throw new ConfigTypeException(e.getKey(), e.getArgs(), e);
        }
    }

    @Override
    public Object
    validate(String content)
    throws ConfigServiceException
    {
        if (m_xsSimpleType == null)
        {
            m_xsSimpleType = buildSimpleType();
        }

        if (content == null)
        {
            throw new ConfigServiceException("ad-validate-content-is-null", new Object []{ content });
        }

        if (m_type == IAttributeMap.class || m_type == IAttributeList.class)
        {
            throw new ConfigServiceException("ad-validate-content-invalid-type");
        }

        try
        {
            Object value = m_xsSimpleType.validate(content, new ValidationState(), new ValidatedInfo());

            if (getType() == Integer.class)
            {
                BigDecimal converted = new BigDecimal(value.toString());
                value = new Integer(converted.intValue());
            }
            else
            if (getType() == Long.class)
            {
                BigDecimal converted = new BigDecimal(value.toString());
                value = new Long(converted.longValue());
            }
            else
            if (getType() == BigDecimal.class)
            {
                value = new BigDecimal(value.toString());
            }
            else
            if (getType() == IConfigElement.class)
            {
                value = new ConfigReference(Util.url2Name((String) value), m_configServer);
            }
            else
            if (getType() == byte[].class)
            {
                value = HexBin.decode(value.toString()); 
            }
            else
            if (getType() == Date.class)
            {
                value = Util.convertToDate((int[])value);
            }

            return value;
        }
        catch (InvalidDatatypeValueException e)
        {
            throw new ConfigTypeException(e.getKey(), e.getArgs(), e);
        }
    }

    public void toAttributeSet(IAttributeSet attrSet)
    {
        try
        {
            attrSet.setStringAttribute(DESCRIPTION_ATTRIBUTE, getDescription());
            if (getType()!= null)
            {
                attrSet.setStringAttribute(TYPE_ATTRIBUTE, getType().getName());
            }
            IAttributeSet attributeProperties = attrSet.createAttributeSet(PROPERTIES_ATTRIBUTE);
            m_properties.toAttributeSet(attributeProperties);
            m_attributeDescriptions.toAttributeSet(attrSet.createAttributeSet(ATTRIBUTE_DESCRIPTIONS_ATTRIBUTE));
        }
        catch (Exception e)
        {
            throw new ConfigServiceRuntimeException("ad-to-attr-set-failed", e);
        }
    }


    protected XSSimpleType
    buildSimpleType()
    throws ConfigServiceException
    {
        try
        {
            PropertyMapImpl properties = m_properties;
            Object tmp = properties.getProperty(IAttributeDescription.SIMPLE_TYPE_NAME);
            if (tmp == null)
            {
                throw new ConfigServiceException("ad-build-simple-type-not-atomic");
            }

            String simpleTypeName = (String) tmp;
            XSSimpleType baseType = SchemaDVFactory.getInstance().getBuiltInType(simpleTypeName);

            XSSimpleType simpleType = SchemaDVFactory.getInstance().createTypeRestriction(null, null, (short) 0, baseType, null);

            Object property = null;
            XSFacets facets = new XSFacets();
            short presentFacets = 0;

            if ((property = properties.getProperty(IAttributeDescription.WHITESPACE_FACET)) != null)
            {
                facets.whiteSpace = ((Long)property).shortValue();
                presentFacets |= XSSimpleType.FACET_WHITESPACE;
            }

            if ((property = properties.getProperty(IAttributeDescription.LENGTH_FACET)) != null)
            {
                facets.length = ((Long)property).intValue();
                presentFacets |= XSSimpleType.FACET_LENGTH;
            }

            if ((property = properties.getProperty(IAttributeDescription.MIN_LENGTH_FACET)) != null)
            {
                facets.minLength = ((Long)property).intValue();
                presentFacets |= XSSimpleType.FACET_MINLENGTH;
            }

            if ((property = properties.getProperty(IAttributeDescription.MAX_LENGTH_FACET)) != null)
            {
                facets.maxLength = ((Long)property).intValue();
                presentFacets |= XSSimpleType.FACET_MAXLENGTH;
            }

            if ((property = properties.getProperty(IAttributeDescription.TOTAL_DIGITS_FACET)) != null)
            {
                facets.totalDigits = ((Long)property).intValue();
                presentFacets |= XSSimpleType.FACET_TOTALDIGITS;
            }

            if ((property = properties.getProperty(IAttributeDescription.FRACTION_DIGITS_FACET)) != null)
            {
                facets.fractionDigits = ((Long)property).intValue();
                presentFacets |= XSSimpleType.FACET_FRACTIONDIGITS;
            }

            if ((property = properties.getProperty(IAttributeDescription.PATTERN_FACETS)) != null)
            {
                AttributeListImpl enumerationDesc = (AttributeListImpl) property;
                ListIterator iterator = enumerationDesc.listIterator();
                while (iterator.hasNext())
                {
                    AttributeMapImpl regexDesc = (AttributeMapImpl) iterator.next();
                    if (facets.pattern == null)
                    {
                        facets.pattern = (String) regexDesc.getAttribute(IAttributeDescription.PATTERN_REGEX);
                    }
                    else
                    {
                        facets.pattern = facets.pattern + " | " + (String) regexDesc.getAttribute(IAttributeDescription.PATTERN_REGEX);
                    }
                }
                presentFacets |= XSSimpleType.FACET_PATTERN;
            }

            if ((property = properties.getProperty(IAttributeDescription.EMUMERATION_FACET)) != null)
            {
                AttributeListImpl enumerationDesc = (AttributeListImpl) property;
                ListIterator iterator = enumerationDesc.listIterator();
                while (iterator.hasNext())
                {
                    String enumeration = (String) iterator.next();
                    if(facets.enumeration == null)
                    {
                        facets.enumeration = new java.util.Vector();
                    }
                    facets.enumeration.add(enumeration);
                }
                presentFacets |= XSSimpleType.FACET_ENUMERATION;
            }

            if ((property = properties.getProperty(IAttributeDescription.MAX_INCLUSIVE_FACET)) != null)
            {
                facets.maxInclusive = property.toString();
                presentFacets |= XSSimpleType.FACET_MAXINCLUSIVE;
            }

            if ((property = properties.getProperty(IAttributeDescription.MAX_EXCLUSIVE_FACET)) != null)
            {
                facets.maxExclusive = property.toString();
                presentFacets |= XSSimpleType.FACET_MAXEXCLUSIVE;
            }

            if ((property = properties.getProperty(IAttributeDescription.MIN_EXCLUSIVE_FACET)) != null)
            {
                facets.minExclusive = property.toString();
                presentFacets |= XSSimpleType.FACET_MINEXCLUSIVE;
            }

            if ((property = properties.getProperty(IAttributeDescription.MIN_INCLUSIVE_FACET)) != null)
            {
                facets.minInclusive = property.toString();
                presentFacets |= XSSimpleType.FACET_MININCLUSIVE;
            }

            ValidationContext validationContext = new ValidationState();

            simpleType.applyFacets(facets, presentFacets, (short) 0, validationContext);

            return simpleType;
        }
        catch (InvalidDatatypeFacetException e)
        {
            throw new ConfigTypeException(e.getKey(), e.getArgs(), e);
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("ad-build-simple-type-failed", e);
        }
    }

    /*  Set the owner of this description.
     *  The owner of this discription must contain it.
     *  @exception  ConfigServiceException thrown if this description
     *              is already contained in another config type.
     */
    protected void
    setOwner(IConfigType configType)
    throws ConfigServiceException
    {
        if (m_owner != null && configType != null)
        {
            new ConfigServiceException("ad-set-owner-already-owned", new Object[]{ configType.getName(), m_owner.getName() });
        }

        ((AttributeDescriptionMapImpl)m_attributeDescriptions).setOwner(configType);

        m_owner = configType;
    }

}