/*
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.mx.config.impl;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import com.sonicsw.mx.config.ConfigAttributeException;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.ConfigServiceRuntimeException;
import com.sonicsw.mx.config.IAttributeDescription;
import com.sonicsw.mx.config.IConfigPath;
import com.sonicsw.mx.config.IConfigType;

import com.sonicsw.mf.common.config.IAttributeSet;

public class AttributeDescriptionMapImpl
extends java.util.HashMap
{

    /** The config type that contains and owns this attribute description map.
     *  An attribute description map can reside within only one config type
     *  at a time.
     */
    protected IConfigType m_owner = null;

    /** The Config Server this Attribute Description is associated with.
     */
    protected ConfigServer m_configServer = null;

     public AttributeDescriptionMapImpl(ConfigServer configServer)
     {
        m_configServer = configServer;
     }

     public AttributeDescriptionMapImpl(IAttributeSet descriptions, ConfigServer configServer)
     {
        m_configServer = configServer;
        HashMap descrMap = descriptions.getAttributes();
        Iterator keysIT = descrMap.keySet().iterator();
        while (keysIT.hasNext())
        {
            String key = (String)keysIT.next();
            IAttributeSet value = (IAttributeSet)descrMap.get(key);
            put(key, new AttributeDescriptionImpl(value, m_configServer));
        }
     }

    @Override
    public String
    toString()
    {
        return toString(0);
    }

    public String
    toString(int indent)
    {
        StringBuffer buffer = new StringBuffer();
        StringBuffer indentBuf = new StringBuffer();
        for (int i = 0; i < indent; i++)
        {
            indentBuf.append("    ");
        }
        String indentStr = indentBuf.toString();

        Iterator attrNames = keySet().iterator();
        while (attrNames.hasNext())
        {
            String attrName = (String) attrNames.next();
            IAttributeDescription attrDesc = getAttributeDescription(attrName);
            buffer.append(indentStr).append(attrName + " " + ((AttributeDescriptionImpl)attrDesc).toString(indent));
        }

        return buffer.toString();
    }

    @Override
    public Object
    clone()
    {
        /*  First make shallow copy of map */
        AttributeDescriptionMapImpl copy = (AttributeDescriptionMapImpl) super.clone();

        /*  Clones are not initially owned by any config type. */
        copy.m_owner = null;

        /*  Deep copy attribute descriptions in copied map. */
        copy.putAll(this);

        return copy;
    }

    public IAttributeDescription
    getAttributeDescription(String name)
    {
        return (IAttributeDescription) get(name);
    }

    public void
    setAttributeDescription(String name, IAttributeDescription value)
    throws ConfigAttributeException
    {
        try
        {
            put(name, value);
        }
        catch (Exception e)
        {
            throw new ConfigAttributeException(new ConfigPathImpl(name), "adm-set-attr-desc-failed", e);
        }
    }

    public IAttributeDescription
    removeAttributeDescription(String name)
    throws ConfigAttributeException
    {
        try
        {
            return (IAttributeDescription) remove(name);
        }
        catch (ConfigServiceRuntimeException e)
        {
            throw (ConfigAttributeException) e.getLinkedException();
        }
    }

    public IAttributeDescription
    getAttributeDescription(IConfigPath path)
    {
        switch (path.size())
        {
        case 0:
            return null;
        case 1:
            return getAttributeDescription(path.getLastComponent());
        default:
            IAttributeDescription attrDesc = getAttributeDescription(path.getFirstComponent());
            if (attrDesc != null)
            {
                 return attrDesc.getAttributeDescription(path.subPath(1));
            }
            /*  Intermediate path component either does not exist or is not a IAttributeDescriptionMap: return null. */
            return null;
        }
    }

    public void
    setAttributeDescription(IConfigPath path, IAttributeDescription value)
    throws ConfigAttributeException
    {
        try
        {
            switch (path.size())
            {
            case 0:
                return;
            case 1:
                setAttributeDescription(path.getLastComponent(), value);
                return;
            default:
                IAttributeDescription attrDesc = getAttributeDescription(path.getFirstComponent());
                if (attrDesc != null)
                {
                    ((AttributeDescriptionImpl)attrDesc).setAttributeDescription(path.subPath(1), value);
                }
                return;
            }
        }
        catch (Exception e)
        {
            throw new ConfigAttributeException(path, "adm-set-attr-desc-cp-failed", e);
        }
    }

    public IAttributeDescription
    removeAttributeDescription(IConfigPath path)
    throws ConfigAttributeException
    {
        try
        {
            switch (path.size())
            {
            case 0:
                return null;
            case 1:
                return removeAttributeDescription(path.getLastComponent());
            default:
                IAttributeDescription attrDesc = getAttributeDescription(path.getFirstComponent());
                if (attrDesc != null)
                {
                    return ((AttributeDescriptionImpl)attrDesc).removeAttributeDescription(path.subPath(1));
                }
                /*  Intermediate path component either does not exist or is not a IAttributeDescriptionMap: return null. */
                return null;
            }
        }
        catch (Exception e)
        {
            throw new ConfigAttributeException(path, "adm-remove-attr-desc-cp-failed", e);
        }
    }

    /**
     *
     */
    public AttributeDescriptionMapImpl
    getAttributeDescriptions(Set descriptionNames)
    {
        AttributeDescriptionMapImpl map = new AttributeDescriptionMapImpl(m_configServer);
        Iterator names = descriptionNames.iterator();
        while (names.hasNext())
        {
            String name = (String) names.next();
            map.put(name,get(name));
        }
        return map;
    }

    /**
     *
     */
    public void
    setAttributeDescriptions(AttributeDescriptionMapImpl attributeDescriptions)
    throws ConfigAttributeException
    {
        Set nameSet = getAttributeDescriptionNames();
        Iterator namesIT = nameSet.iterator();
        while (namesIT.hasNext())
        {
            String attrName = (String) namesIT.next();
            IAttributeDescription value = (IAttributeDescription)getAttributeDescription(attrName);
            setAttributeDescription(attrName,  value);
        }
        super.putAll(attributeDescriptions);
    }

    /**
     *
     */
    public AttributeDescriptionMapImpl
    removeAttributeDescriptions(Set attributeNames)
    throws ConfigAttributeException
    {
        AttributeDescriptionMapImpl map = new AttributeDescriptionMapImpl(m_configServer);
        Iterator names = attributeNames.iterator();
        while (names.hasNext())
        {
            String name = (String) names.next();
            Object attribute = remove(name);
            map.put(name,attribute);
        }
        return map;
    }

    public Set
    getAttributeDescriptionNames()
    {
        Set names = new NameSetImpl();
        names.addAll(keySet());
        return names;
    }

    public void toAttributeSet(IAttributeSet dsAttributeDescriptionSet)
    {
        Iterator names= getAttributeDescriptionNames().iterator();
        while (names.hasNext())
        {
            String attrName = (String)names.next();
            try
            {
                IAttributeSet attrSet = dsAttributeDescriptionSet.createAttributeSet(attrName);
                ((AttributeDescriptionImpl)getAttributeDescription(attrName)).toAttributeSet(attrSet);
            }
            catch (Exception e)
            {
                throw new ConfigServiceRuntimeException("adm-to-attr-set-failed", e);
            }
        }

    }

   /*
     *
     *  Methods overriding behavior in java.util.HashMap. This Prevents keys
     *  that are not of type java.lang.String and values that are not supported types
     *  from being added to map.
     *
     */


    /**
     * Associates the specified value with the specified key in this map.
     * If the map previously contained a mapping for this key, the old value is replaced.
     */
    @Override
    public final Object
    put(Object key, Object value)
    throws  ConfigServiceRuntimeException
    {
        try
        {
            AttributeDescriptionImpl desc
                = (AttributeDescriptionImpl) Util.validateAttributeDescriptionValue(value);

            desc.setOwner(m_owner);

            desc =  (AttributeDescriptionImpl) super.put(Util.validateAttributeDescriptionName(key), desc);

            if (desc != null)
            {
                desc.setOwner(null);
            }

            if (m_owner != null)
            {
                ((ConfigTypeImpl)m_owner).setModified();
            }

            return desc;
        }
        catch (ConfigServiceException e)
        {
            throw new ConfigServiceRuntimeException(e);
        }
    }

    /**
     * Copies all of the mappings from the specified map to this map. These
     * mappings will replace any mappings that this map had for any of the keys currently in the
     * specified map.
     */
    @Override
    public void
    putAll(Map mappings)
    throws  ConfigServiceRuntimeException
    {
        Iterator entries = entrySet().iterator();
        while(entries.hasNext())
        {
            Map.Entry entry = (Map.Entry) entries.next();
            put(entry.getKey(), entry.getValue());
        }
    }

    @Override
    public Object
    remove(Object key)
    throws ConfigServiceRuntimeException
    {
        try
        {
            AttributeDescriptionImpl desc = (AttributeDescriptionImpl) super.remove(key);

            if (desc != null)
            {
                desc.setOwner(null);
            }

            if (m_owner != null && desc != null)
            {
                ((ConfigTypeImpl)m_owner).setModified();
            }

            return desc;
        }
        catch(ConfigServiceException e)
        {
            throw new ConfigServiceRuntimeException(new ConfigAttributeException(new ConfigPathImpl((String) key), "adm-remove-failed", e));
        }
    }

    /*  Two attribute description maps are equal iff they are
        the same object.    */
    @Override
    public boolean equals(Object obj)
    {
        return (this == obj);
    }
    
    @Override
    public int hashCode() {
        return super.hashCode();
    }

    /* End methods overriding behavior in java.util.HashMap.
     */

    /*  Set the owner of this description map.
     *  The owner of this discription map must contain it.
     *  @exception  ConfigServiceException thrown if this description map
     *              is already contained in another config type.
     */
    protected void
    setOwner(IConfigType configType)
    throws ConfigServiceException
    {
        if (m_owner != null && configType != null)
        {
            new ConfigServiceException("ad-set-owner-already-owned", new Object[]{ configType.getName(), m_owner.getName() });
        }

        /*  Change ownership of all contained descriptions. */
        Iterator descriptions = values().iterator();
        while (descriptions.hasNext())
        {
            AttributeDescriptionImpl description = (AttributeDescriptionImpl) descriptions.next();
            description.setOwner(configType);
        }

        m_owner = configType;
    }

}