/*
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.mx.config.impl;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import com.sonicsw.mx.config.ConfigAttributeException;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.ConfigServiceRuntimeException;
import com.sonicsw.mx.config.IAttributeDescription;
import com.sonicsw.mx.config.IAttributeList;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IAttributeMetaData;
import com.sonicsw.mx.config.IConfigElement;
import com.sonicsw.mx.config.IConfigPath;
import com.sonicsw.mx.config.IConfigPrototype;

import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.security.ManagementPermissionDeniedException;

/**
 *
 */
public class AttributeListImpl
extends java.util.ArrayList
implements IAttributeList
{
    /** Attribute List's Identity.
     *  This object is used in comparing two attribute lists for identity
     *  in the 'equals' method and generating a hash code for the map.
     */
    protected Object m_identity = new Object();

    /** The config element that contains and owns this attribute list.
     *  And attribute list can reside within only one config element
     *  at a time.
     */
    protected IConfigElement m_owner = null;

    /** Parent
     *  The config object containing this attribute list.
     */
    protected Object m_parent = null;

    /** Attribute Name
     *  The name associated with this attribute list when
     *  it resides within a map or list.
     */
    protected String m_attributeName = null;

    /** Contained Lists
     *  Set of attribute lists directly contained within this list.
     */
    protected HashSet m_lists = new HashSet();

    /** Contained Maps
     *  Set of attribute maps directly contained within this list.
     */
    protected HashSet m_maps = new HashSet();

    /** Contained Refs
     *  Set of references directly contained within this list.
     */
    protected IdentityHashMap m_refs = new IdentityHashMap();

    /** Attribute Description
     *  Type description of list and its entries;
     */
    protected IAttributeDescription m_attrDescription = null;

    /** Config Server
     *  The config server this attribute list is associated with.
     */
    protected ConfigServer m_configServer = null;

    protected AttributeListImpl(ConfigServer configServer)
    throws ConfigServiceRuntimeException
    {
        m_configServer = configServer;
    }

    protected AttributeListImpl(IAttributeDescription attrDescription)
    throws ConfigServiceRuntimeException
    {
        m_attrDescription = attrDescription;
        m_configServer = ((AttributeDescriptionImpl)attrDescription).m_configServer;
    }

    protected AttributeListImpl(com.sonicsw.mf.common.config.IAttributeList dsList,
                                ConfigServer configServer)
    throws ConfigServiceRuntimeException
    {
        this(dsList, null, configServer);
    }

    protected AttributeListImpl(com.sonicsw.mf.common.config.IAttributeList dsList,
                                IAttributeDescription attrDescription, ConfigServer configServer)
    throws ConfigServiceRuntimeException
    {
        m_attrDescription = attrDescription;
        m_configServer    = configServer;

        int count = dsList.getCount();
        for (int i = 0; i < count; i++)
        {
// Rather than mask attributes from the template we want everything to show
// through to the instance attribute list...this mimics the underlying DS
// implementation of the IAttributeList
//
//            if (!dsList.getItemMetaData(i).isFromTemplate())
            {
                Object value = dsList.getItem(i);
                add(i, value);
            }
        }
    }

    public void
    toDSAttributeList(com.sonicsw.mf.common.config.IAttributeList dsList)
    throws ConfigAttributeException
    {
        for (int i = 0; i < size(); i++)
        {
            addDSListItem(_get(i), dsList);
        }
    }

    private void
    addDSListItem(Object attribute, com.sonicsw.mf.common.config.IAttributeList dsList)
    throws ConfigAttributeException
    {
        try
        {
            if (attribute instanceof IAttributeMap)
            {
                ((AttributeMapImpl)attribute).toAttributeSet(dsList.addNewAttributeSetItem());
            }
            else
            if (attribute instanceof IAttributeList)
            {
                ((AttributeListImpl)attribute).toDSAttributeList(dsList.addNewAttributeListItem());
            }
            else
            if (attribute instanceof String)
            {
                dsList.addStringItem((String)attribute);
            }
            else
            if (attribute instanceof Integer)
            {
                dsList.addIntegerItem((Integer)attribute);
            }
            else
            if (attribute instanceof Long)
            {
                dsList.addLongItem((Long)attribute);
            }
            else
            if (attribute instanceof BigDecimal)
            {
                dsList.addDecimalItem((BigDecimal)attribute);
            }
            else
            if (attribute instanceof Date)
            {
                dsList.addDateItem((Date)attribute);
            }
            else
            if (attribute instanceof Boolean)
            {
                dsList.addBooleanItem((Boolean)attribute);
            }
            else
            if (attribute instanceof ConfigReference)
            {
                dsList.addReferenceItem(new Reference(((ConfigReference)attribute).getElementName()));
            }
            else
            if (attribute instanceof byte[])
            {
                dsList.addBytesItem((byte[])attribute);
            }
        }
        catch (Exception e)
        {
            throw new ConfigAttributeException(getAttributeName().append(String.valueOf(size())), "al-add-ds-list-item-failed", e);
        }
    }

    @Override
    public Object
    clone()
    throws ConfigServiceRuntimeException
    {
        return clone(m_configServer);
    }

    /**
     * Creates a near-identical copy of the <code>AttributeListImpl</code>.
     * A different identity is assigned and while the list's owner (container)
     * is cleared (because the clone is free-floating) the list's instance owner
     * is maintained so that the link to any template element is kept, i.e. the
     * attribute list can get at the underlying temlplate entries (if applicable).
     *
     * @param configServer  The <code>ConfigServer</code> to which the cloned
     *                      list is to be binded.
     * @return              The cloned attribute list.
     * @throws ConfigServiceRuntimeException
     */
    protected Object
    clone(ConfigServer configServer)
    throws ConfigServiceRuntimeException
    {
        /*  First make shallow copy of list */
        AttributeListImpl copy = (AttributeListImpl) super.clone();

        copy.m_configServer = configServer;

        /*  Clones have new identity */
        copy.m_identity = new Object();

        /*  Clones are not initially owned by any config element */
        copy.m_owner = null;

        /*  Create new and empty map and list sets; they will be populated
            with newly cloned maps and lists of this list. */
        copy.m_lists = new HashSet();
        copy.m_maps = new HashSet();
        copy.m_refs = new IdentityHashMap();

        /*  Deep copy contents of list.    */
        for (int i = 0; i < size(); i++)
        {
            copy.set(i, _get(i));
        }

        return copy;
    }

    /*  Validates that the structure of the Attribute List contains
        all required values.
     */
    @Override
    public void
    validateComplete()
    throws ConfigAttributeException
    {
        if (m_attrDescription != null)
        {
            Iterator iterator = null;

            /*  Validate the structure of all embedded attribute maps.   */
            iterator = m_maps.iterator();
            while (iterator.hasNext())
            {
                ((IAttributeMap) iterator.next()).validateComplete();
            }

            /*  Validate the structure of all embedded attribute lists.   */
            iterator = m_lists.iterator();
            while (iterator.hasNext())
            {
                ((IAttributeList) iterator.next()).validateComplete();
            }

            /*  Validate that the list has the required number of entries */

            iterator = m_attrDescription.getAttributeDescriptionNames().iterator();
            if (!iterator.hasNext())
            {
                throw new ConfigAttributeException(getAttributeName(), "al-validate-complete-invalid-config-type");
            }

            String attrName = (String)iterator.next();
            IAttributeDescription attrDesc = m_attrDescription.getAttributeDescription(attrName);

            Object tmp = attrDesc.getProperty(IAttributeDescription.MIN_OCCURS_PROPERTY);
            if (tmp != null && tmp instanceof Long)
            {   /* Determine if minimum required entries are present. */
                long min = ((Long)tmp).longValue();
                if (size() < min)
                {
                    throw new ConfigAttributeException(getAttributeName(), "al-validate-complete-min-entries", new Object[] { new Integer(size()), new Long(min) });
                }
            }

            tmp = attrDesc.getProperty(IAttributeDescription.MAX_OCCURS_PROPERTY);
            if (tmp != null && tmp instanceof Long)
            {
                long max = ((Long)tmp).longValue();
                if (!tmp.equals(IAttributeDescription.LENGTH_UNBOUNDED) &&
                     size() > max)
                {
                    throw new ConfigAttributeException(getAttributeName(), "al-validate-complete-max-entries", new Object[] { new Integer(size()), new Long(max) });
                }
            }
        }
    }

    @Override
    public void
    addAttribute(Object attribute)
    throws ConfigAttributeException
    {
        try
        {
            add(attribute);
        }
        catch (ConfigServiceRuntimeException e)
        {
            throw (ConfigAttributeException) e.getLinkedException();
        }
   }

    @Override
    public void
    addAttribute(int index, Object attribute)
    throws ConfigAttributeException
    {
        try
        {
            add(index, attribute);
        }
        catch (ConfigServiceRuntimeException e)
        {
            throw (ConfigAttributeException) e.getLinkedException();
        }
    }

    @Override
    public void
    setAttribute(int index, Object attribute)
    throws ConfigAttributeException
    {
        try
        {
            set(index, attribute);
        }
        catch (ConfigServiceRuntimeException e)
        {
            throw (ConfigAttributeException) e.getLinkedException();
        }
    }

    @Override
    public void setAttribute(IConfigPath path, Object attribute)
    throws ConfigAttributeException
    {
        try
        {
            int index;
            switch (path.size())
            {
            case 0:
                return;
            case 1:
                try
                {
                    index = Integer.parseInt(path.getFirstComponent());
                }
                catch (NumberFormatException e)
                {   /*  Intermediate path component is not an index: Throw exception. */
                    throw new ConfigAttributeException(getAttributeName().append(path), "al-set-attr-cp-invalid-index", new Object[] { path.getFirstComponent(), getAttributeName() });
                }
                setAttribute(index, attribute);
                return;
            default:
                try
                {
                    index = Integer.parseInt(path.getFirstComponent());
                }
                catch (NumberFormatException e)
                {   /*  Intermediate path component is not an index: Throw exception. */
                    throw new ConfigAttributeException(getAttributeName().append(path), "al-set-attr-cp-invalid-index", new Object[] { path.getFirstComponent(), getAttributeName() });
                }
                Object tmp = getAttribute(index);
                if (tmp != null)
                {
                    if (tmp instanceof IAttributeMap)
                    {
                        ((AttributeMapImpl)tmp).setAttribute(path.subPath(1), attribute);
                    }
                    else if (tmp instanceof IAttributeList)
                    {
                        ((AttributeListImpl)tmp).setAttribute(path.subPath(1), attribute);
                    }
                    else
                    {   /*  Not a collection type.  */
                        throw new ConfigAttributeException(getAttributeName().append(path), "al-set-attr-cp-path-cmpnt-not-collection", new Object[] { getAttributeName().append(path.getFirstComponent()) });
                    }
                }
                else
                {   /*  Intermediate path component does not exist: Throw exception. */
                    throw new ConfigAttributeException(getAttributeName().append(path), "al-set-attr-cp-path-cmpnt-does-not-exist", new Object[] { getAttributeName().append(path.getFirstComponent()) });
                }
                return;
            }
        }
        catch (ConfigAttributeException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigAttributeException(getAttributeName().append(path), "al-set-attr-cp-failed", new Object[] { getAttributeName().append(path.getFirstComponent()) }, e);
        }
    }

    @Override
    public Object getAttribute(int index)
    {
        return get(index);
    }

    @Override
    public IAttributeMetaData
    getAttributeMetaData(int index)
    {
        if (m_parent instanceof IConfigPrototype)
        {
            return AttributeMetaData.s_isFromPrototype;
        }
        else
        {
            return AttributeMetaData.s_isFromConfigBean;
        }
    }

    @Override
    public Object
    getAttribute(IConfigPath path)
    {
        int index;
        switch (path.size())
        {
        case 0:
            return null;
        case 1:
            try
            {
                index = Integer.parseInt(path.getFirstComponent());
            }
            catch (NumberFormatException e)
            {   /*  Intermediate path component is not an index: return null. */
                return null;
            }
            return getAttribute(index);
        default:
            try
            {
                index = Integer.parseInt(path.getFirstComponent());
            }
            catch (NumberFormatException e)
            {   /*  Intermediate path component is not an index: return null. */
                return null;
            }
            Object tmp = getAttribute(index);
            if (tmp != null)
            {
                if (tmp instanceof IAttributeMap)
                {
                    return ((AttributeMapImpl)tmp).getAttribute(path.subPath(1));
                }
                else if (tmp instanceof IAttributeList)
                {
                    return ((AttributeListImpl)tmp).getAttribute(path.subPath(1));
                }
            }
            /*  Intermediate path component does not exist: return null */
            return null;
        }
    }

    @Override
    public IAttributeMetaData
    getAttributeMetaData(IConfigPath path)
    {
        int index;
        switch (path.size())
        {
        case 0:
            return null;
        case 1:
            try
            {
                index = Integer.parseInt(path.getLastComponent());
            }
            catch (NumberFormatException e)
            {   /*  Intermediate path component is not an index: return null. */
                return null;
            }
            return getAttributeMetaData(index);
        default:
            Object tmp = getAttribute(path.subPath(0, path.size() - 1));
            if (tmp != null)
            {
                if (tmp instanceof IAttributeMap)
                {
                    return ((AttributeMapImpl)tmp).getAttributeMetaData(path.getLastComponent());
                }
                else if (tmp instanceof IAttributeList)
                {
                    try
                    {
                        index = Integer.parseInt(path.getLastComponent());
                    }
                    catch (NumberFormatException e)
                    {   /*  Intermediate path component is not an index: return null. */
                        return null;
                    }
                    return ((AttributeListImpl)tmp).getAttributeMetaData(index);
                }
            }
        }
        return null;
    }

    @Override
    public Object
    removeAttribute(int index)
    throws ConfigAttributeException
    {
        try
        {
            return remove(index);
        }
        catch (ConfigServiceRuntimeException e)
        {
            throw (ConfigAttributeException) e.getLinkedException();
        }
    }

    @Override
    public Object
    removeAttribute(IConfigPath path)
    throws ConfigAttributeException
    {
        try
        {
            int index;
            switch (path.size())
            {
            case 0:
                return null;
            case 1:
                try
                {
                    index = Integer.parseInt(path.getFirstComponent());
                }
                catch (NumberFormatException e)
                {   /*  Intermediate path component is not an index: Throw exception. */
                    throw new ConfigAttributeException(getAttributeName().append(path), "al-remove-attr-cp-invalid-index", new Object[] { path.getFirstComponent(), getAttributeName() });
                }
                removeAttribute(index);
            default:
                try
                {
                    index = Integer.parseInt(path.getFirstComponent());
                }
                catch (NumberFormatException e)
                {   /*  Intermediate path component is not an index: Throw exception. */
                    throw new ConfigAttributeException(getAttributeName().append(path), "al-remove-attr-cp-invalid-index", new Object[] { path.getFirstComponent(), getAttributeName() });
                }
                Object tmp = get(index);
                if (tmp != null)
                {
                    if (tmp instanceof IAttributeMap)
                    {
                        return ((AttributeMapImpl)tmp).removeAttribute(path.subPath(1));
                    }
                    else if (tmp instanceof IAttributeList)
                    {
                        return ((AttributeListImpl)tmp).removeAttribute(path.subPath(1));
                    }
                }
                else
                {   /*  Intermediate path component does not exist: Throw exception. */
                    throw new ConfigAttributeException(getAttributeName().append(path), "al-remove-attr-cp-path-cmpnt-does-not-exist", new Object[] { getAttributeName().append(path.getFirstComponent()) });
                }
                return null;
            }
        }
        catch (ConfigAttributeException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigAttributeException(getAttributeName().append(path), "al-remove-attr-cp-failed", new Object[] { getAttributeName().append(path.getFirstComponent()) }, e);
        }
    }

    @Override
    public boolean
    removeAttribute(Object attribute)
    throws ConfigAttributeException
    {
        try
        {
            return remove(attribute);
        }
        catch (ConfigServiceRuntimeException e)
        {
            throw (ConfigAttributeException) e.getLinkedException();
        }
    }

    @Override
    public void
    addAttributes(IAttributeList attributes)
    throws ConfigAttributeException
    {
        try
        {
            addAll(attributes);
        }
        catch (ConfigServiceRuntimeException e)
        {
            throw (ConfigAttributeException) e.getLinkedException();
        }
    }

    @Override
    public void
    addAttributes(int index, IAttributeList attributes)
    throws ConfigAttributeException
    {
        try
        {
            addAll(index,attributes);
        }
        catch (ConfigServiceRuntimeException e)
        {
            throw (ConfigAttributeException) e.getLinkedException();
        }
   }

    @Override
    public boolean
    removeAttributes(IAttributeList attributes)
    throws ConfigAttributeException
    {
        try
        {
            return removeAll(attributes);
        }
        catch (ConfigServiceRuntimeException e)
        {
            throw (ConfigAttributeException) e.getLinkedException();
        }
    }

    @Override
    public IAttributeDescription
    getAttributeDescription()
    {
        return m_attrDescription;
    }

    @Override
    public IAttributeMap
    createAttributeMap(String name)
    throws ConfigAttributeException
    {
        try
        {
            if (m_attrDescription != null)
            {
                if (name == null)
                {
                    throw new ConfigAttributeException(getAttributeName(), "al-create-am-is-null", new Object[] { name });
                }
                IAttributeDescription attrDesc = m_attrDescription.getAttributeDescription(name);
                if (attrDesc == null)
                {
                    throw new ConfigAttributeException(getAttributeName(), "al-create-am-not-defined", new Object[] { name });
                }
                else
                if (attrDesc.getType() != IAttributeMap.class)
                {
                    throw new ConfigAttributeException(getAttributeName(), "al-create-am-not-map", new Object[] { name });
                }
                return new AttributeMapImpl(attrDesc);
            }
            else
            {
                return new AttributeMapImpl(m_configServer);
            }
        }
        catch (ConfigAttributeException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigAttributeException(getAttributeName(), "al-create-am-failed", new Object[] { name }, e);
        }
    }

    @Override
    public IAttributeList
    createAttributeList(String name)
    throws ConfigAttributeException
    {
        try
        {
            if (m_attrDescription != null)
            {
                if (name == null)
                {
                    throw new ConfigAttributeException(getAttributeName(), "al-create-al-is-null", new Object[] { name });
                }
                IAttributeDescription attrDesc = m_attrDescription.getAttributeDescription(name);
                if (attrDesc == null)
                {
                    throw new ConfigAttributeException(getAttributeName(), "al-create-al-not-defined", new Object[] { name });
                }
                else
                if (attrDesc.getType() != IAttributeList.class)
                {
                    throw new ConfigAttributeException(getAttributeName(), "al-create-al-not-list", new Object[] { name });
                }
                return new AttributeListImpl(attrDesc);
            }
            else
            {
                return new AttributeListImpl(m_configServer);
            }
        }
        catch (ConfigAttributeException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigAttributeException(getAttributeName(), "al-create-al-failed", new Object[] { name }, e);
        }
    }

    /*
     *
     *  Methods overriding behavior in java.util.ArrayList. This Prevents list
     *  values that are not supported types from being added to list.
     *
     */

    @Override
    public final void
    add(int index, Object element)
    throws ConfigServiceRuntimeException
    {
        try
        {

            element = Util.validateAttributeValue(getAttributeName().append(String.valueOf(index)),
                                                  element,
                                                  m_configServer,
                                                  m_attrDescription);

            setOwnership(element);

            super.add(index, element);

            if (m_owner != null)
            {
                ((ConfigElementImpl)m_owner).setModified();
            }
        }
        catch (ConfigAttributeException e)
        {
            throw new ConfigServiceRuntimeException(e);
        }
        catch (Exception e)
        {
            throw new ConfigServiceRuntimeException(new ConfigAttributeException(getAttributeName().append(String.valueOf(index)), "al-add-failed", e));
        }
    }

    @Override
    public boolean
    add(Object element)
    throws ConfigServiceRuntimeException
    {
        add(size(), element);
        return true;
    }

    @Override
    public boolean
    addAll(int index, Collection collection)
    throws ConfigServiceRuntimeException
    {
        Iterator values = collection.iterator();
        while (values.hasNext())
        {
            add(index++, values.next());
        }
        return true;
    }

    @Override
    public boolean
    addAll(Collection collection)
    throws ConfigServiceRuntimeException
    {
        Iterator values = collection.iterator();
        while (values.hasNext())
        {
            add(values.next());
        }
        return true;
    }

    @Override
    public void
    clear()
    {
        while (!isEmpty())
        {
            remove(0);
        }
    }

    public Object
    _get(IConfigPath path)
    {
        int index;
        switch (path.size())
        {
        case 0:
            return null;
        case 1:
            try
            {
                index = Integer.parseInt(path.getFirstComponent());
            }
            catch (NumberFormatException e)
            {   /*  Intermediate path component is not an index: return null. */
                return null;
            }
            return _get(index);
        default:
            try
            {
                index = Integer.parseInt(path.getFirstComponent());
            }
            catch (NumberFormatException e)
            {   /*  Intermediate path component is not an index: return null. */
                return null;
            }
            Object tmp = _get(index);
            if (tmp != null)
            {
                if (tmp instanceof IAttributeMap)
                {
                    return ((AttributeMapImpl)tmp)._get(path.subPath(1));
                }
                else if (tmp instanceof IAttributeList)
                {
                    return ((AttributeListImpl)tmp)._get(path.subPath(1));
                }
            }
            /*  Intermediate path component does not exist: return null */
            return null;
        }
    }

    public Object
    _get(int index)
    {
        return super.get(index);
    }


    @Override
    public Object
    get(int index)
    {
        try
        {
            Object obj = super.get(index);
            if (obj instanceof ConfigReference)
            {
                try
                {
                    obj = ((ConfigReference)obj).getConfigElement();
                }
                catch (ConfigServiceException e)
                {
                    if(e.getLinkedException() instanceof ManagementPermissionDeniedException){
                        obj  = IdentityOnlyConfigBeanImpl.createIdentityOnlyConfigBean(m_configServer,((ConfigReference)obj));
                    } else {
                    obj = null;
                    }
                }
            }
            return obj;
        }
        catch (Exception e)
        {
            throw new ConfigServiceRuntimeException(new ConfigAttributeException(getAttributeName().append(String.valueOf(index)), "al-get-failed", e));
        }
    }

    // MQ-34477: component collection is not listed in the view when added to
    // collection monitor component. Prior to Java 7 the JDK's implementation of
    // iterator.next() invoked get(); this is no longer the case with Java 7.
    // AttributeListImpl overrides get() to unpack ConfigReference instances,
    // but with Java 7 iterator.next() fails to make use of this code.
    // To address this, iterator.next() has now been overridden to unpack
    // ConfigReferences itself.
    @Override
    public Iterator iterator() {
        final Iterator iterator = super.iterator();
        return new Iterator() {

            @Override
            public boolean hasNext() {
                return iterator.hasNext();
            }

            @Override
            public Object next() {
                try {
                    Object obj = iterator.next();
                    if (obj instanceof ConfigReference) {
                        try {
                            obj = ((ConfigReference) obj).getConfigElement();
                        }
                        catch (ConfigServiceException e) {
                            if (e.getLinkedException() instanceof ManagementPermissionDeniedException) {
                                obj = IdentityOnlyConfigBeanImpl.createIdentityOnlyConfigBean(m_configServer, ((ConfigReference) obj));
                            } else {
                                obj = null;
                            }
                        }
                    }
                    return obj;
                }
                catch (Exception e) {
                    throw new ConfigServiceRuntimeException(new ConfigAttributeException(getAttributeName(), "al-get-failed", e));
                }
            }

            @Override
            public void remove() {
                iterator.remove();
            }
        };
    }

    @Override
    public int
    indexOf(Object elem)
    {
        if (elem == null)
        {
            for (int i = 0; i < size(); i++)
            {
                if (get(i) == null)
                {
                    return i;
                }
            }
        }
        else
        {
            for (int i = 0; i < size(); i++)
            {
                Object item = get(i);

                if ((item != null) && item.equals(elem))
                {
                    return i;
                }
            }
        }
        return -1;
    }

    @Override
    public int
    lastIndexOf(Object elem)
    {
        if (elem == null)
        {
            for (int i = size() - 1; i >= 0; i--)
            {
                if (get(i) == null)
                {
                    return i;
                }
            }
        }
        else
        {
            for (int i = size() - 1; i >= 0; i--)
            {
                if (get(i).equals(elem))
                {
                    return i;
                }
            }
        }
        return -1;
    }

    @Override
    public Object
    remove(int index)
    throws ConfigServiceRuntimeException
    {
        try
        {
            Object obj = super.remove(index);

            removeOwnership(obj);

            if (m_owner != null)
            {
                ((ConfigElementImpl)m_owner).setModified();
            }

            if (obj instanceof ConfigReference)
            {
                obj = ((ConfigReference)obj).getConfigElement();
            }
            return obj;
        }
        catch (Exception e)
        {
            throw new ConfigServiceRuntimeException(new ConfigAttributeException(getAttributeName().append(String.valueOf(index)),"al-remove-failed", e));
        }
    }

    @Override
    public boolean
    remove(Object obj)
    {
        int index = indexOf(obj);
        if (index < 0)
        {   /* obj is not in list */
            return false;
        }
        else
        {   /* obj is in list */
            remove(index);
            return true;
        }
    }

    @Override
    public boolean
    removeAll(Collection c)
    {
        Iterator objs = c.iterator();
        while (objs.hasNext())
        {
            remove(objs.next());
        }
        return true;
    }

    @Override
    public boolean
    retainAll(Collection c)
    {
        throw new UnsupportedOperationException();
    }

    @Override
    public Object
    set(int index, Object element)
    throws ConfigServiceRuntimeException
    {
        try
        {
            element = Util.validateAttributeValue(getAttributeName().append(String.valueOf(index)),
                                                  element,
                                                  m_configServer,
                                                  m_attrDescription);

            setOwnership(element);

            Object obj = super.set(index, element);

            removeOwnership(obj);

            if (m_owner != null)
            {
                ((ConfigElementImpl)m_owner).setModified();
            }

            return obj;
        }
        catch (ConfigAttributeException e)
        {
            throw new ConfigServiceRuntimeException(e);
        }
        catch (Exception e)
        {
            throw new ConfigServiceRuntimeException(new ConfigAttributeException(getAttributeName().append(String.valueOf(index)), "al-set-failed", e));
        }
    }

    @Override
    public Object[]
    toArray()
    {
        Object[] result = new Object[size()];

        for (int i = 0; i < size(); i++)
        {
            result[i] = get(i);
        }

        return result;
    }

    @Override
    public Object[] toArray(Object a[])
    {
        if (a.length < size())
        {
            a = (Object[])java.lang.reflect.Array.newInstance(a.getClass().getComponentType(), size());
        }

        for (int i = 0; i < size(); i++)
        {
            a[i] = get(i);
        }

        if (a.length > size())
        {
            a[size()] = null;
        }

        return a;
    }

    @Override
    public boolean
    equals(Object obj)
    {
        if (obj != null && (obj.getClass().equals(this.getClass())))
        {
            return m_identity == ((AttributeListImpl) obj).m_identity;
        }
        return false;
    }

    @Override
    public int
    hashCode()
    {
        return m_identity.hashCode();
    }

    /* End methods overriding behavior in java.util.HashMap.
     */

    protected void
    getSubElements(Set subElements)
    throws ConfigServiceException
    {
        Iterator refs = m_refs.keySet().iterator();
        while (refs.hasNext())
        {
            ConfigReference ref = (ConfigReference) refs.next();
            /*  Resolve referenced bean only if already loaded and add
                it to sub-element list. */
            IConfigElement element = ref.getLocalConfigElement();
            if (element != null)
            {
                if (!subElements.contains(element))
                {
                    subElements.add(element);
                    ((ConfigElementImpl)element).getSubElements(subElements);
                }
           }
        }

        Iterator maps = m_maps.iterator();
        while (maps.hasNext())
        {
            AttributeMapImpl map = (AttributeMapImpl) maps.next();
            map.getSubElements(subElements);
        }

        Iterator lists = m_lists.iterator();
        while (lists.hasNext())
        {
            AttributeListImpl list = (AttributeListImpl) lists.next();
            list.getSubElements(subElements);
        }
    }

    /*  Methods enforcing Config Object Ownership
     */

    public Object
    getParent()
    {
        return m_parent;
    }

    protected void
    setParent(String attributeName, Object parent)
    throws ConfigServiceException
    {
        if (parent != null &&
            !(parent instanceof AttributeMapImpl) &&
            !(parent instanceof AttributeListImpl))
        {
            throw new ConfigAttributeException(getAttributeName().append(attributeName), "al-set-parent-failed");
        }

        if (parent == this)
        {
            m_parent = null;
        }
        else
        {
            m_parent = parent;
            m_attributeName = attributeName;
        }
    }

     /*  Return a reference to the config element that contains and owns this list.
     *  If this list is not contained in any config element, a <code>null</code>
     *  is returned.
     */
    public IConfigElement
    getOwner()
    {
        return m_owner;
    }

    /*  Set the owner of this list.
     *  The owner of this list must contain it.
     *  @exception  ConfigServiceException thrown if this list
     *              is already contained in another config element.
     */
    protected void
    setOwner(IConfigElement configElement)
    throws ConfigServiceException
    {
        if (m_owner != null && configElement != null)
        {
            new ConfigServiceException("al-set-owner-already-owned", new Object[]{ configElement.getName(), m_owner.getName() });
        }

        /*  Change ownership of all contained maps and lists */
        Iterator maps = m_maps.iterator();
        while (maps.hasNext())
        {
            AttributeMapImpl map = (AttributeMapImpl) maps.next();
            map.setOwner(configElement);
        }
        Iterator lists = m_lists.iterator();
        while (lists.hasNext())
        {
            AttributeListImpl list = (AttributeListImpl) lists.next();
            list.setOwner(configElement);
        }
        Iterator refs = m_refs.keySet().iterator();
        while (refs.hasNext())
        {
            ConfigReference ref = (ConfigReference) refs.next();
            ref.setOwner(configElement);
        }

        m_owner = configElement;
    }

    /*  Set the ownership of obj to this list's config object.
     *  @exception  ConfigServiceException thrown if obj is already
     *              contained in another config element.
     */
    private void
    setOwnership(Object obj)
    throws ConfigServiceException
    {
        if (obj instanceof AttributeMapImpl)
        {
            ((AttributeMapImpl)obj).setOwner(m_owner);
            ((AttributeMapImpl)obj).setParent("", this);
            m_maps.add(obj);
        }
        else if (obj instanceof AttributeListImpl)
        {
            ((AttributeListImpl)obj).setOwner(m_owner);
            ((AttributeListImpl)obj).setParent("", this);
            m_lists.add(obj);
        }
        else if (obj instanceof ConfigReference)
        {
            ((ConfigReference)obj).setOwner(m_owner);
            m_refs.put(obj, obj);
        }
    }

    /*  Remove the ownership from obj.
     */
    private void
    removeOwnership(Object obj)
    throws ConfigServiceException
    {
        if (obj instanceof AttributeMapImpl)
        {
            ((AttributeMapImpl)obj).setOwner(null);
            ((AttributeMapImpl)obj).setParent("", null);
            m_maps.remove(obj);
        }
        else if (obj instanceof AttributeListImpl)
        {
            ((AttributeListImpl)obj).setOwner(null);
            ((AttributeListImpl)obj).setParent("", null);
            m_lists.remove(obj);
        }
        else if (obj instanceof ConfigReference)
        {
            ((ConfigReference)obj).setOwner(null);
            m_refs.remove(obj);
        }
    }

    /*  End methods enforcing Config Object Ownership
     */

    @Override
    public String
    toString()
    {
        return toString(0, false);
    }

    public String
    toString(int indent, boolean isFromPrototype)
    {
        StringBuffer buffer = new StringBuffer();
        StringBuffer indentBuf = new StringBuffer();
        for (int i = 0; i < indent; i++)
        {
            indentBuf.append("    ");
        }
        String indentStr = indentBuf.toString();

        for (int i = 0; i < size(); i++)
        {
            Object attrValue = _get(i);

            String attrSource = null;
            if (isFromPrototype)
            {
                attrSource = ":P";
            }
            else
            {
                attrSource = ":B";
            }

            String className = attrValue.getClass().getName();
            String typeName  = className.substring(className.lastIndexOf(".") + 1);

            if (attrValue instanceof IAttributeMap)
            {
                buffer.append(indentStr).append(i + "  [" + typeName + attrSource + "] =\n");
                buffer.append(indentStr).append("{\n");
                buffer.append(((AttributeMapImpl)attrValue).toString(indent + 1, isFromPrototype));
                buffer.append(indentStr).append("}\n");
            }
            else
            if (attrValue instanceof IAttributeList)
            {
                buffer.append(indentStr).append(i + "  [" + typeName + attrSource + "] =\n");
                buffer.append(indentStr).append("{\n");
                buffer.append(((AttributeListImpl)attrValue).toString(indent + 1, isFromPrototype));
                buffer.append(indentStr).append("}\n");
            }
            else
            {
                buffer.append(indentStr).append(i + "  [" + typeName + attrSource + "] = " + Util.toString(attrValue) + "\n");
            }
        }
        return buffer.toString();
    }

    @Override
    public String toXML()
    {
        StringBuffer buffer = new StringBuffer();

        // get the tag name from the attribute description. This is actually
        // a map but for now just rely on their only being one entry in there.
        Iterator descIter = m_attrDescription.getAttributeDescriptionNames().iterator();

        if(descIter == null || !descIter.hasNext())
        {
            return null;
        }

        String description = (String)descIter.next();

        for(Iterator iter = iterator(); iter.hasNext(); )
        {
            String content = elementToString(description, iter.next());

            if(content != null)
            {
                buffer.append(content);
            }
        }
        return buffer.toString();
    }

    private String elementToString(String name, Object value)
    {
        String header  = "<" + name + ">";
        String footer  = "</" + name + ">\r\n";
        String content = null;

        boolean crContent = false;

        if(value instanceof ConfigBeanImpl)
        {
            content = Util.name2Url(((ConfigBeanImpl)value).getName());
        }
        else if(value instanceof AttributeMapImpl)
        {
            content = ((AttributeMapImpl)value).toXML();
            crContent = true;
        }
        else if(value instanceof AttributeListImpl)
        {
            content = ((AttributeListImpl)value).toXML();
            crContent = true;
        }
        else
        {
            if(!Util.isDefaultValue(name, value, m_attrDescription))
            {
                content = Util.toString(value);
            }
        }

        String ret = null;

        if(content != null && content.length() > 0)
        {
            if(crContent)
            {
                ret = header + "\r\n" + content + footer;
            }
            else
            {
                ret = header + content + footer;
            }
        }
        return ret;
    }

    public IConfigPath
    getAttributeName()
    {
        IConfigPath attributeName = null;
        if (m_parent != null)
        {
            if (m_parent instanceof AttributeMapImpl)
            {
                attributeName = ((AttributeMapImpl)m_parent).getAttributeName();
                attributeName.append(m_attributeName);
            }
            else // m_parent instanceof AttributeListImpl
            {
                attributeName = ((AttributeListImpl)m_parent).getAttributeName();
                int index = ((AttributeListImpl)m_parent).indexOf(this);
                attributeName.append(String.valueOf(index));
            }
        }
        else
        {
                attributeName = new ConfigPathImpl();
        }
        return attributeName;
    }

}