/*
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.mx.config.impl;

import java.math.BigDecimal;
import java.util.AbstractCollection;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import com.sonicsw.mx.config.AttributeChangeFilter;
import com.sonicsw.mx.config.ConfigAttributeException;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.ConfigServiceRuntimeException;
import com.sonicsw.mx.config.IAnnotationExtension;
import com.sonicsw.mx.config.IAttributeDescription;
import com.sonicsw.mx.config.IAttributeList;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IAttributeMetaData;
import com.sonicsw.mx.config.IConfigChangeListener;
import com.sonicsw.mx.config.IConfigElement;
import com.sonicsw.mx.config.IConfigPath;
import com.sonicsw.mx.config.IConfigPrototype;

import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.security.ManagementPermissionDeniedException;

/**
 *
 */
public class AttributeMapImpl
extends java.util.HashMap
implements IAttributeMap
{
    /**
     * This class acts as a placeholder to mask out locally removed attributes
     * when the <code>AttributeMapImpl</code> is owned (contained) by an
     * instance element. i.e. when an entry is removed, we actually add a
     * <code>RemovedAttribute</code> in its place. This stops the underlying
     * template attribute from automatically showing through.
     */
    public class RemovedAttribute
    {
        @Override
        public String toString()
        {
            return "REMOVED_ATTRIBUTE";
        }
    }

    /** Attribute Map's Identity.
     *  This object is used in comparing two attribute maps for identity
     *  in the 'equals' method and generating a hash code for the map.
     */
    protected Object m_identity = new Object();

    /** The config element that contains and owns this attribute map.
     *  And attribute map can reside within only one config element
     *  at a time.
     */
    protected IConfigElement m_owner = null;

    /** The prototype-related config element that contains and owns this
     *  attribute map. Ownership is required not only to track modifications
     *  to an element, but also to maintain the link with a template element.
     *  This instance owner is kept in-sync with the regular owner...they
     *  only digress when the attribute map is cloned...the instance owner
     *  is maintained in order to keep the tie with (any) underlying template
     *  element.
     */
    protected IConfigElement m_instanceOwner = null;

    /** Parent
     *  The config object containing this attribute map.
     */
    protected Object m_parent = null;

    /** Attribute Name
     *  The name associated with this attribute list when
     *  it resides within a map or list.
     */
    protected String m_attributeName = null;

    /** Contained Lists
     *  Set of attribute lists directly contained within this map.
     */
    protected HashSet m_lists = new HashSet();

    /** Contained Maps
     *  Set of attribute maps directly contained within this map.
     */
    protected HashSet m_maps = new HashSet();

    protected boolean m_override = false;

    /** Contained Refs
     *  Set of references directly contained within this map.
     */
    protected IdentityHashMap m_refs = new IdentityHashMap();

    /** Attribute Description
     *  Type description of map and its entries;
     */
    protected IAttributeDescription m_attrDescription = null;

    protected ConfigServer m_configServer = null;


    protected AttributeMapImpl(ConfigServer configServer)
        throws ConfigServiceRuntimeException
    {
        m_configServer = configServer;
    }

    protected AttributeMapImpl(IAttributeDescription attrDescription)
        throws ConfigServiceRuntimeException
    {
        m_attrDescription = attrDescription;
        m_configServer    = ((AttributeDescriptionImpl)attrDescription).m_configServer;
    }

    protected AttributeMapImpl(com.sonicsw.mf.common.config.IAttributeSet attrSet,
                               ConfigServer configServer)
        throws ConfigServiceRuntimeException
    {
        this(attrSet, null, configServer, true);
    }

    protected AttributeMapImpl(com.sonicsw.mf.common.config.IAttributeSet attrSet,
                               IAttributeDescription attrDescription,
                               ConfigServer configServer,
                               boolean filterDefaults)
        throws ConfigServiceRuntimeException
    {
        m_attrDescription = attrDescription;
        m_configServer    = configServer;
        HashMap attrs     = attrSet.getAttributes();
        Iterator attrKeys = attrs.keySet().iterator();

        while (attrKeys.hasNext())
        {
            String key = (String)attrKeys.next();

            // If the attribute is from the template then we don't want to
            // add it to our AttributeMap. Otherwise add it.
            if (!attrSet.getAttributeMetaData(key).isFromTemplate())
            {
                // Only populate config element with state that is not from template
                put(key, attrs.get(key), true, filterDefaults);
            }
        }

        String[] removedAttributes = attrSet.getDeletedAttributesInThisSubclassed();
        if (removedAttributes != null)
        {
            for (int i = 0; i < removedAttributes.length; i++)
            {
                put(removedAttributes[i], new RemovedAttribute(), true, true);
            }
        }
    }

    protected AttributeMapImpl(AttributeMapImpl attrMap)
        throws ConfigServiceRuntimeException
    {
        m_configServer = attrMap.m_configServer;

        putAll(attrMap);  // Deep copy contends of map
    }

    public void toAttributeSet(com.sonicsw.mf.common.config.IAttributeSet set)
        throws ConfigAttributeException
    {
        toAttributeSet(set, false);
    }

    public void toAttributeSet(com.sonicsw.mf.common.config.IAttributeSet set, boolean overlay)
        throws ConfigAttributeException
    {
       String key = null;
        Iterator it = super.keySet().iterator();
        try
        {
            while (it.hasNext())
            {
                key = (String)it.next();
                Object value = _get(key);
                setDSAttribute(key, value, set, overlay);
            }
        }
        catch (ConfigAttributeException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigAttributeException(getAttributeName().append(key), "am-to-attr-set-failed", e);
        }
    }

    private void
    setDSAttribute(String key, Object value, com.sonicsw.mf.common.config.IAttributeSet dsSet, boolean overlay)
    throws ConfigAttributeException
    {
        if (value == null)
        {
            throw new ConfigAttributeException(getAttributeName().append(key), "am-set-ds-attr-is-null");
        }

        try
        {
            if (value instanceof IAttributeMap)
            {
                Object tmp = dsSet.getAttribute(key);
                if (tmp != null && tmp instanceof com.sonicsw.mf.common.config.IAttributeSet && overlay)
                {
                    ((AttributeMapImpl)value).toAttributeSet((com.sonicsw.mf.common.config.IAttributeSet)tmp, true);
                }
                else
                {
                    ((AttributeMapImpl)value).toAttributeSet(dsSet.createAttributeSet(key));
                }
            }
            else
            if (value instanceof com.sonicsw.mx.config.IAttributeList)
            {
                ((AttributeListImpl)value).toDSAttributeList(dsSet.createAttributeList(key));
            }
            else
            if (value instanceof String)
            {
                dsSet.setStringAttribute(key, (String)value);
            }
            else
            if (value instanceof Integer)
            {
                dsSet.setIntegerAttribute(key, (Integer)value);
            }
            else
            if (value instanceof Long)
            {
                dsSet.setLongAttribute(key, (Long)value);
            }
            else
            if (value instanceof BigDecimal)
            {
                dsSet.setDecimalAttribute(key, new BigDecimal(value.toString()));
            }
            else
            if (value instanceof Date)
            {
                dsSet.setDateAttribute(key, (Date)value);
            }
            else
            if (value instanceof Boolean)
            {
                dsSet.setBooleanAttribute(key, (Boolean)value);
            }
            else
            if (value instanceof ConfigReference)
            {
                dsSet.setReferenceAttribute(key, new Reference(((ConfigReference) value).getElementName()));
            }
            else
            if (value instanceof byte[])
            {
                dsSet.setBytesAttribute(key, (byte[]) value);
            }
            else
            if (value instanceof RemovedAttribute)
            {
                dsSet.mergeUtilHint_markAttributeDeleted(key);
            }
            else
            {
                throw new ConfigAttributeException(getAttributeName().append(key), "am-set-ds-attr-invalid-value", new Object[] { value.getClass().getName() });
            }
        }
        catch (ConfigAttributeException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigAttributeException(getAttributeName().append(key), "am-set-ds-attr-failed", e);
        }
    }

    @Override
    public Object clone()
        throws ConfigServiceRuntimeException
    {
        return clone(m_configServer);
    }

    /**
     * Creates a near-identical copy of the <code>AttributeMapImpl</code>.
     * A different identity is assigned and while the map's owner (container)
     * is cleared (because the clone is free-floating) the map's instance owner
     * is maintained so that the link to any template element is kept, i.e. the
     * attribute map can get at the underlying temlplate entries (if applicable).
     *
     * @param configServer  The <code>ConfigServer</code> to which the cloned
     *                      map is to be binded.
     * @return              The cloned attribute map.
     * @throws ConfigServiceRuntimeException
     */
    protected Object clone(ConfigServer configServer)
        throws ConfigServiceRuntimeException
    {
        // First make shallow copy of map
        AttributeMapImpl copy = (AttributeMapImpl) super.clone();

        // clear out underlying HashMap contents - we are going to explicitly
        // deep copy the map contents...this also avoids the side-effect of
        // removing ownership from map items in the original!
        copy._clearMap();

        try
        {
            copy.m_configServer = configServer;

            // Clones have new identity
            copy.m_identity = new Object();

            // Clones are not initially owned by any config element
            copy.m_owner = null;
            copy.m_instanceOwner = m_owner;

            /*  Create new and empty map and list sets; they will be populated
             with newly cloned maps and lists of this map. */
            copy.m_lists = new HashSet();
            copy.m_maps = new HashSet();
            copy.m_refs = new IdentityHashMap();

            // Deep copy contents of map
            Iterator i = _keySet().iterator();
            while (i.hasNext())
            {
                String iKey = (String) i.next();
                copy.put(iKey, _get(iKey));
            }
        }
        catch (Exception e)
        {
            ConfigServiceRuntimeException we = new ConfigServiceRuntimeException("Failed to clone");
            we.setLinkedException(e);

            throw we;
        }

        return copy;
    }

    /*  Validates that the structure of the Attribute Map contains
        all required values.
     */
    @Override
    public final void validateComplete()
        throws ConfigAttributeException
    {
        if (m_attrDescription != null)
        {
            Iterator iterator = null;

            /*  Validate the structure of all embedded attribute maps.   */
            iterator = m_maps.iterator();
            while (iterator.hasNext())
            {
                ((IAttributeMap) iterator.next()).validateComplete();
            }

            /*  Validate the structure of all embedded attribute lists.   */
            iterator = m_lists.iterator();
            while (iterator.hasNext())
            {
                ((IAttributeList) iterator.next()).validateComplete();
            }

            if (m_attrDescription.getProperty(IAttributeDescription.UNENUM_MAP_PROPERTY) == null)
            {   /*  This is an enumerated attribute map: validate that all
                    required attributes are present. */
                iterator = m_attrDescription.getAttributeDescriptionNames().iterator();
                while (iterator.hasNext())
                {
                    String attrName = (String)iterator.next();
                    IAttributeDescription attrDesc = m_attrDescription.getAttributeDescription(attrName);

                    /*  NB: super.get() is used here to determine if the attribute
                        value has actually been set; this.get() might return
                        a default or fixed value from the config type for an
                        unset attribute.    */
                    Object attrValue = super.get(attrName);

                    if (attrValue == null &&
                        getProtoMap() != null)
                    {
                        attrValue = ((AttributeMapImpl)getProtoMap())._get(attrName);
                    }

                    /*
                     * BugFix :: Sonic00039709 :: Incase of instance from template it is not validating correctly.
                     * It has value RemovedAttribute instead of null.
                     */
                    if (attrValue == null || attrValue instanceof RemovedAttribute)
                    {   /*  This attribute has not been set on this map:
                            determine if it is required.    */
                        Object tmp = attrDesc.getProperty(IAttributeDescription.MIN_OCCURS_PROPERTY);
                        if (tmp != null && tmp instanceof Long)
                        {
                            if(((Long)tmp).longValue() > 0)
                            {   /*  This is a required attribute that has not
                                    been set: determine if this attribute has a valid
                                    default or fixed value which allows it to be absent
                                    in a configuration.   */
                                if (attrDesc.getType() == IAttributeMap.class)
                                {   /*  This is a required attribute map which must be present
                                        in a configuration: throw an exception. */
                                    /* REVISIT: This is a temporary fix for SONIC00009313 */
                                    IAttributeMap attrMap = createAttributeMap(attrName);
                                    setAttribute(attrName, attrMap);
                                    attrMap = (IAttributeMap) _get(attrName);
                                    attrMap.validateComplete();
                                    //throw new ConfigServiceException("Required attribute missing: '" + getAttributeName() + "." + attrName + "'");
                                }
                                else
                                if (attrDesc.getType() == IAttributeList.class)
                                {   /*  This is a required attribute list which must be present
                                        in a configuration: throw an exception. */
                                    /* REVISIT: This is a temporary fix for SONIC00009313 */
                                    IAttributeList attrList = createAttributeList(attrName);
                                    setAttribute(attrName, attrList);
                                    attrList = (IAttributeList) _get(attrName);
                                    attrList.validateComplete();
                                    //throw new ConfigServiceException("Required attribute missing: '" + getAttributeName() + "." + attrName + "'");
                                }
                                else
                                if ((tmp = attrDesc.getProperty(IAttributeDescription.FIXED_PROPERTY)) != null)
                                {   /*  This attribute has a fixed value: ok to be missing from a configuration. */
                                    /*  REVISIT: This is a temporaty fix for SONIC00009339 */
                                    setAttribute(attrName, tmp);
                                }
                                else
                                if ((tmp = attrDesc.getProperty(IAttributeDescription.DEFAULT_PROPERTY)) != null)
                                {   /*  This attribute has a default value: ok to be missing from a configuration. */
                                }
                                else
                                {   /*  This is a required attribute with no
                                        default or fixed value: throw an exception.    */
                                    throw new ConfigAttributeException(getAttributeName(), "am-validate-complete-req-attr-missing", new Object[] { attrName });
                                }
                            }
                            /*  else this is an optional attribute that has not
                                been set: OK.   */
                        }
                    }
                }
            }
            else
            {   /*  This is an unenumerated attribute map: validate that the
                    required number of entries are present. */

                iterator = m_attrDescription.getAttributeDescriptionNames().iterator();
                if (!iterator.hasNext())
                {
                    throw new ConfigAttributeException(getAttributeName(), "am-validate-complete-invalid-config-type");
                }

                String attrName = (String)iterator.next();
                IAttributeDescription attrDesc = m_attrDescription.getAttributeDescription(attrName);

                Object tmp = attrDesc.getProperty(IAttributeDescription.MIN_OCCURS_PROPERTY);
                if (tmp != null && tmp instanceof Long)
                {   /* Determine if minimum required entries are present. */
                    long min = ((Long)tmp).longValue();
                    if (size() < min)
                    {
                        throw new ConfigAttributeException(getAttributeName(), "am-validate-complete-min-entries", new Object[] { new Integer(size()), new Long(min) });
                    }
                }

                tmp = attrDesc.getProperty(IAttributeDescription.MAX_OCCURS_PROPERTY);
                if (tmp != null && tmp instanceof Long)
                {
                    long max = ((Long)tmp).longValue();
                    if (!tmp.equals(IAttributeDescription.LENGTH_UNBOUNDED) &&
                         size() > max)
                    {
                        throw new ConfigAttributeException(getAttributeName(), "am-validate-complete-max-entries", new Object[] { new Integer(size()), new Long(max) });
                    }
                }

            }
        }
    }

    @Override
    public String toString()
    {
        return toString(0, false);
    }

    public String toString(int indent, boolean isFromPrototype)
    {
        StringBuffer buffer = new StringBuffer();
        StringBuffer indentBuf = new StringBuffer();
        for (int i = 0; i < indent; i++)
        {
            indentBuf.append("    ");
        }
        String indentStr = indentBuf.toString();

        Iterator attrNames = getAllAttributeNames().iterator();
        while (attrNames.hasNext())
        {
            String attrName = (String) attrNames.next();
            Object attrValue = getInternal(attrName);

            if (attrValue == null)
            {
                continue;
            }

            String attrSource = null;
            boolean attributeIsFromPrototype = false;
            if (isFromPrototype)
            {
                attrSource = ":P";
                attributeIsFromPrototype = true;
            }
            else
            {
                IAttributeMetaData metaData = getAttributeMetaData(attrName);
                if (metaData.isFromConfigBean())
                {
                    attrSource = ":B";
                }
                else
                if (metaData.isFromPrototype())
                {
                    attrSource = ":P";
                    attributeIsFromPrototype = true;
                }
                else
                {
                    attrSource = ":T";
                }
            }

            String className = attrValue.getClass().getName();
            String typeName  = className.substring(className.lastIndexOf(".") + 1);

            if (attrValue instanceof IAttributeMap)
            {
                buffer.append(indentStr).append(attrName + "  [" + typeName + attrSource + "] =\n");
                buffer.append(indentStr).append("{\n");
                buffer.append(((AttributeMapImpl)attrValue).toString(indent + 1, attributeIsFromPrototype));
                buffer.append(indentStr).append("}\n");
            }
            else
            if (attrValue instanceof IAttributeList)
            {
                buffer.append(indentStr).append(attrName + "  [" + typeName + attrSource + "] =\n");
                buffer.append(indentStr).append("{\n");
                buffer.append(((AttributeListImpl)attrValue).toString(indent + 1, attributeIsFromPrototype));
                buffer.append(indentStr).append("}\n");
            }
            else
            {
                buffer.append(indentStr).append(attrName + "  [" + typeName + attrSource + "] = " + Util.toString(attrValue) + "\n");
            }
        }
        return buffer.toString();
    }

    @Override
    public Object getAttribute(String attributeName)
    {
        return get(attributeName);
    }

    @Override
    public IAttributeMetaData getAttributeMetaData(String attributeName)
    {
        if (_get(attributeName) != null)
        {
            return AttributeMetaData.s_isFromConfigBean;
        }
        else
        if(getProtoMap() != null && ((AttributeMapImpl)getProtoMap())._get(attributeName) != null)
        {
            return AttributeMetaData.s_isFromPrototype;
        }
        else
        if (get(attributeName) != null)
        {
            return AttributeMetaData.s_isFromConfigType;
        }
        return null;
    }

    /**
     *
     */
    @Override
    public Object
    getAttribute(IConfigPath attributeName)
    {
        switch (attributeName.size())
        {
        case 0:
            return null;
        case 1:
            return get(attributeName.getLastComponent());
        default:
            Object tmp = getFromInstanceOrTypeOnly(attributeName.getFirstComponent());
            if (tmp != null)
            {
                if (tmp instanceof IAttributeMap)
                {
                    return ((AttributeMapImpl)tmp).getAttribute(attributeName.subPath(1));
                }
                else if (tmp instanceof IAttributeList)
                {
                    return ((AttributeListImpl)tmp).getAttribute(attributeName.subPath(1));
                }
            }
            /*  Intermediate path component either does not exist or is not a composite type: return null. */
            return null;
        }
    }

    @Override
    public IAttributeMetaData
    getAttributeMetaData(IConfigPath path)
    {
        switch (path.size())
        {
        case 0:
            return null;
        case 1:
            return getAttributeMetaData(path.getLastComponent());
        default:
            Object tmp = null;
            IAttributeMetaData metaData = getAttributeMetaData(path.getFirstComponent());
            if (metaData == null)
            {
                return null;
            }
            else
            if (metaData.isFromPrototype())
            {
                tmp = getAttribute(path);
                if (tmp == null)
                {
                    return null;
                }
                return AttributeMetaData.s_isFromPrototype;
            }
            else
            if (metaData.isFromConfigType())
            {
                tmp = getAttribute(path);
                if (tmp == null)
                {
                    return null;
                }
                return AttributeMetaData.s_isFromConfigType;
            }
            /* else metaData.isFromConfigBean() */
            tmp = getAttribute(path.getFirstComponent());
            if (tmp != null)
            {
                if (tmp instanceof IAttributeMap)
                {
                    return ((AttributeMapImpl)tmp).getAttributeMetaData(path.subPath(1));
                }
                else if (tmp instanceof IAttributeList)
                {
                    return ((AttributeListImpl)tmp).getAttributeMetaData(path.subPath(1));
                }
            }
        }
        return null;
    }

    @Override
    public void setAttribute(String attributeName,
                             Object attributeValue)
        throws ConfigAttributeException
    {
        setAttribute(attributeName, attributeValue, true);
    }

    protected void setAttribute(String attributeName,
                                Object attributeValue,
                                boolean modify)
        throws ConfigAttributeException
    {
        try
        {
            put(attributeName,attributeValue, modify);
        }
        catch (ConfigServiceRuntimeException e)
        {
            throw (ConfigAttributeException) e.getLinkedException();
        }
    }

    @Override
    public final void setAttribute(IConfigPath attributeName,
                             Object attributeValue)
        throws ConfigAttributeException
    {
        try
        {
            switch (attributeName.size())
            {
            case 0:
                return;
            case 1:
                setAttribute(attributeName.getLastComponent(), attributeValue);
                return;
            default:
                Object tmp = getFromInstanceOrTypeOnly(attributeName.getFirstComponent());
                if (tmp != null)
                {
                    if (tmp instanceof IAttributeMap)
                    {
                        ((AttributeMapImpl)tmp).setAttribute(attributeName.subPath(1), attributeValue);
                    }
                    else if (tmp instanceof IAttributeList)
                    {
                        ((AttributeListImpl)tmp).setAttribute(attributeName.subPath(1), attributeValue);
                    }
                    else
                    {   /*  Not a collection type.  */
                        throw new ConfigAttributeException(getAttributeName().append(attributeName), "am-set-attr-cp-path-cmpnt-not-collection", new Object[] { getAttributeName().append(attributeName.getFirstComponent()) });
                    }
                }
                else
                {   /*  Intermediate path component does not exist: Throw exception. */
                    throw new ConfigAttributeException(getAttributeName().append(attributeName), "am-set-attr-cp-path-cmpnt-does-not-exist", new Object[] { getAttributeName().append(attributeName.getFirstComponent()) });
                }
                return;
            }
        }
        catch (ConfigAttributeException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigAttributeException(getAttributeName().append(attributeName), "am-set-attr-cp-failed", new Object[] { getAttributeName().append(attributeName.getFirstComponent()) }, e);
        }
    }

    @Override
    public Object
    removeAttribute(String attributeName)
    throws ConfigAttributeException
    {
        try
        {
            return remove(attributeName);
        }
        catch (ConfigServiceRuntimeException e)
        {
            throw (ConfigAttributeException) e.getLinkedException();
        }
    }

    @Override
    public Object
    removeAttribute(IConfigPath attributeName)
    throws ConfigAttributeException
    {
        try
        {
            switch (attributeName.size())
            {
            case 0:
                return null;
            case 1:
                return removeAttribute(attributeName.getLastComponent());
            default:
                Object tmp = getAttribute(attributeName.getFirstComponent());
                if (tmp != null)
                {
                    if (tmp instanceof IAttributeMap)
                    {
                        return ((AttributeMapImpl)tmp).removeAttribute(attributeName.subPath(1));
                    }
                    else if (tmp instanceof IAttributeList)
                    {
                        return ((AttributeListImpl)tmp).removeAttribute(attributeName.subPath(1));
                    }
                }
                /*  Intermediate path component either does not exist or is not a composite type: return null. */
                return null;
            }
        }
        catch (ConfigAttributeException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigAttributeException(getAttributeName().append(attributeName), "am-remove-attr-cp-failed", new Object[] { getAttributeName().append(attributeName.getFirstComponent()) }, e);
        }
    }

    @Override
    public IAttributeMap
    getAttributes(Set attributeNames)
    {
        IAttributeMap map = new AttributeMapImpl(m_configServer);
        Iterator names = attributeNames.iterator();
        while (names.hasNext())
        {
            String name = (String) names.next();
            map.put(name,get(name));
        }
        return map;
    }

    @Override
    public void
    setAttributes(IAttributeMap attributes)
    throws ConfigAttributeException
    {
        try
        {
            putAll(attributes);
        }
        catch (ConfigServiceRuntimeException e)
        {
            throw (ConfigAttributeException) e.getLinkedException();
        }
   }

    @Override
    public IAttributeMap
    removeAttributes(Set attributeNames)
    throws ConfigAttributeException
    {
        try
        {
            IAttributeMap map = new AttributeMapImpl(m_configServer);
            Iterator names = attributeNames.iterator();
            while (names.hasNext())
            {
                String name = (String) names.next();
                Object attribute = remove(name);
                map.put(name,attribute);
            }
            return map;
        }
        catch (ConfigServiceRuntimeException e)
        {
            throw (ConfigAttributeException) e.getLinkedException();
        }
    }

    @Override
    public Set
    getAttributeNames()
    {
        HashSet names = new HashSet();

        if (getProtoMap() != null)
        {
            names.addAll(getProtoMap().keySet());
        }

        Iterator it = super.keySet().iterator();
        while (it.hasNext())
        {
            Object key = it.next();
            if (_get(key) instanceof RemovedAttribute)
            {
                names.remove(key);
                continue;
            }
            names.add(key);
        }
        return new NamesSet(names);
    }

    public Set
    getAllAttributeNames()
    {
        Set names = new HashSet();

        if (getProtoMap() != null)
        {
            names.addAll(getProtoMap().keySet());
        }

        Iterator it = super.keySet().iterator();
        while (it.hasNext())
        {
            Object key = it.next();
            names.add(key);
        }

        if (m_attrDescription != null)
        {
            names.addAll(m_attrDescription.getAttributeDescriptionNames());
        }

        return new NamesSet(names);
    }

    // This is an internal NameSet that is returned on calls to things like
    // keySet(). It ensures that when items are deleted from this Set they are
    // also deleted from the underlying AttributeMap
    private class NamesSet extends HashSet
    {
        public NamesSet(Set names)
        {
            super(names);
        }

        @Override
        public Iterator iterator()
        {
            return new NamesIterator(this, super.iterator());
        }

        @Override
        public boolean remove(Object o)
        {
            // On a remove from the NameSet we want to remove from this HashSet
            // and also remove from the userlying AttributeMapImpl
            // return HashMap.this.removeEntryForKey(o) != null;
            AttributeMapImpl.this.remove(o);
            return super.remove(o);
        }

        @Override
        public void clear()
        {
            // On a remove from the NameSet we want to remove from this HashSet
            // and also remove from the userlying AttributeMapImpl
            AttributeMapImpl.this.clear();
            super.clear();
        }
    }

    // The iterator used on the above NamesSet. It ensures that when items are
    // deleted from this Iterator they are also deleted from the underlying
    // NamesSet (and therefore also from the AttributeMap)
    // This Iterator simple wraps around the underlying NamesSet iterator.
    private class NamesIterator implements Iterator
    {
        private NamesSet m_names    = null;
        private Iterator m_iterator = null;
        private Object   m_current  = null;

        NamesIterator(NamesSet names, Iterator iter)
        {
            m_names    = names;
            m_iterator = iter;
        }

        @Override
        public boolean hasNext()
        {
            return m_iterator.hasNext();
        }

        @Override
        public Object next()
        {
            return m_current = m_iterator.next();
        }

        @Override
        public void remove()
        {
            if (m_current == null)
            {
                throw new IllegalStateException();
            }

            m_iterator.remove();
            m_names.remove(m_current);
            m_current = null;
        }
    }

    @Override
    public IAttributeDescription
    getAttributeDescription()
    {
        return m_attrDescription;
    }

    @Override
    public IAttributeMap
    createAttributeMap(String name)
    throws ConfigAttributeException
    {
        try
        {
            if (m_attrDescription != null)
            {
                IAttributeDescription attrDesc = null;
                if (m_attrDescription.getProperty(IAttributeDescription.UNENUM_MAP_PROPERTY) != null)
                {
                    Iterator iterator = m_attrDescription.getAttributeDescriptionNames().iterator();
                    if (!iterator.hasNext())
                    {
                        throw new ConfigAttributeException(getAttributeName(),
                            "am-create-am-not-defined", new Object[]
                            {name});
                    }

                    String attrName = (String)iterator.next();
                    attrDesc = m_attrDescription.getAttributeDescription(attrName);
                }
                else
                {
                    if (name == null)
                    {
                        throw new ConfigAttributeException(getAttributeName(),
                            "am-create-am-is-null", new Object[]
                            {name});
                    }
                    attrDesc = m_attrDescription.
                        getAttributeDescription(name);
                }

                if (attrDesc == null)
                {
                    throw new ConfigAttributeException(getAttributeName(),
                        "am-create-am-not-defined", new Object[]
                        {name});
                }
                else
                if (attrDesc.getType() != IAttributeMap.class)
                {
                    throw new ConfigAttributeException(getAttributeName(),
                        "am-create-am-not-map", new Object[]
                        {name});
                }
                return new AttributeMapImpl(attrDesc);
            }
            else
            {
                return new AttributeMapImpl(m_configServer);
            }
        }
        catch (ConfigAttributeException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigAttributeException(getAttributeName(), "am-create-am-failed", new Object[] { name }, e);
        }
    }

    private IAttributeList
    createAttributeList(String name, IAttributeList list, boolean modify)
    throws ConfigAttributeException
    {
        IAttributeList newList = createAttributeList(name);

        short oldState = (m_owner != null) ? ((ConfigElementImpl)m_owner).getState() : -1;

        newList.addAttributes(list);

        if (modify && oldState > -1)
        {
            if (m_owner != null)
            {
                ((ConfigElementImpl)m_owner).setState(oldState);
            }
        }

        return newList;
    }

    @Override
    public IAttributeList
    createAttributeList(String name)
    throws ConfigAttributeException
    {
        try
        {
            if (m_attrDescription != null)
            {
                if (name == null)
                {
                    throw new ConfigAttributeException(getAttributeName(), "am-create-al-is-null", new Object[] { name });
                }
                IAttributeDescription attrDesc = m_attrDescription.getAttributeDescription(name);
                if (attrDesc == null)
                {
                    throw new ConfigAttributeException(getAttributeName(), "am-create-al-not-defined", new Object[] { name });
                }
                else
                if (attrDesc.getType() != IAttributeList.class)
                {
                    throw new ConfigAttributeException(getAttributeName(), "am-create-al-not-list", new Object[] { name });
                }
                return new AttributeListImpl(attrDesc);
            }
            else
            {
                return new AttributeListImpl(m_configServer);
            }
        }
        catch (ConfigAttributeException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigAttributeException(getAttributeName(), "am-create-al-failed", new Object[] { name }, e);
        }
    }

    public Object _get(Object key)
    {
        return super.get(key);
    }

    public Object _get(IConfigPath attributeName)
    {
        switch (attributeName.size())
        {
        case 0:
            return null;
        case 1:
            return _get(attributeName.getLastComponent());
        default:
            Object tmp = _get(attributeName.getFirstComponent());
            if (tmp != null)
            {
                if (tmp instanceof IAttributeMap)
                {
                    return ((AttributeMapImpl)tmp)._get(attributeName.subPath(1));
                }
                else if (tmp instanceof IAttributeList)
                {
                    return ((AttributeListImpl)tmp)._get(attributeName.subPath(1));
                }
            }
            /*  Intermediate path component either does not exist or is not a composite type: return null. */
            return null;
        }
    }

    /*
     *  Methods overriding behavior in java.util.HashMap. This Prevents keys
     *  that are not of type java.lang.String and values that are not supported types
     *  from being added to map.
     */
    @Override
    public Object get(Object key)
    {
        try
        {
            Object tmp = getInternal(key);
            if (tmp instanceof ConfigReference)
            {
                try
                {
                    tmp = ((ConfigReference)tmp).getConfigElement();
                }
                catch (ConfigServiceException e)
                {
                    if (e.getLinkedException() instanceof ManagementPermissionDeniedException) {
                        tmp = IdentityOnlyConfigBeanImpl.createIdentityOnlyConfigBean(m_configServer, ((ConfigReference) tmp));
                    } else {
                        tmp = null;
                    }
                }
            }
            else
            if (tmp instanceof RemovedAttribute)
            {
                tmp = null;
            }
            return tmp;
        }
        catch (Exception e)
        {
            throw new ConfigServiceRuntimeException(new ConfigAttributeException(getAttributeName().append((String) key), "am-get-failed", e));
        }
    }

    public Object
    getInternal(Object key)
    {
        try
        {
            Object tmp = super.get(key);

            if (tmp == null)
            {
                tmp = getFromPrototype(key);
            }

            if (tmp == null)
            {
                tmp = getFromType(key);
            }

            return tmp;
        }
        catch (Exception e)
        {
            throw new ConfigServiceRuntimeException(new ConfigAttributeException(getAttributeName().append((String) key), "am-get-failed", e));
        }
    }

    protected Object getFromInstanceOrTypeOnly(Object key)
    {
        Object tmp = super.get(key);

        if (tmp == null)
        {
            tmp = getFromType(key);
        }

        if (tmp == null)
        {
            tmp = getFromPrototype(key);

            if (tmp instanceof IAttributeMap)
            {
                return tmp;
            }

            if (tmp instanceof IAttributeList)
            {
                return tmp;
            }

            tmp = null;
        }

        if (tmp instanceof ConfigReference)
        {
            try
            {
                tmp = ((ConfigReference)tmp).getConfigElement();
            }
            catch (ConfigServiceException e)
            {
                tmp = null;
            }
        }
        else if (tmp instanceof RemovedAttribute)
        {
            tmp = null;
        }

        return tmp;
    }

    protected Object
    getFromPrototype(Object key)
    {
        try
        {
            Object tmp = null;

            if (getProtoMap() != null)
            {
                tmp = ((AttributeMapImpl)getProtoMap())._get(key);
            }

            if (tmp instanceof IAttributeMap)
            {
                try
                {
                    IAttributeMap map = createAttributeMap((String) key);
                    ((AttributeMapImpl)map).m_override = true;

                    // Calling set attribute here marks the owner as modified
                    // which it really isn't...we are only really going to
                    // modify it if we add any children under it.
                    //
                    // Solution - pass false into setAttribute so that it
                    // will skip the setModified bit
                    setAttribute((String) key, map, false);
                    map = (IAttributeMap) _get((String) key);

                    // We just created an empty/fresh AttributeMap as a temporary
                    // insertion point for the instance...but it doesn't have any
                    // connection to the instance...so we enforce that here...
                    ((AttributeMapImpl)map).m_instanceOwner = m_instanceOwner;

                    return map;
                }
                catch (ConfigServiceException e)
                {
                    new ConfigServiceRuntimeException(new ConfigAttributeException(getAttributeName().append((String) key), "am-get-failed", e));
                }
            }
            else if (tmp instanceof IAttributeList)
            {
                try
                {
                    IAttributeList list = createAttributeList((String) key, (IAttributeList)tmp, false);

                    // Calling set attribute here marks the owner as modified
                    // which it really isn't...we are only really going to
                    // modify it if we add any children under it.
                    //
                    // Solution - pass false into setAttribute so that it
                    // will skip the setModified bit
                    setAttribute((String) key, list, false);
                    list = (IAttributeList) _get((String) key);
                    return list;
                }
                catch (ConfigServiceException e)
                {
                    new ConfigServiceRuntimeException(new ConfigAttributeException(getAttributeName().append((String) key), "am-get-failed", e));
                }
            }
            return tmp;
        }
        catch (Exception e)
        {
            throw new ConfigServiceRuntimeException(new ConfigAttributeException(getAttributeName().append((String) key), "am-get-failed", e));
        }
    }

    protected Object
    getFromType(Object key)
    {
        try
        {
            Object tmp = null;

            if (key instanceof String &&
                m_attrDescription != null &&
                m_attrDescription.getProperty(IAttributeDescription.UNENUM_MAP_PROPERTY) == null)
            {   /*  Attribute is not set and this is a typed enumerated map:
                    attempt to retrieve a fixed or default value from the
                    attribute's description; if this attribute is a required
                    attribute map or list attribute create it on demand.   */

                IAttributeDescription attrDesc = m_attrDescription.getAttributeDescription((String) key);

                if (attrDesc != null)
                {
                    /*  Validate that this attribute is not optional: an optinal
                        attribute must have its value explicitly set inorder to
                        return a value */
                    tmp = attrDesc.getProperty(IAttributeDescription.MIN_OCCURS_PROPERTY);

                    if (tmp != null && tmp instanceof Long)
                    {
                        if(((Long)tmp).longValue() < 1)
                        {   /*  This is an optional attribute and its value
                                has not been explicitely set: return null   */
                            return null;
                        }
                    }

                    /*  Create required attribute maps and list on demand   */
                    if (attrDesc.getType() == IAttributeMap.class)
                    {
                        try
                        {
                            IAttributeMap map = createAttributeMap((String) key);
                            setAttribute((String) key, map, false);
                            map = (IAttributeMap) _get((String) key);
                            return map;
                        }
                        catch (ConfigServiceException e)
                        {
                            new ConfigServiceRuntimeException(new ConfigAttributeException(getAttributeName().append((String) key), "am-get-failed", e));
                        }
                    }
                    else
                    if (attrDesc.getType() == IAttributeList.class)
                    {
                        try
                        {
                            IAttributeList list = createAttributeList((String) key);
                            setAttribute((String) key, list, false);
                            list = (IAttributeList) _get((String) key);
                            return list;
                        }
                        catch (ConfigServiceException e)
                        {
                            new ConfigServiceRuntimeException(new ConfigAttributeException(getAttributeName().append((String) key), "am-get-failed", e));
                        }
                    }

                    /*  Otherwise attempt to retreive a fixed or default value */
                    if ((tmp = attrDesc.getProperty(IAttributeDescription.FIXED_PROPERTY)) != null)
                    {
                        return tmp;
                    }

                    if ((tmp = attrDesc.getProperty(IAttributeDescription.DEFAULT_PROPERTY)) != null)
                    {
                        return tmp;
                    }
                }
                /*  else this attribute is not defined: return null.    */
            }
            return tmp;
        }
        catch (Exception e)
        {
            throw new ConfigServiceRuntimeException(new ConfigAttributeException(getAttributeName().append((String) key), "am-get-failed", e));
        }
    }

    /**
     * Removes all mappings from this map.
     *
     * In the special case of an attribute map in an instance element, we
     * do not remove any RemovedAttribute entries as they are actively masking
     * attributes from the linked template. Likewise, for an instance, entries
     * are removed using the specialized remove method that will populate
     * locally removed attributes with the RemovedAttribute placeholder in order
     * to mask out the corresponding attribute from the template.
     */
    @Override
    public void
    clear()
    {
        boolean isInstance = m_instanceOwner == null ? false : m_instanceOwner.isPrototypeInstance();
        Object[] objs = super.keySet().toArray();
        for (int i = 0; i < objs.length; i++)
        {
            if (isInstance && _get(objs[i]) instanceof RemovedAttribute)
            {
                continue;
            }

            remove(objs[i], isInstance);
        }
    }

    public void
    _clear()
    {
        Object[] objs = super.keySet().toArray();
        for (int i = 0; i < objs.length; i++)
        {
            remove(objs[i], false);
        }
    }

    private void _clearMap()
    {
        super.clear();
    }

    @Override
    public boolean
    containsValue(Object value)
    {
        Iterator values = values().iterator();
        if (value == null)
        {
            while(values.hasNext())
            {
                if (values.next() == null)
                {
                    return true;
                }
            }
        }
        else
        {
            while(values.hasNext())
            {
                Object obj = values.next();

                if (obj == null)
                {
                    continue;
                }

                if(obj.equals(value))
                {
                    return true;
                }
            }
        }
        return false;
    }

    @Override
    public Object put(Object key, Object value)
        throws ConfigServiceRuntimeException
    {
        return put(key, value, true);
    }

    protected Object put(Object key, Object value, boolean modify)
        throws ConfigServiceRuntimeException
    {
        boolean isInstance = (m_instanceOwner != null) ?
                              m_instanceOwner.isPrototypeInstance() :
                              false;

        return put(key, value, modify, !isInstance);
    }

    /**
     * Associates the specified value with the specified key in this map.
     * If the map previously contained a mapping for this key, the old value is replaced.
     */
    private Object
    put(Object key, Object value, boolean modify, boolean filterDefaults)
    throws  ConfigServiceRuntimeException
    {
        try
        {
            //  Ensure key is valid before subsequent String casts are made.
            Util.validateAttributeName(key);

            value = Util.validateAttributeValue(getAttributeName().append((String)key),
                                                value,
                                                m_configServer,
                                                m_attrDescription,
                                                filterDefaults);

            // if we are filtering default values from the attribute map and
            // the value we are setting is the same as the default then
            // we want to remove this attribute from the attribute map.
            //
            // Unless this attributeMap is part of a template !!
            //
            if (filterDefaults && m_configServer.m_filterDefaults &&
                Util.isDefaultValue((String) key, value, m_attrDescription))
            {
                removeAttribute((String) key);
                return value;
            }

            setOwnership((String) key, value);

            Object obj = super.put(key, value);

            removeOwnership(obj);

            if (m_owner != null && modify)
            {
                ((ConfigElementImpl)m_owner).setModified();
            }

            return obj;
        }
        catch (ConfigAttributeException e)
        {
            throw new ConfigServiceRuntimeException(e);
        }
        catch (Exception e)
        {
            throw new ConfigServiceRuntimeException(new ConfigAttributeException(getAttributeName().append((String) key), "am-put-failed", e));
        }
    }

    /**
     * Copies all of the mappings from the specified map to this map. These
     * mappings will replace any mappings that this map had for any of the keys currently in the
     * specified map.
     */
    @Override
    public final void
    putAll(Map mappings)
    throws  ConfigServiceRuntimeException
    {
        Iterator entries = mappings.entrySet().iterator();
        while(entries.hasNext())
        {
            Map.Entry entry = (Map.Entry) entries.next();
            put(entry.getKey(), entry.getValue());
        }
    }

    @Override
    public Object
    remove(Object key)
    throws ConfigServiceRuntimeException
    {
        return remove(key, true);
    }

    protected Object
    remove(Object key, boolean populateRemovedAttributes)
    throws ConfigServiceRuntimeException
    {
        try
        {
            Object obj = null;
            if (m_instanceOwner != null &&
                m_instanceOwner.isPrototypeInstance() &&
                populateRemovedAttributes)
            {
                obj = super.put(key, new RemovedAttribute());
            }
            else
            {
                obj = super.remove(key);
            }

            removeOwnership(obj);

            if (obj instanceof ConfigReference)
            {
                obj = ((ConfigReference)obj).getConfigElement();
            }

            if (m_owner != null)
            {
                ((ConfigElementImpl)m_owner).setModified();
            }

            return obj;
        }
        catch (Exception e)
        {
            throw new ConfigServiceRuntimeException(new ConfigAttributeException(getAttributeName().append((String) key), "am-remove-failed", e));
        }
    }

    @Override
    public boolean
    equals(Object obj)
    {
        if (obj != null && (obj.getClass().equals(this.getClass())))
        {
            return m_identity == ((AttributeMapImpl) obj).m_identity;
        }
        return false;
    }

    @Override
    public int
    hashCode()
    {
        return m_identity.hashCode();
    }

    @Override
    public Set
    keySet()
    {
        return getAttributeNames();
    }

    protected Set
    _keySet()
    {
        return super.keySet();
    }

    /**
     * Returns the number of key-value mappings in this map.
     *
     * If the map is part of an instance linked to a prototype, then the number
     * returned is the intersection of the key-value mappings in this map and
     * its linked prototype map.
     *
     * @return  the number of key-value mappings
     */
    @Override
    public int
    size()
    {
        return keySet().size();
    }

    /**
     * Returns the number of key-value mappings in this map.
     *
     * This method returns the underlying HashMap size...if the map is an
     * instance then ONLY the local instance attribute count is returned.
     *
     * @return  the number of key-value mappings
     */
    public int
    _size()
    {
        return super.size();
    }

    /** Overload values method so that returned collection uses our
     *  'ValueIterator' class; this in to ensure that ConfigElements, not
     *  ConfigReferences, are returned by interator's 'next' method.
     */
    Values m_values = null;
    @Override
    public Collection values()
    {
        Collection vs = m_values;
        return (vs != null ? vs : (m_values = new Values()));
    }

    private class Values
    extends AbstractCollection
    {
        @Override
        public Iterator iterator() {
            return new ValueIterator();
        }
        @Override
        public int size() {
            return AttributeMapImpl.this.size();
        }
        @Override
        public boolean contains(Object o) {
            return containsValue(o);
        }
        @Override
        public void clear() {
            AttributeMapImpl.this.clear();
        }
    }

    private abstract class MapIterator
    implements Iterator
    {
        Iterator m_names = null;
        Object m_current = null;

        MapIterator()
        {
            m_names = keySet().iterator();
        }

        @Override
        public boolean
        hasNext()
        {
            return m_names.hasNext();
        }

        @Override
        public void remove()
        {
            if (m_current == null)
            {
                throw new IllegalStateException();
            }

            AttributeMapImpl.this.remove(m_current);
            m_current = null;
        }
    }

    private class ValueIterator
    extends MapIterator
    {

        @Override
        public Object next()
        {
            return m_current = get(m_names.next());
        }

    }

    /* End methods overriding behavior in java.util.HashMap.
     */

    /*  Methods enforcing Config Object Ownership
     */

    public Object
    getParent()
    {
        return m_parent;
    }

    protected void
    setParent(String attributeName, Object parent)
    throws ConfigAttributeException
    {
        if (parent != null &&
            !(parent instanceof AttributeMapImpl) &&
            !(parent instanceof AttributeListImpl))
        {
            throw new ConfigAttributeException(getAttributeName().append(attributeName), "am-set-parent-failed");
        }
        if (parent == this)
        {
            m_parent = null;
        }
        else
        {
            m_parent = parent;
            m_attributeName = attributeName;
        }
    }

    protected IAttributeMap
    getProtoMap()
    {
        IAttributeMap map = null;

        if (m_instanceOwner != null)
        {
            IConfigPrototype prototype = m_instanceOwner.getPrototype();

            if (prototype != null)
            {
                try
                {
                  if (getAttributeName().size() == 0)
                {
                    return null;
                }
                else
                {
                    map = (IAttributeMap)prototype.getAttribute(getAttributeName().subPath(1));
                }
                }
                catch (Throwable e)
                {
                  e.printStackTrace();
                }
            }
        }

        return map;
    }

    /*  Return a reference to the config element that contains and owns this map.
     *  If this map is not contained in any config element, a <code>null</code>
     *  is returned.
     */
    public IConfigElement
    getOwner()
    {
        return m_owner;
    }

    public IConfigElement
    getInstanceOwner()
    {
        return m_instanceOwner;
    }

    protected void
    getSubElements(Set subElements)
    throws ConfigServiceException
    {
        Iterator refs = m_refs.keySet().iterator();
        while (refs.hasNext())
        {
            ConfigReference ref = (ConfigReference) refs.next();
            /*  Resolve referenced bean only if already loaded and add
                it to sub-element list. */
            IConfigElement element = ref.getLocalConfigElement();
            if (element != null)
            {
                if (!subElements.contains(element))
                {
                    subElements.add(element);
                    ((ConfigElementImpl)element).getSubElements(subElements);
                }
            }
        }

        Iterator maps = m_maps.iterator();
        while (maps.hasNext())
        {
            AttributeMapImpl map = (AttributeMapImpl) maps.next();
            map.getSubElements(subElements);
        }

        Iterator lists = m_lists.iterator();
        while (lists.hasNext())
        {
            AttributeListImpl list = (AttributeListImpl) lists.next();
            list.getSubElements(subElements);
        }
    }

    /*  Set the owner of this map.
     *  The owner of this map must contain it.
     *  @exception  ConfigServiceException thrown if this map
     *              is already contained in another config element.
     */
    protected final void
    setOwner(IConfigElement configElement)
    throws ConfigServiceException
    {
        if (m_owner != null && configElement != null)
        {
            new ConfigServiceException("am-set-owner-already-owned", new Object[]{ configElement.getName(), m_owner.getName() });
        }

        /*  Change ownership of all contained maps, lists and references */
        Iterator maps = m_maps.iterator();
        while (maps.hasNext())
        {
            AttributeMapImpl map = (AttributeMapImpl) maps.next();
            map.setOwner(configElement);
        }
        Iterator lists = m_lists.iterator();
        while (lists.hasNext())
        {
            AttributeListImpl list = (AttributeListImpl) lists.next();
            list.setOwner(configElement);
        }
        Iterator refs = m_refs.keySet().iterator();
        while (refs.hasNext())
        {
            ConfigReference ref = (ConfigReference) refs.next();
            ref.setOwner(configElement);
        }

        m_owner = configElement;

        if ((m_owner != null) && (m_owner != m_instanceOwner))
        {
            m_instanceOwner = m_owner;
        }
    }

    /*  Set the ownership of obj to this list's config object.
     *  @exception  ConfigServiceException thrown if obj is already
     *              contained in another config element.
     */
    private void
    setOwnership(String attributeName, Object obj)
    throws ConfigServiceException
    {
        if (obj instanceof AttributeMapImpl)
        {
            ((AttributeMapImpl)obj).setOwner(m_owner);
            ((AttributeMapImpl)obj).setParent(attributeName, this);
            m_maps.add(obj);
        }
        else if (obj instanceof AttributeListImpl)
        {
            ((AttributeListImpl)obj).setOwner(m_owner);
            ((AttributeListImpl)obj).setParent(attributeName, this);
            m_lists.add(obj);
        }
        else if (obj instanceof ConfigReference)
        {
            ((ConfigReference)obj).setOwner(m_owner);
            m_refs.put(obj, obj);
        }
    }

    /*  Remove the ownership from obj.
     */
    private void
    removeOwnership(Object obj)
    throws ConfigServiceException
    {
        if (obj instanceof AttributeMapImpl)
        {
            if (((AttributeMapImpl)obj).m_override)
            {
                return;
            }

            ((AttributeMapImpl)obj).setOwner(null);
            ((AttributeMapImpl)obj).setParent(null, null);
            m_maps.remove(obj);
        }
        else if (obj instanceof AttributeListImpl)
        {
            ((AttributeListImpl)obj).setOwner(null);
            ((AttributeListImpl)obj).setParent(null, null);
            m_lists.remove(obj);
        }
        else if (obj instanceof ConfigReference)
        {
            ((ConfigReference)obj).setOwner(null);
            m_refs.remove(obj);
        }
    }

    /*  End methods enforcing Config Object Ownership
     */

    @Override
    public String toXML()
    {
        StringBuffer buffer = new StringBuffer();

        for(Iterator iter = super.keySet().iterator(); iter.hasNext(); )
        {
            String key = (String)iter.next();

            if (key.equals("_MF_SYSTEM_ATTRIBUTES") || key.equals(IAnnotationExtension.TOOL_ANNOTATION))
            {
                continue;
            }

            String content = elementToString(key, get(key));

            if(content != null)
            {
                buffer.append(content);
            }
        }
        return buffer.toString();
    }

    private String elementToString(String name, Object value)
    {
        Set descSet = m_attrDescription.getAttributeDescriptionNames();

        String innerName = name;
        String header    = null;
        String content   = null;
        String footer    = null;

        // check to see if the element name is specified in the descriptions
        if(descSet.contains(name))
        {
            header = "<" + name + ">";
        }
        else
        {
            // it isn't so check that there is only a single description
            // and if there use that as the element type

            if(descSet.size() == 1)
            {
                String[] names = (String[])descSet.toArray(new String[descSet.size()]);

                innerName = names[0];

                header = "<" + innerName + " name=\"" + name + "\">";
            }
        }

        boolean crContent = false;

        if(value instanceof ConfigBeanImpl)
        {
            content = Util.name2Url(((ConfigBeanImpl)value).getName());
        }
        else
        if(value instanceof AttributeMapImpl)
        {
            content = ((AttributeMapImpl)value).toXML();
            crContent = true;
        }
        else
        if(value instanceof AttributeListImpl)
        {
            content = ((AttributeListImpl)value).toXML();
            crContent = true;
        }
        else
        {
            if(!Util.isDefaultValue(name, value, m_attrDescription))
            {
                content = Util.toString(value);
            }
        }

        footer = "</" + innerName + ">\r\n";

        String ret = null;

        if(content != null && content.length() > 0)
        {
            if(crContent)
            {
                ret = header + "\r\n" + content + footer;
            }
            else
            {
                ret = header + content + footer;
            }
        }
        return ret;
    }

    public IConfigPath
    getAttributeName()
    {
        IConfigPath attributeName = null;
        if (m_parent != null)
        {
            if (m_parent instanceof AttributeMapImpl)
            {
                attributeName = ((AttributeMapImpl)m_parent).getAttributeName();
                attributeName.append(new String[] { m_attributeName });
            }
            else // m_parent instanceof AttributeListImpl
            {
                attributeName = ((AttributeListImpl)m_parent).getAttributeName();
                int index = ((AttributeListImpl)m_parent).indexOf(this);
                attributeName.append(String.valueOf(index));
            }
        }
        else
        {
                attributeName = new ConfigPathImpl();
        }

        return attributeName;
    }

    @Override
    public void
    addConfigChangeListener(IConfigPath attributeName, IConfigChangeListener listener, Object handback)
    throws ConfigServiceException
    {
        if (m_owner != null)
        {
            if (m_owner.getName() == null)
            {
                throw new ConfigServiceException("am-add-listener-name-null");
            }
            AttributeChangeFilter filter = new AttributeChangeFilter();
            filter.enableAttribute(getAttributeName().subPath(1).append(attributeName));
            m_owner.getConfigServer().addConfigChangeListener(m_owner.getName(),listener, filter, handback);
        }
        else
        {
            throw new ConfigServiceException("am-add-listener-not-owned");
        }
    }

    @Override
    public void
    removeConfigChangeListener(IConfigPath attributeName, IConfigChangeListener listener, Object handback)
    throws ConfigServiceException
    {
        if (m_owner != null)
        {
            if (m_owner.getName() == null)
            {
                throw new ConfigServiceException("am-remove-listener-name-null");
            }
            AttributeChangeFilter filter = new AttributeChangeFilter();
            filter.enableAttribute(getAttributeName().subPath(1).append(attributeName));
            m_owner.getConfigServer().removeConfigChangeListener(m_owner.getName(),listener, filter, handback);
        }
        else
        {
            throw new ConfigServiceException("am-remove-listener-not-owned");
        }
    }
}
