package com.sonicsw.mx.config.impl;

import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import com.sonicsw.mx.config.ConfigServiceException;

/**
 * <p>Title: Cache.java</p>
 * <p>Description: Cache provides a way of holding a lookup table of objects
 * that are weakly referenced...so that when an object is no longer referenced
 * it can be "automatically" removed from the cache by scavanging.</p>
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: Sonic Software Corporation</p>
 * @author bcollins, hjones
 * @version 1.0
 */
public class Cache
{
    private HashMap m_map = new HashMap();
    private ReferenceQueue m_refQueue = new ReferenceQueue();

    public Set getObjects()
    {
        scavange();
        HashSet set = new HashSet();
        Iterator it = m_map.values().iterator();

        while (it.hasNext())
        {
            CacheRef ref = (CacheRef) it.next();

            if (ref.get() != null)
            {
                set.add(ref.get());
            }
        }

        return set;
    }

    /**
     * A version-dependant lookup for an object keyed under a specific path
     * and with a specific version. Only an object that directly matches the
     * specific path will be returned.
     *
     * @param path     The path to the expected object.
     * @param version  The version of the object required.
     * @return         The object if it is found, otherwise null.
     */
    public Object lookup(String path, String version)
    {
        scavange();

        CacheKey key = new CacheKey(path, version);
        CacheRef ref = (CacheRef)m_map.get(key);

        Object value = (ref != null) ? ref.get() : null;

        if (value == null)
        {
            m_map.remove(key);
        }

        return value;

    }

    /**
     * Finds all objects in the cache that have a key name starting with path.
     *
     * This method can search for all objects under a given path (folder) and
     * is version-independent.
     *
     * @param path  The path (to a folder or concrete object) under which all
     *              matching (starts with) objects will be returned.
     * @return      An array of objects that matched the path criteria.
     */
    public Object[] lookup(String path)
    {
        scavange();

        List     list = new ArrayList();
        Iterator i    = m_map.keySet().iterator();

        while (i.hasNext())
        {
            CacheKey key = (CacheKey)i.next();

            if (key.getName().startsWith(path))
            {
                Object obj = m_map.get(key);

                if (obj != null)
                {
                    list.add(((CacheRef)obj).get());
                }
            }
        }

        return list.toArray();
    }

    /**
     * Adds the supplied object into the cache using the path and version as
     * the key.
     *
     * @param path     The path under which the object can be found.
     * @param version  The version of the object being cached.
     * @param obj      The object being cached.
     *
     * @throws ConfigServiceException  An exception if the object already
     *                                 exists in the cache.
     */
    public void add(String path, String version, Object obj)
        throws ConfigServiceException
    {
        scavange();

        CacheKey key = new CacheKey(path, version);
        Object   tmp = m_map.put(key, new CacheRef(key, obj, m_refQueue));

        if (tmp != null)
        {   /*  Instance of named already exists in cache:
             *  replace previous instance in cache and throw exception.
             */
            m_map.put(key, tmp);
            throw new ConfigServiceException("Named instance of config objects already exists");
        }
    }

    /**
     * Tries to remove the object keyed in the cache under the given path and
     * version.
     *
     * @param path     The path of the object.
     * @param version  The version of the object.
     */
    public void remove(String path, String version)
    {
        scavange();

        m_map.remove(new CacheKey(path, version));
    }

    /**
     * The size of the cache.
     *
     * @return  The size of the cache (which must be a positive number >=0).
     */
    public int size()
    {
        return m_map.size();
    }

    private void scavange()
    {
        // Ensure that the garbage collector enqueues any weakly reachable
        // references
        Reference ref = null;

        while ((ref = m_refQueue.poll()) != null)
        {
            m_map.remove(((CacheRef)ref).getKey());
        }
    }

    @Override
    public String toString()
    {
        StringBuffer buffer = new StringBuffer();

        String className = this.getClass().getName();
        String typeName  = className.substring(className.lastIndexOf(".") + 1);

        buffer.append("[" + typeName + "] =\n");
        buffer.append("{\n");

        Iterator it = m_map.keySet().iterator();
        while (it.hasNext())
        {
            buffer.append("  ").append(it.next().toString()).append("\n");
        }

        buffer.append("}\n");

        return buffer.toString();
    }

    //-------------------------------------------------------------------------

    static class CacheRef extends WeakReference
    {
        private CacheKey m_key;

        public CacheRef(CacheKey key, Object referent)
        {
            super(referent);
            m_key = key;
        }

        public CacheRef(CacheKey key, Object referent, ReferenceQueue queue)
        {
            super(referent, queue);
            m_key = key;
        }

        public CacheKey getKey()
        {
            return m_key;
        }
    }

    //-------------------------------------------------------------------------

    static class CacheKey
    {
        private String m_name;
        private String m_version;

        public CacheKey(String name, String version)
        {
            m_name = (name == null) ? "" : name;
            m_version = (version == null) ? "" : version;
        }

        public String getName()
        {
            return m_name;
        }

        public String getVersion()
        {
            return m_version;
        }

        @Override
        public boolean equals(Object obj)
        {
            if (obj != null && this.getClass() == obj.getClass())
            {
                CacheKey key = (CacheKey) obj;
                if (m_name.equals(key.m_name) && m_version.equals(key.m_version))
                {
                    return true;
                }
            }
            return false;
        }

        @Override
        public int hashCode()
        {
            return m_name.hashCode();
        }

        @Override
        public String toString()
        {
            return "(" + m_name + "," + m_version + ")";
        }
    }
}
