/*
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.mx.config.impl;

import java.util.Map;

import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.ConfigServiceRuntimeException;
import com.sonicsw.mx.config.IAnnotationExtension;
import com.sonicsw.mx.config.IAttributeDescription;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigPrototype;
import com.sonicsw.mx.config.IConfigType;

import com.sonicsw.mf.common.dirconfig.IDirElement;

public class ConfigBeanImpl extends ConfigElementImpl implements IConfigBean
{
    IConfigType m_configType = null;


    protected ConfigBeanImpl(String       name,
                             String       type,
                             String       version,
                             ConfigServer configServer)
        throws ConfigServiceException
    {
        this(name, type, version, configServer, false);
    }

    protected ConfigBeanImpl(String       name,
                             String       type,
                             String       version,
                             ConfigServer configServer,
                             boolean      isInstance)
        throws ConfigServiceException
    {
        super(name, type, version, configServer);

        if ((type == null) || (version == null))
        {
            throw new ConfigServiceException("cb-init-failed-no-ct");
        }

        m_configType = configServer.loadConfigType(type, version);   // Load config type
        if (m_configType == null)
        {
            throw new ConfigServiceException("cb-init-failed-ct-is-null", new Object[] {type, version});
        }

        // Retrieve type's attribute description to initialize and constrain
        // this bean's base attribute map
        m_attrDescription = m_configType;
        
        if (!isInstance && ((ConfigTypeImpl)m_configType).hasInitialValues())
        {
            ((ConfigTypeImpl)m_configType).applyInitialValuesTo(this);
        }
    }
 
    protected ConfigBeanImpl(IDirElement  dsBeanElement,
                             ConfigServer cs)
        throws ConfigServiceException
    {
        super(cs);

        // Retrieve name of config type from element identity
        String typeName = dsBeanElement.getIdentity().getType();
        String typeVersion = dsBeanElement.getIdentity().getReleaseVersion();
        if ((typeName == null) || (typeVersion == null))
        {
            throw new ConfigServiceException("cb-init-failed-no-ct");
        }

        m_configType = cs.loadConfigType(typeName, typeVersion);   //  Load config type
        if (m_configType == null)
        {
            throw new ConfigServiceException("cb-init-failed-ct-is-null", new Object[] {typeName, typeVersion});
        }

        // Assign config type to attribute description to initialize and
        // constrain this bean's base attribute map
        m_attrDescription = m_configType;

        // Initialize this bean's base config element
        super.init(dsBeanElement);

        validateComplete();
    }

    /** Construct a copy of a Config Bean
     *
     * @param name the name of copied config bean.
     * @param configBean the config bean to copy.
     * @param isPrototype if true, the copied bean is a prototype.
     * @throws ConfigServiceException if failed to construct the config bean.
     */
    protected ConfigBeanImpl(String         name,
                             ConfigBeanImpl configBean,
                             boolean        isPrototype)
        throws ConfigServiceException
    {
        super(name, configBean, isPrototype);     // Deep copy config element

        m_configType = configBean.m_configType;   // Same config type

        // Retrieve type's attribute description to initialize and constrain
        // this bean's base attribute map
        m_attrDescription = m_configType;
    }

    @Override
    protected void refreshFromDSElement(IDirElement dsElement)
        throws ConfigServiceException
    {
        if (dsElement == null)
        {
            throw new ConfigServiceException("cb-refresh-from-dse-ce-is-null", new Object[] { m_name });
        }

        try
        {
            String elementType = dsElement.getIdentity().getType();
            if (!elementType.equals(m_type))
            {
                throw new ConfigServiceException("cb-refresh-from-dse-ce-type-changed", new Object[] { m_name, m_type, elementType });
            }

            String typeVersion = dsElement.getIdentity().getReleaseVersion();
            if (!typeVersion.equals(m_version))
            {
                // ConfigType version has changed - reinitialize configType reference
                IConfigType configType = m_configServer.loadConfigType(m_type, typeVersion);

                 if (configType == null)
                {
                    throw new ConfigServiceException("cb-refresh-from-dse-ct-is-null", new Object[] {m_name, m_type, typeVersion});
                }

                 m_configType = configType;

                 /*  Assign config type to attribute description to initialize and
                     constrain this bean's base attribute map  */
                 m_attrDescription = m_configType;
             }

             super.refreshFromDSElement(dsElement);
        }
        catch (ConfigServiceRuntimeException e)
        {
            throw (ConfigServiceException) e.getLinkedException();
        }
    }

    @Override
    public IConfigType getConfigType()
    {
        return m_configType;
    }

    /**
     * Creates a new configuration object marked as a template (prototype) from
     * which instances can be created.
     *
     * @param name                     The name (logical path) of the new
     *                                 prototype configuration.
     * @return                         The new prototype (template) - an exact
     *                                 copy of this element.
     * @throws ConfigServiceException  An exception if it failed to create a
     *                                 prototype.
     */
    @Override
    public IConfigPrototype createPrototype(String name)
        throws ConfigServiceException
    {
        return new ConfigBeanPrototype(name, this);
    }

    @Override
    public String toXML()
    {
        StringBuffer buffer = new StringBuffer("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\r\n");

        String pVersion = null;

        try
        {
            Map meta = m_configServer.getMetaAttributes(m_name);

            if(meta != null)
            {
                pVersion = (String)meta.get(ConfigServerUtility.PRODUCT_VERSION);
            }
        }
        catch(Throwable e)
        {
        }

        buffer.append("<").append(m_configType.getName());
        buffer.append(" xmlns=\"" + m_configType.getProperty(IAttributeDescription.SCHEMA_NAMESPACE)+ "\"");
        buffer.append(" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"");
        buffer.append(" xsi:schemaLocation=\"http://www.sonicsw.com/sonicmq " + m_configType.getProperty(IConfigType.SCHEMA_LOCATION_HINT)+ "\"");
        buffer.append(" name=\"").append(Util.name2Url(m_name)).append('\"');
        buffer.append(" version=\"").append(m_configType.getVersion()).append('\"');
        if(pVersion != null)
        {
            buffer.append(" productVersion=\"").append(pVersion).append('\"');
        }
        buffer.append(">\r\n");

        if(getAnnotation() != null)
        {
        	buffer.append("<").append(IAnnotationExtension.TOOL_ANNOTATION);
        	buffer.append(">\r\n");
        	buffer.append(IAnnotationExtension.START_CDATA).append(getAnnotation());
        	buffer.append(IAnnotationExtension.END_CDATA).append("\r\n");
        	buffer.append("</").append(IAnnotationExtension.TOOL_ANNOTATION);
        	buffer.append(">\r\n");
        }
        
        String content = super.toXML();

        if(content != null && content.length() > 0)
        {
            buffer.append(content);
        }

        buffer.append("</").append(m_configType.getName()).append(">");

        return buffer.toString();
    }
}