/*
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.mx.config.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import com.sonicsw.mx.config.ConfigFactory;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.ConfigServiceRuntimeException;
import com.sonicsw.mx.config.IAnnotationExtension;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IConfigElement;
import com.sonicsw.mx.config.IConfigPath;
import com.sonicsw.mx.config.IConfigPrototype;
import com.sonicsw.mx.config.IConfigServer;

import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.MergeUtil;
import com.sonicsw.mf.common.config.impl.Identity;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;


public class ConfigElementImpl extends AttributeMapImpl implements IConfigElement
{
    /*  Identity Element Type String
        This is the string populated in to the 'type' field
        of the identity element of a directory service element
        used to store a generic config element. */
    protected static final String IDENTITY_ELEMENT_TYPE_STRING = "ConfigElement";

    /*  Identity Element Release Version String
        This is the string populated in to the 'releaseVersion' field
        of the identity element of a directory service element
        used to store a new generic config element. */
    protected static final String IDENTITY_ELEMENT_RELEASE_VERSION_STRING = "1.0";

    // LOCAL STATE

    // Local - A config element in this state is new but not yet added to transaction
    protected static final short LOCAL_STATE = 0;

    // New - A config element in this state is new and not yet stored in the DS
    protected static final short NEW_STATE = 1;

    // Unmodified - Config element has not been modified since it was loaded
    protected static final short UNMODIFIED_STATE = 2;

    // Modified - Config element has been modified since it was loaded
    //            and these changes have not been saved to the config server
    protected static final short MODIFIED_STATE = 3;

    // Removed - Config element is about to removed from the DS.
    protected static final short REMOVED_STATE = 4;

    // Deleted - Config element has been removed from the DS.
    protected static final short DELETED_STATE = 5;


    protected static final String[] STATES_DEBUG = { "LOCAL", "NEW", "UNMODIFIED", "MODIFIED", "REMOVED", "DELETED" };


    /*  Config Element Name.
        Name uniquely identifying this config element. */
    protected String m_name = null;

    /*  Config Element Type.
        The name of the config type associated with this config element. */
    protected String m_type = null;

    /*  Config Element Version.
        The version of the config type associated with this config element. */
    protected String m_version = null;

    /*  Attachment InputStream
        Used to attach binary data to this config element. */
    protected InputStream m_blob = null;
    
    // When an 
    protected long m_creationTime = -1;
    
    /*  Meta Attributes associated with the logical name for this element */
    protected Map m_metaAttributes = null;
    protected boolean m_metaChanged = false;

    // Config Prototype - if non-null, the prototype from which this element was created
    protected IConfigPrototype m_configPrototype = null;

    /*  Directory Element Name currently storing the state of this Config Element.
        The name of the directory element that contains the last saved state of
        this config element. In most cases this element has the same name as this
        config element, however in the case where this config element has been
        renamed within a transaction, this field will contain the original name
        of this config element in the transaction.
        */
    protected String m_dirElementName = null;

    /*  Config Element State
        Indicates the current lifecycle state of this config element:
        new, modified, unmodified, removed, deleted. */
    protected short m_state = NEW_STATE;


    protected String m_annotation = null;
    
    protected ConfigElementImpl(ConfigServer configServer)
    {
        super(configServer);
    }

    protected ConfigElementImpl(String       name,
                                ConfigServer configServer)
        throws ConfigServiceException
    {
        this(name, IDENTITY_ELEMENT_TYPE_STRING, IDENTITY_ELEMENT_RELEASE_VERSION_STRING, configServer);
    }

    protected ConfigElementImpl(String       name,
                                String       type,
                                String       version,
                                ConfigServer configServer)
        throws ConfigServiceException
    {
        super(configServer);

        if (configServer == null)
        {
            throw new ConfigServiceException("ce-init-cs-is-null");
        }

        m_name          = validateName(name);
        m_type          = type;
        m_version       = version;
        m_configServer  = configServer;
        m_owner         = this;
        m_instanceOwner = this;

        m_configServer.elementCreated(this);
    }

    /** Construct a copy of a config element.
     *
     * @param name the name of copied config element.
     * @param configElement the config element to copy.
     * @param isPrototype if true, the copied element is a prototype.
     * @throws ConfigServiceException if failed to construct the config element.
     */
    protected ConfigElementImpl(String            name,
                                ConfigElementImpl configElement,
                                boolean           isPrototype)
        throws ConfigServiceException
    {
        super(configElement);   // Deep copy attribute map

        m_name         = validateName(name);
        m_type         = configElement.m_type;
        m_version      = configElement.m_version;
        m_configServer = configElement.m_configServer;

        setOwner(this);   // Set ownership of attribute map

        m_configServer.elementCreated(this);
    }

    /*
     * Reconstitute a ConfigElement from the IDirElement from the DS.
     */
    protected ConfigElementImpl(IDirElement  dsElement,
                                ConfigServer configServer)
        throws ConfigServiceException
    {
        super(configServer);

        init(dsElement);
    }

    protected final void init(IDirElement dsElement)
        throws ConfigServiceException
    {
        m_owner         = this;
        m_instanceOwner = this;
        m_name          = dsElement.getIdentity().getName();
        m_type          = dsElement.getIdentity().getType();
        m_state         = UNMODIFIED_STATE;
        
        if (dsElement.getSuperElementName() != null)
        {
            IConfigElement configElement = m_configServer.loadConfigElement(dsElement.getSuperElementName());

            if (configElement instanceof IConfigPrototype)
            {
                m_configPrototype = (IConfigPrototype) configElement;
            }
            else
            {
                throw new ConfigServiceException("ce-init-super-is-not-proto");
            }
        }

        refreshFromDSElement(dsElement);

        m_configServer.elementLoaded(this);
    }

    protected void refreshFromDSElement(IDirElement dsElement)
        throws ConfigServiceException
    {
        if (dsElement == null)
        {
            throw new ConfigServiceException("ce-refresh-from-dse-ce-is-null");
        }

        m_configServer.debugPrintln("refreshFromDSElement " + dsElement.getIdentity().getName() +
                                    " version=" + dsElement.getIdentity().getVersion());

        try
        {
            // Clear previous state.
            _clear();

            m_metaAttributes = null;

            /* Reinitialize version in case of version upgrade. */
            m_version = dsElement.getIdentity().getReleaseVersion();
            m_creationTime = dsElement.getIdentity().getCreationTimestamp();
            
            IAttributeSet mapSet = dsElement.getAttributes();

            HashMap attributes = mapSet.getAttributes();
            Iterator keysIt = attributes.keySet().iterator();
            while (keysIt.hasNext())
            {
                String key = (String)keysIt.next();

                if (key.equals("_MF_SYSTEM_ATTRIBUTES"))
                {
                	IAttributeSet systemAttr = (IAttributeSet)attributes.get("_MF_SYSTEM_ATTRIBUTES");
                	String value = (String)systemAttr.getAttribute(IAnnotationExtension.TOOL_ANNOTATION);
                	
                	if(value == null)
                	{
                		if (m_annotation != null)
                        {
                            m_annotation = null;
                        }
                		
                		continue; // do not copy internal DS state, only copy TOOL_ANNOTATION value
                	}
                	
                	int indx = value.indexOf(IAnnotationExtension.START_CDATA);
                	String annotation = null; 
                	
                	if (indx == -1)
                	{		
                		annotation = value;
                	}
                	else
                	{
                	    annotation = value.substring(IAnnotationExtension.START_CDATA.length());
                		annotation = (annotation.length() == IAnnotationExtension.END_CDATA.length()) ? "" : annotation.substring(0, annotation.length() - IAnnotationExtension.END_CDATA.length());
                	}
                	
                	if (!annotation.equals(m_annotation))
                    {
                        setAnnotation(annotation.trim());
                    }
                }

                /*
                 * Only populate config element with state that is not from
                 * the template or if the attribute is an attribute list.
                 */
                if ((mapSet.getAttribute(key)instanceof com.sonicsw.mf.common.config.IAttributeList) ||
                    !mapSet.getAttributeMetaData(key).isFromTemplate())
                {
                    put(key, mapSet.getAttribute(key), false);
                }

            }

            String[] removedAttributes = mapSet.getDeletedAttributesInThisSubclassed();
            
            for (int i = 0; i < removedAttributes.length; i++)
            {
                put(removedAttributes[i], new RemovedAttribute(), false);
            }

            m_configServer.elementRefreshed(this);
        }
        catch (ConfigServiceRuntimeException e)
        {
            throw (ConfigServiceException)e.getLinkedException();
        }
    }

    @Override
    public Object clone()
    {
        /*  Generate a name for clone. */
        int idx = m_name.lastIndexOf('/');
        String name = m_name.substring(0, idx + 1) + ConfigFactory.createGUID();

        return clone(m_configServer, name);
    }

    protected Object clone(ConfigServer configServer, String name)
    {
        ConfigElementImpl copy = null;

        try
        {
            /*  Deep copy attribute map */
            copy = (ConfigElementImpl) super.clone(configServer);

            copy.m_name = name;

            copy.m_state = NEW_STATE;

            /* Clone does not initially have a blob attached */
            copy.m_blob = null;

            /* Clone does not have a prototype */
            copy.m_configPrototype = null;

            /*  Set ownership of attribute map */
            copy.setOwner(copy);

            configServer.elementCreated(copy);

            return copy;
        }
        catch (ConfigServiceException e)
        {
            if (copy != null)
            {
                try { configServer.elementRemoved(copy); } catch (Exception ex) { /* Ignored */ }
            }
            e.printStackTrace(System.err);
            throw new ConfigServiceRuntimeException("ce-clone-failed", e);
        }
    }

    @Override
    public Object clonePrototypeInstance(String name)
    {
        if (!isPrototypeInstance())
        {
            throw new ConfigServiceRuntimeException("ce-clone-pi-not-pi");
        }

        ConfigElementImpl copy = null;

        try
        {
            synchronized (m_configServer)
            {
                copy = (ConfigElementImpl)getPrototype().newInstance(name);

                Iterator it = _keySet().iterator();
                while (it.hasNext())
                {
                    Object key = it.next();
                    copy.put(key, _get(key));
                }

                return copy;
            }
        }
        catch (ConfigServiceException e)
        {
            if (copy != null)
            {
                try { m_configServer.elementRemoved(copy); } catch (Exception ex) { /* Ignored */ }
            }

            throw new ConfigServiceRuntimeException("ce-clone-pi-failed", e);
        }

    }

    @Override
    public String toString()
    {
        StringBuffer buffer = new StringBuffer();

        String className = this.getClass().getName();
        String typeName  = className.substring(className.lastIndexOf(".") + 1);

        buffer.append(getName() + "  [" + typeName + "] =\n");
        buffer.append("{\n");
        buffer.append(this.toString(1, false));
        buffer.append("}\n");

        return buffer.toString();
    }

    @Override
    public boolean isRemoved()
    {
        return m_state == REMOVED_STATE || m_state == DELETED_STATE;
    }

    protected void setRemoved()
        throws ConfigServiceException
    {
        if (!isRemoved())
        {
            m_configServer.elementRemoved(this);
        }
    }

    public boolean isDeleted()
    {
        return m_state == DELETED_STATE;
    }

    @Override
    public boolean isModified()
    {
        return ((m_state == MODIFIED_STATE) || (m_state == NEW_STATE));
    }

    protected void setModified()
    {
        try
        {
            m_configServer.elementModified(this);
        }
        catch (ConfigServiceException e)
        {
            /* Ignore */
        }
    }

    @Override
    public boolean isNew()
    {
        return ((m_state == NEW_STATE) || (m_state == LOCAL_STATE));
    }

    public boolean isLocal()
    {
        return (m_state == LOCAL_STATE);
    }

    protected short getState()
    {
        synchronized (m_configServer)
        {
            return m_state;
        }
    }

    protected void setState(short state)
    {
        synchronized (m_configServer)
        {
            if(ConfigServer.DEBUG)
            {
                if (state != m_state)
                {
                    m_configServer.debugPrintln("setState for " + getName() + " from(" + STATES_DEBUG[m_state] + ") to(" + STATES_DEBUG[state] + ")");
                }
            }

            m_state = state;
        }
    }

    protected String getDirectoryElementName()
    {
       return m_dirElementName;
    }

    protected void setDirectoryElementName(String name)
    {
        m_dirElementName = name;
    }

    @Override
    public String getName()
    {
        return m_name;
    }

    @Override
    public void setName(String name)
        throws ConfigServiceException
    {
        try
        {
            synchronized (m_configServer)
            {
                validateName(name);

                m_configServer.elementRenamed(m_name, name, this);

                m_name = name;
            }
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("ce-set-name-failed", e);
        }
    }

    protected void rename(String oldPrefix, String newPrefix)
        throws ConfigServiceException
    {
        if (!m_name.startsWith(oldPrefix))
        {
            throw new ConfigServiceException("");
        }

        String suffix = m_name.substring(oldPrefix.length());
        m_name = newPrefix + suffix;
    }

    /**
     * Get the InputStream for the blob attached to this configElement.
     */
    @Override
    public InputStream getInputStream()
        throws ConfigServiceException
    {
        m_configServer.debugPrintln("ConfigElementImpl::getInputStream(" + m_name + ")");
        boolean isReadable = true;
        
        if (m_blob != null) //Sonic00025418
        {
            try 
            {
                int size = m_blob.available(); //floating InputStream is useless,it doesn't contain copy of data
                if (m_blob.available() == 0)
                {
                    isReadable = false;
                }
            }
            catch (IOException ie)
            {
                isReadable = false; //let give one more change to read a blob
            }
            
        }
        
        if (m_blob == null || !isReadable)
        {
            IBlob blob = (IBlob)((ConfigServer)m_configServer).getBlob(m_name);
            
            if (blob != null)
            {
                m_blob = blob.getBlobStream();
            }
        }
        
        return m_blob;
    }

    /**
     * Set an InputStream for this config element. The contents of the
     * InputStream will be stored to the DS when the configElement is stored.
     */
    @Override
    public void setInputStream(InputStream stream)
        throws ConfigServiceException
    {
        m_configServer.debugPrintln("ConfigElementImpl::setInputStream(" + m_name + ")");
        
        // What we're doing here is basically creating an in-memory copy of the input stream
        // by reading the stream, storing the bytes and then creating a new input stream.
        // Inefficient, but the only way to guarantee that the contents will be preserved
        // given that the stream parameter could be flushed, closed and the underlying
        // data source removed!
        //
        try
        {
            // Lets pretend that most files are small :) It will grow as needed.
            ByteArrayOutputStream oStream = new ByteArrayOutputStream(4096);
            byte[] buffer = new byte[4096];
            int len = 0;
            
            while ((len = stream.read(buffer)) >= 0)
            {
                oStream.write(buffer, 0, len);
            }

            m_blob = new ByteArrayInputStream(oStream.toByteArray());
            
            oStream.close();
            setModified();
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("ce-set-input-stream-failed", e);
        }
    }

    @Override
    public IConfigPath[] getReferences()
        throws ConfigServiceException
    {
        return m_configServer.getReferences(m_name);
    }

    @Override
    public boolean isMetaAttributesModified()
    {
        return m_metaChanged;
    }

    @Override
    public HashMap getMetaAttributes()
        throws ConfigServiceException
    {
        if (m_metaAttributes == null)
        {
            m_metaAttributes = Util.splitToolMetaAttributes(m_configServer.getMetaAttributes(m_name));
        }

        return (m_metaAttributes != null) ? Util.mapToHashMap(m_metaAttributes) : null;
    }

    @Override
    public void setMetaAttributes(HashMap attributes)
//        throws ConfigServiceException
    {
        if ((attributes == null) || attributes.isEmpty())
        {
//            throw new ConfigServiceException("ce-set-meta-attributes-failed", new Object[] { m_name });
            return;
        }

        m_metaAttributes = attributes;
        m_metaChanged = true;

        setModified();
    }

    @Override
    public IConfigPrototype getPrototype()
    {
        return m_configPrototype;
    }

    @Override
    public boolean isPrototypeInstance()
    {
        return (m_configPrototype != null);
    }

    @Override
    protected IAttributeMap getProtoMap()
    {
        return m_configPrototype;
    }

    @Override
    public void unshare()
        throws ConfigServiceException
    {
        if (!isPrototypeInstance())
        {
            throw new ConfigServiceException("ce-unshare-failed");
        }

        m_configServer.unshareConfigElement(this);
    }

    /**
     * Creates a new configuration object marked as a template (prototype) from
     * which instances can be created.
     *
     * @param name                     The name (logical path) of the new
     *                                 prototype configuration.
     * @return                         The new prototype (template) - an exact
     *                                 copy of this element.
     * @throws ConfigServiceException  An exception if it failed to create a
     *                                 prototype.
     */
    @Override
    public IConfigPrototype createPrototype(String name)
        throws ConfigServiceException
    {
         return new ConfigElementPrototype(name, this);
    }

    protected IBasicElement doneUpdate()
        throws ConfigServiceException
    {
        IBasicElement storeThis = null;

        if ((m_name == null) || (m_name.length() == 0))
        {
            throw new ConfigServiceException("ce-done-update-no-name");
        }

        if (isNew())
        {
            boolean     isPrototype     = (this instanceof IConfigPrototype);
            IDirElement localDirElement = null;
            boolean overlay = false;

            if (isPrototypeInstance())
            {
                localDirElement = ElementFactory.createInstanceElement(m_dirElementName, m_type, m_version, getPrototype().getName());

                // Get the template (prototype) attributes into the element
                ((ConfigElementImpl)getPrototype()).toAttributeSet(localDirElement.getAttributes());
                overlay = true;
            }
            else
            {
                localDirElement = ElementFactory.createElement(m_dirElementName, m_type, m_version, isPrototype);  
            }
            
            // Get the attributes into the element
            // If this element is an instance then the attributes are overlayed
            // ontop of the existing template attributes
            //
            toAttributeSet(localDirElement.getAttributes(), overlay);

            try
            {
            	//Set annotation for element
                if(m_annotation != null )
                {
                	if (isPrototypeInstance() || (this instanceof IConfigPrototype))
                    {
                        throw new ConfigServiceException("ce-done-update-failed", new Object[] { m_name }, new Exception("Annotation are not supported for template and instances"));
                    }
            		
                }
              
                addAnnotationToElement(localDirElement);
                storeThis = localDirElement.doneUpdate();
            }
            catch (Exception e)
            {
                throw new ConfigServiceException("ce-done-update-failed", new Object[] { m_name }, e);
            }
        }
        else  // Update
        {
            IDirElement loadedDirElement = m_configServer.loadDirElement(m_dirElementName);

            if (loadedDirElement == null)
            {
                throw new ConfigServiceException("ce-done-update-element-removed", new Object[] { m_name });
            }

            ((Identity)loadedDirElement.getIdentity()).setName(m_name);

            /* Create and Populate a DS element with config element's state. */
            IDirElement localDirElement = ElementFactory.createElement(null,null,null);
            // If instance then this indicate to merge utils that we are only
            // updating overridden state of instance
            boolean isInstance = isPrototypeInstance() ? true : false;

            toAttributeSet(localDirElement.getAttributes());
            
            /* Merge local state into loaded DS element and generate delta. */
            try
            {
                MergeUtil.mergeModifications(loadedDirElement, localDirElement, isInstance);

                if(m_annotation != null )
                {
                	if (isPrototypeInstance() || (this instanceof IConfigPrototype))
                    {
                        throw new ConfigServiceException("ce-done-update-failed", new Object[] { m_name }, new Exception("Annotations are not supported for template and instances"));
                    }
                	
                }
                
                addAnnotationToElement(loadedDirElement);
                
                storeThis = loadedDirElement.doneUpdate();
            }
            catch (Exception e)
            {
                throw new ConfigServiceException("ce-done-update-failed", new Object[] { m_name }, e);
            }
        }
        return storeThis;
    }

    @Override
    public void refresh()
        throws ConfigServiceException
    {
        try
        {
            m_configServer.refreshElement(this);

            // We need to null the blob so that it will be fetched fresh the
            // next time
            m_blob = null;
        }
        catch (ConfigServiceException e)
        {
            if (e.getErrorKey().equals("ce-refresh-from-dse-ce-is-null"))
            {   /*  This config element has been removed from config server:
                    remove it from the cache and mark it as being removed. */
                setRemoved();
            }
            else
            {
                throw e;
            }
        }
    }

    public Set getSubElements()
        throws ConfigServiceException
    {
        HashSet subElements = new HashSet();
        super.getSubElements(subElements);
        if (m_configPrototype != null)
        {
            subElements.add(m_configPrototype);
            subElements.addAll(((ConfigElementImpl)m_configPrototype).getSubElements());
        }
        return subElements;
    }

    public String getDSDirectory()
    {
        return (m_name != null) ? m_name.substring(0, m_name.lastIndexOf("/")+1) : null;
    }

    @Override
    public IConfigServer getConfigServer()
    {
        return m_configServer;
    }

    static protected String validateName(String name)
        throws ConfigServiceException
    {
        String error = null;

        if (name == null || name.length() <= 0)
        {
            error = "Cannot be an empty string";
        }
        else {
        	
	        if (name.charAt(0) != '/')
	        {
	            error = "Name does not begin with a '/'";
	        }
	
	        if (name.indexOf("$") == 0)
	        {
	            error = "Cannot contain '$'";
	        }
	
	        if (name.indexOf("\\") >= 0)
	        {
	            error = "Cannot contain '\\'";
	        }
	
	        if (name.indexOf("::") >= 0 )
	        {
	            error = "Cannot contain '::'";
	        }
	
	        if (name.indexOf(".*.") >= 0 )
	        {
	            error = "Cannot contain '.*.'";
	        }
	
	        if (name.indexOf(".#.") >= 0 )
	        {
	            error = "Cannot contain '.#.'";
	        }
	
	        if (name.indexOf("<") >= 0 )
	        {
	            error = "Cannot contain '<'";
	        }
	
	        if (name.indexOf(">") >= 0 )
	        {
	            error = "Cannot contain '>'";
	        }
	
	        if (name.indexOf("&") >= 0 )
	        {
	            error = "Cannot contain '&'";
	        }
        }
        if(error != null)
        {
            throw new ConfigServiceException("Name is invalid: " + error);
        }

        return name;
    }

    @Override
    public IConfigPath getAttributeName()
    {
        return new ConfigPathImpl(m_name);
    }
    
    /**
     * Gets the date/time when this file element was created.
     *
     * @return  The <code>long</code> value representing the creation date/time.
     */
    @Override
    public long getCreationTime()
    {
        return m_creationTime;
    }
    
    @Override
    public void  setAnnotation(String value)
    {
    	m_annotation = value;
    	
    	if (getState() == NEW_STATE)
        {
            return;
        }
    	
    	setModified();
    }
    
	@Override
    public String getAnnotation()
	{
		return m_annotation;
	}
	
	private void addAnnotationToElement(IDirElement element)
		throws ConfigException
	{
		IAttributeSet systemAttributes = (IAttributeSet)element.getAttributes().getAttribute("_MF_SYSTEM_ATTRIBUTES");
		
		if (systemAttributes == null)
		{
			systemAttributes = (IAttributeSet)element.getAttributes().createAttributeSet("_MF_SYSTEM_ATTRIBUTES");
		}
		
		if (m_annotation == null)
		{
			systemAttributes.deleteAttribute(IAnnotationExtension.TOOL_ANNOTATION);
		}
		else
		{
			systemAttributes.setStringAttribute(IAnnotationExtension.TOOL_ANNOTATION, IAnnotationExtension.START_CDATA + m_annotation + IAnnotationExtension.END_CDATA);
		}
	}
}
