/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.mx.config.impl;

import java.util.Collection;
import java.util.Iterator;
import java.util.StringTokenizer;
import java.util.Vector;

import com.sonicsw.mx.config.IAttributeList;
import com.sonicsw.mx.config.IConfigElement;
import com.sonicsw.mx.config.IConfigPath;

import com.sonicsw.mf.common.config.query.AttributeName;

/**
 * The ConfigPathImpl represents the path through a series of nested
 * AttributeMap's to a particular point in either a IConfigBean or
 * IConfigType structure.
 */
public class ConfigPathImpl
extends Vector
implements IConfigPath
{
  public static final char   DEFAULT_SEPARATOR = '.';
  private static final String SUBSTITUTE        = "%SUB%";
  private static final String ILL_ARG        = "Component of an IConfigPath must be a String.";

  protected String[] m_path;
  protected char     m_separator;
  private   String   m_buffer;

  /**   Construct an empty configuration path.
   */
  public ConfigPathImpl()
  {
    super();
  }

  /**   Construct a configuration path that is a copy of path.
   */
  public ConfigPathImpl(IConfigPath path)
  {
    super((Collection)path);
  }

  /**   Construct a configuration path from the given
   *    array of components.
   */
  public ConfigPathImpl(String[] path)
  {
      addPathElements(path);
  }

  private void addPathElements(String[] path) {
      for (int i = 0; i < path.length; i++)
      {
          super.add(path[i]);
      }
  }

  /**   Construct a configuration path from the given string of components.
   *    Components in the string are separated by '.' character.
   */
  public ConfigPathImpl(String path)
  {
      String[] components = split(path == null ? "" : path);
      addPathElements(components);
  }

  /**   Construct a configuration path from an AttributeName
   */
  public ConfigPathImpl(AttributeName attributeName)
  {
      preparePathImpl(attributeName);
  }
  
  private void preparePathImpl(AttributeName attributeName) {

      super.add(attributeName.getElementName());
      for (int i = 0; i < attributeName.getComponentCount(); i++)
      {
          Object obj = attributeName.getComponent(i);
          if (obj instanceof Integer)
          {
              super.add(((Integer)obj).toString());
          }
          else
              if (obj instanceof String)
              {
                  super.add(obj);
              }
      }
  }

  protected ConfigPathImpl(Collection path)
  {
    super(path);
  }

  /**   Returns a clone of this configuration path.
   *    @return copy of this configuration path.
   */
  @Override
public Object clone()
  {
    return super.clone();
  }

  /**   Append the supplied array of components to the end of this configuration path.
   *
   *    @param  path Array of components to appended to this path.
   *    @return This configuration path with <code>path</code> appended to it.
   */
  @Override
public IConfigPath append(String[] path)
  {
    for (int i = 0; i < path.length; i++)
    {
        super.add(path[i]);
    }
    return this;
  }

  /**   Append the supplied string of components to the end of this configuration path.
   *    Components in the string are separated by '.' character.
   *
   *    @param  path String of components to appended to this path.
   *    @return This configuration path with <code>path</code> appended to it.
   */
  @Override
public IConfigPath append(String path)
  {
    return append(split(path));
  }

  /**   Append the supplied path to the end of this configuration path.
   *
   *    @param path Configuration path to append to this path
   *    @return This configuration path with <code>path</code> appended to it.
   */
  @Override
public IConfigPath append(IConfigPath path)
  {
    addAll((Collection)path);
    return this;
  }

  /**   Append a placeholder component to the end of this configuration path.
   *    Placeholder components are special markers placed in a configuration
   *    path to indicate where subsequent substitutions will occur.
   *
   *    @return This configuration path with a placeholder component appended to it.
   *    @see #substituteComponent
   */
  @Override
public IConfigPath appendPlaceholder()
  {
    return append(SUBSTITUTE);
  }

  /**   Insert the supplied string of components into this configuration path at the specified inded.
   *
   *    @param index The index at which to insert <code>path</code>.
   *    @param path The string of components to insert into this path
   *    @return This configuration path with <code>path</code> appended to it.
   */
  @Override
public IConfigPath insert(int index, String path)
  {
    return insert(index, split(path));
  }

  /**   Insert the supplied array of components into this configuration path at the specified inded.
   *
   *    @param index The index at which to insert <code>path</code>.
   *    @param path The array of components to insert into this path
   *    @return This configuration path with <code>path</code> appended to it.
   */
  @Override
public IConfigPath insert(int index, String[] path)
  {
    for (int i = 0; i < path.length; i++)
    {
        super.add(index++, path[i]);
    }
    return this;
  }

  /**   Insert the supplied path into this configuration path at the specified inded.
   *
   *    @param index The index at which to insert <code>path</code>.
   *    @param path The configuration path to insert into this path
   *    @return This configuration path with <code>path</code> appended to it.
   */
  @Override
public IConfigPath insert(int index, IConfigPath path)
  {
    addAll(index, (Collection)path);
    return this;
  }

  /**   Returns the first component in this configuration path.
   *
   *    @return The first component in path.
   */
  @Override
public String getFirstComponent()
  {
    return (String) firstElement();
  }

  /**   Returns the component at the specified index in this configuration path.
   *
   *    @param index The index of requested component.
   *    @return The component at the specified index.
   */
  @Override
public String getComponent(int index)
  {
    return (String) get(index);
  }

  /**   Set the supplied component at the specified index in this configuration path.
   *    The previous component at that index is removed and discarded.
   *
   *    @param index The index in path where the component is set.
   *    @param component The component set in path.
   */
  @Override
public void setComponent(int index, String component)
  {
    setElementAt(component, index);
  }

  /**   Inserts the supplied component at the specified index in configuration path.
   *    Each component in this path with an index greater
   *    or equal to the specified index is shifted upward to have an
   *    index one greater than the value it had previously.
   *
   *    @param index The index in path where the component is inserted.
   *    @param component The component inserted into path.
   */
  @Override
public void insertComponent(int index, String component)
  {
    super.insertElementAt(component, index);
  }

  /**   Removes the component at the specified index in configuration path.
   *    Each component in this path with an index greater
   *    or equal to the specified index is shifted downward to have an
   *    index one less than the value it had previously.
   *
   *    @param index The index of the component removed.
   */
  @Override
public void removeComponent(int index)
  {
    super.removeElementAt(index);
  }

  /**   Returns the last component in path.
   *
   *    @return The last component in path.
   */
  @Override
public String getLastComponent()
  {
    return (String) lastElement();
  }

  /**   Creates a copy of this configuration path with the
   *    first placeholder component substitued with the
   *    the supplied component
   *
   *    @param component    Component to be substituted.
   *    @return             A configuration path with <code>component</code> substituted.
   */
  @Override
public IConfigPath substitute(String component)
  {
    return substitute(split(component));
  }

  /**   Creates a new ConfigPathImpl based on the current instance
   *    but substitutes the  elements with the
   *    supplied new elements.
   *
   *    @param element  An array of components to be substituted.
   *    @return         A configuration path with <code>components</code> substituted.
   */
  @Override
public IConfigPath substitute(String[] components)
  {
    ConfigPathImpl res = null;

    try
    {
      res = (ConfigPathImpl)this.clone();

      int j   = 0;

      for (int i = 0; i < res.size(); i++)
      {
        if (res.getComponent(i).equals(SUBSTITUTE) && (j < components.length))
        {
            res.setComponent(i, components[j++]);
        }
      }
    }
    catch (Exception e)
    {
      e.printStackTrace();
    }

    return res;
  }

  /**   Returns a new configuration path that is a sub-path of this path.
   *    The subpath begins with the component at the specified index and extends
   *    to the end of this path.
   *
   *    @param beginIndex   The beginning index, inclusive.
   *    @return             The specified sub-path.
   */
    @Override
    public IConfigPath subPath(int beginIndex)
    {
        return new ConfigPathImpl(subList(beginIndex, size()));
    }

  /**   Returns a new configuration path that is a sub-path of this path.
   *    The sub-path begins at the specified beginIndex and extends to the
   *    componenet at index endIndex - 1. Thus the length of the sub-path is
   *    endIndex-beginIndex.
   *
   *    @param beginIndex   The beginning index, inclusive.
   *    @param endIndex     The ending index, exclusive.
   *    @return             The specified sub-path.
   */
    @Override
    public IConfigPath subPath(int beginIndex, int endIndex)
    {
        return new ConfigPathImpl(subList(beginIndex, endIndex));
    }

  /**   Returns a string representation of the object.
   *
   *    @return The string representation of the object.
   */
  @Override
public String toString()
  {
    StringBuffer buffer = new StringBuffer();
    Iterator strings = iterator();
    if (strings.hasNext())
    {
        buffer.append((String)strings.next());
    }

    while (strings.hasNext())
    {
        buffer.append(DEFAULT_SEPARATOR + (String)strings.next());
    }
    return buffer.toString();
  }

  /*    Overridden Vector Class Methods
   *
   *    These methods ensure that only string elements are placed in
   *    in ConfigPaths.
   */

    @Override
    public void add(int index, Object element)
    {
        super.add(index, validate(element));
    }

    @Override
    public synchronized boolean add(Object o)
    {
        return super.add(validate(o));
    }

    @Override
    public synchronized boolean addAll(Collection c)
    {
        Iterator components = c.iterator();
        while (components.hasNext())
        {
            super.add(validate(components.next()));
        }
        return true;
    }

   @Override
public synchronized boolean addAll(int index, Collection c)
   {
        Iterator components = c.iterator();
        while (components.hasNext())
        {
            super.add(index++,validate(components.next()));
        }
        return true;
   }

   @Override
public synchronized void addElement(Object obj)
   {
        super.addElement(validate(obj));
   }

   @Override
public synchronized void insertElementAt(Object obj, int index)
   {
        super.insertElementAt(validate(obj),index);
   }

    @Override
    public synchronized Object set(int index, Object element)
    {
        return super.set(index,validate(element));
    }

    @Override
    public synchronized void setElementAt(Object obj, int index)
    {
        super.setElementAt(validate(obj),index);
    }

    /*  End of Overridden Vector Class Methods
     */

    public static final String[] split(String path)
    {
        StringTokenizer st        = new StringTokenizer(path, new String(new char[] {DEFAULT_SEPARATOR}));
        String[]        component = new String[st.countTokens()];
        int             i         = 0;

        while (st.hasMoreTokens())
        {
            component[i++] = st.nextToken().trim();
        }

        return component;
    }

    /**
     *
     * @deprecated  Use toAttributeName(IConfigElement) instead because this
     *              method does not handle conversion of path components which
     *              are numbers correctly
     */
    public AttributeName toAttributeName()
    {
        return toAttributeName(null);
    }

    /**
     * Converts the ConfigPath into an AttributeName object which is basically
     * the same thing but with a slightly different format.
     *
     * The big thing to note is the way we have to handle building parts of
     * the name if it relates to an AttributeList...we must call a different
     * method with an integer value. This requires that we know what each
     * path part maps onto....and do the necessary conversion - NASTY.
     *
     * @param element  The element for which this path conversion is relevant
     * @return         A valid <code>AttributeName</code> object
     */
    /* package protected */ AttributeName toAttributeName(IConfigElement element)
    {
        AttributeName attributeName = new AttributeName();
        boolean       convertToNumber = false;

        for (int i = 0; i < size(); i++)
        {
            String component = (String)this.elementAt(i);
            boolean nextConvert = false;  // Always start at an AttributeMap or ConfigElement
                                          // so no need for conversion
            if (element != null)
            {
                IConfigPath subPath = subPath(0, i);
                Object value = element.getAttribute(subPath);

                if (value instanceof IAttributeList)
                {
                    nextConvert = true;
                }
            }

            if (convertToNumber)
            {
                try
                {
                    int index = Integer.parseInt(component);
                    attributeName.setNextComponent(index);

                    convertToNumber = nextConvert;
                    continue;
                }
                catch (NumberFormatException e)
                {
                }
            }

            attributeName.setNextComponent(component);

            convertToNumber = nextConvert;
        }

        // Lets set the element name on each attribute - don't think
        // this is used though because we pass the element name in
        // to the revertToTemplate call.
        if (element != null)
        {
            attributeName.setElementName(element.getName());
        }

        return attributeName;
    }

    private Object validate(Object obj)
    {
        if (obj instanceof String)
        {
            return obj;
        }
        else
        {
            throw new IllegalArgumentException(ILL_ARG);
        }
    }

}