package com.sonicsw.mx.config.impl;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import com.sonicsw.mx.config.ConfigChange;
import com.sonicsw.mx.config.ConfigServerFactory;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigBeanFile;
import com.sonicsw.mx.config.IConfigChangeFilter;
import com.sonicsw.mx.config.IConfigChangeListener;
import com.sonicsw.mx.config.IConfigElement;
import com.sonicsw.mx.config.IConfigPath;
import com.sonicsw.mx.config.IConfigServer;
import com.sonicsw.mx.config.IConfigType;

import com.sonicsw.mf.common.IDSTransaction;
import com.sonicsw.mf.common.IDirectoryAdminNotifications;
import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.IDirectoryFileSystemNotifications;
import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IDeltaAttributeSet;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.IElementChangeHandler;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.NotModifiedAttException;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.config.query.FromDirectory;
import com.sonicsw.mf.common.config.query.FromElementList;
import com.sonicsw.mf.common.config.query.Query;
import com.sonicsw.mf.common.config.query.Select;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDeltaDirElement;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.security.ConfigurePermissionDeniedException;


public class ConfigServer implements IConfigServer
{
    protected static final boolean DEBUG = System.getProperty("debug.cl", "false").equalsIgnoreCase("true");

    /*  Directory Service
        Used by clients running outside of a container. */
    protected IDirectoryFileSystemService m_ds = null;

    private ConfigChangeDispatcher m_configChangeDispatcher = null;
    private IElementChangeHandler  m_typeChangeHandler = null;

    /*  Config Type Cache
        Maintains cache of all named instances of config types either created
        by or loaded into the JVM by this config server.    */
    protected Cache m_typeCache = new Cache();

    /*  Config Element Cache
        Maintains cache of all named instances of config types either created
        by or loaded into the JVM by this config server.    */
    protected Cache m_elementCache = new Cache();

    protected IConfigServer m_parent = null;

    /*  Writeable Connection Mode
        Enables updating of config elements loaded through this connection. */
    final protected boolean m_writeable;

    protected boolean m_closed = false;
    protected boolean m_filterDefaults = true;

    public ConfigServer()
    {
        m_writeable = false;
    }

    public ConfigServer(IDirectoryFileSystemService ds, boolean writeable)
        throws ConfigServiceException
    {
        this(ds, writeable, true);
    }

    public ConfigServer(IDirectoryFileSystemService ds, boolean writeable, boolean listen)
        throws ConfigServiceException
    {
        if (!(ds instanceof IDirectoryAdminService))
        {
            throw new ConfigServiceException("cs-connect-failed");
        }

        try
        {
            m_ds        = ds;
            m_writeable = writeable;

            if (listen)
            {
                registerElementChangeListener();
            }
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            m_ds = null;
            throw new ConfigServiceException("cs-connect-failed", e);
        }
    }

    @Override
    protected void finalize()
        throws Throwable
    {
        debugPrintln("finalize");

        super.finalize();
    }

    @Override
    public void close()
        throws ConfigServiceException
    {
    	close(true);
    }

    /**
     * In the case where we are closing the entire connection to the Domain
     * (JMS Connector disconnect) then we don't need to unregister change
     * listeners because the connector will clear everything out itself...
     * and trying to unregister on an already closing agent will cause a
     * timeout!
     */
    public void close(boolean unregisterChangeListener)
        throws ConfigServiceException
    {
        if (m_closed == false)
        {
            m_closed = true;

            synchronized (this)
            {
           		debugPrintln("close");

        		if (unregisterChangeListener)
                {
                    unregisterElementChangeListener();
                }

        		m_ds = null;
        		m_parent = null;
           }
        }
	}

    @Override
    public IConfigServer connect()
        throws ConfigServiceException
    {
        IConfigServer server = ConfigServerFactory.connectConfigServer(m_ds, isWriteable(), isListening());

        ((ConfigServer)server).setServerParent(this);

        return server;
    }

    @Override
    public IConfigServer connectTransacted()
        throws ConfigServiceException
    {
        IConfigServer server = ConfigServerFactory.connectTransactedConfigServer(m_ds);

        ((ConfigServer)server).setServerParent(this);

        return server;
    }

    protected void setServerParent(IConfigServer parent)
    {
        m_parent = parent;
    }

    public boolean isListening()
    {
        return (m_configChangeDispatcher != null);
    }

    @Override
    public boolean isWriteable()
    {
        return m_writeable;
    }

    @Override
    public boolean isTransacted()
    {
        return false;
    }

    public String getConfigServerURL()
    {
        return null;
    }

    public void setFilterDefaults(boolean filterDefaults)
    {
        m_filterDefaults = filterDefaults;
    }

    @Override
    synchronized public void storeConfigElements(IConfigElement[] configElements,
                                                 String[]         deleteElements)
        throws ConfigServiceException
    {
        List elements = new ArrayList();
        List ceList   = new ArrayList();

        if (configElements != null)
        {
            for (int iCE = 0; iCE < configElements.length; iCE++)
            {
                ceList.add(configElements[iCE]);
                ceList.addAll(((ConfigElementImpl)configElements[iCE]).getSubElements());
            }
        }

        Iterator i = ceList.iterator();

        while (i.hasNext())
        {
            IConfigElement configElement = (IConfigElement)i.next();

            if (configElement.isRemoved())
            {
                throw new ConfigServiceException("cs-store-ces-already-removed", new Object[] {configElement.getName()});
            }

            if (configElement.isModified())
            {
                try
                {
                    configElement.validateComplete();

                    if (!elements.contains(configElement))
                    {
                        elements.add(configElement);
                    }
                }
                catch (ConfigServiceException e)
                {
                    throw e;
                }
                catch (Exception e)
                {
                    throw new ConfigServiceException("cs-store-ces-failed", e);
                }
            }
        }

        configElements = (IConfigElement[])elements.toArray(new IConfigElement[elements.size()]);

        _storeConfigElements(configElements, deleteElements);
    }

    protected void _storeConfigElements(IConfigElement[] configElements,
                                        String[]         deleteElements)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-store-ces-not-connected");
        }

        try
        {
            IDSTransaction txn = m_ds.createTransaction();

            if (deleteElements != null)
            {
                for (int i = 0; i < deleteElements.length; i++)
                {
                    txn.addDeleteElement(deleteElements[i]);
                }
            }

            for (int i=0; i<configElements.length; i++)
            {
                if (((ConfigElementImpl)configElements[i]).m_configServer != this)
                {
                    throw new ConfigServiceException("cs-store-ces-wrong-cs", new Object[] { configElements[i].getName() });
                }

                InputStream   blob  = null;
                if (((ConfigElementImpl)configElements[i]).m_blob != null)
                {
                    blob = ((ConfigElementImpl)configElements[i]).m_blob;
                }
                IBasicElement delta = ((ConfigElementImpl)configElements[i]).doneUpdate();

                // If we have an attachment blob then adding the attachment into
                // the transaction also handles the delta.
                if (blob != null)
                {
                    txn.addAttachBlob(delta, blob);
                }
                else
                {
                    if (delta instanceof IDeltaDirElement)
                    {
                        txn.addUpdateElement((IDeltaDirElement)delta, true);
                    }
                    else if (delta instanceof IDirElement)
                    {
                        txn.addCreateElement((IDirElement)delta);
                    }
                }

                if (configElements[i].isMetaAttributesModified())
                {
                    Map metaAttributes = Util.combineToolMetaAttributes((Map)configElements[i].getMetaAttributes());

                    txn.addSetAttributes(configElements[i].getName(), Util.mapToHashMap(metaAttributes));
                }
            }

            synchronized(this)
            {
                debugPrintln("ConfigServer:_storeConfigElements transaction =", txn);

                m_ds.executeTransaction(txn);

                for (int i = 0; i < configElements.length; i++)
                {
                    elementStored((ConfigElementImpl)configElements[i]);
                }

                if (deleteElements != null)
                {
                    for (int i = 0; i < deleteElements.length; i++)
                    {
                        elementRemoved(deleteElements[i]);
                    }
                }
            }
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-store-ces-failed", e);
        }
    }

    synchronized public void storeConfigType(IConfigType configType)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-store-ct-not-connected", new Object[] { configType.getName() } );
        }

        try
        {
            IConfigType[] configTypes = new IConfigType[] { configType };
            storeConfigTypes(configTypes, null);
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-store-ct-failed", new Object[] { configType.getName() }, e );
        }
    }

    synchronized public void storeConfigTypes(IConfigType[] configTypes,
                                              String[]      deleteTypes)
        throws ConfigServiceException
    {
        HashSet elements = new HashSet();

        for (int i=0; i<configTypes.length; i++)
        {
            if (((ConfigTypeImpl)configTypes[i]).isRemoved())
            {
                throw new ConfigServiceException("cs-store-cts-already-removed", new Object[] { configTypes[i].getName() });
            }

            if (((ConfigTypeImpl)configTypes[i]).isModified())
            {
                try
                {
                    elements.add(configTypes[i]);
                }
                catch (Exception e)
                {
                    throw new ConfigServiceException("cs-store-cts-failed", e);
                }
            }
            Iterator iterator = ((ConfigTypeImpl)configTypes[i]).getSubElements().iterator();

            while (iterator.hasNext())
            {
                ConfigTypeImpl configType = (ConfigTypeImpl) iterator.next();
                if (configType.isRemoved())
                {
                    throw new ConfigServiceException("cs-store-cts-already-removed", new Object[] { configTypes[i].getName() });
                }

                if (configType.isModified())
                {
                    try
                    {
                        elements.add(configType);
                    }
                    catch (Exception e)
                    {
                        throw new ConfigServiceException("cs-store-cts-failed", e);
                    }
                }
            }
        }

        configTypes = (IConfigType[]) elements.toArray(new IConfigType[0]);
        _storeConfigTypes(configTypes, deleteTypes);
    }

    protected void _storeConfigTypes(IConfigType[] configTypes,
                                     String[]      deleteTypes)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-store-cts-not-connected");
        }

        try
        {
            IBasicElement[] deltas = new IBasicElement[configTypes.length];

            for (int i=0; i<configTypes.length; i++)
            {
                IBasicElement delta = ((ConfigTypeImpl)configTypes[i]).doneUpdate();
                createDirectory(((ConfigTypeImpl)configTypes[i]).getDSDirectory());
                deltas[i] = delta;
            }

            synchronized(this)
            {
                ((IDirectoryAdminService)m_ds).setElements(deltas, deleteTypes, null);
            }
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-store-cts-failed", e);
        }
    }

    protected void refreshElement(IConfigElement configElement)
        throws ConfigServiceException
    {
        try
        {
            ((ConfigElementImpl)configElement).refreshFromDSElement(
                m_ds.getFSElement(configElement.getName(), true, true));
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-refresh-ce-failed", e);
        }
    }

    protected void refreshType(IConfigType configType)
        throws ConfigServiceException
    {
        try
        {
            String      dsName     = ((ConfigTypeImpl) configType).getDSName();
            IDirElement dirElement = m_ds.getFSElement(dsName, true, true);

            ((ConfigTypeImpl)configType).refreshFromDSElement(dirElement, this);
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-refresh-ct-failed", e);
        }
    }

    @Override
    synchronized public void storeConfigElement(IConfigElement configElement)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-store-ce-not-connected", new Object[] { configElement.getName() } );
        }

        try
        {
            IConfigElement[] configElements = new IConfigElement[] { configElement };
            storeConfigElements(configElements, null);
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-store-ce-failed", new Object[] { configElement.getName() }, e );
        }
    }

    protected IDirElement loadDirElement(String dirElementName)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-load-de-not-connected", new Object[] { dirElementName });
        }

        try
        {
            IDirElement element = m_ds.getFSElement(dirElementName, m_writeable, true);

            if (element == null)
            {
                debugPrintln("loadDirElement for '" + dirElementName + "' > element == null");
            }

            return element;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-load-de-cant-load-element", new Object[] { dirElementName }, e);
        }
    }

    protected IDirElement loadDirTypeElement(String dirElementName)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-load-de-not-connected", new Object[] { dirElementName });
        }

        try
        {
            IDirElement element = ((IDirectoryAdminService)m_ds).getElement(dirElementName, m_writeable, true);

            if (element == null)
            {
                debugPrintln("loadDirTypeElement for '" + dirElementName + "' > type element == null");
            }

            return element;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-load-de-cant-load-element", new Object[] { dirElementName }, e);
        }
    }
    
    protected IConfigElement loadConfigElement(String  configElementName, boolean localOnly)
        throws ConfigServiceException
    {
        try
        {
            IConfigElement ce = null;

            /*  See if config bean has already been loaded */
                        
            ce = (IConfigElement) m_elementCache.lookup(configElementName, "");
      
            if (ce != null)
            { /* Already loaded */
                return ce;
            }
            
            if (localOnly)
            {   /*  Element is not locally loaded: return null. */
                return null;
            }

            return dirElement2ConfigElement(loadDirElement(configElementName));
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-load-ce-failed", new Object[] { configElementName }, e);
        }
    }
    
    @Override
    synchronized public IConfigElement loadConfigElement(String configElementName)
        throws ConfigServiceException
    {
        return loadConfigElement(configElementName, false);
    }

    @Override
    synchronized public IConfigElement loadLocalConfigElement(String configElementName)
        throws ConfigServiceException
    {
        return loadConfigElement(configElementName, true);
    }

    @Override
    synchronized public Set loadConfigElements(Query query)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-load-ces-not-connected");
        }

        if (query.getSelect() != null)
        {
            throw new ConfigServiceException("cs-load-ces-query-w-select");
        }

        try
        {
            IDirElement[] elements = m_ds.getFSElements(query, m_writeable, true);
            HashSet set = new HashSet(elements.length);
            for (int i = 0; i< elements.length; i++)
            {
                // See if config bean has already been loaded
                IConfigElement configElement = (IConfigElement) m_elementCache.lookup(elements[i].getIdentity().getName(), "");
                if (configElement != null)
                {
                    set.add(configElement);  // Already loaded
                    continue;
                }

                set.add(dirElement2ConfigElement(elements[i]));
            }
            return set;
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-load-ces-failed", e);
        }
    }

    @Override
    synchronized public Set loadConfigElements(String path)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-load-ces-not-connected");
        }

        try
        {
            IDirElement[] elements = m_ds.getFSElements(path, m_writeable);
            HashSet set = new HashSet(elements.length);
            for (int i = 0; i < elements.length; i++)
            {
                // See if config bean has already been loaded
                IConfigElement configElement = (IConfigElement) m_elementCache.lookup(elements[i].getIdentity().getName(), "");
                if (configElement != null)
                {
                    set.add(configElement); // Already loaded
                    continue;
                }

                set.add(dirElement2ConfigElement(elements[i]));
            }
            return set;
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-load-ces-failed", e);
        }
    }

    protected IConfigBean loadConfigBean(String configBeanName, boolean localOnly)
        throws ConfigServiceException
    {
        IConfigElement configElement = null;
        try
        {
            configElement = loadConfigElement(configBeanName, localOnly);
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-load-cb-failed", new Object[] { configBeanName }, e);
        }

        if (configElement instanceof IConfigBean)
        {
            return (IConfigBean) configElement;
        }
        else
        if (configElement == null)
        {
            return null;
        }
        throw new ConfigServiceException("cs-load-cb-not-cb", new Object[] { configBeanName });
    }

    @Override
    public Set listConfigElements(Query query)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-list-ce-not-connected");
        }

        if (query.getSelect() != null)
        {
            throw new ConfigServiceException("cs-list-ce-query-w-select");
        }

        try
        {
            // optimize the query to return no attributes. We don't need it
            // for just getting the name --> less network io
            Query identityOnlyQuery = new Query().
              setFrom(query.getFrom()).
              setWhere(query.getWhere()).
              setSelect(new Select(new AttributeName[]{}));
            IDirElement[] elements = m_ds.getFSElements(identityOnlyQuery, false, true);
            HashSet set = new HashSet(elements.length);

            for (int i = 0; i < elements.length; i++)
            {
                set.add(elements[i].getIdentity().getName());
            }

            return set;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-list-ce-failed", e);
        }
    }

    @Override
    synchronized public IConfigType loadConfigType(String configBeanTypeName,
                                                   String version)
        throws ConfigServiceException
    {
        return loadConfigType(configBeanTypeName, version, false);
    }

    synchronized public IConfigType loadConfigTypeFromLocal(String configBeanTypeName,
                                                            String version)
        throws ConfigServiceException
    {
        return loadConfigType(configBeanTypeName, version, true);
    }

    protected IConfigType loadConfigType(String  configBeanTypeName,
                                         String  version,
                                         boolean localOnly)
        throws ConfigServiceException
    {
        /*  See if config type has already been loaded */
        IConfigType configType = (IConfigType) m_typeCache.lookup(configBeanTypeName, version);
        if (configType != null)
         {
            return configType;  // Already loaded
        }

        // See if the type is cached in the parent
        if (localOnly && (m_parent != null))
        {
            configType = ((ConfigServer)m_parent).loadConfigType(configBeanTypeName, version, localOnly);
        }

        if (configType != null)
         {
            return configType;  // Already loaded
        }

        if (localOnly)
         {
            return null;  // Type is not locally loaded: return null
        }

        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-load-ct-not-connected", new Object[] { configBeanTypeName, version });
        }

        try
        {
            configType = dirElement2ConfigType(loadDirTypeElement(ConfigTypeImpl.convertTypeName2DSName(configBeanTypeName, version)));
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-load-ct-failed", new Object[] { configBeanTypeName, version, ConfigTypeImpl.convertTypeName2DSName(configBeanTypeName, version) }, e );
        }

        return configType;
    }

    @Override
    synchronized public Set loadConfigTypes(Query query)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-load-cts-not-connected");
        }

        if (query.getSelect() != null)
        {
            throw new ConfigServiceException("cs-load-cts-query-w-select");
        }

        try
        {
            IDirElement[] elements = ((IDirectoryAdminService)m_ds).getElements(resetFromClause(query), m_writeable, true);
            HashSet set = new HashSet();

            for (int i = 0; i< elements.length; i++)
            {
                // See if type bean has already been loaded
                IConfigType configType = (IConfigType) m_typeCache.lookup(elements[i].getIdentity().getName(),
                                                                          elements[i].getIdentity().getReleaseVersion());
                if (configType != null)
                {   // Already loaded
                    set.add(configType);
                    continue;
                }

                configType = dirElement2ConfigType(elements[i]);

                if (configType != null)
                {
                    set.add(configType);
                }

                // Not a config type: ignore
            }

            return set;
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-load-cts-failed", e);
        }
    }

    @Override
    synchronized public void removeConfigElements(String[] configElements)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-remove-ces-not-connected");
        }

        try
        {
            IDSTransaction txn = m_ds.createTransaction();
            for (int i = 0; i < configElements.length; i++)
            {
                txn.addDeleteElement(configElements[i]);
            }

            debugPrintln("ConfigServer:removeConfigElements transaction =", txn);

            m_ds.executeTransaction(txn);

            for (int i = 0; i < configElements.length; i++)
            {
                IConfigElement element = loadLocalConfigElement(configElements[i]);

                if (element != null)
                {
                    ((ConfigElementImpl) element).setRemoved();
                }
            }
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-remove-ces-failed", e);
        }
    }

    @Override
    synchronized public void removeConfigElement(String configElementName)
        throws ConfigServiceException
    {
        try
        {
            removeConfigElements(new String[] { configElementName } );
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-remove-ce-failed", new Object[] { configElementName }, e);
        }
    }

    synchronized public void removeConfigType(String configTypeName,
                                              String version)
        throws ConfigServiceException
    {
        try
        {
            IConfigType type = loadConfigTypeFromLocal(configTypeName, version);

            if (type != null)
            {
                ((ConfigTypeImpl) type).setRemoved();
            }

            ((IDirectoryAdminService)m_ds).deleteElement(ConfigTypeImpl.convertTypeName2DSName(configTypeName, version), null);
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-remove-ct-failed", new Object[] { configTypeName, version }, e);
        }
    }

    @Override
    public boolean pathExists(String path)
    {
        try
        {
        	return _pathExists(path);
        }
        catch (ConfigServiceException e)
        {
            return false;
        }
    }

    public boolean _pathExists(String path)
    throws ConfigServiceException
	{
    	if (path == null)
        {
            throw new ConfigServiceException("_pathExists needs non-null path parameter");
        }

    	if (path.equals("/"))
        {
            return true;
        }

    	try
		{
    	    return (m_ds.getMetaAttributes(path) != null);
		}
    	catch (Exception e)
		{
    		throw new ConfigServiceException("_pathExists failure", e);
		}
	}

    @Override
    public void createPath(String path)
        throws ConfigServiceException
    {
        createPath(path, false);
    }

    @Override
    public void createPath(String path, boolean existingOK)
        throws ConfigServiceException
    {
        if (path.charAt(0) != '/')
        {
            throw new ConfigServiceException("cs-create-path-invalid-path", new Object[] { path });
        }

        try
        {
            StringTokenizer tokens = new StringTokenizer(path, "/");
            StringBuffer buffer = new StringBuffer();

            while (tokens.countTokens() > 1)
            {
                buffer.append("/").append(tokens.nextToken());
                m_ds.createFolder(buffer.toString(), true);
            }

            buffer.append("/").append(tokens.nextToken());
            m_ds.createFolder(buffer.toString(), existingOK);
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-create-path-failed", new Object[] { path }, e);
        }
    }

    @Override
    public void createFolder(String path)
        throws ConfigServiceException
    {
        createFolder(path, null, false);
    }

     @Override
    public void createFolder(String path, Map metaAttributes, boolean createParentDirs)
         throws ConfigServiceException
     {
         if (m_ds == null)
        {
            throw new ConfigServiceException("cs-create-folder-not-connected", new Object[] { path });
        }

        try
        {
            IDSTransaction txn = m_ds.createTransaction();

            if (createParentDirs)
            {
                createParentFolders(path, txn);
            }

            txn.addCreateFolder(path);

            if (metaAttributes != null)
            {
                txn.addSetAttributes(path, Util.mapToHashMap(Util.combineToolMetaAttributes(metaAttributes)));
            }

            debugPrintln("ConfigServer:createFolder transaction =", txn);

            m_ds.executeTransaction(txn);
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-create-folder-failed", new Object[] { path }, e);
        }
    }

    private void createParentFolders(String path, IDSTransaction txn)
    {
        int lastSlash = path.lastIndexOf("/", path.length() - 2);

        if (lastSlash > 0)
        {
            String parentPath = path.substring(0, lastSlash);
            createParentFolders(parentPath, txn);
            txn.addCreateFolder(parentPath, true);
        }
    }

    public void createFolder(String directoryName, IDSTransaction txn)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-create-folder-not-connected", new Object[] { directoryName });
        }

        // if we can list the directory then it exists, otherwise create it
        try
        {
            m_ds.listFolders(directoryName);
        }
        catch (DirectoryServiceException e)
        {
            // assumes there's a slash at the end of the directory specification
            int lastSlash = directoryName.lastIndexOf("/", directoryName.length() - 2);

            if (lastSlash != -1)
            {
                createFolder(directoryName.substring(0, lastSlash + 1), txn);
                txn.addCreateFolder(directoryName);
            }
        }
        catch (ConfigurePermissionDeniedException e)
        {
            throw new ConfigServiceException("cs-create-folder-failed",new Object[]{directoryName},e);
        }
    }

    @Override
    public void deleteFolder(String path)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-delete-folder-not-connected", new Object[] { path });
        }

        try
        {
            m_ds.deleteFolder(path);
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-delete-folder-failed", new Object[] { path }, e);
        }
    }

    public void createDirectory(String directoryName)
    throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-create-dir-not-connected", new Object[] { directoryName });
        }

        // if we can list the directory then it exists, otherwise create it
        try
        {
            ((IDirectoryAdminService)m_ds).listDirectories(directoryName);
        }
        catch (DirectoryServiceException e)
        {
            // assumes there's a slash at the end of the directory specification
            int lastSlash = directoryName.lastIndexOf("/", directoryName.length() - 2);

            if (lastSlash != -1)
            {
                createDirectory(directoryName.substring(0, lastSlash + 1));
                try
                {
                    ((IDirectoryAdminService)m_ds).createDirectory(directoryName);
                }
                catch (DirectoryServiceException createException)
                {
                    throw new ConfigServiceException("cs-create-dir-failed", new Object[] { directoryName }, createException);
                }
                catch (ConfigurePermissionDeniedException e1)
                {
                    throw new ConfigServiceException("cs-create-dir-failed",new Object[] { directoryName },e1);

                }
            }
        }
        catch (ConfigurePermissionDeniedException e)
        {
            throw new ConfigServiceException("cs-create-dir-failed",new Object[] { directoryName },e);
        }
    }

    @Override
    public void rename(String oldName, String newName)
        throws ConfigServiceException
    {
        try
        {
            m_ds.rename(oldName, newName);
            renameCacheElements(oldName, newName);
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-rename-failed", new Object[] { oldName, newName }, e);
        }
    }

    protected void renameCacheElements(String oldPrefix, String newPrefix)
        throws ConfigServiceException
    {
        Iterator it = m_elementCache.getObjects().iterator();
        while (it.hasNext())
        {
            ConfigElementImpl configElement = (ConfigElementImpl) it.next();
            if (configElement.getName().startsWith(oldPrefix))
            {
                /* Remove from cache under old name */
                getElementCache().remove(configElement.getName(), "");

                configElement.rename(oldPrefix, newPrefix);

                /* Add to cache under new name */
                getElementCache().add(configElement.getName(), "", configElement);
            }
        }
    }

    @Override
    public Set list(String path)
    throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-list-not-connected", new Object[] { path });
        }

        try
        {
            HashMap[] entries = m_ds.listFSAll(path);

            HashSet set = new HashSet(entries.length);

            for (int i = 0; i < entries.length; i++)
            {
                set.add(Util.splitToolMetaAttributes((Map)entries[i]));
            }

            return set;
        }
        catch (DirectoryServiceException e)
        {
            throw new ConfigServiceException("cs-list-failed", new Object[] { path }, e);
        }
        catch (ConfigurePermissionDeniedException e)
        {
            throw new ConfigServiceException("cs-list-failed", new Object[] { path }, e);
        }
    }

    @Override
    public Map getMetaAttributes(String path)
        throws ConfigServiceException
    {

        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-get-meta-attrs-not-connected", new Object[] { path });
        }

        try
        {
            return Util.splitToolMetaAttributes((Map)m_ds.getMetaAttributes(path));
        }
        catch (DirectoryServiceException e)
        {
            throw new ConfigServiceException("cs-get-meta-attrs-failed", new Object[] { path }, e);
        }
        catch (ConfigurePermissionDeniedException e)
        {
            throw new ConfigServiceException("cs-get-meta-attrs-failed", new Object[] { path }, e);
        }
    }

    @Override
    public void setMetaAttributes(String path, Map metaAttributes)
        throws ConfigServiceException
    {
        if (metaAttributes == null)
        {
            throw new ConfigServiceException("cs-set-meta-attrs-is-null", new Object[] {path});
        }

        try
        {
            m_ds.setMetaAttributes(path, Util.mapToHashMap(Util.combineToolMetaAttributes((Map)metaAttributes)));
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-set-meta-attrs-failed", new Object[] {path}, e);
        }
    }

    @Override
    public Set listFolders(String path)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-list-folders-not-connected", new Object[] { path });
        }

        try
        {
            HashMap[] directories = m_ds.listFolders(path);

            HashSet set = new HashSet(directories.length);

            for (int i = 0; i < directories.length; i++)
            {
                set.add(directories[i].get(FOLDER_NAME));
            }

            return set;
        }
        catch (DirectoryServiceException e)
        {
            throw new ConfigServiceException("cs-list-folders-failed", new Object[] { path }, e);
        }
        catch (ConfigurePermissionDeniedException e)
        {
            throw new ConfigServiceException("cs-list-folders-failed", new Object[] { path }, e);
        }
    }

    @Override
    public Set listConfigElements(String path)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-list-ces-not-connected", new Object[] { path });
        }

        try
        {
            HashMap[] elements = m_ds.listFSElements(path);

            HashSet set = new HashSet(elements.length);

            for (int i = 0; i < elements.length; i++)
            {
                IElementIdentity id = (IElementIdentity) elements[i].get(ELEMENT_IDENTITY);

                if(id != null)
                {
                    set.add(id.getName());
                }
            }
            return set;
        }
        catch (DirectoryServiceException e)
        {
            throw new ConfigServiceException("cs-list-ces-failed", new Object[] { path }, e);
        }
        catch (ConfigurePermissionDeniedException e)
        {
            throw new ConfigServiceException("cs-list-ces-failed", new Object[] { path }, e);
        }
    }

    public IBlob getBlob(String name)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-get-blob-not-connected", new Object[] { name });
        }

        try
        {
            return m_ds.getFSBlob(name, m_writeable);
        }
        catch (DirectoryServiceException e)
        {
            throw new ConfigServiceException("cs-get-blob-failed", new Object[] { name }, e);
        }
        catch (ConfigurePermissionDeniedException e)
        {
            throw new ConfigServiceException("cs-get-blob-failed", new Object[] { name }, e);
        }
    }

    @Override
    public String storageToLogical(String name)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-storage-to-logical-not-connected", new Object[] { name });
        }

        try
        {
            return m_ds.storageToLogical(name);
        }
        catch (DirectoryServiceException e)
        {
            throw new ConfigServiceException("cs-storage-to-logical-failed", new Object[] { name }, e);
        }
        catch (ConfigurePermissionDeniedException e)
        {
            throw new ConfigServiceException("cs-storage-to-logical-failed", new Object[] { name }, e);
        }
    }

    @Override
    public String logicalToStorage(String name)
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-logical-to-storage-not-connected", new Object[] { name });
        }

        try
        {
            return m_ds.logicalToStorage(name);
        }
        catch (DirectoryServiceException e)
        {
            throw new ConfigServiceException("cs-logical-to-storage-failed", new Object[] { name }, e);
        }
        catch (ConfigurePermissionDeniedException e)
        {
            throw new ConfigServiceException("cs-logical-to-storage-failed", new Object[] { name }, e);
        }
    }

    private Query resetFromClause(Query query)
    {
        if (query.getFrom() instanceof FromDirectory)
        {
            query.setFrom(new FromDirectory("/mx/configTypes"));
        }
        else
        if (query.getFrom() instanceof FromElementList)
        {
            String[] list = ((FromElementList)query.getFrom()).getElementNames();
            String[] newList = new String[list.length];

            for (int i = 0; i < list.length; i++)
            {
               int indx = list[i].lastIndexOf("/");
               newList[i] = "/mx/configTypes" + list[i].substring(indx);
            }
            query.setFrom(new FromElementList(newList));
        }

        return query;
    }

    //------------------------------------------------------------------------
    //
    // DS Change Notification handling
    //
    //------------------------------------------------------------------------

    protected final void registerElementChangeListener()
        throws ConfigServiceException
    {
        if (m_ds == null)
        {
            throw new ConfigServiceException("cs-register-escl-not-connected");
        }

        if ((m_configChangeDispatcher == null) && (m_ds instanceof IDirectoryFileSystemNotifications))
        {
            m_configChangeDispatcher = new ConfigChangeDispatcher(this);
            ((IDirectoryFileSystemNotifications)m_ds).registerFSElementChangeHandler(m_configChangeDispatcher);
            ((IDirectoryFileSystemNotifications)m_ds).registerNameChangeHandler(m_configChangeDispatcher);
        }

        // Must register logical listener before physical otherwise DS gets locked in physical mode
        // Its a DS implementation side-affect.
        if ((m_typeChangeHandler == null) && (m_ds instanceof IDirectoryAdminNotifications))
        {
            m_typeChangeHandler = new ConfigTypeChangeHandler(this);
            ((IDirectoryAdminNotifications)m_ds).registerElementChangeHandler(m_typeChangeHandler);
        }
    }

    protected void unregisterElementChangeListener()
        throws ConfigServiceException
    {
        if ((m_typeChangeHandler != null) && (m_ds instanceof IDirectoryAdminNotifications))
        {
            try
            {
                ((IDirectoryAdminNotifications)m_ds).unregisterElementChangeHandler(m_typeChangeHandler);
                m_typeChangeHandler = null;
            }
            catch (Exception e)
            {
                throw new ConfigServiceException("cs-unregister-escl-failed", e);
            }
        }

        if ((m_configChangeDispatcher != null) && (m_ds instanceof IDirectoryFileSystemNotifications))
        {
            try
            {
                ((IDirectoryFileSystemNotifications)m_ds).unregisterFSElementChangeHandler(m_configChangeDispatcher);
                ((IDirectoryFileSystemNotifications)m_ds).unregisterNameChangeHandler(m_configChangeDispatcher);
                m_configChangeDispatcher = null;
            }
            catch (Exception e)
            {
                throw new ConfigServiceException("cs-unregister-escl-failed", e);
            }
        }
    }

    class ConfigTypeChangeHandler implements IElementChangeHandler
    {
        private ConfigServer m_cs;

        public ConfigTypeChangeHandler(ConfigServer cs)
        {
            m_cs = cs;
        }

        @Override
        public void handleElementChange(IElementChange ec)
        {
            synchronized (m_cs)
            {
                IBasicElement element = ec.getElement();
                String physicalPath = element.getIdentity().getName();

                // We are only going to deal with ConfigType elements here so
                // filter anything else out...
                //
                if (physicalPath.startsWith("/mx"))
                {
                    if ((ec.getChangeType() == IElementChange.ELEMENT_UPDATED) &&
                        hasInitialValue(element))
                    {
                        try
                        {
                            String iValue = getInitialValue(element);

                            // Extract the type and version information from the
                            // physical path. It has the format:
                            //    /mx/configTypes/<version>/<type>
                            //
                            int    index   = physicalPath.lastIndexOf('/');
                            String type    = physicalPath.substring(index + 1);
                            String tmp     = physicalPath.substring(0, index);
                            String version = tmp.substring(tmp.lastIndexOf('/') + 1);

                            IConfigType configType = m_cs.loadConfigTypeFromLocal(type, version);

                            // If the type is cached then we want to update the initial value
                            //
                            if (configType != null)
                            {
                                ((ConfigTypeImpl)configType).setInitialValues(getInitialValue(element), false);
                            }
                        }
                        catch (Exception e)
                        {
                        }
                    }
                }
            }
        }

        private boolean hasInitialValue(IBasicElement element)
        {
            if (element instanceof IDeltaElement)
            {
                Object obj = ((IDeltaElement)element).getDeltaAttributes();

                if (obj instanceof IDeltaAttributeSet)
                {
                    IDeltaAttributeSet set = (IDeltaAttributeSet)obj;

                    if (inArray(set.getDeletedAttributesNames(),  ConfigTypeImpl.INITIAL_VALUES) ||
                        inArray(set.getModifiedAttributesNames(), ConfigTypeImpl.INITIAL_VALUES) ||
                        inArray(set.getNewAttributesNames(),      ConfigTypeImpl.INITIAL_VALUES))
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        private boolean inArray(String[] items, String item)
        {
            for (int i = 0; i < items.length; i++)
            {
                if (items[i].equals(item))
                {
                    return true;
                }
            }
            return false;
        }

        private String getInitialValue(IBasicElement element)
        {
            String res = null;

            if (element instanceof IDeltaElement)
            {
                Object obj = ((IDeltaElement)element).getDeltaAttributes();

                if (obj instanceof IDeltaAttributeSet)
                {
                    IDeltaAttributeSet set = (IDeltaAttributeSet)obj;

                    if (!inArray(set.getDeletedAttributesNames(), ConfigTypeImpl.INITIAL_VALUES))
                    {
                        try
                        {
                            res = set.getNewValue(ConfigTypeImpl.INITIAL_VALUES).toString();
                        }
                        catch (NotModifiedAttException e)
                        {
                        }
                    }
                }
            }

            return res;
        }
    }

    //------------------------------------------------------------------------

    @Override
    public IConfigElement createConfigElement(String name)
        throws ConfigServiceException
    {
        try
        {
            return new ConfigElementImpl(name, this);
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-create-ce-failed", new Object[] { name }, e);
        }
    }

    /**
     * Creates a new config bean, i.e. a typed config element with the given
     * path (name).
     *
     * Note: a config bean has to be typed and versioned.
     *
     * @param path                     The fully qualified path that will be
     *                                 the logical location of the new element
     *                                 in the Directory Service.
     * @param type                     The type of element (<code>IConfigBean</code>)
     *                                 being created.
     * @param version                  Types are versioned so we must specify
     *                                 which particular configuration version
     *                                 of the type is to be created.
     * @return                         The newly created <code>IConfigBean</code>.
     * @throws ConfigServiceException  An exception if the operation failed.
     * @deprecated                     Use other createconfigBean method.
     */
    @Override
    public IConfigBean createConfigBean(String name, String type, String version)
        throws ConfigServiceException
    {
        return createConfigBean(name, type, version, false);
    }

    @Override
    public IConfigBean createConfigBean(String  name,
                                        String  type,
                                        String  version,
                                        boolean isTemplate)
        throws ConfigServiceException
    {
        try
        {
            if (isTemplate)
            {
                return new ConfigBeanPrototype(name, type, version, this);
            }

            return new ConfigBeanImpl(name, type, version, this);
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-create-cb-failed", new Object[] { name }, e);
        }
    }

    @Override
    public IConfigBeanFile createConfigBeanFile(String name, String user)
        throws ConfigServiceException
    {
        try
        {
            return new ConfigBeanFile(name, user, this);
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-create-cbf-failed", new Object[] { name }, e);
        }
    }

    public IConfigType createConfigType(String name, String version)
        throws ConfigServiceException
    {
        try
        {
            return new ConfigTypeImpl(name, version, this);
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-create-ct-failed", new Object[] { name, version }, e);
        }
    }

    @Override
    public Collection parseConfigTypes(String namespaceURI, String schemaDocURL)
        throws ConfigServiceException
    {
        try
        {
            return ConfigTypeDocument.parse(namespaceURI, schemaDocURL, this, null);
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-parse-ct-failed", new Object[] { schemaDocURL }, e);
        }
    }

    @Override
    public IConfigBean parseConfigBean(String beanDocumentURL)
        throws ConfigServiceException
    {
        try
        {
            return ConfigBeanDocument.parse(beanDocumentURL, this);
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-parse-cb-failed", new Object[] { beanDocumentURL }, e);
        }
    }

    @Override
    public String exportConfigBean(String name)
        throws ConfigServiceException
    {
        IConfigBean bean = null;

        try
        {
            bean = loadConfigBean(name, false);
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-export-cb-does-not-exist", new Object[] { name } );
        }

        return exportConfigBean(bean);
    }

    @Override
    public String exportConfigBean(IConfigBean bean)
    {
        return bean.toXML();
    }
    
    @Override
    public String exportDSBootConfiguration(IConfigElement configElement)
    {    	
    	try {
    		 String storageName = logicalToStorage(configElement.getName());
			return ((IDirectoryAdminService)m_ds).exportDSBootFileString(storageName);
		} catch (DirectoryServiceException e) {
			return null;
		} catch (ConfigServiceException e) {
			return null;
		}
    }

    @Override
    public String exportContainerBootConfiguration(IConfigElement containerConfigElement,
                                                   String         domainName) throws ConfigServiceException
    {
        String storageName = logicalToStorage(containerConfigElement.getName());

        com.sonicsw.mf.common.config.IElement containerDirElement
            = ElementFactory.createElement(storageName,
                                           ((ConfigElementImpl)containerConfigElement).m_type,
                                           ((ConfigElementImpl)containerConfigElement).m_version);

        //<IL> Sonic00020818, We need a summary of prototype and instance attributes that have been changed.
        //Otherwise produced container boot file will be invalid during container startup
        if (containerConfigElement.isPrototypeInstance())
        {
            ((ConfigElementImpl)containerConfigElement.getPrototype()).toAttributeSet(containerDirElement.getAttributes());
        }
        
        ((ConfigElementImpl)containerConfigElement).toAttributeSet(containerDirElement.getAttributes(), containerConfigElement.isPrototypeInstance());
        
        return com.sonicsw.mf.common.util.Container.exportContainerBootConfiguration(containerDirElement, domainName);
    }
   
    protected Cache getTypeCache()
    {
        return m_typeCache;
    }

    protected Cache getElementCache()
    {
        return m_elementCache;
    }

    synchronized protected ConfigBeanImpl createBeanFromPrototype(ConfigBeanPrototype prototype,
                                                                  String              name)
        throws ConfigServiceException
    {
        try
        {
            ConfigBeanImpl res = new ConfigBeanImpl(name, prototype.getConfigType().getName(), prototype.getConfigType().getVersion(), this, true);

            res.m_configPrototype = prototype;

            return res;
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-create-bfp-failed", new Object[] {name, prototype.getName()} , e);
        }
    }

    synchronized protected ConfigElementImpl createElementFromPrototype(ConfigElementPrototype prototype,
                                                                        String                 name)
        throws ConfigServiceException
    {
        try
        {
            ConfigElementImpl res = new ConfigElementImpl(name, this);

            res.m_configPrototype = prototype;

            return res;
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-create-efp-failed", new Object[] {name, prototype.getName()} , e);
        }
    }

    protected void unshareConfigElement(ConfigElementImpl configElement)
        throws ConfigServiceException
    {

        try
        {
            /* Save any changes to DS */
            storeConfigElement(configElement);

            /*  Remove subclassing from element at DS level */
            m_ds.unSubclassFSElement(configElement.getName());

            /* Remove instance from prototype */
            if (configElement.m_configPrototype instanceof ConfigElementPrototype)
            {
                ((ConfigElementPrototype)configElement.m_configPrototype).removeInstance(configElement);
            }
            else
            {
                ((ConfigBeanPrototype)configElement.m_configPrototype).removeInstance(configElement);
            }
            configElement.m_configPrototype = null;

            /* Repopulate config element with state from DS. */
            configElement.refresh();
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-unshare-ce-failed", new Object[] { configElement.getName() }, e);
        }
    }

    protected IConfigType dirElement2ConfigType(IDirElement dirElement)
        throws ConfigServiceException
    {
        try
        {
            if (dirElement != null &&
                dirElement.getIdentity().getType() != null &&
                dirElement.getIdentity().getType().equals(ConfigTypeImpl.IDENTITY_ELEMENT_TYPE_STRING))
            {   // This is a dir element for a config type
                return new ConfigTypeImpl(dirElement, this);
            }
            else
            {
                return null;
            }
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-de-2-ct-failed", e);
        }
    }

    protected IConfigElement dirElement2ConfigElement(IDirElement dirElement)
        throws ConfigServiceException
    {
        try
        {
            if (dirElement == null)
            {
                return null;
            }

            if (dirElement.getIdentity().getType() == null ||
                dirElement.getIdentity().getType().equals("") ||
                dirElement.getIdentity().getType().equals(ConfigElementImpl.IDENTITY_ELEMENT_TYPE_STRING))
            {   /*  This is a dir element for a config element  */
                if (dirElement.isTemplate())
                {   /*  This is a dir element for a config element prototype */
                    return new ConfigElementPrototype(dirElement, this);
                }
                else
                {
                    return new ConfigElementImpl(dirElement, this);
                }
            }
            else
            if (dirElement.getIdentity().getType().equals(ConfigBeanFile.IDENTITY_ELEMENT_TYPE_STRING))
            {   /*  This is a dir element for a config bean file */
                return new ConfigBeanFile(dirElement, this);
            }
            else
            if (dirElement.getIdentity().getType().equals(ConfigTypeImpl.IDENTITY_ELEMENT_TYPE_STRING))
            {   /*  This is a dir element for a config type */
                return null;
            }
            else
            {   /*  This is a dir element for a config bean */
                if (dirElement.isTemplate())
                {   /*  This is a dir element for a config bean prototype */
                    return new ConfigBeanPrototype(dirElement, this);
                }
                else
                {
                    return new ConfigBeanImpl(dirElement, this);
                }
            }
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-de-2-ce-failed", e);
        }
    }

    //-------------------------------------------------------------------------
    //
    // Config Change Listener implementation
    //
    // Listeners registered with config server to receive configuration
    // change notification.
    //
    //-------------------------------------------------------------------------

    protected Map m_listeners = new HashMap();

    class ListenerRegistration
    {
        IConfigChangeListener m_listener = null;

        IConfigChangeFilter m_filter = null;

        Object m_handback = null;

        public ListenerRegistration(IConfigChangeListener listener,
                                    IConfigChangeFilter filter,
                                    Object handback)
        {
            m_listener = listener;
            m_filter = filter;
            m_handback = handback;
        }

        @Override
        public boolean equals(Object obj)
        {
            if (obj instanceof ListenerRegistration)
            {
                ListenerRegistration reg = (ListenerRegistration) obj;
                if(m_listener.equals(reg.m_listener) &&
                   m_filter == null ? reg.m_filter == null : m_filter.equals(reg.m_filter) &&
                   m_handback == null ? reg.m_handback == null : m_handback.equals(reg.m_handback))
                {
                    return true;
                }
            }
            return false;
        }

        @Override
        public int hashCode()
        {
            return m_listener.hashCode() +
                   (m_filter == null ? 0 : m_filter.hashCode()) +
                   (m_handback == null ? 0 : m_handback.hashCode());
        }

    }

    @Override
    public void addConfigChangeListener(String                configElementName,
                                        IConfigChangeListener listener,
                                        IConfigChangeFilter   filter,
                                        Object                handback)
    {
        if (listener == null)
        {
            throw new IllegalArgumentException();
        }

        synchronized (m_listeners)
        {
            Map elementListeners = (Map) m_listeners.get(configElementName);

            if (elementListeners == null)
            {
                elementListeners = new HashMap();
                m_listeners.put(configElementName, elementListeners);
            }

            Set listenerReg = (Set) elementListeners.get(listener);

            if (listenerReg == null)
            {
                listenerReg = new HashSet();
                elementListeners.put(listener, listenerReg);
            }

            listenerReg.add(new ListenerRegistration(listener, filter, handback));
        }
    }

    @Override
    public void removeConfigChangeListener(String                configElementName,
                                           IConfigChangeListener listener,
                                           IConfigChangeFilter   filter,
                                           Object                handback)
        throws ConfigServiceException
    {
        synchronized (m_listeners)
        {
            Map elementListeners = (Map) m_listeners.get(configElementName);
            if (elementListeners == null)
            {
                throw new ConfigServiceException("listener not registered");
            }

            Set listenerReg = (Set) elementListeners.get(listener);
            if (listenerReg == null)
            {
                throw new ConfigServiceException("listener not registered");
            }

            listenerReg.remove(new ListenerRegistration(listener, filter, handback));

            if (listenerReg.isEmpty())
            {
                elementListeners.remove(listener);

                if (elementListeners.isEmpty())
                {
                    m_listeners.remove(configElementName);
                }
            }
        }
    }

    @Override
    public void removeConfigChangeListener(String                configElementName,
                                           IConfigChangeListener listener)
        throws ConfigServiceException
    {
        synchronized (m_listeners)
        {
            Map elementListeners = (Map) m_listeners.get(configElementName);
            if (elementListeners == null)
            {
                throw new ConfigServiceException("listener not registered");
            }

            Set listenerReg = (Set) elementListeners.remove(listener);
            if (listenerReg == null)
            {
                throw new ConfigServiceException("listener not registered");
            }
        }
    }

    synchronized protected void sendConfigChange(ConfigChange configChange)
    {
        synchronized (m_listeners)
        {
            Map elementListeners = (Map)m_listeners.get(configChange.getName());
            if (elementListeners != null)
            {
                dispatchConfigChange(elementListeners, configChange);
            }

            elementListeners = (Map)m_listeners.get(null);
            if (elementListeners != null)
            {
                dispatchConfigChange(elementListeners, configChange);
            }
        }
    }

    protected void dispatchConfigChange(Map          elementListeners,
                                        ConfigChange configChange)
    {
        Iterator listenerRegs = elementListeners.values().iterator();

        while (listenerRegs.hasNext())
        {
            Set listenerReg = (Set) listenerRegs.next();
            Iterator registrations = listenerReg.iterator();
            while (registrations.hasNext())
            {
                ListenerRegistration registration = (ListenerRegistration) registrations.next();
                if(registration.m_filter == null ||
                   registration.m_filter.isConfigChangeEnabled(configChange))
                {
                    switch (configChange.getChangeType())
                    {
                    case ConfigChange.ELEMENT_ADDED:
                        registration.m_listener.elementAdded(new ConfigChange(configChange, registration.m_handback));
                        break;
                    case ConfigChange.ELEMENT_DELETED:
                        registration.m_listener.elementDeleted(new ConfigChange(configChange, registration.m_handback));
                        break;
                    case ConfigChange.ELEMENT_REPLACED:
                        registration.m_listener.elementReplaced(new ConfigChange(configChange, registration.m_handback));
                        break;
                    case ConfigChange.ELEMENT_UPDATED:
                        registration.m_listener.elementUpdated(new ConfigChange(configChange, registration.m_handback));
                        break;
                    case ConfigChange.FOLDER_ADDED:
                        registration.m_listener.folderAdded(new ConfigChange(configChange, registration.m_handback));
                        break;
                    case ConfigChange.FOLDER_DELETED:
                        registration.m_listener.folderDeleted(new ConfigChange(configChange, registration.m_handback));
                        break;
                    case ConfigChange.RENAME:
                        registration.m_listener.rename(new ConfigChange(configChange, registration.m_handback));
                        break;
                    case ConfigChange.META_ATTRIBUTES_CHANGE:
                        registration.m_listener.metaAttributesChanged(new ConfigChange(configChange, registration.m_handback));
                        break;
                    }
                }
            }
        }
    }

    @Override
    public synchronized void commit()
        throws ConfigServiceException
    {
        throw new ConfigServiceException("cs-commit-not-transacted");
    }

    @Override
    public synchronized void rollback()
        throws ConfigServiceException
    {
        throw new ConfigServiceException("cs-rollback-not-transacted");
    }

    @Override
    public synchronized void flush()
        throws ConfigServiceException
    {
        throw new ConfigServiceException("cs-flush-not-transacted");
    }

    @Override
    public synchronized IConfigServer subtransaction()
        throws ConfigServiceException
    {
        throw new ConfigServiceException("cs-subtransaction-not-transacted");
    }

    protected void elementCreated(ConfigElementImpl configElement)
        throws ConfigServiceException
    {
        getElementCache().add(configElement.getName(), "", configElement);
        configElement.setDirectoryElementName(configElement.getName());
        configElement.setState(ConfigElementImpl.NEW_STATE);
    }

    protected void elementLoaded(ConfigElementImpl configElement)
        throws ConfigServiceException
    {
        getElementCache().add(configElement.getName(), "", configElement);
        configElement.setDirectoryElementName(configElement.getName());
        configElement.setState(ConfigElementImpl.UNMODIFIED_STATE);
    }

    protected void elementStored(ConfigElementImpl configElement)
        throws ConfigServiceException
    {
        configElement.setDirectoryElementName(configElement.getName());
        configElement.setState(ConfigElementImpl.UNMODIFIED_STATE);
    }

    protected void elementRefreshed(ConfigElementImpl configElement)
        throws ConfigServiceException
    {
        configElement.setState(ConfigElementImpl.UNMODIFIED_STATE);
    }

    protected void elementModified(ConfigElementImpl configElement)
        throws ConfigServiceException
    {
         if (!configElement.isNew())
        {
            configElement.setState(ConfigElementImpl.MODIFIED_STATE);
        }
    }

    protected void elementRenamed(String            oldName,
                                  String            newName,
                                  ConfigElementImpl configElement)
        throws ConfigServiceException
    {
        try
        {
            if (!configElement.isNew())
            {
                m_ds.rename(oldName, newName);
            }
            else
            {
                /* New Element uses new name when creating delta */
                configElement.setDirectoryElementName(newName);
            }
            /* Add to cache under new name */
            getElementCache().add(newName, "", configElement);
            /* Remove from cache under old name */
            getElementCache().remove(oldName, "");
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-element-rename-failed", new Object[] { oldName, newName }, e);
        }
    }

    protected void elementRemoved(ConfigElementImpl configElement)
        throws ConfigServiceException
    {
        getElementCache().remove(configElement.getName(), "");
        configElement.setState(ConfigElementImpl.REMOVED_STATE);
    }

    protected void elementRemoved(String configElementName)
        throws ConfigServiceException
    {
        ConfigElementImpl configElement = (ConfigElementImpl) getElementCache().lookup(configElementName, "");
        if (configElement != null)
        {
            getElementCache().remove(configElementName, "");
            configElement.setState(ConfigElementImpl.REMOVED_STATE);
        }
    }
    

    @Override
    public IConfigPath[] getReferences(String configElementName)
        throws ConfigServiceException
    {
        try
        {
            AttributeName[] attributeNames = m_ds.getReferences(configElementName);
            IConfigPath[] configPaths = new IConfigPath[attributeNames.length];

            for (int i = 0; i < configPaths.length; i++)
            {
                configPaths[i] = new ConfigPathImpl(attributeNames[i]);
            }

            return configPaths;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-get-references-failed", new Object[] { configElementName }, e);
        }
    }

    public void revertToTemplate(String        configElement,
                                 IConfigPath[] attributes)
        throws ConfigServiceException
    {
        try
        {
            AttributeName[] attributeNames = new AttributeName[attributes.length];

            // We're going to need this in order to convert each path to
            // an AttributeName object...
            IConfigElement ce = loadConfigElement(configElement);

            for (int i = 0; i < attributes.length; i++)
            {
                attributeNames[i] = ((ConfigPathImpl)attributes[i]).toAttributeName(ce);
                debugPrintln("reverting " + attributeNames[i].toString() + " (" + configElement + ")");
            }

            m_ds.revertToTemplate(configElement, attributeNames);
        }
        catch (ConfigServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-revert-to-template-failed", e);
        }
    }

    protected void debugPrintln(String message)
    {
        debugPrintln(message, null);
    }

    protected void debugPrintln(String message, IDSTransaction txn)
    {
        if (!DEBUG)
        {
            return;
        }

        StringBuffer sb = new StringBuffer();
        if (isTransacted())
        {
            sb.append("txn");
        }
        sb.append("CL(").append(hashCode()).append("): ");
        sb.append(message);

        System.err.println(sb.toString());

        if (txn != null)
        {
            System.err.print(Util.txnToString(txn));
            dumpCache();
        }
    }

    protected void dumpCache()
    {
        if (!DEBUG)
        {
            return;
        }

        System.err.println("CL(" + hashCode() + "): Cache");

        Iterator it = m_elementCache.getObjects().iterator();
        while (it.hasNext())
        {
            ConfigElementImpl element = (ConfigElementImpl) it.next();
            System.err.println("    " + element.getName() + " State=" + ConfigElementImpl.STATES_DEBUG[element.getState()]);
        }
    }
}
