package com.sonicsw.mx.config.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import org.apache.xerces.impl.Constants;
import org.apache.xerces.impl.XMLEntityManager;
import org.apache.xerces.impl.dv.DatatypeException;
import org.apache.xerces.impl.dv.ValidationContext;
import org.apache.xerces.impl.dv.XSSimpleType;
import org.apache.xerces.impl.xpath.regex.RegularExpression;
import org.apache.xerces.impl.xs.SchemaGrammar;
import org.apache.xerces.impl.xs.XSAttributeDecl;
import org.apache.xerces.impl.xs.XSAttributeUseImpl;
import org.apache.xerces.impl.xs.XSElementDecl;
import org.apache.xerces.parsers.XMLGrammarPreparser;
import org.apache.xerces.util.SymbolTable;
import org.apache.xerces.util.XMLGrammarPoolImpl;
import org.apache.xerces.xni.XNIException;
import org.apache.xerces.xni.grammars.XMLGrammarDescription;
import org.apache.xerces.xni.parser.XMLErrorHandler;
import org.apache.xerces.xni.parser.XMLInputSource;
import org.apache.xerces.xni.parser.XMLParseException;
import org.apache.xerces.xs.StringList;
import org.apache.xerces.xs.XSAttributeDeclaration;
import org.apache.xerces.xs.XSAttributeUse;
import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.apache.xerces.xs.XSConstants;
import org.apache.xerces.xs.XSElementDeclaration;
import org.apache.xerces.xs.XSModel;
import org.apache.xerces.xs.XSModelGroup;
import org.apache.xerces.xs.XSNamedMap;
import org.apache.xerces.xs.XSObjectList;
import org.apache.xerces.xs.XSParticle;
import org.apache.xerces.xs.XSSimpleTypeDefinition;
import org.apache.xerces.xs.XSTypeDefinition;

import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.ConfigTypeException;
import com.sonicsw.mx.config.IAttributeDescription;
import com.sonicsw.mx.config.IAttributeList;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IConfigElement;
import com.sonicsw.mx.config.IConfigPath;
import com.sonicsw.mx.config.IConfigType;

public class ConfigTypeDocument
{
    //IL
    public static final String SYMBOL_TABLE =
        Constants.XERCES_PROPERTY_PREFIX + Constants.SYMBOL_TABLE_PROPERTY;

    /** Property identifier: grammar pool. */
    public static final String GRAMMAR_POOL =
        Constants.XERCES_PROPERTY_PREFIX + Constants.XMLGRAMMAR_POOL_PROPERTY;

    // feature ids

    /** Namespaces feature id (http://xml.org/sax/features/namespaces). */
    protected static final String NAMESPACES_FEATURE_ID = "http://xml.org/sax/features/namespaces";

    /** Validation feature id (http://xml.org/sax/features/validation). */
    protected static final String VALIDATION_FEATURE_ID = "http://xml.org/sax/features/validation";

    /** Schema validation feature id (http://apache.org/xml/features/validation/schema). */
    protected static final String SCHEMA_VALIDATION_FEATURE_ID = "http://apache.org/xml/features/validation/schema";

    /** Schema full checking feature id (http://apache.org/xml/features/validation/schema-full-checking). */
    protected static final String SCHEMA_FULL_CHECKING_FEATURE_ID = "http://apache.org/xml/features/validation/schema-full-checking";

    //
    private static final String CONFIG_BEAN_ELEMENT_NAME = "config-bean";
    private static final String ANY_TYPE_NAME = "anyType";
    private static final String ATTRIBUTE_MAP_TYPE_NAME = "attributeMap";
    private static final String ATTRIBUTE_LIST_TYPE_NAME = "attributeList";
    private static final String DESCRIPTION_ATTRIBUTE = "description";

    //  Error Messages

    private static final String INV_CFG_TYPE = "Invalid configuration type: ";
    private static final String INV_CFG_TYPE_SIMPLE_CONTENT = " : simple content: configuration bean element can not contain text.";
    private static final String INV_CFG_TYPE_MIXED_CONTENT = " : mixed content: configuration bean element can not contain text.";
    private static final String INV_CFG_TYPE_UNKNOWN_CONTENT = " : unknown content type.";
    private static final String INV_CFG_TYPE_SIMPLE_TYPE = " : simple type: configuration bean element can not contain text.";

    private static final String INV_ATTR_MAP = "Invalid attribute map type: ";
    private static final String INV_ATTR_MAP_RPT_ELMT = " : repeated element: attribute element can occur in attribute map at most once.";
    private static final String INV_ATTR_MAP_OPTRPT_GRP = " : optional or repeated all group: a group of attribute elements must occur in attribute map exactly once.";
    private static final String INV_ATTR_MAP_SEQ_GRP = " : sequence group: attribute elements of an attribute map must be specified within a choice group.";
    private static final String INV_ATTR_MAP_WILD_CARD = " : wild card: an attribute map can not contain a wild card element.";
    private static final String INV_ATTR_MAP_CHC_GRP = " : choice group: attribute elements of an attribute map must be specified within a choice group.";
    private static final String INV_ATTR_MAP_EMPT_GRP = " : empty group.";
    private static final String INV_ATTR_MAP_UNK_GRP = " : unknown group type.";

    private static final String INV_ATTR = "Invalid attribute type: ";
    private static final String INV_ATTR_MIXED_CONTENT = " : mixed content: attribute element can not contain both text and child elements.";
    private static final String INV_ATTR_UNK_CONTENT = " : unknown content type.";
    private static final String INV_ATTR_LIST_TYPE = " : list simple type: list simple types for attribute values are not supported";
    private static final String INV_ATTR_UNION_TYPE = " : union simple type: union simple types for attribute values are not supported";
    private static final String INV_ATTR_WILDCARD_TYPE = " : wildcard simple type: wildcard simple types for attribute values are not supported";
    private static final String INV_ATTR_UNK_TYPE = " : unknown simple type.";
    private static final String INV_ATTR_WILDCARD_CONTENT = " : wild card: an attribute map can not contain a wild card element.";
    private static final String INV_ATTR_CHOICE_CONTENT = " : choice group: attribute elements of an attribute map must be specified within an all group.";
    private static final String INV_ATTR_EMPTY_CONTENT = " : empty group.";
    private static final String INV_ATTR_UNKNOWN_CONTENT = " : unknown group type.";
    private static final String INV_ATTR_PROPERTY_NOT_FIXED = " : no fixed value: user defined attribute properties must have a fixed value.";
    private static final String INV_ATTR_DESCRIPTION_NOT_STRING = " : description not a string: attribute description must be a string.";
    private static final String INV_ATTR_UNSUPPORTED_PRIM_TYPE = "unsupported primitive type";
    private static final String INV_ATTR_INVALID_PRIM_TYPE = "invalid primitive type";

    private static final String INV_ATTR_LIST = "Invalid attribute list type: ";
    private static final String INV_ATTR_LIST_MULT_ELEMS = " : multiple element types: attribute lists can only contain a single type of element.";


    protected static final BigDecimal MAX_INT  = BigDecimal.valueOf(Integer.MAX_VALUE);
    protected static final BigDecimal MIN_INT  = BigDecimal.valueOf(Integer.MIN_VALUE);
    protected static final BigDecimal MAX_LONG = BigDecimal.valueOf(Long.MAX_VALUE);
    protected static final BigDecimal MIN_LONG = BigDecimal.valueOf(Long.MIN_VALUE);

    //-------------------------------------------------------------------------

    // The config server to which the config type is stored.
    static protected volatile ConfigServer s_configServer = null;

    static final ValidationContext s_noFacetContext = new ValidationContext()
    {
        @Override
        public boolean useNamespaces() { return false; }
    
        @Override
        public boolean needFacetChecking() { return false; }
        @Override
        public boolean needExtraChecking() { return false; }
        @Override
        public boolean needToNormalize() { return true; }
        @Override
        public boolean isEntityDeclared (String name) { return false; }
        @Override
        public boolean isEntityUnparsed (String name) { return false; }
        @Override
        public boolean isIdDeclared (String name) { return false; }
        @Override
        public void    addId(String name) { }
        @Override
        public void    addIdRef(String name) { }
        @Override
        public String  getSymbol (String symbol) { return null; }
        @Override
        public String  getURI(String prefix) { return null; }
    };

    //-------------------------------------------------------------------------

    static class ErrorHandler implements XMLErrorHandler
    {
        @Override
        public void warning(String domain, String key, XMLParseException ex)
            throws XNIException
        {
            error(domain, key, ex);
        }

        @Override
        public void fatalError(String domain, String key, XMLParseException ex)
        throws XNIException
        {
            error(domain, key, ex);
        }

        @Override
        public void error(String domain, String key, XMLParseException ex)
            throws XNIException
        {
            throw new XNIException(stripScheme(ex.getExpandedSystemId()) + ":" +
                                   (ex.getLineNumber() + 1) + ":" + ex.getColumnNumber() + ", " +
                                   ex.getMessage() + 
                                   " See W3C Recommendation '" + domain + "'",
                                   ex);
        }

        /**
         * Simple method that strips the scheme of the front of the uri to the resource.
         * 
         * @param uri  A uri representation of the resource (xml file).
         * @return     The uri untouched if no scheme is found otherwise the uri with
         *             the scheme stripped off.
         */
        private String stripScheme(String uri)
        {
            int index = uri.indexOf(":");
            
            return (index >= 0) ? uri.substring(index+1) : uri;
        }
    }

    //-------------------------------------------------------------------------

    static class ConfigSchema
    {
        protected XSModel xsModel = null;
        protected String version = null;
        protected String location = null;
        
        public ConfigSchema(SchemaGrammar grammar, String version, String locationHint)
        {
            xsModel = grammar.toXSModel();
            this.version = version;
            location = locationHint;
        }

        public XSNamedMap getGlobalAttrDecls()
        {
            return xsModel.getComponents(XSConstants.ATTRIBUTE_DECLARATION);
        }

        public XSNamedMap getGlobalAttrGrpDecls()
        {
            return xsModel.getComponents(XSConstants.ATTRIBUTE_GROUP);
        }

        public XSNamedMap getGlobalElemDecls()
        {
            return xsModel.getComponents(XSConstants.ELEMENT_DECLARATION);
        }

        public XSNamedMap getGlobalGroupDecls()
        {
            return xsModel.getComponents(XSConstants.MODEL_GROUP_DEFINITION);
        }

        public XSNamedMap getGlobalNotationDecls()
        {
            return xsModel.getComponents(XSConstants.NOTATION_DECLARATION);
        }

        public XSNamedMap getGlobalIDConstraintDecls()
        {
            return xsModel.getComponents(XSConstants.IDENTITY_CONSTRAINT);
        }

        public XSNamedMap getGlobalTypeDecls()
        {
            return xsModel.getComponents(XSConstants.TYPE_DEFINITION);
        }

        /** Returns the version of the element declaration designated by the
         *  given tag name or null if the tag name does not exist or does not have
         *  a version.
         *
         *  @param elementTagName the tag name of the element declaration
         *  @return version of element declaration.
         */
        public String getElementVersion(String elementTagName)
        {
            return version;
        }

        /** Returns the location of the schema document in which the declaration
         *  designated by the given tag name was declared or null if the tag name
         *  does not exist.
         *
         *  @param elementTagName the tag name of the element declaration
         *  @return location of schema document in which the element was declared.
         */
        public String getElementDocumentLocation(String elementTagName)
        {
            return getLocation(elementTagName);
        }
        
        private String getLocation(String elementTagName)
        {
            location = location.replace('\\', '/');
            
            if(elementTagName.equals("MF_COMPONENT"))
            {
                if(location.indexOf("/mf/")== -1)
                {
                    return "../mf/Component.xsd";
                }
                return "./Component.xsd";
            }
            
            if(elementTagName.equals("MF_APPLICATION_COMPONENT"))
            {
                if(location.indexOf("/mf/")== -1)
                {
                    return "../mf/ApplicationComponent.xsd";
                }
                return "./ApplicationComponent.xsd";
            }
            
            if(elementTagName.equals("MF_FRAMEWORK_COMPONENT"))
            {
                if(location.indexOf("/mf/")== -1)
                {
                    return "../mf/FrameworkComponent.xsd";
                }
                return "./FrameworkComponent.xsd";
            }
            
            return location;
        }
        
        
    }

    //-------------------------------------------------------------------------

    private static String quoteSpaces(String url)
    {
        StringBuffer buf = new StringBuffer();
        for (int i = 0; i < url.length(); i++)
        {
            char c = url.charAt(i);

            if (c == ' ')
            {
                buf.append("%20");
            }
            else
            {
                buf.append(c);
            }
        }
        return buf.toString();
    }

    private static ConfigSchema parseSchema(String namespaceURI, String schemaDocURL)
        throws ConfigServiceException
    {
        /*  Create Objects necessary to construct SchemaGrammar. */
        
        SymbolTable symbolTable = new SymbolTable();
        
        /* 'intern' URL strings */
        namespaceURI = symbolTable.addSymbol(namespaceURI);
        schemaDocURL = symbolTable.addSymbol(schemaDocURL);

        XMLGrammarPreparser preparser = new XMLGrammarPreparser(symbolTable);
        preparser.registerPreparser(XMLGrammarDescription.XML_SCHEMA, null);
        
        XMLGrammarPoolImpl grammarPool = new XMLGrammarPoolImpl();
        XMLEntityManager entityResolver = new XMLEntityManager();
        preparser.setErrorHandler(new ErrorHandler());
        preparser.setEntityResolver(entityResolver);
        preparser.setProperty(GRAMMAR_POOL, grammarPool);
        preparser.setFeature(NAMESPACES_FEATURE_ID, true);
        preparser.setFeature(VALIDATION_FEATURE_ID, true);
        
        // note we can set schema features just in case...
        preparser.setFeature(SCHEMA_VALIDATION_FEATURE_ID, true); 
        preparser.setFeature(SCHEMA_FULL_CHECKING_FEATURE_ID, false); //our schema isn't valid for full checking features
   
        String locationHint = schemaDocURL;
        String version = ConfigBeanDocument.parseVersion(schemaDocURL);
        
        /* Create XML Input Source */
        XMLInputSource xmlInputSource = null;
        try
        {
            if (!schemaDocURL.startsWith("file:"))
            {
                schemaDocURL = "file:" + schemaDocURL;
            }

            URL url = new URL(schemaDocURL);
            xmlInputSource = new XMLInputSource(null, schemaDocURL, null, url.openStream(), null);  
        }
        catch (IOException e)
        {
            throw new ConfigServiceException("cant-open-schema-document", new Object[]{ schemaDocURL }, e);
        }

        /*  Parsr schema document to construct a grammar. */
        SchemaGrammar schemaGrammar = null;
        
        try
        {
            schemaGrammar = (SchemaGrammar)preparser.preparseGrammar(XMLGrammarDescription.XML_SCHEMA, xmlInputSource);
        }
        catch (IOException e)
        {
            throw new ConfigServiceException("parse-failed", new Object[]{ schemaDocURL }, e);
        }
 

        if (schemaGrammar != null)
        {
            return new ConfigSchema(schemaGrammar, version, locationHint);
        }

        return null;
    }

    private static boolean isConfigBeanDecl(XSElementDeclaration element)
    {
        element = element.getSubstitutionGroupAffiliation();
     
        while (element != null && !element.getName().equals(CONFIG_BEAN_ELEMENT_NAME))
        {
            element = element.getSubstitutionGroupAffiliation();
        }
        
        return (element != null);
    }

    private static boolean isAttributeMapDecl(XSElementDeclaration element)
    {
        XSTypeDefinition baseType = element.getTypeDefinition().getBaseType();
        while (!baseType.getName().equals(ANY_TYPE_NAME))
        {
            if (baseType.getName().equals(ATTRIBUTE_MAP_TYPE_NAME))
            {
                return true;
            }

            baseType = baseType.getBaseType();
        }
        return false;
    }

    private static boolean isAttributeListDecl(XSElementDecl element)
    {
        XSTypeDefinition baseType = element.getTypeDefinition().getBaseType();
        while (!baseType.getName().equals(ANY_TYPE_NAME))
        {
            if (baseType.getName().equals(ATTRIBUTE_LIST_TYPE_NAME))
            {
                return true;
            }

            baseType = baseType.getBaseType();
        }
        return false;
    }

    public static void buildConfigType(ConfigTypeImpl configType, XSElementDeclaration configTypeDeclaration)
        throws ConfigServiceException
    {
        // Set Type
        configType.setType(IAttributeMap.class);

        configType.setProperty(IAttributeDescription.NILLABLE_PROPERTY, new Boolean(configTypeDeclaration.getNillable()));
        if (configTypeDeclaration.getConstraintType() == XSConstants.VC_FIXED)
        {
            configType.setProperty(IAttributeDescription.FIXED_PROPERTY, convertValue(((XSElementDecl)configTypeDeclaration).fDefault.actualValue, configTypeDeclaration.getTypeDefinition()));
        }
        if (configTypeDeclaration.getConstraintType() == XSConstants.VC_DEFAULT)
        {
            Object obj = convertValue(((XSElementDecl)configTypeDeclaration).fDefault.actualValue, configTypeDeclaration.getTypeDefinition());
            System.err.println("ConfigType: name = " + configTypeDeclaration.getName() + ": " +IAttributeDescription.DEFAULT_PROPERTY + " = " + obj);
            configType.setProperty(IAttributeDescription.DEFAULT_PROPERTY, obj /*convertValue(configTypeDeclaration.fDefault.actualValue, configTypeDeclaration.getTypeDefinition())*/);
        }
        configType.setProperty(IAttributeDescription.SCHEMA_TYPE_NAME, configTypeDeclaration.getTypeDefinition().getName());
        configType.setProperty(IAttributeDescription.SCHEMA_NAMESPACE, configTypeDeclaration.getNamespace());

        // Ensure Complex Type
        if (configTypeDeclaration.getTypeDefinition().getTypeCategory() == XSTypeDefinition.SIMPLE_TYPE)
        {
            throw new ConfigServiceException(INV_CFG_TYPE_SIMPLE_CONTENT + configTypeDeclaration.getName() + INV_CFG_TYPE_SIMPLE_TYPE);
        }
        XSComplexTypeDefinition type = (XSComplexTypeDefinition) configTypeDeclaration.getTypeDefinition();

        // Ensure Group Type
        if (!(type.getParticle().getTerm() instanceof XSModelGroup))
        {
            throw new ConfigServiceException(INV_CFG_TYPE + configTypeDeclaration.getName() + ": the element declaring this config type does not contain model group: the item in a config type must be declared within a model group.");
        }
        XSModelGroup modelGroup = (XSModelGroup) type.getParticle().getTerm();

        // Ensure All Group Type
        if (modelGroup.getCompositor() != XSModelGroup.COMPOSITOR_ALL)
        {
            throw new ConfigServiceException(INV_CFG_TYPE + configTypeDeclaration.getName() + ": the element declaring this config type does not contain sequence model group: the items in a config type must be declared within an all model group.");
        }

        buildAttributeMapDescription((AttributeDescriptionMapImpl)configType.getAttributeDescriptions(), modelGroup);
    }

    private static void buildAttributeMapDescription(AttributeDescriptionMapImpl mapDescription, XSModelGroup modelGroup)
        throws ConfigServiceException
    {
        // Ensure All Group Type
        if (modelGroup.getCompositor() != XSModelGroup.COMPOSITOR_ALL)
        {
            throw new ConfigServiceException(INV_ATTR_LIST /* + mapDeclaration.getName() */ + ": element declaration's type does not contain sequence model group: the items in an attribute map type must be declared within an all model group.");
        }

        _buildAttributeMapDescription(mapDescription, modelGroup, true);
    }

    private static void buildUnenumAttributeMapDescription(AttributeDescriptionMapImpl mapDescription,
                                                           XSModelGroup modelGroup)
        throws ConfigServiceException
    {
        // Ensure Choice Group Type
        if (modelGroup.getCompositor() != XSModelGroup.COMPOSITOR_CHOICE)
        {
            throw new ConfigServiceException(INV_ATTR_LIST /* + mapDeclaration.getName() */ + ": element declaration's type does not contain sequence model group: the items in an attribute map type must be declared within an all model group.");
        }

        _buildAttributeMapDescription(mapDescription, modelGroup, false);
    }

    private static void _buildAttributeMapDescription(AttributeDescriptionMapImpl mapDescription,
                                                      XSModelGroup modelGroup,
                                                      boolean enumeratedMap)
        throws ConfigServiceException
    {
        XSObjectList particles = modelGroup.getParticles();
        for(int i = 0; i < particles.getLength(); i++)
        {
            XSParticle particle = (XSParticle) particles.item(i);

            /*  Enforce special content constraint on attribute maps;
                Attribute maps can only contain a unique set of elements;
                Repeated attributes are not supported in attribute maps:
                -   Element particle can not occur more than once. */
            if (enumeratedMap && particle.getMaxOccurs() > 1)
            {
                throw new ConfigServiceException(INV_ATTR_MAP /* + complexTypeDecl.fName */ + INV_ATTR_MAP_RPT_ELMT);
            }

            if (!(particle.getTerm() instanceof XSElementDeclaration))
            {
                throw new ConfigServiceException(INV_ATTR_LIST /* + mapDeclaration.getName() */ + ": model group contains non-element particle: the items in an attribute map type must be declared as elements directly within an all model group.");
            }

            XSElementDeclaration mapItemDeclaration = (XSElementDeclaration) particle.getTerm();
            AttributeDescriptionImpl mapItemDescription = new AttributeDescriptionImpl(s_configServer);
            buildAttributeDescription(mapItemDescription, mapItemDeclaration);
            mapItemDescription.setProperty(IAttributeDescription.MIN_OCCURS_PROPERTY, new Long(particle.getMinOccurs()));
            mapItemDescription.setProperty(IAttributeDescription.MAX_OCCURS_PROPERTY, new Long(particle.getMaxOccurs()));

            ((AttributeDescriptionMapImpl)mapDescription).setAttributeDescription(mapItemDeclaration.getName(), mapItemDescription);
        }
    }

    private static void buildAttributeDescription(AttributeDescriptionImpl attributeDescription,
                                                  XSElementDeclaration attributeDeclaration)
        throws ConfigServiceException
    {
        if (attributeDeclaration.getTypeDefinition().getTypeCategory() == XSTypeDefinition.COMPLEX_TYPE)
        {   /* COMPLEX_ TYPE*/

            /* Determine the element's content type */
            XSComplexTypeDefinition type = (XSComplexTypeDefinition) attributeDeclaration.getTypeDefinition();
            switch (type.getContentType())
            {
                case XSComplexTypeDefinition.CONTENTTYPE_SIMPLE:
                {   /* Describes an attribute element that may have attributes
                    and contains text representing a simple value: Build an Atomic
                    Attribute Description. */
                    buildAtomicAttributeDescription(attributeDescription, attributeDeclaration);
                    break;
                }
                case XSComplexTypeDefinition.CONTENTTYPE_ELEMENT:
                {   /* Describes an attribute element that may have attributes
                    and contains other elements and no text: Build either an Attribute
                    Map or List Description. */
                    buildCompositeAttributeDescription(attributeDescription, attributeDeclaration);
                    break;
                }
                case XSComplexTypeDefinition.CONTENTTYPE_EMPTY:
                {
                    /* Describes an an element that may have attributes and
                    not contents: Build an Empty Attribute Description. */
                    buildEmptyAttributeDescription(attributeDescription, attributeDeclaration);
                    break;
                }
                case XSComplexTypeDefinition.CONTENTTYPE_MIXED:
                    /* CONTENTTYPE_MIXED: Describes an element that may have attributes and
                    can contains both child elements and text: config bean elements can
                    not contain both text and child elements. */
                    throw new ConfigServiceException(INV_ATTR + attributeDeclaration.getName() + INV_ATTR_MIXED_CONTENT);
                default :
                    throw new ConfigServiceException(INV_ATTR + attributeDeclaration.getName() + INV_ATTR_UNK_CONTENT);
            }
        }
        else
        {   /* SIMPLE_TYPE: Describes an element that has no attributes and contains
            text representing a simple value: Build an Atomic Attribute Description. */
            XSTypeDefinition xsType = attributeDeclaration.getTypeDefinition();
            XSSimpleTypeDefinition type = (XSSimpleTypeDefinition) attributeDeclaration.getTypeDefinition();
            buildAtomicAttributeDescription(attributeDescription, type);
        }

        addElementProperties(attributeDescription, attributeDeclaration);
    }

    private static void addElementProperties(IAttributeDescription attributeDescription, XSElementDeclaration attributeDeclaration)
        throws ConfigServiceException
    {
        ((AttributeDescriptionImpl)attributeDescription).setProperty(IAttributeDescription.NILLABLE_PROPERTY, new Boolean(attributeDeclaration.getNillable()));
        if (attributeDeclaration.getConstraintType() == XSConstants.VC_FIXED)
        {
            ((AttributeDescriptionImpl)attributeDescription).setProperty(IAttributeDescription.FIXED_PROPERTY, convertValue(((XSElementDecl)attributeDeclaration).fDefault.actualValue, attributeDeclaration.getTypeDefinition()));
        }
        else if (attributeDeclaration.getConstraintType() == XSConstants.VC_DEFAULT)
        {
            Object obj = convertValue(((XSElementDecl)attributeDeclaration).fDefault.actualValue, attributeDeclaration.getTypeDefinition());
            ((AttributeDescriptionImpl)attributeDescription).setProperty(IAttributeDescription.DEFAULT_PROPERTY, obj);
        }
        ((AttributeDescriptionImpl)attributeDescription).setProperty(IAttributeDescription.SCHEMA_TYPE_NAME, attributeDeclaration.getTypeDefinition().getName());
        ((AttributeDescriptionImpl)attributeDescription).setProperty(IAttributeDescription.SCHEMA_NAMESPACE, attributeDeclaration.getTypeDefinition().getNamespace());
    }

    private static void addUserDefinedAttributeProperties(AttributeDescriptionImpl attributeDescription, XSElementDeclaration attributeDeclaration)
        throws ConfigServiceException
    {
        XSComplexTypeDefinition type = (XSComplexTypeDefinition) attributeDeclaration.getTypeDefinition();
        XSObjectList attrUses = type.getAttributeUses();
        for (int i = 0; i < attrUses.getLength(); i++)
        {
            XSAttributeUse attrUse = (XSAttributeUse) attrUses.item(i);
            XSAttributeDeclaration attrDecl = attrUse.getAttrDeclaration();

            Object value = null;
            if (attrUse.getConstraintType() == XSConstants.VC_FIXED)
            {   /* Note: this intentionally overrides fixed setting of above. */
                value = convertValue(((XSAttributeUseImpl)attrUse).fDefault.actualValue, attrDecl.getTypeDefinition());
            }
            else
            if (attrDecl.getConstraintType() == XSConstants.VC_FIXED)
            {
                value = convertValue(((XSAttributeDecl)attrDecl).getValInfo().actualValue, attrDecl.getTypeDefinition());
            }
            else {
                continue;  // Attribute w/out fixed value: ignore
            }

            if (attrDecl.getName().equals(DESCRIPTION_ATTRIBUTE))
            {
                if (!(value instanceof String))
                {
                    throw new ConfigServiceException(INV_ATTR + attributeDeclaration.getName() + INV_ATTR_DESCRIPTION_NOT_STRING);
                }

                attributeDescription.setDescription((String) value);
            }
            else
            {
                attributeDescription.setProperty(attrDecl.getName(), value);
            }
        }
    }

    private static void buildEmptyAttributeDescription(AttributeDescriptionImpl attributeDescription,
                                                       XSElementDeclaration attributeDeclaration)
        throws ConfigServiceException
    {

        if (isAttributeMapDecl(attributeDeclaration))
        {
            attributeDescription.setType(IAttributeMap.class);
        }
        else
        {
            attributeDescription.setType(IAttributeList.class);
        }

        // Add xml attributes to attribute description
        addUserDefinedAttributeProperties(attributeDescription, attributeDeclaration);
    }

    private static void buildAtomicAttributeDescription(AttributeDescriptionImpl attributeDescription,
                                                        XSElementDeclaration attributeDeclaration)
        throws ConfigServiceException
    {
        XSComplexTypeDefinition type = (XSComplexTypeDefinition) attributeDeclaration.getTypeDefinition();
        switch (type.getSimpleType().getVariety())
        {
            case XSSimpleType.VARIETY_ATOMIC:
            {
                buildAtomicAttributeDescription(attributeDescription, type.getSimpleType());
                break;
            }
            case XSSimpleType.VARIETY_LIST:
            {   /* List Simple Type: TODO: can we support this??? */
                throw new ConfigServiceException(INV_ATTR + type.getName() + INV_ATTR_LIST_TYPE);
            }
            case XSSimpleType.VARIETY_UNION:
            {   /* Union Simple Type: TODO: can we support this??? */
                throw new ConfigServiceException(INV_ATTR + type.getName() + INV_ATTR_UNION_TYPE);
            }
            case XSSimpleType.VARIETY_ABSENT:
            {   /* Wild Card Simple Type: TODO: can we support this??? */
                throw new ConfigServiceException(INV_ATTR + type.getName() + INV_ATTR_WILDCARD_TYPE);
            }
            default :
                throw new Error(INV_ATTR + type.getName() + INV_ATTR_UNK_TYPE);
        }

        /* Add xml attributes to attribute description. */
        addUserDefinedAttributeProperties(attributeDescription, attributeDeclaration);
    }

    private static void buildAtomicAttributeDescription(AttributeDescriptionImpl attributeDescription,
                                                        XSSimpleTypeDefinition attributeType)
        throws ConfigServiceException
    {

        attributeDescription.setType(getAtomicClassType(attributeType));
        attributeDescription.setProperty(IAttributeDescription.SIMPLE_TYPE_NAME, getSimpleTypeName(((XSSimpleType)attributeType).getPrimitiveKind()));
        if ((attributeType.getDefinedFacets() & XSSimpleType.FACET_WHITESPACE) != 0)
        {
            try
            {
                attributeDescription.setProperty(IAttributeDescription.WHITESPACE_FACET, new Long(((XSSimpleType)attributeType).getWhitespace()));
            }
            catch (DatatypeException e)
            {
                throw new ConfigTypeException(e.getKey(), e.getArgs(), e);
            }
        }
        if ((attributeType.getDefinedFacets() & XSSimpleType.FACET_LENGTH) != 0)
        {
            attributeDescription.setProperty(IAttributeDescription.LENGTH_FACET, new Long(attributeType.getLexicalFacetValue(XSSimpleTypeDefinition.FACET_LENGTH) == null ? "0" : attributeType.getLexicalFacetValue(XSSimpleTypeDefinition.FACET_LENGTH)));
        }
        if ((attributeType.getDefinedFacets() & XSSimpleType.FACET_MINLENGTH) != 0)
        {
            attributeDescription.setProperty(IAttributeDescription.MIN_LENGTH_FACET, new Long(attributeType.getLexicalFacetValue(XSSimpleTypeDefinition.FACET_MINLENGTH) == null ? "0" :attributeType.getLexicalFacetValue(XSSimpleTypeDefinition.FACET_MINLENGTH)));
        }
        if ((attributeType.getDefinedFacets() & XSSimpleType.FACET_MAXLENGTH) != 0)
        {
            attributeDescription.setProperty(IAttributeDescription.MAX_LENGTH_FACET, new Long(attributeType.getLexicalFacetValue(XSSimpleTypeDefinition.FACET_MAXLENGTH) == null ? "0" :attributeType.getLexicalFacetValue(XSSimpleTypeDefinition.FACET_MAXLENGTH)));
        }
        if ((attributeType.getDefinedFacets() & XSSimpleType.FACET_TOTALDIGITS) != 0)
        {
            attributeDescription.setProperty(IAttributeDescription.TOTAL_DIGITS_FACET, new Long(attributeType.getLexicalFacetValue(XSSimpleTypeDefinition.FACET_TOTALDIGITS) == null ? "0" : attributeType.getLexicalFacetValue(XSSimpleTypeDefinition.FACET_TOTALDIGITS)));
        }
        if ((attributeType.getDefinedFacets() & XSSimpleType.FACET_FRACTIONDIGITS) != 0)
        {
            attributeDescription.setProperty(IAttributeDescription.FRACTION_DIGITS_FACET, new Long(attributeType.getLexicalFacetValue(XSSimpleTypeDefinition.FACET_FRACTIONDIGITS) == null ? "0" : attributeType.getLexicalFacetValue(XSSimpleTypeDefinition.FACET_FRACTIONDIGITS)));
        }
        if ((attributeType.getDefinedFacets() & XSSimpleType.FACET_PATTERN) != 0)
        {
            StringList pattern = attributeType.getLexicalPattern();
            if (pattern.getLength() > 0)
            {
                attributeDescription.setProperty(IAttributeDescription.PATTERN_FACETS, buildPatternDescription(pattern));
            }
        }
        if ((attributeType.getDefinedFacets() & XSSimpleType.FACET_ENUMERATION) != 0)
        {
            attributeDescription.setProperty(IAttributeDescription.EMUMERATION_FACET, buildEnumerationDescription(attributeType.getLexicalEnumeration()));
        }
        if ((attributeType.getDefinedFacets() & XSSimpleType.FACET_MAXINCLUSIVE) != 0)
        {
            try
            {
                String content = attributeType.getLexicalFacetValue(XSSimpleTypeDefinition.FACET_MAXINCLUSIVE);
                Object obj = ((XSSimpleType)attributeType).validate(content, s_noFacetContext, null);
                obj = convertValue(obj, attributeType);
                attributeDescription.setProperty(IAttributeDescription.MAX_INCLUSIVE_FACET, obj);
            }
            catch (DatatypeException e)
            {
                throw new ConfigTypeException(e.getKey(), e.getArgs(), e);
            }
        }
        if ((attributeType.getDefinedFacets() & XSSimpleType.FACET_MAXEXCLUSIVE) != 0)
        {
            try
            {
                String content = attributeType.getLexicalFacetValue(XSSimpleTypeDefinition.FACET_MAXEXCLUSIVE);
                Object obj = ((XSSimpleType)attributeType).validate(content, s_noFacetContext, null);
                obj = convertValue(obj, attributeType);
                attributeDescription.setProperty(IAttributeDescription.MAX_EXCLUSIVE_FACET, obj);
            }
            catch (DatatypeException e)
            {
                throw new ConfigTypeException(e.getKey(), e.getArgs(), e);
            }
        }
        if ((attributeType.getDefinedFacets() & XSSimpleType.FACET_MINEXCLUSIVE) != 0)
        {
            try
            {
                String content = attributeType.getLexicalFacetValue(XSSimpleTypeDefinition.FACET_MINEXCLUSIVE);
                Object obj = ((XSSimpleType)attributeType).validate(content, s_noFacetContext, null);
                obj = convertValue(obj, attributeType);
                attributeDescription.setProperty(IAttributeDescription.MIN_EXCLUSIVE_FACET, obj);
            }
            catch (DatatypeException e)
            {
                throw new ConfigTypeException(e.getKey(), e.getArgs(), e);
            }
        }
        if ((attributeType.getDefinedFacets() & XSSimpleType.FACET_MININCLUSIVE) != 0)
        {
            try
            {
                String content = attributeType.getLexicalFacetValue(XSSimpleTypeDefinition.FACET_MININCLUSIVE);
                Object obj = ((XSSimpleType)attributeType).validate(content, s_noFacetContext, null);
                obj = convertValue(obj, attributeType);
                attributeDescription.setProperty(IAttributeDescription.MIN_INCLUSIVE_FACET, obj);
            }
            catch (DatatypeException e)
            {
                throw new ConfigTypeException(e.getKey(), e.getArgs(), e);
            }
        }
    }

    private static void buildAttributeListDescription(AttributeDescriptionMapImpl listDescription,
                                                      XSModelGroup modelGroup)
        throws ConfigServiceException
    {
        /* Ensure Sequence */
        if (modelGroup.getCompositor() != XSModelGroup.COMPOSITOR_SEQUENCE)
        {
            throw new ConfigServiceException(INV_ATTR_LIST /* + listDeclaration.getName() */ + ": element declaration's type does not contain sequence model group: the item in an attribute list must be declared within a sequence model group.");
        }

        /* Ensure Single Particle in Group */
        /*  Enforce special constraint on the content of an attribute list element;
            An attribute list can only contain a single element type that can
            repeat any number of times:
            -   Sequence particle must contain a single element
                particle which describes the element type of list
                item. */
        if (modelGroup.getParticles().getLength() != 1)
        {
            throw new ConfigServiceException(INV_ATTR_LIST /* + listDeclaration.getName() */ + ": element declaration's type does not contain single particle : an attribute list type must be declared with a single list item.");
        }
        XSParticle listItemParticle = (XSParticle) modelGroup.getParticles().item(0);

        /* Ensure Particle is Element Particle */
        if (!(listItemParticle.getTerm() instanceof XSElementDeclaration))
        {
            throw new ConfigServiceException(INV_ATTR_LIST /* + listDeclaration.getName() */ + ": element declaration's type does not contain element declaration : the type of list item in an attribute list must be declared using a single element declaration.");
        }
        XSElementDeclaration listItemDeclaration = (XSElementDeclaration) listItemParticle.getTerm();

        /*  Build attribute description for list item */
        AttributeDescriptionImpl listItemDescription = new AttributeDescriptionImpl(s_configServer);
        buildAttributeDescription(listItemDescription, listItemDeclaration);

        /*  Valid range of sizes for attribute list are specified
            by list item particle */
        listItemDescription.setProperty(IAttributeDescription.MIN_OCCURS_PROPERTY, new Long(listItemParticle.getMinOccurs()));
        listItemDescription.setProperty(IAttributeDescription.MAX_OCCURS_PROPERTY, new Long(listItemParticle.getMaxOccurs()));

        /*  Add list item description to list description */
        listDescription.setAttributeDescription(listItemDeclaration.getName(), listItemDescription);
    }

    private static void buildCompositeAttributeDescription(AttributeDescriptionImpl compositeAttributeDescription,
                                                           XSElementDeclaration compositeAttributeDeclaration)
        throws ConfigServiceException
    {
        /* Ensure Complex Type */
        if (compositeAttributeDeclaration.getTypeDefinition().getTypeCategory() == XSTypeDefinition.SIMPLE_TYPE)
        {
            throw new ConfigServiceException(INV_ATTR + compositeAttributeDeclaration.getName() + ": element declaration is a simple type: composite attribute type must be declared using a complex element type.");
        }
        XSComplexTypeDefinition type = (XSComplexTypeDefinition) compositeAttributeDeclaration.getTypeDefinition();

        /* Ensure Group Type */
        if (!(type.getParticle().getTerm() instanceof XSModelGroup))
        {
            throw new ConfigServiceException(INV_ATTR + compositeAttributeDeclaration.getName() + ": element declaration's type does not contain model group: the item in a composite attribute type must be declared within a model group.");
        }
        XSModelGroup modelGroup = (XSModelGroup) type.getParticle().getTerm();

        switch(modelGroup.getCompositor())
        {
            case XSModelGroup.COMPOSITOR_ALL:
            {   /*  Build Description of Attribute Map. */
                compositeAttributeDescription.setType(IAttributeMap.class);
                AttributeDescriptionMapImpl mapDescription = (AttributeDescriptionMapImpl) compositeAttributeDescription.getAttributeDescriptions();
                buildAttributeMapDescription(mapDescription, modelGroup);
                break;
            }
            case XSModelGroup.COMPOSITOR_CHOICE:
            {   /*  Build Description of Unenumerated Attribute Map. */
                compositeAttributeDescription.setType(IAttributeMap.class);
                AttributeDescriptionMapImpl mapDescription = (AttributeDescriptionMapImpl) compositeAttributeDescription.getAttributeDescriptions();
                buildUnenumAttributeMapDescription(mapDescription, modelGroup);
                compositeAttributeDescription.setProperty(IAttributeDescription.UNENUM_MAP_PROPERTY, new Boolean(true));
                break;
            }
            case XSModelGroup.COMPOSITOR_SEQUENCE:
            {
                compositeAttributeDescription.setType(IAttributeList.class);
                AttributeDescriptionMapImpl listDescription = (AttributeDescriptionMapImpl) compositeAttributeDescription.getAttributeDescriptions();
                buildAttributeListDescription(listDescription, modelGroup);
                break;
            }
            default:
            {
                throw new ConfigServiceException(INV_ATTR_LIST + compositeAttributeDeclaration.getName() + ": element declaration's type contains unknown model group type: the item in a composite attribute type must be declared within a model group.");
            }
        }

        /* Add xml attributes to attribute description. */
        addUserDefinedAttributeProperties(compositeAttributeDescription, compositeAttributeDeclaration);
    }

    private static AttributeListImpl buildEnumerationDescription(StringList enumeration)
        throws ConfigServiceException
    {
        AttributeListImpl enumerationDesc = new AttributeListImpl(s_configServer);
        for (int i = 0; i < enumeration.getLength(); i++)
        {
            enumerationDesc.addAttribute(enumeration.item(i));
        }
        return enumerationDesc;
    }

    private static AttributeListImpl buildPatternDescription(StringList pattern)
        throws ConfigServiceException
    {
        AttributeListImpl patternDesc = new AttributeListImpl(s_configServer);
        for (int i = 0; i < pattern.getLength(); i++)
        {
            RegularExpression regex = new RegularExpression(pattern.item(i));
            AttributeMapImpl regexDesc = buildRegexDescription(regex);
            patternDesc.addAttribute(regexDesc);
        }
        return patternDesc;
    }

    private static AttributeMapImpl buildRegexDescription(RegularExpression regex)
    throws ConfigServiceException
    {
        AttributeMapImpl regexDesc = null;
        regexDesc = new AttributeMapImpl(s_configServer);
        regexDesc.setAttribute(IAttributeDescription.PATTERN_REGEX, regex.getPattern());
        regexDesc.setAttribute(IAttributeDescription.PATTERN_OPTION, regex.getOptions());
        return regexDesc;
    }

    private static Class getAtomicClassType(XSSimpleTypeDefinition simpleType)
        throws ConfigServiceException
    {
        switch(((XSSimpleType)simpleType).getPrimitiveKind())
        {
            case XSSimpleType.PRIMITIVE_STRING:
            {
                return String.class;
            }
            case XSSimpleType.PRIMITIVE_BOOLEAN:
            {
                return Boolean.class;
            }
            case XSSimpleType.PRIMITIVE_DECIMAL:
            {
                if (isIntType(simpleType))
                {
                    return Integer.class;
                }
                else
                if (isLongType(simpleType))
                {
                    return Long.class;
                }
                else
                {
                    return BigDecimal.class;
                }
            }
            case XSSimpleType.PRIMITIVE_DATETIME:
            {
                return Date.class;
            }
            case XSSimpleType.PRIMITIVE_HEXBINARY :
            {
                return byte[].class;
            }
            case XSSimpleType.PRIMITIVE_ANYURI:
            {
                return IConfigElement.class;
            }
            case XSSimpleType.PRIMITIVE_DOUBLE:
            case XSSimpleType.PRIMITIVE_FLOAT:
            case XSSimpleType.PRIMITIVE_DURATION:
            case XSSimpleType.PRIMITIVE_TIME:
            case XSSimpleType.PRIMITIVE_DATE:
            case XSSimpleType.PRIMITIVE_GYEARMONTH:
            case XSSimpleType.PRIMITIVE_GYEAR:
            case XSSimpleType.PRIMITIVE_GMONTHDAY:
            case XSSimpleType.PRIMITIVE_GDAY:
            case XSSimpleType.PRIMITIVE_GMONTH:
            case XSSimpleType.PRIMITIVE_BASE64BINARY:
            case XSSimpleType.PRIMITIVE_QNAME:
            case XSSimpleType.PRIMITIVE_NOTATION:
                throw new ConfigServiceException(INV_ATTR + INV_ATTR_UNSUPPORTED_PRIM_TYPE);
            default :
                throw new Error(INV_ATTR + INV_ATTR_INVALID_PRIM_TYPE);
        }
    }

    private static String getSimpleTypeName(short typeCode)
        throws ConfigServiceException
    {
        switch (typeCode)
        {
            case XSSimpleType.PRIMITIVE_STRING    : return "string";
            case XSSimpleType.PRIMITIVE_BOOLEAN   : return "boolean";
            case XSSimpleType.PRIMITIVE_DECIMAL   : return "decimal";
            case XSSimpleType.PRIMITIVE_DOUBLE    : return "double";
            case XSSimpleType.PRIMITIVE_DATETIME  : return "dateTime";
            case XSSimpleType.PRIMITIVE_HEXBINARY : return "hexBinary";
            case XSSimpleType.PRIMITIVE_ANYURI    : return "anyURI";
            case XSSimpleType.PRIMITIVE_FLOAT:
            case XSSimpleType.PRIMITIVE_DURATION:
            case XSSimpleType.PRIMITIVE_TIME:
            case XSSimpleType.PRIMITIVE_DATE:
            case XSSimpleType.PRIMITIVE_GYEARMONTH:
            case XSSimpleType.PRIMITIVE_GYEAR:
            case XSSimpleType.PRIMITIVE_GMONTHDAY:
            case XSSimpleType.PRIMITIVE_GDAY:
            case XSSimpleType.PRIMITIVE_GMONTH:
            case XSSimpleType.PRIMITIVE_BASE64BINARY:
            case XSSimpleType.PRIMITIVE_QNAME:
            case XSSimpleType.PRIMITIVE_NOTATION:
                throw new ConfigServiceException(INV_ATTR + INV_ATTR_UNSUPPORTED_PRIM_TYPE);
            default :
                throw new Error(INV_ATTR + INV_ATTR_INVALID_PRIM_TYPE);
        }
    }

    /*  This method converts from the runtime value used by
     *  Xerces to represent a schema type to the runtime value
     *  used by the config layer.
     */
    private static Object convertValue(Object obj, XSTypeDefinition type)
    {
        if (isIntType(type))
        {
            return convert2Integer(obj);
        } 

        if (isLongType(type))
        {
            return convert2Long(obj);
        } 
            
        if (isReferenceType(type))
        {
            String converted = obj.toString();
            return new ConfigReference(converted, s_configServer);
        }

        if (isByteArrayType(type))
        {
            String converted = obj.toString();
            return converted.getBytes();
        }

        return obj;
    }

   
    private static Integer convert2Integer(Object value)
    {
        BigDecimal converted = new BigDecimal(value.toString());
        BigDecimal tmp = MAX_INT.min(converted);
        tmp = MIN_INT.max(tmp);
        return new Integer(tmp.intValue());
    }

    private static Long convert2Long(Object value)
    {
        BigDecimal converted = new BigDecimal(value.toString());
        BigDecimal tmp = MAX_LONG.min(converted);
        tmp = MIN_LONG.max(tmp);
        return new Long(tmp.longValue());
    }
    
    
    static boolean isIntType(XSTypeDefinition simpleType)
    {
        XSTypeDefinition type = simpleType;
        while (type != null)
        {
            if (type.getName() != null)
            {
                if (type.getName().equals("anyType"))
                {
                    break;
                }

                if (type.getName().equals("int") ||
                    type.getName().equals("unsignedInt"))
                {
                    return true;
                }
            }
            type = type.getBaseType();
        }
        return false;
    }

    static boolean isLongType(XSTypeDefinition simpleType)
    {
        XSTypeDefinition type = simpleType;
        while (type != null)
        {
            if (type.getName() != null)
            {
                if (type.getName().equals("anyType"))
                {
                    break;
                }

                if (type.getName().equals("integer"))
                {
                    return true;
                }
            }
            type = type.getBaseType();
        }
        return false;
    }

    static boolean isReferenceType(XSTypeDefinition simpleType)
    {
        XSTypeDefinition type = simpleType;
        while (type != null)
        {
            if (type.getName() != null)
            {
                if (type.getName().equals("anyType"))
                {
                    break;
                }

                if (type.getName().equals("anyURI"))
                {
                    return true;
                }
            }
            type = type.getBaseType();
        }
        return false;
    }

    static boolean isByteArrayType(XSTypeDefinition simpleType)
    {
        XSTypeDefinition type = simpleType;
        while (type != null)
        {
            if (type.getName() != null)
            {
                if (type.getName().equals("anyType"))
                {
                    break;
                }

                if (type.getName().equals("hexBinary"))
                {
                    return true;
                }
            }
            type = type.getBaseType();
        }
        return false;
    }

    static boolean isDecimalType(XSTypeDefinition simpleType)
    {
        XSTypeDefinition type = simpleType;
        while (type != null)
        {
            if (type.getName() != null)
            {
                if (type.getName().equals("anyType"))
                {
                    break;
                }

                if (type.getName().equals("decimal"))
                {
                    return true;
                }
            }
            type = type.getBaseType();
        }
        return false;
    }

    private ConfigTypeDocument()
    {
    }

    public static Collection parse(String namespaceURI,
                                   String schemaDocURL,
                                   ConfigServer configServer,
                                   String path)
        throws ConfigServiceException
    {
        
        if (namespaceURI == null)
        {
            throw new IllegalArgumentException("namespaceURI is null");
        }

        if (schemaDocURL == null)
        {
            throw new IllegalArgumentException("schemaDocURL is null");
        }

        if (configServer == null)
        {
            throw new IllegalArgumentException("configServer is null");
        }

        s_configServer = configServer;

        List configTypes = new ArrayList();

        try
        {
            ConfigSchema configSchema = parseSchema(namespaceURI, schemaDocURL);
            if (configSchema != null)
            {
                // Strip out non-ConfigBean defn's and then order (based on type heirarchy)
                Collection globalElemDecls = orderElements(stripNonBeanDecl(configSchema.getGlobalElemDecls()));
                Iterator i = (globalElemDecls != null) ? globalElemDecls.iterator() : Collections.EMPTY_LIST.iterator();

                while (i.hasNext())
                {
                    // Iterate through global element declarations looking for config bean declarations
                    XSElementDecl elemDecl = (XSElementDecl)i.next();

                    String version = configSchema.getElementVersion(elemDecl.fName);
                    String schemaLocationHint = configSchema.getElementDocumentLocation(elemDecl.fName);

                    // Un-versioned Element Declaration; can not build config type
                    if (version == null)
                    {
                        throw new ConfigServiceException("Element declaration '" + elemDecl.fName + "' does not have a version.");
                    }

                    if (configServer.getTypeCache().lookup(elemDecl.fName, version) != null)
                    {
                        continue;
                    }

                    configServer.debugPrintln("Importing type : " + elemDecl.getName() + " v" + version + ", location = " + schemaLocationHint);

                    ConfigTypeImpl configType = new ConfigTypeImpl(elemDecl.getName(), version, s_configServer);
                    buildConfigType(configType, elemDecl);

                    if (path != null)
                    {
                        buildAnnotations(configType, path);
                    }

                    // Add schema document location property
                    configType.setProperty(IConfigType.SCHEMA_LOCATION_HINT, schemaLocationHint);

                    configTypes.add(configType);
                }
            }
            
            return configTypes;
        }
        catch (XNIException e)
        {
            throw new ConfigServiceException("parse-failed", new Object[] { schemaDocURL }, e);
        }
    }

    /**
     * Takes a collection of element declarations and orderes them according
     * to the immediate substutution group affiliation. This results in an
     * ordered collection of elements such that the super affiliations (types)
     * come first in the list and the dependent sub-affiliations (sub-types)
     * are last.
     * 
     * @param map  Collection of unordered element declarations (XSElementDecl)
     * @return     Ordered collection of element declarations
     */
    private static Collection orderElements(Collection map)
    {
        List res = new ArrayList(map);
        Comparator c = new Comparator()
        {
            @Override
            public int compare(Object o1, Object o2)
            {
                XSElementDecl e1 = (XSElementDecl)o1;
                XSElementDecl e2 = (XSElementDecl)o2;
                
                if (e2.getSubstitutionGroupAffiliation().getName().equals(e1.getName()))
                {
                    return -1;
                }
                
                if (e1.getSubstitutionGroupAffiliation().getName().equals(e2.getName()))
                {
                    return +1;
                }

                return +1;
            }
            
            @Override
            public boolean equals(Object o)
            {
                return false;
            }
            
            @Override
            public int hashCode() {
                return super.hashCode();
            }
        };
        
        Collections.sort(res, c);
        
        return res;
    }
    
    /**
     * Strips out / removes any non-ConfigBean element declations.
     * 
     * @param map  Collection of element declarations (XSElementDecl)
     * @return     Collection of element declations minus those
     *             declarations that are not identified as ConfigBeans
     */
    private static Collection stripNonBeanDecl(XSNamedMap map)
    {
        List res = new ArrayList();
        
        for (int i = 0; i < map.getLength(); i++)
        {
            XSElementDecl elemDecl = (XSElementDecl)map.item(i);

            if (isConfigBeanDecl(elemDecl))
            {
                res.add(elemDecl);
            }
        }
        
        return res;
    }
    
    private static void buildAnnotations(ConfigTypeImpl configType, String path)
    {
        Properties data = null;

        String fileName = path + File.separator + configType.getName() + ".properties";
        try
        {
            FileInputStream in = new FileInputStream(new File(fileName));
            data = new Properties();
            data.load(in);
            in.close();
        }
        catch(Exception e)
        {
            System.err.println("WARNING: Can't find property file for " +
                               configType.getName() + " type in the specified path "  + path);
        }

        if (data == null)
        {
            return;
        }

        Enumeration names = data.propertyNames();

        System.out.println("Imported javadocs ...");

        while(names.hasMoreElements())
        {
            String key = (String)names.nextElement();
            boolean isTooltip = key.endsWith("tooltip");
            key = isTooltip ? key.substring(0, key.lastIndexOf(".")) : key;
            String value = data.getProperty(key);
            IConfigPath configPath = new ConfigPathImpl(key);
            AttributeDescriptionImpl desc = (AttributeDescriptionImpl)configType.getAttributeDescription(configPath);
            try
            {
                if (desc != null)
                {
                    if (isTooltip)
                    {
                        desc.setProperty(IAttributeDescription.TO0LTIP_ANNOTATION, value);
                    }
                    else
                    {
                        desc.setProperty(IAttributeDescription.SCHEMA_ANNOTATION, value);
                    }
                }
            }
            catch(ConfigServiceException e) {}
        }
        System.out.println("    type = " + configType.getName() + " version = " + configType.getVersion());
    }
}
