/*
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.mx.config.impl;

import java.io.InputStream;
import java.util.HashSet;
import java.util.Set;

import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.ConfigServiceRuntimeException;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigServer;
import com.sonicsw.mx.config.IConfigType;

import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.INextVersionToken;
import com.sonicsw.mf.common.config.MergeUtil;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;

public class ConfigTypeImpl
extends AttributeDescriptionImpl
implements IConfigType
{
    /*  Config Type Name.
        Name uniquely identifying this config type. */
    protected String m_name = "";

    /*  Config Type Version.
        The version of this config type. */
    protected String m_version = "";

    /*  Super Config Type.
        Reference to the base config type from which this config type derives */
    protected IConfigType m_super = null;

    protected InputStream m_inputStream = null;

    /*  Removed Flag.
        Indicates whether this config type has been removed from the config
        server. A removed config type can not be saved or removed again. */
    protected boolean m_isRemoved = false;

    /*  New Flag.
        Indicates whether this config element is new and not yet stored in DS. */
    protected boolean m_isNew = true;

    /*  Modified Flag.
        Indicates whether this config element has been modifed. */
    protected boolean m_isModified = false;

    
    protected static final String TYPE_NAME_PREFIX = "/mx/configTypes/";

    /*  Super Version Attribute Name.
        Name of the attribute the DS Set of the config type
        storing the config type's super version attribute. */
    protected static final String SUPER_VERSION_ATTRIBUTE = "superVersion";

    /*  Super Attribute Name.
        Name of the attribute the DS Set of the config type
        storing the config type's super attribute. */
    protected static final String SUPER_ATTRIBUTE = "super";

    /*  Identity Element Type String
        This is the string populated in to the 'type' field
        of the identity element of a directory service element
        used to store a config type. */
    protected static final String IDENTITY_ELEMENT_TYPE_STRING = "MAConfigBeanTypeType";

    protected static final String INITIAL_VALUES = "initialValues";
    protected String m_initialValues = null;
    
    
    protected ConfigTypeImpl(String name, String version, ConfigServer configServer)
    throws ConfigServiceException
    {
        super(configServer);
        try
        {
            m_name = name;
            m_version = version;

            /*  Add this config element to config element cache if valid
                name is set. */
            if (!m_name.equals(""))
            {
                m_configServer.getTypeCache().add(m_name, m_version, this);
            }
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("ct-init-failed", e);
        }
   }

    /*
     * Reconstruct a ConfigType out of a DirElement.
     */

    protected ConfigTypeImpl(IDirElement el, ConfigServer configServer)
    throws ConfigServiceException
    {
        try
        {
            init(el, configServer);

            /*  Add this config element to config element cache if valid
                name is set. */
            if (!m_name.equals(""))
            {
                m_configServer.getTypeCache().add(m_name, m_version, this);
            }
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("ct-init-failed", e);
        }
    }

    protected final void
    init(IDirElement dirElement, ConfigServer configServer)
    throws ConfigServiceException
    {
        /*  Reinitialize references to DS state. */
        super.init(dirElement.getAttributes(), configServer);

        m_isNew = false;
        m_isModified = false;

        /*  Reinitialize Name and Version. */
        String dsName = dirElement.getIdentity().getName();
        m_version = dirElement.getIdentity().getReleaseVersion();
        m_name = convertDSName2TypeName(dsName, m_version);

        m_initialValues = (String)dirElement.getAttributes().getAttribute(INITIAL_VALUES);
        
        /* Reinitialize reference to Super Config Type. */
        String superName;
        String superVersion;
        if ((superName=(String)dirElement.getAttributes().getAttribute(SUPER_ATTRIBUTE)) != null &&
            (superVersion=(String)dirElement.getAttributes().getAttribute(SUPER_VERSION_ATTRIBUTE)) != null)
        {
            m_super = configServer.loadConfigType(superName, superVersion);
        }
    }

    void refreshFromDSElement(IDirElement configTypeDSElement, ConfigServer configServer)
        throws ConfigServiceException
    {
         if (configTypeDSElement == null)
        {
            throw new ConfigServiceException("ct-refresh-from-dse-ce-is-null");
        }

        init(configTypeDSElement, configServer);
    }

    @Override
    public String
    toString()
    {
        StringBuffer buffer = new StringBuffer();

        String className = this.getClass().getName();
        String typeName  = className.substring(className.lastIndexOf(".") + 1);

        buffer.append(getName() + ":" + getVersion() + "  [" + typeName + "] =\n");
        buffer.append("{\n");
        buffer.append(this.toString(1));
        buffer.append("}\n");

        return buffer.toString();
    }

    @Override
    public Object
    clone()
    {
        try
        {
            /*  Deep copy attribute description. */
            ConfigTypeImpl copy = (ConfigTypeImpl) super.clone();

            /* Clones do not initially have streams attached */
            copy.m_inputStream = null;

            /*  Clones are initially unnamed. */
            copy.m_name = "";

            /* Clone is new */
            copy.m_isNew = true;

            return copy;
        }
        catch (Exception e)
        {
            throw new ConfigServiceRuntimeException("ct-clone-failed", e);
        }
    }

    public boolean
    isRemoved()
    {
        return m_isRemoved;
    }

    protected void
    setRemoved()
    throws ConfigServiceException
    {
        if (isRemoved())
        {
            return;
        }

        if (!m_name.equals(""))
        {
            m_configServer.getTypeCache().remove(m_name, getVersion());
        }

        m_isRemoved = true;
    }

    public boolean
    isModified()
    {
        return (m_isModified || m_isNew);
    }

    protected void
    setModified()
    {
        m_isModified = true;
    }

    public boolean
    isNew()
    {
        return m_isNew;
    }

    @Override
    public String
    getName()
    {
        return m_name;
    }

    @Override
    public String
    getVersion()
    {
        return m_version;
    }

    public String
    getDSName()
    {
        return ConfigTypeImpl.convertTypeName2DSName(m_name, getVersion());
    }

    public void
    refresh()
    throws ConfigServiceException
    {
        try
        {
            m_configServer.refreshType(this);
        }
        catch (ConfigServiceException e)
        {
            if (e.getErrorKey().equals("ct-refresh-from-dse-ce-is-null"))
            {   /*  This config type has been removed from config server:
                    remove it from the cache and mark it as being removed. */
                setRemoved();
            }
            else
            {
                throw e;
            }
        }
    }

    public void setName(String name) throws ConfigServiceException
    {
        try
        {
            /*  Add this config element to config element cache if valid
                name is set. */
            if (!isNullOrEmpty(name))
            {
                m_configServer.getTypeCache().add(name, m_version, this);
            }

            if (!isNullOrEmpty(m_name))
            {
                m_configServer.getTypeCache().remove(m_name, m_version);
            }

            m_name = name;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("ct-set-name-failed", e);
        }
    }

    public void setVersion(String version) throws ConfigServiceException
    {
        try
        {
            /*  Add this config element to config element cache if valid
                name is set. */
            if (!isNullOrEmpty(m_name))
            {
                m_configServer.getTypeCache().add(m_name, version, this);
                m_configServer.getTypeCache().remove(m_name, m_version);
            }

            m_version = version;
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("ct-set-version-failed", e);
        }
    }
    
    public static boolean isNullOrEmpty(String str)
    {
        return str == null || str.trim().isEmpty();
    }

    @Override
    public IConfigType getSuper()
    {
        return m_super;
    }

    protected void setSuper(IConfigType baseConfigType)
        throws ConfigServiceException
    {
        m_super = baseConfigType;
    }

    public void setInitialValues(String initialValues)
    {
        setInitialValues(initialValues, true);
    }
    
    void setInitialValues(String initialValues, boolean modify)
    {
        m_initialValues = initialValues;
        
        if (modify)
        {
            setModified();
        }
    }
    public boolean hasInitialValues()
    {
        return (m_initialValues != null);
    }

    public void applyInitialValuesTo(IConfigBean newBean)
        throws ConfigServiceException
    {
        InitialValuesImpl.apply(newBean, m_initialValues);
    }
    
    @Override
    public void toAttributeSet(IAttributeSet attrSet)
    {
        super.toAttributeSet(attrSet);
        
        if (m_super != null)
        {
            try
            {
                attrSet.setStringAttribute(SUPER_ATTRIBUTE, m_super.getName());
                attrSet.setStringAttribute(SUPER_VERSION_ATTRIBUTE, m_super.getVersion());
            }
            catch (Exception e)
            {
                throw new ConfigServiceRuntimeException("ct-to-attr-set-failed", e);
            }
        }
        
        if (m_initialValues != null)
        {
            try
            {
                attrSet.setStringAttribute(INITIAL_VALUES, m_initialValues);
            }
            catch (Exception e)
            {
                throw new ConfigServiceRuntimeException("ct-to-attr-set-failed", e);
            }
        }
    }

    public IBasicElement
    doneUpdate()
    throws ConfigServiceException
    {
        IBasicElement storeThis = null;
        if ((m_name == null) || (m_name.length() == 0))
        {
            throw new ConfigServiceException("ce-done-update-no-name");
        }

        if (isNew())
        {   /* Store new DS element */

            /* Create and Populate a DS element with config element's state. */
            IDirElement localDirElement = ElementFactory.createElement(convertTypeName2DSName(m_name, m_version),
                                                                       IDENTITY_ELEMENT_TYPE_STRING, m_version);
            toAttributeSet(localDirElement.getAttributes());

            /* Generate delta */
            try
            {
                storeThis = localDirElement.doneUpdate();
            }
            catch (Exception e)
            {
                throw new ConfigServiceException("ce-done-update-failed", e);
            }
        }
        else
        {   /* Update stored DS element */

            /* Load DS element from DS */
            IDirElement loadedDirElement = m_configServer.loadDirTypeElement(convertTypeName2DSName(m_name, m_version));

            /* Create and Populate a DS element with config element's state. */
            IDirElement localDirElement = (IDirElement) loadedDirElement.createWritableClone();
            toAttributeSet(localDirElement.getAttributes());

            /* Merge local state into loaded DS element and generate delta. */
            try
            {
                MergeUtil.mergeModifications(loadedDirElement, localDirElement);
                storeThis = loadedDirElement.doneUpdate();
            }
            catch (Exception e)
            {
                throw new ConfigServiceException("ce-done-update-failed", e);
            }

        }

        m_isModified = false;
        m_isNew = false;

        return storeThis;
    }

    // TODO: Eliminate
    protected IDirElement
    getNextVersion(INextVersionToken token)
    {
        return null; //return m_configTypeDSElement.getNextVersion(token);
    }

    public Set
    getSubElements()
    throws ConfigServiceException
    {
        return new HashSet();
    }

    public String getDSDirectory()
    {
        return TYPE_NAME_PREFIX +  getVersion() + "/";
    }

    static public String
    convertTypeName2DSName(String typeName, String version)
    {
        return TYPE_NAME_PREFIX + version + "/" + typeName;
    }

    static public String
    convertDSName2TypeName(String dsName, String version)
    {
        String prefix = TYPE_NAME_PREFIX + version + "/";
        return dsName.substring(prefix.length());
    }

    @Override
    public IConfigServer
    getConfigServer()
    {
        return m_configServer;
    }
}