/*
 * Copyright (c) 2003-2004 Sonic Software Corporation. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. ("Confidential Information"). You shall not disclose
 * such Confidential Information and shall use it only in accordance with the
 * terms of the license agreement you entered into with Sonic.
 * 
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
 * LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
 * DERIVATIVES.
 * 
 * CopyrightVersion 1.0
 */

package com.sonicsw.mx.config.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.StringReader;
import java.io.Writer;
import java.util.Iterator;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.xerces.dom.DeferredTextImpl;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import com.sonicsw.mx.config.ConfigFactory;
import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.IAttributeDescription;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigPath;
import com.sonicsw.mx.config.IConfigServer;
import com.sonicsw.mx.config.IConfigType;
import com.sonicsw.mx.config.IInitialValues;

public class InitialValuesImpl implements IInitialValues
{
    private static final int    TYPE_MAP              = 0;
    private static final int    TYPE_LIST             = 1;
    private static final int    TYPE_VALUE            = 2;

    private static final String DOCUMENT_ELEMENT_NAME = "initialValues";
    private static final String TYPE_ATTR_NAME        = "type";
    private static final String VERSION_ATTR_NAME     = "version";

    private File                m_file                = null;
    private Document            m_document            = null;
    private IConfigServer       m_server              = null;
    
    public InitialValuesImpl(String type, String version, IConfigServer server)
       throws IllegalArgumentException, ParserConfigurationException
    {
        if ((type == null) || (type.trim().length() == 0))
        {
            throw new IllegalArgumentException("Type is not defined");
        }
        
        if ((version == null) || (version.trim().length() == 0))
        {
            throw new IllegalArgumentException("Version is not defined");
        }

        m_server = server;
        
        try
        {
            confirmType(type, version);
        }
        catch (Exception e)
        {
            throw new IllegalArgumentException(e.getMessage());
        }
        
        DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
        m_document = builder.getDOMImplementation().createDocument("www.sonicsw.com", DOCUMENT_ELEMENT_NAME, null);

        m_document.getDocumentElement().setAttribute(TYPE_ATTR_NAME,    type);
        m_document.getDocumentElement().setAttribute(VERSION_ATTR_NAME, version);
    }

    public InitialValuesImpl(File file, IConfigServer server) throws Exception
    {
        m_file = file;
        m_document = getDocument(new String(readFile(file.toString())));
        m_server = server;
        
        if (getType() == null)
        {
            throw new Exception("Type information missing from '" + file + "'");
        }

        if (getVersion() == null)
        {
            throw new Exception("Version information missing from '" + file + "'");
        }
        
        confirmType(getType(), getVersion());
    }

    //------------------------------------------------------------------------
    //
    // IInitialValues implementation
    //
    //------------------------------------------------------------------------
    
    @Override
    public final String getType()
    {
        String value = m_document.getDocumentElement().getAttribute(TYPE_ATTR_NAME);

        return ((value != null) && (value.trim().length() > 0)) ? value.trim() : null;
    }

    @Override
    public final String getVersion()
    {
        String value = m_document.getDocumentElement().getAttribute(VERSION_ATTR_NAME);

        return ((value != null) && (value.trim().length() > 0)) ? value.trim() : null;
    }

    @Override
    public boolean isNew()
    {
        return (m_file == null);
    }

    public Iterator pathIterator()
    {
        return null;
    }

    @Override
    public Object get(IConfigPath attrPath)
    {
        Element node = m_document.getDocumentElement();
        
        for (int i = 0; i < attrPath.size(); i++)
        {
            String attrName = attrPath.getComponent(i);
            
            node = getChildElement(node, attrName);

            if (node == null)
            {
                return null;
            }
        }
        
        return getValueFromNode(node);
    }

    @Override
    public void set(IConfigPath attrPath, Object value)
    {
        Element node = m_document.getDocumentElement();
        
        for (int i = 0; i < attrPath.size(); i++)
        {
            String attrName = attrPath.getComponent(i);
            
            Element childNode = getChildElement(node, attrName);
            
            if (childNode == null)
            {
                childNode = m_document.createElement(attrName);
                node.appendChild(childNode);
            }
            
            node = childNode;
        }

        Node valueNode = m_document.createTextNode(attrPath.getLastComponent());
        valueNode.setNodeValue(value.toString());
        node.appendChild(valueNode);
    }

    @Override
    public Object remove(IConfigPath attrPath)
    {
        Element node = m_document.getDocumentElement();
        
        for (int i = 0; i < attrPath.size(); i++)
        {
            String attrName = attrPath.getComponent(i);
            
            node = getChildElement(node, attrName);

            if (node == null)
            {
                return null;
            }
        }
        
        Object res = getValueFromNode(node);
        Node parent = node.getParentNode();
        
        if (parent != null)
        {
            parent.removeChild(node);

            NodeList parentChildren = parent.getChildNodes();
            
            for (int j = parentChildren.getLength()-1; j >= 0; j--)
            {
                Node aChild = (Node)parentChildren.item(j);
                
                // Flush out any Deferred Text nodes...
                if (aChild instanceof DeferredTextImpl)
                {
                    parent.removeChild(aChild);
                }
            }
            
            if (!parent.hasChildNodes() && (parent.getParentNode() != null))
            {
                parent.getParentNode().removeChild(parent);
            }
        }
        
        return res;
    }

    @Override
    public void validate() throws ConfigServiceException
    {
    }

    @Override
    public void save() throws Exception
    {
        if (isNew())
        {
            throw new Exception("File is new - use saveAs to specify file name");
        }
        
        saveAs(m_file);
    }
    
    @Override
    public void saveAs(File file) throws Exception
    {
        m_file = file;
        
        writeFile(m_file, toString());
    }

    //------------------------------------------------------------------------

    @Override
    public String toString()
    {
        StringBuffer sb = new StringBuffer();

        sb.append("<?xml version=\"1.0\" ?>").append("\r\n");

        toString(sb, 0, m_document.getDocumentElement());

        return sb.toString();
    }

    //------------------------------------------------------------------------
    //
    // Private Methods
    //
    //------------------------------------------------------------------------
    
    private void confirmType(String type, String version)
        throws Exception
    {
        try
        {
            m_server.loadConfigType(type, version);
        }
        catch (ConfigServiceException e)
        {
            throw new Exception("ConfigType '" + type + "' v" + version + " is not in the DS");
        }
    }

    private Element getChildElement(Element parent, String name)
    {
        NodeList child = parent.getChildNodes();
        
        for (int i = 0; i < child.getLength(); i++)
        {
            Node aChild = child.item(i);
            
            if (aChild.getNodeName().equals(name))
            {
                return (Element)aChild;
            }
        }
        
        return null;
    }

    private void toString(StringBuffer sb, int indent, Element parent)
    {
        sb.append(pad(indent)).append("<").append(parent.getNodeName());

        Object value = getValueFromNode(parent);
        
        if (value != null)
        {
            sb.append(">").append(value).append("</").append(parent.getNodeName()).append(">").append("\r\n");
        }
        else
        {
            NamedNodeMap attr = parent.getAttributes();

            if ((attr != null) && (attr.getLength() > 0))
            {
                for (int i = 0; i < attr.getLength(); i++)
                {
                    Attr aAttr = (Attr) attr.item(i);
                    sb.append(" ");
                    sb.append(aAttr.getName()).append("=").append('"').append(aAttr.getValue()).append('"');
                }
            }
        
            sb.append(">").append("\r\n");
        

            NodeList children = ((Element)parent).getChildNodes();
            
            for (int i = 0; i < children.getLength(); i++)
            {
                if (children.item(i) instanceof Element)
                {
                    toString(sb, indent+1, (Element)children.item(i));
                }
            }
            
            sb.append(pad(indent)).append("</").append(parent.getNodeName()).append(">").append("\r\n");
        }
    }
    
    private String pad(int indent)
    {
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < indent; i++)
        {
            sb.append("    ");
        }
        return sb.toString();
    }
    
    protected static Element getDocumentElement(String content) throws Exception
    {
        Document document = getDocument(content);

        return document.getDocumentElement();
    }

    protected static byte[] readFile(String filename) throws Exception
    {
        File file = new File(filename);
        FileInputStream fis = null;

        try
        {
            if (!file.exists())
            {
                throw new Exception("File '" + file + "' does not exist");
            }

            fis = new FileInputStream(file);

            byte[] buf = new byte[(fis.available())];

            fis.read(buf);

            return buf;
        }
        finally
        {
            if (fis != null)
            {
                fis.close();
            }
        }
    }

    protected static void writeFile(File file, String content) throws Exception
    {
        // create any intermediate directories to get to the file
        File parent = file.getParentFile();

        parent.mkdirs();

        Writer writer = null;

        try
        {
            writer = new FileWriter(file);

            writer.write(content);
            writer.flush();
        }
        finally
        {
            if (writer != null)
            {
                writer.close();
            }
        }
    }

    public static final IConfigType attach(IConfigServer server, String url) throws Exception
    {
        String content = new String(readFile(url));
        Element element = getDocumentElement(content);
        String type = element.getAttribute("type");
        String version = element.getAttribute("version");

        if (type == null)
        {
            throw new Exception("Type information missing from '" + url + "'");
        }

        if (version == null)
        {
            throw new Exception("Version information missing from '" + url + "'");
        }

        IConfigType configType = server.loadConfigType(type, version);

        if (configType == null)
        {
            throw new Exception("Type '" + type + "' v" + version + " not found in DS");
        }

        ((ConfigTypeImpl)configType).setInitialValues(content);

        return configType;
    }
    
    public static final IConfigType attach(IConfigType configType, String url) throws Exception
    {
        String content = new String(readFile(url));
        Element element = getDocumentElement(content);
        String type = element.getAttribute("type");
        String version = element.getAttribute("version");

        if (type == null)
        {
            throw new Exception("Type information missing from '" + url + "'");
        }

        if (version == null)
        {
            throw new Exception("Version information missing from '" + url + "'");
        }

        if (configType == null)
        {
            throw new Exception("Type '" + type + "' v" + version + " not found in DS");
        }

        ((ConfigTypeImpl)configType).setInitialValues(content);

        return configType;
    }

    public static final void apply(IConfigBean newBean, String content) throws ConfigServiceException
    {
        try
        {
            Element root = getDocumentElement(content);
            NodeList child = root.getChildNodes();

            for (int i = 0; i < child.getLength(); i++)
            {
                if (child.item(i) instanceof Element)
                {
                    _apply(newBean, (Element) child.item(i));
                }
            }
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("apply", e);
        }
    }

    private static void _apply(IAttributeMap parentMap, Element element) throws ConfigServiceException
    {
        try
        {
            int attrType = getAttributeTypeFromNode(element);
            String name = element.getNodeName();

            switch (attrType)
            {
                case TYPE_MAP:
                {
                    IAttributeMap childMap = (IAttributeMap) parentMap.getAttribute(name);

                    if (childMap == null)
                    {
                        parentMap.setAttribute(name, parentMap.createAttributeMap(name));

                        // Have to refresh because of internal cloning issue
                        childMap = (IAttributeMap) parentMap.getAttribute(name);
                    }

                    // Recusively deal with any child maps / lists / values
                    NodeList children = element.getChildNodes();
                    for (int i = 0; i < children.getLength(); i++)
                    {
                        if (children.item(i) instanceof Element)
                        {
                            _apply(childMap, (Element) children.item(i));
                        }
                    }
                    break;
                }
                case TYPE_LIST:
                {
                    throw new Exception("AttributeList not supported!");
                }
                case TYPE_VALUE:
                {
                    IAttributeDescription ad = parentMap.getAttributeDescription().getAttributeDescription(name);

                    Object value = ad.validate(getValueFromNode(element));

                    if (value instanceof ConfigReference)
                    {
                        System.out.println("Config Reference Attribute " + ((ConfigReference) value).getElementName());
                    }

                    parentMap.setAttribute(name, value);
                    break;
                }
            }
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("test", e);
        }
    }

    private static int getAttributeTypeFromNode(Element element) throws Exception
    {
        // Check first to see if the xml file explicitly states that a node is a
        // given type
        String typeAttr = element.getAttribute("type");
        if ((typeAttr != null) && (typeAttr.trim().length() > 0))
        {
            if (typeAttr.equals("map"))
            {
                return TYPE_MAP;
            }

            if (typeAttr.equals("list"))
            {
                return TYPE_LIST;
            }

            throw new Exception("Not a supported type '" + typeAttr + "'");
        }

        NodeList children = element.getChildNodes();

        // No children (this means no text value either) - we will assume that
        // it IS an AttributeMap
        if ((children == null) || (children.getLength() == 0))
        {
            return TYPE_MAP;
        }

        // One child of type Text means that its an Attribute Value
        if ((children.getLength() == 1) && (children.item(0) instanceof Text))
        {
            return TYPE_VALUE;
        }

        // Otherwise we have an AttributeMap
        return TYPE_MAP;
    }

    private static String getValueFromNode(Element element)
    {
        NodeList list = element.getChildNodes();

        if ((list == null) || (list.getLength() != 1) || !(list.item(0) instanceof Text))
        {
            return null;
        }

        return ((Text)list.item(0)).getNodeValue().trim();
    }

    //------------------------------------------------------------------------
    //
    // DOM Utilities
    //
    //------------------------------------------------------------------------

    private static DocumentBuilderFactory getFactory()
    {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        
        factory.setValidating(false);
        factory.setNamespaceAware(true);
        factory.setAttribute("http://xml.org/sax/features/validation", Boolean.FALSE);

        return factory;
    }
    
    public static Document getDocument(String content)
        throws Exception
    {
        DocumentBuilder builder = getFactory().newDocumentBuilder();
        
        /**
         * DOMParserErrorHandlerImpl implements ErrorHandler to report SAXParseError during validation.
         * Throw an exception for different types of errors(fatal and recoverable) and warnings.
         */
        builder.setErrorHandler(new ErrorHandler()
        {
            @Override
            public void warning(SAXParseException e) throws SAXException
            {
                throw e;
            }

            @Override
            public void error(SAXParseException e) throws SAXException
            {
                throw e;
            }

            @Override
            public void fatalError(SAXParseException e) throws SAXException
            {
                throw e;
            }
        });
        
        return builder.parse(new InputSource(new StringReader(content)));
    }
    
    //------------------------------------------------------------------------
    
    private static final IConfigPath TEST1_ATTR = ConfigFactory.createConfigPath("MONITORING.STATUS_POLL_INTERVAL");
    private static final IConfigPath TEST2_ATTR = ConfigFactory.createConfigPath("COMPONENTS.entry0.CONFIG_REF");
    private static final IConfigPath TEST3_ATTR = ConfigFactory.createConfigPath("DOMAIN_NAME");
    private static final IConfigPath TEST4_ATTR = ConfigFactory.createConfigPath("COMPONENTS.entry0.DEPLOYMENT_PARAMETERS.aUniqueKey.value");

    public static void main(String[] arg)
    {
        ConfigServerUtility csu = null;
        
        try
        {
            csu = new ConfigServerUtility();
            csu.connect("Domain1", "localhost", "Administrator", "Administrator", true);
            
//            IInitialValues iv = new InitialValuesImpl("MF_CONTAINER", "100", csu.getConfigServer());
            IInitialValues iv = new InitialValuesImpl(new File("C:\\SonicMQ62\\MQ6.2\\schema\\mf\\Container.iv"), csu.getConfigServer());
            System.out.println(iv.toString());
            
            System.out.println(TEST1_ATTR + " = " + iv.get(TEST1_ATTR));
            System.out.println(TEST2_ATTR + " = " + iv.get(TEST2_ATTR));
            iv.remove(TEST1_ATTR);
            iv.remove(TEST3_ATTR);

            iv.set(TEST1_ATTR, "23232");
            iv.set(TEST4_ATTR, "this is a test");
            System.out.println(TEST4_ATTR + " = " + iv.get(TEST4_ATTR));
            System.out.println(iv);
          
//            iv.save();
            iv.saveAs(new File("C:\\SonicMQ62\\MQ6.2\\schema\\mf\\Test.iv"));
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        finally
        {
            if (csu != null)
            {
                try { csu.disconnect(); } catch (ConfigServiceException e) { e.printStackTrace(); }
            }
        }
    }
}