package com.sonicsw.mx.config.impl;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.StringTokenizer;

import com.sonicsw.mx.config.ConfigFactory;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigElement;
import com.sonicsw.mx.config.IConfigPath;
import com.sonicsw.mx.config.IConfigServer;

import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.impl.DSTransaction;
import com.sonicsw.mf.common.config.impl.ElementIdentity;
import com.sonicsw.mf.common.config.query.BooleanExpression;
import com.sonicsw.mf.common.config.query.EqualExpression;
import com.sonicsw.mf.common.config.query.From;
import com.sonicsw.mf.common.config.query.FromElementType;
import com.sonicsw.mf.common.config.query.FromFolder;
import com.sonicsw.mf.common.config.query.NotEqualExpression;
import com.sonicsw.mf.common.config.query.Query;
import com.sonicsw.mf.common.config.query.Where;
import com.sonicsw.mf.common.dirconfig.IDeltaDirElement;
import com.sonicsw.mf.common.dirconfig.IDirElement;

public class TxnConfigServer extends ConfigServer
{
    protected List m_localElements      = new ArrayList();
    protected List m_newElements        = new ArrayList();
    protected List m_unmodifiedElements = new ArrayList();
    protected List m_modifiedElements   = new ArrayList();
    protected List m_removedElements    = new ArrayList();

    protected DSTransaction m_txn = null;


    public TxnConfigServer(IDirectoryFileSystemService ds)
        throws ConfigServiceException
    {
        super(ds, true, false);
        m_txn = (DSTransaction)m_ds.createTransaction();
    }

    @Override
    public void close()
        throws ConfigServiceException
    {
        super.close();

        if (m_localElements != null)
        {
            m_localElements.clear();
            m_localElements = null;
        }

        if (m_newElements != null)
        {
            m_newElements.clear();
            m_newElements = null;
        }

        if (m_unmodifiedElements != null)
        {
            m_unmodifiedElements.clear();
            m_unmodifiedElements = null;
        }

        if (m_modifiedElements != null)
        {
            m_modifiedElements.clear();
            m_modifiedElements = null;
        }

        if (m_removedElements != null)
        {
            m_removedElements.clear();
            m_removedElements = null;
        }

        if (m_txn != null)
        {
            if (m_txn.getActions() != null)
            {
                m_txn.getActions().clear();
            }

            m_txn = null;
        }
    }

    @Override
    public boolean isTransacted()
    {
        return true;
    }

    @Override
    public boolean isWriteable()
    {
        return true;
    }

    //-------------------------------------------------------------------------
    //
    // Query implementation
    //
    // Overridden default DS-side behavior to include querying of locally
    // cached elements...this is necessary because otherwise the query will
    // miss any un-committed operations buffered in the transaction.
    //
    // Supports:
    //          From >
    //                  FromFolder
    //                  FromElementType
    //
    //          Where >
    //                  EqualExpression
    //                  NotEqualExpression
    //
    //-------------------------------------------------------------------------

    private Set loadLocalConfigElements(Query query)
        throws ConfigServiceException
    {
        Set cache = m_elementCache.getObjects();
        Set res   = new HashSet();


        From from = query.getFrom();
        Where where = query.getWhere();

        Set fromSet = new HashSet();

        // Build up set of cached elements that match the From clause...
        //
        if (from instanceof FromFolder)
        {
            String fromFolder = ((FromFolder)from).getFolderName();

            if (!fromFolder.endsWith("/"))
            {
                fromFolder = fromFolder + IMFDirectories.MF_DIR_SEPARATOR;
            }

            Iterator iCache = cache.iterator();

            while (iCache.hasNext())
            {
                IConfigElement eCache = (IConfigElement)iCache.next();

                if (eCache.getName().startsWith(fromFolder))
                {
                    String subString = eCache.getName().substring(fromFolder.length());

                    // The cache element is a match...it is in the query's
                    // from folder
                    //
                    if (subString.indexOf(IMFDirectories.MF_DIR_SEPARATOR) == -1)
                    {
                        fromSet.add(eCache);
                    }
                }
            }
        }
        else
        if (from instanceof FromElementType)
        {
            String   type = ((FromElementType)from).getType();
            Iterator iCache = cache.iterator();

            while (iCache.hasNext())
            {
                IConfigElement eCache = (IConfigElement)iCache.next();

                if ((eCache instanceof IConfigBean) &&
                    ((IConfigBean)eCache).getConfigType().getName().equals(type))
                {
                    fromSet.add(eCache);
                }
            }
        }
        else
        {
            throw new ConfigServiceException("TxnConfigServer doesn't support the '" +
                                     from.getClass().getName() + "' query construct");
        }

        // Now restrict that set by applying the Where clause...
        //
        Iterator iFrom = fromSet.iterator();

        while (iFrom.hasNext())
        {
            IConfigElement eCache = (IConfigElement)iFrom.next();

            if (isLocalWhere(eCache, where))
            {
                debugPrintln("Local Query Matched = " + eCache.getName());
                res.add(eCache);
            }
        }

        return res;
    }

    /**
     * Method applies the 'Where' query operation to the supplied element
     * to determine if the element matches.
     *
     * @param  element  The local Config Layer element to be checked.
     * @param  where    The where clause to apply to the element.
     * @return          Returns true if the element matches the expression(s),
     *                  otherwise false.
     */
    private boolean isLocalWhere(IConfigElement element, Where where)
        throws ConfigServiceException
    {
        boolean res = false;

        if (where != null)
        {
            res = true;

            BooleanExpression[] exp = where.getAndChain();

            for (int i = 0; i < exp.length; i++)
            {
                if (exp[i] instanceof EqualExpression)
                {
                    if (!evaluateEqualExpression((EqualExpression)exp[i], element, false))
                    {
                        res = false;
                        break;
                    }
                }
                else
                if (exp[i] instanceof NotEqualExpression)
                {
                    if (!evaluateEqualExpression((EqualExpression)exp[i], element, true))
                    {
                        res = false;
                        break;
                    }
                }
                else
                {
                    throw new ConfigServiceException("TxnConfigServer doesn't support the '" +
                                            where.getClass().getName() + "' query construct");
                }
            }
        }

        return res;
    }

    protected boolean evaluateEqualExpression(EqualExpression expression,
                                              IConfigElement  element,
                                              boolean         negate)
    {
        IConfigPath path   = ConfigFactory.createConfigPath(expression.getFirstOperand().toString());
        Object      wValue = expression.getSecondOperand();
        Object      oValue = element.getAttribute(path);

        if ((oValue == null) ||                          // Couldn't find attribute
            (oValue.getClass() != wValue.getClass()) ||  // Values aren't the same type Don't support array comparisons!!!
            oValue.getClass().isArray())
        {
            return false;
        }

        debugPrintln("Local Query Compare '" + element.getName() + "', path=" + path + ", " + wValue + " = " + oValue);

        boolean res = oValue.equals(wValue);

        if (!negate && !res)
        {
            return false;
        }

        if (negate && res)
        {
            return false;
        }

        return !negate;
    }

    @Override
    public synchronized Set loadConfigElements(Query query)
        throws ConfigServiceException
    {
        Set res = null;

        try
        {
            res = super.loadConfigElements(query);
        }
        catch (Exception e)
        {
            debugPrintln("loadConfigElements failed on remote query: " + e.getMessage());
            res = new HashSet();
        }

        // Add any locally matched elements into the results set...
        //
        res.addAll(loadLocalConfigElements(query));

        return res;
    }

    @Override
    public Set listConfigElements(Query query)
        throws ConfigServiceException
    {
        Set res = null;

        try
        {
            res = super.listConfigElements(query);
        }
        catch (Exception e)
        {
            debugPrintln("listConfigElements failed on remote query: " + e.getMessage());
            res = new HashSet();
        }

        // Add any locally matched names into the results set...
        //
        Iterator i = loadLocalConfigElements(query).iterator();
        while (i.hasNext())
        {
            res.add(((IConfigElement)i.next()).getName());
        }

        return res;
    }

    //-------------------------------------------------------------------------

    /**
     *
     * @param path
     * @throws ConfigServiceException
     * @deprecated  Use createFolder instead
     */
    @Override
    public void createPath(String path)
        throws ConfigServiceException
    {
        createFolder(path, null, true, false);
    }

    /**
     *
     * @param path
     * @param existingOK
     * @throws ConfigServiceException
     * @deprecated  Use createFolder instead
     */
    @Override
    public void createPath(String path, boolean existingOK)
        throws ConfigServiceException
    {
        createFolder(path, null, true, existingOK);
    }

    /**
     * Creates the specified folder (and any intermediate folders if they don't
     * exist).
     *
     * If the folder already exists then no exception is thrown
     * ie. the call behaves as a no-op.
     *
     * @param  path                    The folder path to create.
     * @throws ConfigServiceException  If the method fails to create the folder
     *                                 then this exception is thrown.
     */
    @Override
    public void createFolder(String folderName)
        throws ConfigServiceException
    {
        createFolder(folderName, null, true, true);
    }

    /**
     * Creates the specified folder and applies any meta attributes.
     *
     * @param  path                    The folder path to create.
     * @param  metaAttributes          The meta attributes to set on the folder.
     * @param  createParentFolders     If true then any missing parent folders
     *                                 are created too.
     * @throws ConfigServiceException  If the method fails to create the folder
     *                                 then this exception is thrown.
     */
    @Override
    public void createFolder(String  path,
                             Map     metaAttributes,
                             boolean createParentFolders)
        throws ConfigServiceException
    {
        createFolder(path, metaAttributes, createParentFolders, true);
    }

    /**
     * Creates the specified folder and applies any meta attributes.
     *
     * @param  path                    The folder path to create.
     * @param  metaAttributes          The meta attributes to set on the folder.
     * @param  createParentFolders     If true then any missing parent folders
     *                                 are created too.
     * @param existing
     * @throws ConfigServiceException  If the method fails to create the folder
     *                                 then this exception is thrown.
     */
    protected void createFolder(String  path,
                                Map     metaAttributes,
                                boolean createParentFolders,
                                boolean existingOK)
        throws ConfigServiceException
    {
        if (path.charAt(0) != IMFDirectories.MF_DIR_SEPARATOR)
        {
            throw new ConfigServiceException("cs-create-path-invalid-path", new Object[] {path});
        }

        try
        {
            StringTokenizer st     = new StringTokenizer(path, String.valueOf(IMFDirectories.MF_DIR_SEPARATOR));
            StringBuffer    buffer = new StringBuffer();

            while (st.hasMoreTokens())
            {
                buffer.append(IMFDirectories.MF_DIR_SEPARATOR);
                buffer.append(st.nextToken());

                boolean exist = st.hasMoreTokens() ? true : existingOK;

                if (createParentFolders || !st.hasMoreTokens())
                {
                    String subPath = buffer.toString();

                    if (isCreateFolderAllowed(m_txn, subPath, exist))
                    {
                        m_txn.addCreateFolder(subPath, exist);
                    }
                }
            }

            if ((metaAttributes != null) && !metaAttributes.isEmpty())
            {
                setMetaAttributes(path, metaAttributes);
            }
        }
        catch (NoSuchElementException e)
        {
            throw new ConfigServiceException("cs-create-path-failed", new Object[] {path} , e);
        }
    }

    @Override
    public void deleteFolder(String folderName)
        throws ConfigServiceException
    {
        m_txn.addDeleteFolder(folderName);
    }

    @Override
    public void storeConfigElement(IConfigElement configElement)
        throws ConfigServiceException
    {
        if (((ConfigElementImpl)configElement).m_configServer != this)
        {
            throw new ConfigServiceException("cs-store-ces-wrong-cs", new Object[] {configElement.getName()});
        }

        if (((ConfigElementImpl)configElement).getState() == ConfigElementImpl.LOCAL_STATE)
        {
            elementAddedToTxn((ConfigElementImpl)configElement);
        }

        Set subElements = ((ConfigElementImpl)configElement).getSubElements();
        Iterator i = subElements.iterator();
        while (i.hasNext())
        {
            ConfigElementImpl subElement = (ConfigElementImpl)i.next();

            if (subElement.getState() == ConfigElementImpl.LOCAL_STATE)
            {
                elementAddedToTxn(subElement);
            }
        }
    }

    @Override
    public void storeConfigElements(IConfigElement[] configElements,
                                    String[] deleteElements)
        throws ConfigServiceException
    {
        for (int i = 0; i < configElements.length; i++)
        {
            storeConfigElement(configElements[i]);
        }

        if (deleteElements != null)
        {
            for (int i = 0; i < deleteElements.length; i++)
            {
                removeConfigElement(deleteElements[i]);
            }
        }
    }

    @Override
    public void removeConfigElement(String configElementName)
        throws ConfigServiceException
    {
        elementRemoved(configElementName);
    }

    @Override
    public void removeConfigElements(String[] configElementNames)
        throws ConfigServiceException
    {
        for (int i = 0; i < configElementNames.length; i++)
        {
            elementRemoved(configElementNames[i]);
        }
    }

    @Override
    public void rename(String oldName, String newName)
        throws ConfigServiceException
    {
        ConfigElementImpl configElement = (ConfigElementImpl)getElementCache().lookup(oldName, "");

        if (configElement != null)
        {
            configElement.setName(newName); // Config element that has been loaded locally
        }
        else
        {
            m_txn.addRename(oldName, newName); // Must be folder or config element that has not been loaded
            renameCacheElements(oldName, newName);
        }
    }

    @Override
    public void setMetaAttributes(String path, Map metaAttributes)
        throws ConfigServiceException
    {
        if (metaAttributes == null)
        {
            throw new ConfigServiceException("tcs-set-meta-attrs-is-null", new Object[] {path});
        }

        m_txn.addSetAttributes(path, Util.mapToHashMap(Util.combineToolMetaAttributes(metaAttributes)));
    }

    void insertCreateElement(String      name,
                             IDirElement delta,
                             Map         meta,
                             InputStream blob,
                             boolean     isInstance)
        throws ConfigServiceException
    {
        try
        {
            ArrayList actions = m_txn.getActions();

            for (int i = 0; i < actions.size(); i++)
            {
                DSTransaction.Action action = (DSTransaction.Action)actions.get(i);

                if (((action instanceof DSTransaction.CreateElement) && ((DSTransaction.CreateElement)action).m_element.getIdentity().getName().equals(name)) ||
                    ((action instanceof DSTransaction.AttachBlob) && ((DSTransaction.AttachBlob)action).m_element.getIdentity().getName().equals(name)))
                {
                    if (blob != null)
                    {
                        actions.set(i, m_txn.createAttachBlob(delta, blob));
                    }
                    else
                    {
                        actions.set(i, m_txn.createCreateElement(delta));
                    }

                    if (meta != null)
                    {
                        m_txn.addSetAttributes(delta.getIdentity().getName(), Util.mapToHashMap(Util.combineToolMetaAttributes(meta)));
                    }

                    return;
                }
            }
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("tcs-insert-create-element-failed", e);
        }
        throw new ConfigServiceException("tcs-insert-create-element-failed");
    }

    @Override
    public synchronized void commit()
        throws ConfigServiceException
    {
        printLists();

        try
        {
            Iterator it = null;

            // Add new elements to transaction
            it = m_newElements.iterator();
            while (it.hasNext())
            {
                ConfigElementImpl configElement = (ConfigElementImpl)it.next();
                configElement.validateComplete();

                InputStream blob  = (configElement.m_blob != null) ? configElement.m_blob : null;
                IDirElement delta = (IDirElement)configElement.doneUpdate();
                Map         meta  = configElement.isMetaAttributesModified() ? configElement.getMetaAttributes() : null;

                insertCreateElement(delta.getIdentity().getName(), delta, meta, blob, configElement.isPrototypeInstance());
            }

            it = m_modifiedElements.iterator();
            while (it.hasNext())
            {
                addToTxn((ConfigElementImpl)it.next());
            }

            DSTransaction txn = adjustOrdering(m_txn);

            debugTxn(txn);

            m_ds.executeTransaction(txn);

            // Transition NEW element to UNMODIFIED state
            Object[] objs = m_newElements.toArray();
            for (int i = 0; i < objs.length; i++)
            {
                elementStored((ConfigElementImpl)objs[i]);
            }

            // Transition MODIFIED element to UNMODIFIED state
            objs = m_modifiedElements.toArray();
            for (int i = 0; i < objs.length; i++)
            {
                elementStored((ConfigElementImpl)objs[i]);
            }

            // Transition REMOVED element to DELETED state
            objs = m_removedElements.toArray();
            for (int i = 0; i < objs.length; i++)
            {
                elementDeleted((ConfigElementImpl)objs[i]);
            }

            m_txn = (DSTransaction)m_ds.createTransaction();
        }
        catch (Exception e)
        {
            try { rollback(); } catch (ConfigServiceException ex) { /* Ignore */ }
            throw new ConfigServiceException("tcs-commit-failed", e);
        }
    }

    private void addToTxn(ConfigElementImpl configElement)
        throws ConfigServiceException, IOException
    {
        configElement.validateComplete();

        IDeltaDirElement delta = (IDeltaDirElement)configElement.doneUpdate();

       if (configElement.m_blob != null)
    {
        m_txn.addAttachBlob(delta, configElement.m_blob);
    }
    else
    {
        m_txn.addUpdateElement(delta, true);
    }

        if (configElement.isMetaAttributesModified())
        {
            Map metaAttributes = Util.combineToolMetaAttributes((Map)configElement.getMetaAttributes());

            m_txn.addSetAttributes(configElement.getName(), Util.mapToHashMap(metaAttributes));
        }
    }

    @Override
    public synchronized void rollback()
        throws ConfigServiceException
    {
        Exception ex = null;

        /*  Transition LOCAL elements to DELETED state.  */
        Object[] objs = m_localElements.toArray();
        for (int i = 0; i < objs.length; i++)
        {
            elementDeleted((ConfigElementImpl)objs[i]);
        }

        /*  Transition NEW elements to DELETED state.  */
        objs = m_newElements.toArray();
        for (int i = 0; i < objs.length; i++)
        {
            elementDeleted((ConfigElementImpl)objs[i]);
        }

        /*  Refresh MODIFIED elements. */
        objs = m_modifiedElements.toArray();
        for (int i = 0; i < objs.length; i++)
        {
            try { ((ConfigElementImpl)objs[i]).refresh(); } catch (Exception e) { ex = e; }
        }

        /*  Refresh UNMODIFIED elements. */
        objs = m_unmodifiedElements.toArray();
        for (int i = 0; i < objs.length; i++)
        {
            try { ((ConfigElementImpl)objs[i]).refresh(); } catch (Exception e) { ex = e; }
        }

        /*  Put REMOVED elements back into element cache and refresh. */
        objs = m_removedElements.toArray();
        for (int i = 0; i < objs.length; i++)
        {
            ConfigElementImpl configElement = (ConfigElementImpl)objs[i];
            getElementCache().add(configElement.getName(), "", configElement);
            try
            {
                configElement.refresh();
            }
            catch (Exception e)
            {
                ex = e;
            }
        }

        m_txn = (DSTransaction)m_ds.createTransaction();

        if (ex != null)
        {
            throw new ConfigServiceException("tcs-rollback-failed", ex);
        }
    }

    @Override
    public synchronized void flush()
        throws ConfigServiceException
    {
        // Transition LOCAL elements to DELETED state
        Object[] objs = m_localElements.toArray();
        for (int i = 0; i < objs.length; i++)
        {
            elementDeleted((ConfigElementImpl)objs[i]);
        }

        // Transition NEW elements to DELETED state
        objs = m_newElements.toArray();
        for (int i = 0; i < objs.length; i++)
        {
            elementDeleted((ConfigElementImpl)objs[i]);
        }

        // Transition MODIFIED elements to DELETED state
        objs = m_modifiedElements.toArray();
        for (int i = 0; i < objs.length; i++)
        {
            elementDeleted((ConfigElementImpl)objs[i]);
        }

        // Transition UNMODIFIED elements to DELETED state
        objs = m_unmodifiedElements.toArray();
        for (int i = 0; i < objs.length; i++)
        {
            elementDeleted((ConfigElementImpl)objs[i]);
        }

        /*  Transition REMOVED elements to DELETED state.  */
        objs = m_removedElements.toArray();
        for (int i = 0; i < objs.length; i++)
        {
            elementDeleted((ConfigElementImpl)objs[i]);
        }

        m_txn = (DSTransaction)m_ds.createTransaction();
    }

    @Override
    public IConfigServer subtransaction()
        throws ConfigServiceException
    {
        return new SubTxnConfigServer(this);
    }

    @Override
    protected void elementCreated(ConfigElementImpl configElement)
        throws ConfigServiceException
    {
        getElementCache().add(configElement.getName(), "", configElement);
        configElement.setDirectoryElementName(configElement.getName());
        removeElementFromLists(configElement);
        m_localElements.add(configElement);
        configElement.setState(ConfigElementImpl.LOCAL_STATE);
    }

    protected void elementAddedToTxn(ConfigElementImpl configElement)
        throws ConfigServiceException
    {
        configElement.setDirectoryElementName(configElement.getName());

        m_txn.addCreateElement((IDirElement)configElement.doneUpdate());

        removeElementFromLists(configElement);
        m_newElements.add(configElement);
        configElement.setState(ConfigElementImpl.NEW_STATE);
    }

    @Override
    protected void elementLoaded(ConfigElementImpl configElement)
        throws ConfigServiceException
    {
        getElementCache().add(configElement.getName(), "", configElement);
        removeElementFromLists(configElement);
        m_unmodifiedElements.add(configElement);
        configElement.setDirectoryElementName(configElement.getName());
        configElement.setState(ConfigElementImpl.UNMODIFIED_STATE);
    }

    @Override
    protected void elementStored(ConfigElementImpl configElement)
        throws ConfigServiceException
    {
        removeElementFromLists(configElement);
        configElement.setState(ConfigElementImpl.UNMODIFIED_STATE);
        configElement.setDirectoryElementName(configElement.getName());
        m_unmodifiedElements.add(configElement);
    }

    @Override
    protected void elementRefreshed(ConfigElementImpl configElement)
        throws ConfigServiceException
    {
        removeElementFromLists(configElement);
        configElement.setState(ConfigElementImpl.UNMODIFIED_STATE);
        m_unmodifiedElements.add(configElement);
    }

    @Override
    protected void elementModified(ConfigElementImpl configElement)
        throws ConfigServiceException
    {
        if (!configElement.isNew())
        {
            removeElementFromLists(configElement);
            configElement.setState(ConfigElementImpl.MODIFIED_STATE);
            m_modifiedElements.add(configElement);
        }
    }

    @Override
    protected void elementRenamed(String            oldName,
                                  String            newName,
                                  ConfigElementImpl configElement)
        throws ConfigServiceException
    {
        try
        {
            /* Add to cache under new name */
            getElementCache().add(newName, "", configElement);
            /* Remove from cache under old name */
            getElementCache().remove(oldName, "");
            if (!configElement.isNew())
            {
                removeElementFromLists(configElement);
                configElement.setState(ConfigElementImpl.MODIFIED_STATE);
                m_modifiedElements.add(configElement);
                m_txn.addRename(oldName, newName);
            }
            else
            {
                if (configElement.getState() == ConfigElementImpl.NEW_STATE)
                {
                    m_txn.addRename(oldName, newName);
                }
            }
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("tcs-element-renamed-failed", e);
        }
    }

    @Override
    protected void elementRemoved(ConfigElementImpl configElement)
        throws ConfigServiceException
    {
        removeElementFromLists(configElement);
        getElementCache().remove(configElement.getName(), "");
        configElement.setState(ConfigElementImpl.REMOVED_STATE);
        m_removedElements.add(configElement);
        m_txn.addDeleteElement(configElement.getName());
    }

    @Override
    protected void elementRemoved(String configElementName)
        throws ConfigServiceException
    {
        ConfigElementImpl configElement = (ConfigElementImpl)getElementCache().
                                          lookup(configElementName, "");
        if (configElement != null)
        {
            removeElementFromLists(configElement);
            getElementCache().remove(configElementName, "");
            configElement.setState(ConfigElementImpl.REMOVED_STATE);
            m_removedElements.add(configElement);
        }
        m_txn.addDeleteElement(configElementName);
    }

    protected void elementDeleted(String configElementName)
        throws ConfigServiceException
    {
        ConfigElementImpl configElement = (ConfigElementImpl)getElementCache().
                                          lookup(configElementName, "");
        if (configElement != null)
        {
            removeElementFromLists(configElement);
            getElementCache().remove(configElementName, "");
            configElement.setState(ConfigElementImpl.DELETED_STATE);
        }
    }

    protected void elementDeleted(ConfigElementImpl configElement)
        throws ConfigServiceException
    {
        removeElementFromLists(configElement);
        getElementCache().remove(configElement.getName(), "");
        configElement.setState(ConfigElementImpl.DELETED_STATE);
    }

    void removeElementFromLists(ConfigElementImpl configElement)
    {
        m_localElements.remove(configElement);
        m_newElements.remove(configElement);
        m_modifiedElements.remove(configElement);
        m_unmodifiedElements.remove(configElement);
        m_removedElements.remove(configElement);
    }

    private void debugTxn(DSTransaction txn)
    {
        if (!DEBUG)
        {
            return;
        }

        List list = txn.getActions();

        for (int i = 0; i < list.size(); i++)
        {
            DSTransaction.Action action = (DSTransaction.Action)list.get(i);

            System.err.println(getActionString(action));
        }
    }

    /**
     * Method determines whether or not its ok to add the create folder operation
     * to the transaction...by checking for existing creates, deletes and renames
     * which might obviate the need for the operation.
     *
     * @param txn           The transaction on which the check is based.
     * @param folder        The new folder that is being created.
     * @param okIfExisting  Indicates whether or not we care if the new folder
     *                      already exists or not.
     * @return              Returns true if the create folder action should
     *                      be added...otherwise false if the create folder
     *                      is somehow already in the transaction.
     */
    private boolean isCreateFolderAllowed(DSTransaction txn,
                                          String        folder,
                                          boolean       okIfExisting)
    {
        List list = txn.getActions();

        for (int i = list.size() - 1; i >= 0; i--)
        {
            DSTransaction.Action action = (DSTransaction.Action)list.get(i);

            if (action instanceof DSTransaction.DeleteFolder)
            {
                // If we find that the same folder is deleted in the transaction
                // then we should add it in again.
                //
                if (((DSTransaction.DeleteFolder)action).m_folderName.equals(folder))
                {
                    return true;
                }
            }

            if (action instanceof DSTransaction.CreateFolder)
            {
                if (((DSTransaction.CreateFolder)action).m_folderName.equals(folder))
                {
                    // If we find the create folder already in the transaction
                    // and this time we don't care if it already exists then
                    // there's no need to add it...
                    //
                    return !okIfExisting;
                }
            }

            if (action instanceof DSTransaction.Rename)
            {
                // The folder we are trying to add/create was previously added
                // and then renamed...
                //
                if (((DSTransaction.Rename)action).m_oldName.equals(folder))
                {
                    return true;
                }

                if (((DSTransaction.Rename)action).m_newName.equals(folder))
                {
                    // The transaction contains a folder that has been renamed
                    // to the folder we are trying to add/create and if
                    return !okIfExisting;
                }
            }
        }

        return true;
    }

    private String getActionString(DSTransaction.Action action)
    {
        StringBuffer sb = new StringBuffer();

        if (action instanceof DSTransaction.CreateElement)
        {
            DSTransaction.CreateElement a = (DSTransaction.CreateElement)action;
            sb.append("CREATE").append("(");
            sb.append(a.m_element.getIdentity().getName());
            sb.append(")");
            sb.append(".").append(a.m_element.getIdentity().getVersion());
        }
        else
        if (action instanceof DSTransaction.DeleteElement)
        {
            DSTransaction.DeleteElement a = (DSTransaction.DeleteElement)action;
            sb.append("DELETE").append("(");
            sb.append(a.m_elementName);
            sb.append(")");
        }
        else
        if (action instanceof DSTransaction.AttachBlob)
        {
            DSTransaction.AttachBlob a = (DSTransaction.AttachBlob)action;
            sb.append("ATTACH").append("(");
            sb.append(a.m_element.getIdentity().getName());
            sb.append(")");
            sb.append(".").append(a.m_element.getIdentity().getVersion());
        }
        else
        if (action instanceof DSTransaction.DetachBlob)
        {
            DSTransaction.DetachBlob a = (DSTransaction.DetachBlob)action;
            sb.append("DETACH").append("(");
            sb.append(a.m_delta.getIdentity().getName());
            sb.append(")");
            sb.append(".").append(a.m_delta.getIdentity().getVersion());
        }
        else
        if (action instanceof DSTransaction.UpdateElement)
        {
            DSTransaction.UpdateElement a = (DSTransaction.UpdateElement)action;
            sb.append("UPDATE").append("(");
            sb.append(a.m_element.getIdentity().getName());
            sb.append(")");
            sb.append(".").append(a.m_element.getIdentity().getVersion());
        }
        else
        if (action instanceof DSTransaction.CreateFolder)
        {
            DSTransaction.CreateFolder a = (DSTransaction.CreateFolder)action;
            sb.append("CREATE_FOLDER").append("(");
            sb.append(a.m_folderName);
            sb.append(", existingOk = ").append(a.m_existingOk);
            sb.append(")");
        }
        else
        if (action instanceof DSTransaction.DeleteFolder)
        {
            DSTransaction.DeleteFolder a = (DSTransaction.DeleteFolder)action;
            sb.append("DELETE_FOLDER").append("(");
            sb.append(a.m_folderName);
            sb.append(")");
        }
        else
        if (action instanceof DSTransaction.Rename)
        {
            DSTransaction.Rename a = (DSTransaction.Rename)action;
            sb.append("RENAME").append("(");
            sb.append(a.m_oldName).append(" > ").append(a.m_newName);
            sb.append(")");
        }
        else
        if (action instanceof DSTransaction.SetAttributes)
        {
            DSTransaction.SetAttributes a = (DSTransaction.SetAttributes)action;
            sb.append("SET_ATTRIBUTES").append("(").append(a.m_name).append(")");
        }
        else
        if (action instanceof DSTransaction.SubclassElement)
        {
            DSTransaction.SubclassElement a = (DSTransaction.SubclassElement)action;
            sb.append("SUBCLASS").append("(");
            sb.append(a.m_delta.getIdentity().getName());
            sb.append(")");
            sb.append(".").append(a.m_delta.getIdentity().getVersion());
        }

        return sb.toString();
    }

    private void printLists()
    {
        java.io.PrintStream out = System.err;

        if (!DEBUG)
        {
            return;
        }

        out.println("-- TRANSACTED SERVER STATE ------------------------------");

        Iterator it = m_localElements.iterator();
        if (it.hasNext())
        {
            out.println("LOCAL:");
        }
        while (it.hasNext())
        {
            ConfigElementImpl ce = (ConfigElementImpl)it.next();
            out.println("    " + ce.getName());
        }

        it = m_newElements.iterator();
        if (it.hasNext())
        {
            out.println("NEW:");
        }
        while (it.hasNext())
        {
            ConfigElementImpl ce = (ConfigElementImpl)it.next();
            out.println("    " + ce.getName());
        }

        it = m_modifiedElements.iterator();
        if (it.hasNext())
        {
            out.println("MODIFIED:");
        }
        while (it.hasNext())
        {
            ConfigElementImpl ce = (ConfigElementImpl)it.next();
            out.println("    " + ce.getName());
        }

        it = m_removedElements.iterator();
        if (it.hasNext())
        {
            out.println("REMOVED:");
        }
        while (it.hasNext())
        {
            ConfigElementImpl ce = (ConfigElementImpl)it.next();
            out.println("    " + ce.getName());
        }
        out.println("---------------------------------------------------------");
    }

    //-------------------------------------------------------------------------

    private DSTransaction adjustOrdering(DSTransaction txn)
        throws Exception
    {
        DSTransaction res     = (DSTransaction)m_ds.createTransaction();
        List          actions = txn.getActions();

try{
        while (!actions.isEmpty())
        {
            DSTransaction.Action action = (DSTransaction.Action)actions.remove(0);

            if (action instanceof DSTransaction.UpdateElement)
            {
                DSTransaction.UpdateElement a = (DSTransaction.UpdateElement)action;
                IConfigBean aBean = (IConfigBean)loadConfigBean(a.m_element.getIdentity().getName(), true);
                int         i = -1;


                if (aBean != null && aBean.isPrototypeInstance())
                {
                    //bean can be null for localonly
                    i = getPrototypeIndex(res.getActions(), aBean.getPrototype().getName());
                }

                if (i == -1)
                {
                    res.addAction(action);
                }
                else
                {
                    debugPrintln("TXN: adjusting instance '" + (aBean == null ? null : aBean.getName()) + "' order to " + i);

                    res.addAction(i, action);
                }
            }
            else
            {
                res.addAction(action);
            }
        }
}catch (Exception e)
{
        e.printStackTrace();
}

        return res;
    }

    /**
     * Searchs a list of transaction actions looking for an element update of
     * a prototype with the given name. If the action is found then its index
     * is returned.
     *
     * @param list           A list of DSTransaction.Action objects
     * @param prototypeName  The logical path (name) of the prototype element
     * @return               Returns the index of the action in the list
     */
    private int getPrototypeIndex(List list, String prototypeName)
    {
        for (int i = 0; i < list.size(); i++)
        {
            DSTransaction.Action action = (DSTransaction.Action)list.get(i);

// AHJ : HAVE TO HANDLER ATTACH BLOB TOO!!!!

            if (action instanceof DSTransaction.UpdateElement)
            {
                DSTransaction.UpdateElement a = (DSTransaction.UpdateElement)action;

                if (a.m_element.getIdentity().getName().equals(prototypeName))
                {
                    return i;
                }
            }
        }

        return -1;
    }

    /*
             // IF we are adding an instance (of a prototype) AND that prototype
             // is already in the transaction list AND the prototype has been
             // modified THEN
             //   The prototype change will result in all related instances getting
             //   a version increment behind the seens (on the DS) so we need to
             //   adjust for this.
             // END IF
      */
    private DSTransaction adjustVersioning(DSTransaction txn)
        throws Exception
    {
        DSTransaction res = (DSTransaction)m_ds.createTransaction();
        Iterator      i   = txn.getActions().iterator();

        while (i.hasNext())
        {
            DSTransaction.Action action = (DSTransaction.Action)i.next();

            if (action instanceof DSTransaction.UpdateElement)
            {
                DSTransaction.UpdateElement a = (DSTransaction.UpdateElement)action;
                IConfigBean bean = (IConfigBean)loadConfigBean(a.m_element.getIdentity().getName(), true);

                if (bean.isPrototypeInstance() &&
                    (getPrototypeIndex(m_txn.getActions(), bean.getPrototype().getName()) != -1) &&
                    bean.getPrototype().isModified())
                {
                    IElementIdentity ei = a.m_element.getIdentity();
                    ((ElementIdentity)ei).setVersion(((ElementIdentity)ei).getVersion() + 1);
                }
            }
            else
            if (action instanceof DSTransaction.AttachBlob)
            {
                DSTransaction.AttachBlob a = (DSTransaction.AttachBlob)action;
                IConfigBean bean = (IConfigBean)loadConfigBean(a.m_element.getIdentity().getName(), true);

                if (bean.isPrototypeInstance() &&
                    (getPrototypeIndex(m_txn.getActions(), bean.getPrototype().getName()) != -1) &&
                    bean.getPrototype().isModified())
                {
                    IElementIdentity ei = a.m_element.getIdentity();
                    ((ElementIdentity)ei).setVersion(((ElementIdentity)ei).getVersion() + 1);
                }
            }

            res.addAction(action);
        }

        return res;
    }

}
