/*
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.mx.config.impl;

import java.math.BigDecimal;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TimeZone;

import org.apache.xerces.impl.dv.util.HexBin;

import com.sonicsw.mx.config.ConfigAttributeException;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.ConfigTypeException;
import com.sonicsw.mx.config.IAttributeDescription;
import com.sonicsw.mx.config.IAttributeList;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IConfigElement;
import com.sonicsw.mx.config.IConfigPath;
import com.sonicsw.mx.config.IConfigServer;
import com.sonicsw.mx.config.util.MessageFormatter;
import com.sonicsw.mf.common.IDSTransaction;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.impl.DSTransaction;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.view.ILogicalNameSpace;

public class Util
{
    private static final String MA_TOOL_ATTRIBUTES = "TOOL_ATTRIBUTES";

    public static final String DEFAULT_PARSER_NAME = "org.apache.xerces.parsers.SAXParser";

    private static final String XDECIMAL = "org.apache.xerces.impl.dv.xs.DecimalDV$XDecimal";
    public static final String LEXICAL_HANDLER_PROPERTY = "http://xml.org/sax/properties/lexical-handler";

    public static Object
    validateAttributeName(Object attributeName)
    throws ConfigServiceException
    {
        if (!(attributeName instanceof String))
        {
            String name = (attributeName == null) ? "null" : attributeName.toString();
            String type = (attributeName == null) ? "na"   : attributeName.getClass().getName();

            throw new ClassCastException("Invalid Attribute Name [" + name + ", " + type + "]");
        }

        return attributeName;
    }

    public static Object
    validateAttributeValue(IConfigPath attributeName,
                           Object attributeValue,
                           IConfigServer configServer,
                           IAttributeDescription parentAttrDesc)
    throws ConfigAttributeException
    {
        return validateAttributeValue(attributeName, attributeValue, configServer, parentAttrDesc, true);
    }

    /*  This method is called before adding an attribute value to
        an attribute map or list to validate that it is a valid type
        and conforms to the map's or list's type constraints   */
    public static Object
    validateAttributeValue(IConfigPath attributeName,
                           Object attributeValue,
                           IConfigServer configServer,
                           IAttributeDescription parentAttrDesc,
                           boolean filterDefaults)
    throws ConfigAttributeException
    {
        IAttributeDescription attrDesc = null;
        Object tmp = null;

        /*  Validate that the name of an attribute that is added to a
            typed attribute map or list is valid.   */
        if (parentAttrDesc != null  && !attributeName.getLastComponent().equals("_MF_SYSTEM_ATTRIBUTES") )
        {   /*  Parent has an attribute description so it must be either a
                typed map or list: do constraint checking. */

            if (parentAttrDesc.getType() == IAttributeList.class ||
                parentAttrDesc.getProperty(IAttributeDescription.UNENUM_MAP_PROPERTY) != null)
            {   /*  Parent is an attribute list or unenumerated attribute map.    */
                Set names = parentAttrDesc.getAttributeDescriptionNames();
                if (names.size() < 1)
                {
                    throw new ConfigAttributeException (attributeName, "config-attr-desc-invalid", new ConfigTypeException("no-entry-in-list-desc"));
                }
                if (names.size() > 1)
                {
                    System.err.println(MessageFormatter.formatMessage(ConfigTypeException.CONFIG_TYPE_ERRORS, "multi-entry-in-list-desc", new Object[] { attributeName.toString() } ));
                }
                String entryDescName = (String) names.iterator().next();
                attrDesc = parentAttrDesc.getAttributeDescription(entryDescName);
            }
            else
            {   /*  Parent is an enumerated attribute map */
                attrDesc = parentAttrDesc.getAttributeDescription(attributeName.getLastComponent());
                if (attrDesc == null)
                {
                    throw new ConfigAttributeException (attributeName, "config-attr-not-defined");
                }
            }

        }

        /*  Validate that the attribute's value is a valid type
            and clone or translate value as necessary. */
        if (attributeValue == null)
        {
            throw new ConfigAttributeException (attributeName, "config-attr-val-is-null");
        }
        else if (attributeValue instanceof Boolean)
        {
            ; // Do nothing - old code was useless... attributeValue = attributeValue;
        }
        else if (attributeValue instanceof Integer)
        {
            ; // Do nothing - old code was useless... attributeValue = attributeValue;
        }
        else if (attributeValue instanceof Long)
        {
            ; // Do nothing - old code was useless... attributeValue = attributeValue;
        }
        else if (attributeValue instanceof BigDecimal)
        {
            ; // Do nothing - old code was useless... attributeValue = attributeValue;
        }
        else if (attributeValue.getClass().getName().equals(XDECIMAL))
        {
            attributeValue = new BigDecimal(attributeValue.toString());
        }
        else if (attributeValue instanceof String)
        {
            ; // Do nothing - old code was useless... attributeValue = attributeValue;
        }
        else if (attributeValue instanceof Date)
        {
            attributeValue = ((Date)attributeValue).clone();
        }
        else if (attributeValue.getClass().equals(byte[].class))
        {
            attributeValue = ((byte[])attributeValue).clone();
        }
        else if (attributeValue.getClass().equals(int[].class))
        {
            attributeValue = convertToDate((int[])attributeValue);
        }
        else if (attributeValue instanceof com.sonicsw.mf.common.config.IAttributeList)
        {
            attributeValue = new AttributeListImpl((com.sonicsw.mf.common.config.IAttributeList)attributeValue,
                                                  attrDesc, (ConfigServer) configServer);
        }
        else if (attributeValue instanceof com.sonicsw.mf.common.config.IAttributeSet)
        {
            attributeValue = new AttributeMapImpl((com.sonicsw.mf.common.config.IAttributeSet)attributeValue,
                                                  attrDesc, (ConfigServer)configServer, filterDefaults);
        }
        else if (attributeValue instanceof IConfigElement)
        {
            attributeValue = new ConfigReference((IConfigElement)attributeValue);
        }
        else if (attributeValue instanceof IAttributeMap)
        {
            attributeValue = ((IAttributeMap)attributeValue).clone();
        }
        else if (attributeValue instanceof IAttributeList)
        {
            attributeValue = ((IAttributeList)attributeValue).clone();
        }
        else if (attributeValue instanceof Reference)
        {
            attributeValue = new ConfigReference(filterReferenceStrings(((Reference)attributeValue).getElementName()), configServer);
        }
        else if (attributeValue instanceof AttributeMapImpl.RemovedAttribute)
        {
            ; // Do nothing - old code was useless... attributeValue = attributeValue;
        }
        else
        {
            throw new ConfigAttributeException (attributeName, "config-attr-val-invalid-type", new Object[] { attributeValue, attributeValue.getClass().getName() });
        }

        /*  Validate that the value of an attribute added to a type list or map
            matches the attribute's type description.   */
        if (attrDesc != null && !attributeValue.getClass().equals(AttributeMapImpl.RemovedAttribute.class))
        {   /*  An attribute description was allocated above so this attribute
                is being added to a typed list or map: do constraint checking. */

            /*  Constrain fixed attributes.  */
            if ((tmp = attrDesc.getProperty(IAttributeDescription.FIXED_PROPERTY)) != null &&
                !Util.areEqual(attributeValue, tmp))
            {   /*  This attribute has a fixed value and it does no match the
                    value being set: throw exception*/
                throw new ConfigAttributeException (attributeName, "config-attr-val-no-match-fixed", new Object[] { attributeValue, tmp });
            }


            /*  For attributes that are either attribute maps or attribute lists,
                validate that the structure of the map or list matches the
                structure specified in the attribute's description.    */
            if (attributeValue instanceof IAttributeMap)
            {
                if (((AttributeMapImpl)attributeValue).m_attrDescription != attrDesc)
                {
                    throw new ConfigAttributeException (attributeName, "config-attr-val-incorrect-map-type");
                }
            }
            else
            if  (attributeValue instanceof IAttributeList)
            {
                if (((AttributeListImpl)attributeValue).m_attrDescription != attrDesc)
                {
                    throw new ConfigAttributeException (attributeName, "config-attr-val-incorrect-list-type");
                }
            }
            else
            {   /*  For atomic attributes, validate that the value conforms
                    to the constraints of the attribute description  */
                try
                {
                    attrDesc.validate(attributeValue);
                }
                catch (ConfigTypeException e)
                {
                    throw new ConfigAttributeException(attributeName, "config-attr-val-failed-validation", new Object[] { attributeValue }, e);
                }
                catch (ConfigAttributeException e)
                {
                    e.setAttributeName(attributeName);
                    throw e;
                }
                catch (ConfigServiceException e)
                {
                    throw new ConfigAttributeException(attributeName, "config-attr-val-failed-validation", new Object[] { attributeValue }, e);
                }
            }
        }

        return attributeValue;
    }

    public static Object
    validateAttributeDescriptionName(Object descriptionName)
    throws ConfigServiceException
    {
        return validateAttributeName(descriptionName);
    }

    public static Object
    validateAttributeDescriptionValue(Object descriptionValue)
    throws ConfigServiceException
    {
        if (descriptionValue instanceof IAttributeDescription)
        {
            return ((AttributeDescriptionImpl)descriptionValue).clone();
        }
        throw new ClassCastException("Invalid Attribute Description Value");
    }

    public static String
    filterReferenceStrings(String elementPath)
    {
        if (elementPath.startsWith(ILogicalNameSpace.DELETED_LABEL))
        {
            elementPath = elementPath.substring(ILogicalNameSpace.DELETED_LABEL.length());
        }
        if (elementPath.startsWith(ILogicalNameSpace.NO_STORAGE_LABEL))
        {
            elementPath = elementPath.substring(ILogicalNameSpace.NO_STORAGE_LABEL.length());
        }
        return elementPath;
    }

    public static String
    url2Name(String elementURL)
    {
        StringBuffer buffer = new StringBuffer();

        for (int newIndex, index = 0; ; index = newIndex + 3)
        {
            newIndex = elementURL.indexOf("%20", index);
            if (newIndex == -1)
            {
                buffer.append(elementURL.substring(index));
                break;
            }
            else
            {
                buffer.append(elementURL.substring(index, newIndex));
                buffer.append(" ");
            }
        }
        return buffer.toString();
    }

    public static String
    name2Url(String name)
    {
        StringBuffer buffer = new StringBuffer(name);

        for (int index = 0; index < buffer.length(); index++)
        {
            if (buffer.charAt(index) == ' ')
            {
                buffer.replace(index, index + 1, "%20");
                index++;
                index++;
            }
        }
        return buffer.toString();
    }

    private static final ThreadLocal<GregorianCalendar> CALENDER_THREAD_LOCAL = new ThreadLocal<GregorianCalendar>() {
        @Override
        protected GregorianCalendar initialValue() {
            return new GregorianCalendar();
        }
    };

    public static Date convertToDate(int[] dateTime)
    {
        if (dateTime.length < 8)
        {
            throw new ClassCastException("Invalid Attribute Value: " + dateTime.getClass().getName());
        }

        GregorianCalendar calendar = CALENDER_THREAD_LOCAL.get();
        calendar.clear();
        calendar.set(dateTime[0],     /* Year */
                       dateTime[1] - 1, /* Month: Note: convert from one based
                                                  to zero based month value. */
                       dateTime[2],     /* Day */
                       dateTime[3],     /* Hour */
                       dateTime[4],     /* Minute */
                       dateTime[5]);    /* Second */
        calendar.add(Calendar.MILLISECOND, dateTime[6]);
        calendar.setTimeZone(TimeZone.getTimeZone("GMT"));
        return calendar.getTime();
    }

    public static int[]
    convertFromDate(Date date)
    {
        int[] dateTime = new int[8];

        GregorianCalendar calendar = CALENDER_THREAD_LOCAL.get();
        calendar.setTime(date);

        dateTime[0] = calendar.get(Calendar.YEAR);
        dateTime[1] = calendar.get(Calendar.MONTH) + 1;
        dateTime[2] = calendar.get(Calendar.DAY_OF_MONTH);
        dateTime[3] = calendar.get(Calendar.HOUR_OF_DAY);
        dateTime[4] = calendar.get(Calendar.MINUTE);
        dateTime[5] = calendar.get(Calendar.SECOND);
        dateTime[6] = calendar.get(Calendar.MILLISECOND);
        dateTime[7] = 'Z';

        return dateTime;
    }

    public static boolean
    areEqual(Object attribute1, Object attribute2)
    {
        if (attribute1 == null || attribute2 == null)
        {
            return false;
        }
        else
        if (!attribute1.getClass().isInstance(attribute2))
        {   /*  attributes are not comparable: can not be equal */
            return false;
        }
        else
        if (attribute1 instanceof byte[])
        {
            return Arrays.equals((byte[]) attribute1, (byte[]) attribute2);
        }
        else
        if (attribute1 instanceof Reference)
        {
            /* Reference or ConfigReference */
            return ((Reference)attribute1).getElementName().equals(((Reference)attribute2).getElementName());
        }
        else
        {
            return attribute1.equals(attribute2);
        }
   }

    public static String toString(Object value)
    {
        StringBuffer buffer = new StringBuffer();
      
        if(value instanceof byte[])
        {
            return HexBin.encode((byte[])value);
        }
        else
        if(value instanceof Date)
        {
            int[] date = convertFromDate((Date)value);

            NumberFormat formatter = NumberFormat.getInstance();
            formatter.setMinimumIntegerDigits(4);
            formatter.setGroupingUsed(false);

            buffer.append(formatter.format(date[0]));
            buffer.append('-');

            formatter.setMinimumIntegerDigits(2);

            buffer.append(formatter.format(date[1]));
            buffer.append('-');
            buffer.append(formatter.format(date[2]));
            buffer.append('T');
            buffer.append(formatter.format(date[3]));
            buffer.append(':');
            buffer.append(formatter.format(date[4]));
            buffer.append(':');
            buffer.append(formatter.format(date[5]));
            buffer.append('.');
            buffer.append(formatter.format(date[6]));
            buffer.append((char)date[7]);
        }
        else
        if (value instanceof Reference)
        {
            buffer.append(name2Url(((Reference)value).getElementName()));
        }
        else
        {
            buffer.append(value);
        }
        return buffer.toString();
    }

    public static void
    validateConfigBeanName(String name)
    throws ConfigServiceException
    {
        if (name.charAt(0) != '/')
        {
            throw new ConfigServiceException("config-bean-name-not-abs", new Object[] { name });
        }
        validateConfigName(name);
    }

    public static void
    validateConfigTypeName(String name)
    throws ConfigServiceException
    {
        validateConfigName(name);
    }

    public static void
    validateConfigName(String name)
    throws ConfigServiceException
    {
        String error = null;

        if (name == null)
        {
            throw new ConfigServiceException("config-bean-name-is-null", new Object[] { name });
        }
        if (name.length() == 0)
        {
            throw new ConfigServiceException("config-bean-name-is-empty", new Object[] { name });
        }
        if (name.indexOf("$") == 0)
        {
            throw new ConfigServiceException("config-bean-name-contains-$", new Object[] { name });
        }
        if (name.indexOf("\\") >= 0)
        {
            throw new ConfigServiceException("config-bean-name-contains-\\", new Object[] { name });
        }
        if (name.indexOf("::") >= 0 )
        {
            throw new ConfigServiceException("config-bean-name-contains-::", new Object[] { name });
        }
        if (name.indexOf(".*.") >= 0 )
        {
            throw new ConfigServiceException("config-bean-name-contains-.*.", new Object[] { name });
        }
        if (name.indexOf(".#.") >= 0 )
        {
            throw new ConfigServiceException("config-bean-name-contains-.#.", new Object[] { name });
        }
        if (name.indexOf("<") >= 0 )
        {
            throw new ConfigServiceException("config-bean-name-contains-<", new Object[] { name });
        }
        if (name.indexOf(">") >= 0 )
        {
            throw new ConfigServiceException("config-bean-name-contains->", new Object[] { name });
        }
        if (name.indexOf("&") >= 0 )
        {
            throw new ConfigServiceException("config-bean-name-contains-&", new Object[] { name });
        }
    }

    public static boolean
    isDefaultValue(String attributeName,
                   Object attributeValue,
                   IAttributeDescription parentAttrDesc)
    {
        if (attributeValue == null || parentAttrDesc == null)
        {
            return false;
        }

        Object tmp = null;
        IAttributeDescription attrDesc = null;

        if (parentAttrDesc.getType() == IAttributeList.class ||
            parentAttrDesc.getProperty(IAttributeDescription.UNENUM_MAP_PROPERTY) != null)
        {   /*  Parent is an attribute list or unenumerated attribute map.    */
            Set names = parentAttrDesc.getAttributeDescriptionNames();
            if (names.size() == 1)
            {   /*  Description has valid number of entry descriptions:
                    retrieve entry description. */
                String entryDescName = (String) names.iterator().next();
                attrDesc = parentAttrDesc.getAttributeDescription(entryDescName);
                if (attrDesc != null &&
                    (tmp = attrDesc.getProperty(IAttributeDescription.DEFAULT_PROPERTY)) != null)
                {   /*  Entry description exists and has a default property
                        defined: determine if set value equals default. */
                    return tmp.equals(attributeValue);
                }
            }
        }
        else
        {   /*  Parent is an enumerated attribute map */
            attrDesc = parentAttrDesc.getAttributeDescription(attributeName);
            if (attrDesc != null &&
                (tmp = attrDesc.getProperty(IAttributeDescription.DEFAULT_PROPERTY)) != null)
            {   /*  Entry description exists and has a default property
                        defined: determine if set value equals default. */
                return tmp.equals(attributeValue);
            }
        }

        return false;
    }

    public static final HashMap mapToHashMap(Map map)
    {
        if (map == null)
        {
            return null;
        }

        if (map instanceof HashMap)
        {
            return (HashMap)map;
        }

        return new HashMap(map);
    }

    /**
     * Hack to split TOOL_ATTRIBUTES meta-attribute into the expected hashmap!
     * Basically removes the TOOL_ATTRIBUTES key-value pair and splits the value
     * into the "actual" meta attributes.
     *
     * @param map  The original HashMap that contains the TOOL_ATTRIBUTES entry
     * @return     The original HashMap with the TOOL_ATTRIBUTES entry replaced
     *             with meta attributes decoded from its value.
     * @deprecated Use the more generic java.util.Map-based method.
     */
    public static HashMap splitToolMetaAttributes(HashMap map)
    throws ConfigServiceException
    {
        return mapToHashMap(splitToolMetaAttributes((Map)map));
    }

    /**
     * Hack to split TOOL_ATTRIBUTES meta-attribute into the expected hashmap!
     * Basically removes the TOOL_ATTRIBUTES key-value pair and splits the value
     * into the "actual" meta attributes.
     *
     * @param map  The original HashMap that contains the TOOL_ATTRIBUTES entry
     * @return     The original HashMap with the TOOL_ATTRIBUTES entry replaced
     *             with meta attributes decoded from its value.
     */
    public static Map splitToolMetaAttributes(Map map)
    throws ConfigServiceException
    {
        if (map == null)
        {
            return null;
        }

        try
        {
            String ma = (String)map.remove(MA_TOOL_ATTRIBUTES);

            if (ma == null)
            {
                return map;
            }

            StringTokenizer st = new StringTokenizer(ma, ";", false);

            while (st.hasMoreTokens())
            {
                String item   = st.nextToken();
                int    index  = item.indexOf('=');

                if(index != -1)
                {
                    String key    = item.substring(0, index);
                    String value  = item.substring(index+1);

                    // If the meta attributes 'map' already contains a value
                    // then we don't want to overwrite it with 'String'
                    // something from the "TOOL_ATTRIBUTES" because its probably
                    // a bug!
                    if (!map.containsKey(key))
                    {
                        map.put(key, value);
                    }
                }
            }
        }
        catch (Exception e)
        {
            throw new ConfigServiceException("cs-list-failed", e);
        }
        return map;
    }

    /**
     * Takes the meta attributes and combines them into 1 ';' delimited
     * TOOL_ATTRIBUTES meta attribute to get around DS meta-attribute
     * limitations!
     *
     * @param map  Individual meta attributes to be set
     * @return     An adjusted meta attribute map
     * @deprecated Use more generic java.util.Map-based method.
     */
    public static HashMap combineToolMetaAttributes(HashMap map)
    {
        return (HashMap)combineToolMetaAttributes((Map)map);
    }

    /**
     * AHJ
     * Takes the meta attributes and combines them into 1 ';' delimited
     * TOOL_ATTRIBUTES meta attribute to get around DS meta-attribute
     * limitations!
     *
     * @param map  Individual meta attributes to be set
     * @return     An adjusted meta attribute map
     */
    public static Map combineToolMetaAttributes(Map map)
    {
        if (map == null)
        {
            return null;
        }

        HashMap      res = new HashMap(map.size());
        Iterator     i   = map.keySet().iterator();
        StringBuffer sb  = new StringBuffer();

        while (i.hasNext())
        {
            String key = (String)i.next();

            if (key.equals(IConfigServer.ELEMENT_IDENTITY) ||
                key.equals(IConfigServer.FOLDER_NAME) ||
                key.equals(IConfigServer.IS_COMPLEX))
            {
                res.put(key, map.get(key));
                continue;
            }

            Object value = map.get(key);

            if (sb.length() > 0)
            {
                sb.append(';');
            }

            sb.append(key).append("=").append(value);
        }

        res.put(MA_TOOL_ATTRIBUTES, sb.toString());

        return res;
    }

    public static String
    txnToString(IDSTransaction txn)
    {
        ArrayList actions = ((DSTransaction)txn).getActions();

        StringBuffer buffer = new StringBuffer();

        for (int i = 0; i < actions.size(); i++)
        {
            DSTransaction.Action action = (DSTransaction.Action)actions.get(i);

            if (action instanceof DSTransaction.AttachBlob)
            {
                DSTransaction.AttachBlob attachBlob = (DSTransaction.AttachBlob) action;
                buffer.append("AttachBlob: element=" + attachBlob.m_element.getIdentity().getName() + ":" + attachBlob.m_element.getIdentity().getVersion());
                buffer.append("\r\n");
            }
            else if (action instanceof DSTransaction.CreateElement)
            {
                DSTransaction.CreateElement createElement = (DSTransaction.CreateElement) action;
                buffer.append("CreateElement: element=" + createElement.m_element.getIdentity().getName() + ":" + createElement.m_element.getIdentity().getVersion());
                buffer.append("\r\n");
            }
            else if (action instanceof DSTransaction.CreateElements)
            {
                DSTransaction.CreateElements createElements=(DSTransaction.CreateElements) action;
                buffer.append("CreateElements: ");
                IDirElement[] elements=createElements.m_elements;
                for (int j=0; j < elements.length; j++)
                {
                    buffer.append(" element=" + elements[j].getIdentity().getName() + ":" + elements[j].getIdentity().getVersion());
                }
                buffer.append("\r\n");
            }
            else if (action instanceof DSTransaction.CreateFolder)
            {
                DSTransaction.CreateFolder createFolder=(DSTransaction.CreateFolder) action;
                buffer.append("CreateFolder: folder=" + createFolder.m_folderName + " existingOk = " + createFolder.m_existingOk);
                buffer.append("\r\n");
            }
            else if (action instanceof DSTransaction.DeleteElement)
            {
                DSTransaction.DeleteElement deleteElement=(DSTransaction.DeleteElement) action;
                buffer.append("DeleteElement: element=" + deleteElement.m_elementName);
                buffer.append("\r\n");
            }
            else if (action instanceof DSTransaction.DeleteFolder)
            {
                DSTransaction.DeleteFolder deleteFolder=(DSTransaction.DeleteFolder) action;
                buffer.append("DeleteFolder: folder=" + deleteFolder.m_folderName);
                buffer.append("\r\n");
            }
            else if (action instanceof DSTransaction.DetachBlob)
            {
                DSTransaction.DetachBlob detachBlob=(DSTransaction.DetachBlob) action;
                buffer.append("DetachBlob: element=" + detachBlob.m_delta.getIdentity().getName() + ":" + detachBlob.m_delta.getIdentity().getVersion());
                buffer.append("\r\n");
            }
            else if (action instanceof DSTransaction.Rename)
            {
                DSTransaction.Rename rename=(DSTransaction.Rename) action;
                buffer.append("Rename: oldName=" + rename.m_oldName + " newName = " + rename.m_newName);
                buffer.append("\r\n");
            }
            else if (action instanceof DSTransaction.SetAttributes)
            {
                DSTransaction.SetAttributes setAttributes=(DSTransaction.SetAttributes) action;
                buffer.append("SetAttributes: name=" + setAttributes.m_name + " : " + setAttributes.m_attributes);
                buffer.append("\r\n");
            }
            else if (action instanceof DSTransaction.SubclassElement)
            {
                DSTransaction.SubclassElement subclassElement=(DSTransaction.SubclassElement) action;
                buffer.append("SubclassElement: newElementPath=" + subclassElement.m_newElementPath);
                buffer.append("\r\n");
            }
            else if (action instanceof DSTransaction.UpdateElement)
            {
                DSTransaction.UpdateElement updateElement=(DSTransaction.UpdateElement) action;
                buffer.append("UpdateElement: element=" + updateElement.m_element.getIdentity().getName() + ":" + updateElement.m_element.getIdentity().getVersion());
                buffer.append("\r\n");
            }
        }
        return buffer.toString();
    }

    public static IConfigElement createConfigElement(IDirElement dirElement, ConfigServer configServer)
    throws ConfigServiceException
    {
        return new ConfigElementImpl(dirElement, configServer);
    }

    public static void printBasicElement(com.sonicsw.mf.common.config.IBasicElement element)
    {
        System.err.println( new BasicElementPrinter(element));
    }

}

class BasicElementPrinter
{
    protected String m_elementType = "DeltaElement";

    protected long m_version = -1;

    protected String m_name;

    protected Set m_newAttributeNames = new java.util.HashSet();

    protected Set m_modifiedAttributeNames = new java.util.HashSet();

    protected Set m_deletedAttributeNames = new java.util.HashSet();

    public BasicElementPrinter(com.sonicsw.mf.common.config.IBasicElement element)
    {
        m_name = element.getIdentity().getName();
        m_version = element.getIdentity().getVersion();
        parseElement(element);
    }

    protected final void
    parseElement(com.sonicsw.mf.common.config.IBasicElement element)
    {
        if (element instanceof com.sonicsw.mf.common.config.IDeltaElement)
        {
            Object obj = ((com.sonicsw.mf.common.config.IDeltaElement)element).getDeltaAttributes();
            if (obj instanceof com.sonicsw.mf.common.config.IDeltaAttributeSet)
            {
                parseDeltaAttributeSet((com.sonicsw.mf.common.config.IDeltaAttributeSet) obj, new ConfigPathImpl());
            }
            else // obj instanceof IAttributeSet
            {   // Entire attribute set of element has been overwritten:
                // add the names of all attributes in the set to the modified attribute names set.
                System.err.println("Delta element contains attribute set.");
                Iterator names = ((com.sonicsw.mf.common.config.IAttributeSet)obj).getAttributes().keySet().iterator();
                while (names.hasNext())
                {
                    m_modifiedAttributeNames.add(new ConfigPathImpl((String) names.next()));
                }
            }
        }
        else
        {

        }
    }

    protected void
    parseDeltaAttributeSet(com.sonicsw.mf.common.config.IDeltaAttributeSet attrSet, IConfigPath path)
    {
        // Add paths for deleted attributes.
        String[] attrNames = attrSet.getDeletedAttributesNames();
        for(int i = 0; i < attrNames.length; i++)
        {
            m_deletedAttributeNames.add(new ConfigPathImpl(path).append(attrNames[i]));
        }

        // Add paths for new attributes.
        attrNames = attrSet.getNewAttributesNames();
        for(int i = 0; i < attrNames.length; i++)
        {
            Object attrValue = null;
            try
            {
                attrValue = attrSet.getNewValue(attrNames[i]);
            }
            catch (Exception e)
            {
                continue;
            }

            if (attrValue instanceof com.sonicsw.mf.common.config.IDeltaAttributeSet)
            {
                parseDeltaAttributeSet((com.sonicsw.mf.common.config.IDeltaAttributeSet) attrValue, new ConfigPathImpl(path).append(attrNames[i]));
            }
            else
            if (attrValue instanceof com.sonicsw.mf.common.config.IDeltaAttributeList)
            {
                // REVISIT: For now any change to attribute list is recognized as a
                // change of entire list.
                m_newAttributeNames.add(new ConfigPathImpl(path).append(attrNames[i]));
            }
            else // New attribute list, map or atomic value
            {
                m_newAttributeNames.add(new ConfigPathImpl(path).append(attrNames[i]));
            }
        }

        // Add paths for modified attributes.
        attrNames = attrSet.getModifiedAttributesNames();
        for(int i = 0; i < attrNames.length; i++)
        {
            Object attrValue = null;
            try
            {
                attrValue = attrSet.getNewValue(attrNames[i]);
            }
            catch (Exception e)
            {
                continue;
            }

            if (attrValue instanceof com.sonicsw.mf.common.config.IDeltaAttributeSet)
            {
                parseDeltaAttributeSet((com.sonicsw.mf.common.config.IDeltaAttributeSet) attrValue, new ConfigPathImpl(path).append(attrNames[i]));
            }
            else
            if (attrValue instanceof com.sonicsw.mf.common.config.IDeltaAttributeList)
            {
                // REVISIT: For now any change to attribute list is recognized as a
                // change of entire list.
                m_modifiedAttributeNames.add(new ConfigPathImpl(path).append(attrNames[i]));
            }
            else
            if(attrValue instanceof com.sonicsw.mf.common.config.IAttributeSet)
            {
                parseAttributeSet((com.sonicsw.mf.common.config.IAttributeSet) attrValue, new ConfigPathImpl(path).append(attrNames[i]));
            }
            else // modified attribute list or atomic value
            {
                m_modifiedAttributeNames.add(new ConfigPathImpl(path).append(attrNames[i]));
            }
        }
    }

    protected void
    parseAttributeSet(com.sonicsw.mf.common.config.IAttributeSet attrSet, IConfigPath path)
    {
        Iterator it = attrSet.getAttributes().keySet().iterator();
        while (it.hasNext())
        {
            String attrName = (String) it.next();
            Object attrValue = attrSet.getAttribute(attrName);
            if (attrValue instanceof com.sonicsw.mf.common.config.IAttributeSet)
            {
                parseAttributeSet((com.sonicsw.mf.common.config.IAttributeSet) attrValue, new ConfigPathImpl(path).append(attrName));
            }
            else
            {
                m_newAttributeNames.add(new ConfigPathImpl(path).append(attrName));
            }
        }

    }

    @Override
    public String
    toString()
    {
        StringBuffer buffer = new StringBuffer();
        buffer.append("IBasicElement:\n");
        buffer.append("{\n");

        buffer.append(m_name +  " : " + m_version + "  [" + m_elementType + "] =\n");

        buffer.append("  NewAttributeNames:\n");
        buffer.append("  {\n");
        Iterator it = m_newAttributeNames.iterator();
        while (it.hasNext())
        {
            buffer.append("    ").append(it.next()).append("\n");
        }
        buffer.append("  }\n");

        buffer.append("  ModifiedAttributeNames:\n");
        buffer.append("  {\n");
        it = m_modifiedAttributeNames.iterator();
        while (it.hasNext())
        {
            buffer.append("    ").append(it.next()).append("\n");
        }
        buffer.append("  }\n");

        buffer.append("  DeletedAttributeNames:\n");
        buffer.append("  {\n");
        it = m_deletedAttributeNames.iterator();
        while (it.hasNext())
        {
            buffer.append("    ").append(it.next()).append("\n");
        }
        buffer.append("  }\n");

        buffer.append("}\n");
        return buffer.toString();
    }

}