package com.sonicsw.mx.config.tools;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.Writer;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.StringTokenizer;

import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.IAttributeDescription;
import com.sonicsw.mx.config.IAttributeList;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IConfigElement;
import com.sonicsw.mx.config.IConfigType;
import com.sonicsw.mx.config.util.ConfigHelper;

public class GenerateBean extends ConfigServerUtility
{
    private static final String DEFAULT_URL       = "localhost";
    private static final String DEFAULT_USER      = "Administrator";
    private static final String DEFAULT_PASSWORD  = "Administrator";            //NOSONAR field change is not required.
    private static final String DEFAULT_DOMAIN    = "Domain1";
    private static final String DEFAULT_PACKAGE   = "com.sonicsw.mf.mgmtapi.config";
    private static final String DEFAULT_PREFIX    = "MF";

    private ArrayList m_argList   = new ArrayList();
    private String    m_url       = DEFAULT_URL;
    private String    m_user      = DEFAULT_USER;
    private String    m_password  = DEFAULT_PASSWORD;
    private String    m_domain    = DEFAULT_DOMAIN;
    private String    m_facPrefix = DEFAULT_PREFIX;
    private String    m_package   = DEFAULT_PACKAGE;
    private String    m_output    = null;
    private boolean   m_impl      = false;
    private boolean   m_factory   = false;

    private PrintWriter m_acWriter;  // Writer for the abstract class
    private PrintWriter m_aiWriter;  // Writer for the abstract interface
    private PrintWriter m_ciWriter;  // Writer for the constants interface
    private PrintWriter m_ucWriter;  // Writer for the user class
    private PrintWriter m_uiWriter;  // Writer for the user interface
    private PrintWriter m_fiWriter;  // Writer for the factory interface

    private ArrayList m_innerClasses = new ArrayList();
    private Set m_writtenConstants = new HashSet();

    public GenerateBean(String[] args) throws Exception
    {
        try
        {
            parseArgs(args);
        }
        catch(Exception e)
        {
            printUsage();

            throw e;
        }

        System.out.println("ConfigBean Generator");
        System.out.println();
        System.out.println("Output   : " + m_output);
        System.out.println("Package  : " + m_package);

        if (m_facPrefix != null)
        {
            System.out.println("Factory  : " + m_facPrefix + "MgmtBeanFactory");
        }

        System.out.println("Generate : " + (String)((m_impl) ? "Interfaces & Implementation" : "Interfaces ONLY"));
        System.out.println();
        System.out.println("Connection (Broker URL=" + m_url + ", User=" + m_user + ", Password=" + m_password + ", Domain=" + m_domain + ")");
        System.out.println();

        connect(m_domain, m_url, m_user, m_password, false);
    }

    public void generate()
    throws Exception
    {
        // Can only generate the factory classes if we're generating from a
        // properties file, hence the factory argument can only be set through
        // the properties file.
        //
        if (m_factory)
        {
            String fiFilename = m_output + File.separator + "gen" + File.separator + "Abstract" + m_facPrefix + "MgmtBeanFactory.java";

            m_fiWriter = new PrintWriter(new FileWriter(fiFilename));

            generateFactoryInterface();
        }

        for (int i = 0; i < m_argList.size(); i++)
        {
            generate((GenerateArg)m_argList.get(i));

            m_innerClasses.clear();
            m_writtenConstants.clear();
        }

        if (m_factory)
        {
            println(m_fiWriter, "}");

            closeWriter(m_fiWriter);


            if (m_impl)
            {
                String      fcFilename = m_output + File.separator + m_facPrefix + "MgmtBeanFactory.java";
                PrintWriter fcWriter   = new PrintWriter(new FileWriter(fcFilename));

                generateFactoryClass(fcWriter);

                closeWriter(fcWriter);
            }
        }
    }

    public void generate(GenerateArg arg)
    throws Exception
    {
        System.out.print(arg.getName() + " (" +arg.m_type + ", cV=" + arg.m_cVersion + ", pV=" + arg.m_pVersion);

        if (arg.m_template)
        {
            System.out.print(", Template");
        }

        if (arg.m_private)
        {
            System.out.print(", Private");
        }

        System.out.println(")");

        String uiFilename = m_output + File.separator + "I" + arg.getClassName() + "Bean.java";
        String ucFilename = m_output + File.separator + "impl" + File.separator + arg.getClassName() + "Bean.java";
        String aiFilename = m_output + File.separator + "gen" + File.separator + "IAbstract" + arg.getClassName() + "Bean.java";
        String acFilename = m_output + File.separator + "gen" + File.separator + "impl" + File.separator + "Abstract" + arg.getClassName() + "Bean.java";
        String ciFilename = m_output + File.separator + "constants" + File.separator + "I" + arg.getClassName() + "Constants.java";

        m_aiWriter = new PrintWriter(new FileWriter(aiFilename));
        m_acWriter = new PrintWriter(new FileWriter(acFilename));
        m_ciWriter = new PrintWriter(new FileWriter(ciFilename));

        if (m_impl)
        {
            m_ucWriter = new PrintWriter(new FileWriter(ucFilename));
            m_uiWriter = new PrintWriter(new FileWriter(uiFilename));
        }

        IConfigType type = getConfigServer().loadConfigType(arg.m_type, arg.m_cVersion);

        if (type == null)
        {
            throw new Exception("Failed to load config type " + arg.m_type + " " + arg.m_cVersion);
        }

        generateClass(arg, type);
        generateConstants(arg, type);

        if (m_factory && (m_fiWriter != null))
        {
            generateDomain(arg, type, m_fiWriter);
        }

        closeWriter(m_acWriter);
        closeWriter(m_aiWriter);
        closeWriter(m_ciWriter);
        closeWriter(m_ucWriter);
        closeWriter(m_uiWriter);
    }

    private void generateClass(GenerateArg arg, IConfigType type)
    throws Exception
    {
        // Generate the header for the abstract interface file.
        //
        printCopywrite(m_aiWriter);
        println(m_aiWriter, "package " + m_package + ".gen;");
        println(m_aiWriter);
        println(m_aiWriter, "import com.sonicsw.ma.mgmtapi.config.*;");
        println(m_aiWriter, "import " + m_package + ".*;");
        println(m_aiWriter);
        println(m_aiWriter, "/**");
        println(m_aiWriter, " * Configuration Bean representing a '" + arg.getName() + "'");
        if (arg.getClassName().equals("GenericComponent"))
        {
            println(m_aiWriter, " * @deprecated");
        }
        println(m_aiWriter, " */");
        println(m_aiWriter, "public interface IAbstract" + arg.getClassName() + "Bean extends IMgmtBeanBase");
        println(m_aiWriter, "{");

        // Generate the header for the implementation interface file.
        //
        if(m_uiWriter != null)
        {
            printCopywrite(m_uiWriter);
            println(m_uiWriter, "package " + m_package + ";");
            println(m_uiWriter);
            println(m_uiWriter, "import " + m_package + ".gen.*;");
            println(m_uiWriter);
            println(m_uiWriter, "/**");
            println(m_uiWriter, " * Configuration Bean representing a '" + arg.getName() + "'");
            if (arg.getClassName().equals("GenericComponent"))
            {
                println(m_uiWriter, " * @deprecated");
            }
            println(m_uiWriter, " */");
            println(m_uiWriter, "public interface I" + arg.getClassName() + "Bean extends IAbstract" + arg.getClassName() + "Bean");
            println(m_uiWriter, "{");
        }

        // Generate the header for the abstract class file.
        //
        printCopywrite(m_acWriter);
        println(m_acWriter, "package " + m_package + ".gen.impl;");
        println(m_acWriter);
        println(m_acWriter, "import com.sonicsw.ma.mgmtapi.config.*;");
        println(m_acWriter, "import com.sonicsw.ma.mgmtapi.config.impl.*;");
        println(m_acWriter, "import " + m_package + ".*;");
        println(m_acWriter, "import " + m_package + ".impl.*;");
        println(m_acWriter, "import " + m_package + ".constants.*;");
        println(m_acWriter, "import " + m_package + ".gen.*;");
        println(m_acWriter);
        println(m_acWriter, "public abstract class Abstract" + arg.getClassName() + "Bean");
        println(m_acWriter, "extends MgmtBeanBase implements I" + arg.getClassName() + "Constants, I" + arg.getClassName() + "Bean");
        println(m_acWriter, "{");
        println(m_acWriter, "    public Abstract" + arg.getClassName() + "Bean(MgmtBeanFactory factory)");
        println(m_acWriter, "    {");
        println(m_acWriter, "        super(factory);");
        println(m_acWriter, "    }");

        // Generate the header for the implementation class file.
        //
        if(m_ucWriter != null)
        {
            printCopywrite(m_ucWriter);
            println(m_ucWriter, "package " + m_package + ".impl;");
            println(m_ucWriter);
            println(m_ucWriter, "import com.sonicsw.ma.mgmtapi.config.*;");
            println(m_ucWriter, "import " + m_package + ".gen.impl.*;");
            println(m_ucWriter, "import " + m_package + ".*;");
            println(m_ucWriter);
            println(m_ucWriter, "public class " + arg.getClassName() + "Bean");
            println(m_ucWriter, "extends Abstract" + arg.getClassName() + "Bean implements I" + arg.getClassName() + "Bean");
            println(m_ucWriter, "{");
            println(m_ucWriter, "    public " + arg.getClassName() + "Bean(MgmtBeanFactory factory)");
            println(m_ucWriter, "    {");
            println(m_ucWriter, "        super(factory);");
            println(m_ucWriter, "    }");
        }

        int indent = 1;

        for(Iterator iter = type.getAttributeDescriptionNames().iterator(); iter.hasNext(); )
        {
            String key = (String)iter.next();
            IAttributeDescription val = type.getAttributeDescription(key);

            generateAttribute(arg.getClassName(), indent, key, val);
        }

        // now iterator through the m_innerClasses until there is nothing left
        while(!m_innerClasses.isEmpty())
        {
            LocalInnerClass innerClass = (LocalInnerClass)m_innerClasses.get(0);
            m_innerClasses.remove(0);

            generateInnerClass(arg.getClassName(), indent, innerClass.m_name, innerClass.m_type, innerClass.m_desc);
        }

        println(m_aiWriter, "}");
        println(m_acWriter, "}");
        if(m_uiWriter != null)
        {
            println(m_uiWriter, "}");
        }
        if(m_ucWriter != null)
        {
            println(m_ucWriter, "}");
        }
    }

    private void generateConstants(GenerateArg arg, IConfigType type)
    throws Exception
    {
        // Generate the header for the contsants file.
        //
        printCopywrite(m_ciWriter);
        println(m_ciWriter, "/**");
        println(m_ciWriter, " * This generated file contains all constants for the configuration type");
        println(m_ciWriter, " * " + arg.m_type + " (v" + arg.m_cVersion + ")");
        println(m_ciWriter, " * This file is automatically generated. Do not edit manually.");
        if (arg.getClassName().equals("GenericComponent"))
        {
            println(m_ciWriter, " * @deprecated");
        }
        println(m_ciWriter, " */");
        println(m_ciWriter, "package " + m_package + ".constants;");
        println(m_ciWriter);
        println(m_ciWriter, "public interface I" + arg.getClassName() + "Constants");
        println(m_ciWriter, "{");

        // Nice to output the underlying type and version of the constants file here...
        //
        println(m_ciWriter, 1, "public static final String DS_TYPE      = \"" + arg.m_type + "\";");
        println(m_ciWriter, 1, "public static final String DS_C_VERSION = \"" + arg.m_cVersion + "\";");
        println(m_ciWriter, 1, "public static final String DS_P_VERSION = \"" + arg.m_pVersion + "\";");
        println(m_ciWriter);

        for(Iterator iter = type.getAttributeDescriptionNames().iterator(); iter.hasNext(); )
        {
            String key = (String)iter.next();
            Object val = type.getAttributeDescription(key);

            generateAttributeNameConstant(key, (IAttributeDescription)val);
        }
        println(m_ciWriter, "}");
    }

    private void generateDomain(GenerateArg arg, IConfigType type, PrintWriter cWriter)
    throws Exception
    {
        // Generate the class file.
        //
        println(cWriter, 1, "// -----------------------------------------------------------------------");
        println(cWriter, 1, "// Domain methods for " + arg.getClassName() + "Bean");
        println(cWriter, 1, "//");
        println(cWriter, 1, "private void register" + arg.getClassName() + "Bean()");
        println(cWriter, 1, "{");
        println(cWriter, 1, "    HashMap map = new HashMap();");
        println(cWriter);
        println(cWriter, 1, "    map.put(IMgmtBeanBase.TYPE_PROPERTY,      " + arg.getClassName() + "Bean.DS_TYPE);");
        println(cWriter, 1, "    map.put(IMgmtBeanBase.C_VERSION_PROPERTY, " + arg.getClassName() + "Bean.DS_C_VERSION);");
        println(cWriter, 1, "    map.put(IMgmtBeanBase.P_VERSION_PROPERTY, " + arg.getClassName() + "Bean.DS_P_VERSION);");
        println(cWriter, 1, "    map.put(IMgmtBeanBase.CLASS_PROPERTY,     " + arg.getClassName() + "Bean.class);");
        println(cWriter, 1, "    map.put(IMgmtBeanBase.TEMPLATE_PROPERTY,  " + booleanObjectString(arg.m_template) + ");");
        println(cWriter);
        println(cWriter, 1, "    registerBean(map);");
        println(cWriter, 1, "}");
        println(cWriter);

        if (!arg.m_private)
        {
            println(cWriter, 1, "/**");
            println(cWriter, 1, " * Get the names of all beans of type I" + arg.getClassName());
            println(cWriter, 1, " * @return an array of bean names");
            if (arg.getClassName().equals("GenericComponent"))
            {
                println(cWriter, 1, " * @deprecated");
            }
            println(cWriter, 1, " */");
            println(cWriter, 1, "public List get" + arg.getClassName() + "BeanNames() throws MgmtException");
            println(cWriter, 1, "{");
            println(cWriter, 1, "    return listConfigElements(" + arg.getClassName() + "Bean.DS_TYPE);");
            println(cWriter, 1, "}");
            println(cWriter);
            println(cWriter, 1, "/**");
            println(cWriter, 1, " * Get the '" + arg.getName() + "' bean with the specified name");
            println(cWriter, 1, " * @param name The name of the bean");
            println(cWriter, 1, " * @return a bean of type I" + arg.getClassName());
            if (arg.getClassName().equals("GenericComponent"))
            {
                println(cWriter, 1, " * @deprecated");
            }
            println(cWriter, 1, " */");
            println(cWriter, 1, "public I" + arg.getClassName() + "Bean get" + arg.getClassName() + "Bean(String name) throws MgmtException");
            println(cWriter, 1, "{");
            println(cWriter, 1, "    " + arg.getClassName() + "Bean bean = new " + arg.getClassName() + "Bean(this);");
            println(cWriter);
            println(cWriter, 1, "    bean.loadBean(name, " + arg.getClassName() + "Bean.DS_TYPE);");
            println(cWriter);
            println(cWriter, 1, "    return bean;");
            println(cWriter, 1, "}");
            println(cWriter);
            println(cWriter, 1, "/**");
            println(cWriter, 1, " * Create a " + arg.getName() + " bean with the specified name");
            println(cWriter, 1, " * @param name the name of the new bean");
            println(cWriter, 1, " * @return a new " + arg.getName() + " bean");
            if (arg.getClassName().equals("GenericComponent"))
            {
                println(cWriter, 1, " * @deprecated");
            }
            println(cWriter, 1, " */");
            println(cWriter, 1, "public I" + arg.getClassName() + "Bean create" + arg.getClassName() + "Bean(String name) throws MgmtException");
            println(cWriter, 1, "{");
            println(cWriter, 1, "    " + arg.getClassName() + "Bean bean = new " + arg.getClassName() + "Bean(this);");
            println(cWriter);
            println(cWriter, 1, "    bean.createBean(name, " + arg.getClassName() + "Bean.DS_TYPE, " + arg.getClassName() + "Bean.DS_C_VERSION, " + arg.getClassName() + "Bean.DS_P_VERSION);");
            println(cWriter);
            println(cWriter, 1, "    return bean;");
            println(cWriter, 1, "}");
            println(cWriter);
            println(cWriter, 1, "/**");
            println(cWriter, 1, " * Save the specified " + arg.getName() + " bean");
            println(cWriter, 1, " * @param bean the " + arg.getName() + " bean to save");
            if (arg.getClassName().equals("GenericComponent"))
            {
                println(cWriter, 1, " * @deprecated");
            }
            println(cWriter, 1, " */");
            println(cWriter, 1, "public void save" + arg.getClassName() + "Bean(I" + arg.getClassName() + "Bean bean) throws MgmtException");
            println(cWriter, 1, "{");
            println(cWriter, 1, "    saveBean(bean);");
            println(cWriter, 1, "}");
            println(cWriter);
            println(cWriter, 1, "/**");
            println(cWriter, 1, " * Delete the specified " + arg.getName() + " bean");
            println(cWriter, 1, " * @param bean the " + arg.getName() + " bean to delete");
            if (arg.getClassName().equals("GenericComponent"))
            {
                println(cWriter, 1, " * @deprecated");
            }
            println(cWriter, 1, " */");
            println(cWriter, 1, "public void delete" + arg.getClassName() + "Bean(I" + arg.getClassName() + "Bean bean) throws MgmtException");
            println(cWriter, 1, "{");
            println(cWriter, 1, "    deleteBean(bean);");
            println(cWriter, 1, "}");
        }

        println(cWriter);
    }

    private void generateInnerClassNameConstants(String name, String type, IAttributeDescription desc)
    throws Exception
    {
        Set names = desc.getAttributeDescriptionNames();

        if (!names.isEmpty())
        {
            println(m_ciWriter);
            println(m_ciWriter, 1, "// " + name + " attribute set/list constants and defaults");
        }

        for(Iterator iter = names.iterator(); iter.hasNext(); )
        {
            String key = (String)iter.next();
            IAttributeDescription val = desc.getAttributeDescription(key);

            generateAttributeNameConstant(key, val);
        }

        if (!names.isEmpty())
        {
            println(m_ciWriter);
        }
    }

    private void generateInnerClass(String outer, int indent, String name, String type, IAttributeDescription desc)
    throws Exception
    {
        boolean unenumMap = false;
        boolean list      = false;

        if(desc.getProperty(IAttributeDescription.UNENUM_MAP_PROPERTY) != null)
        {
            unenumMap = true;
        }

        if(desc.getType() == IAttributeList.class)
        {
            list = true;
        }

        // Generate the header for the interface file.
        //
        println(m_aiWriter);

        if(m_uiWriter != null)
        {
            println(m_uiWriter);
        }

        if(unenumMap)
        {
            println(m_aiWriter, indent, "/**");
            println(m_aiWriter, indent, " * Inner class representing " + type);
            println(m_aiWriter, indent, " * This inner class contains inner data items that are keyed off");
            println(m_aiWriter, indent, " * a name. This class extends IMgmtMapBase");
            println(m_aiWriter, indent, " */");
            println(m_aiWriter, indent, "public interface IAbstract" + type + " extends IMgmtMapBase");

            if(m_uiWriter != null)
            {
                println(m_uiWriter, indent, "/**");
                println(m_uiWriter, indent, " * Inner class representing " + type);
                println(m_uiWriter, indent, " * This inner class contains inner data items that are keyed off");
                println(m_uiWriter, indent, " * a name. This class extends IAbstract" + type);
                println(m_uiWriter, indent, " */");
                println(m_uiWriter, indent, "public interface I" + type + " extends IAbstract" + type);
            }
        }
        else if(list)
        {
            println(m_aiWriter, indent, "/**");
            println(m_aiWriter, indent, " * Inner class representing " + type);
            println(m_aiWriter, indent, " * This inner class contains inner data items that have a zero based");
            println(m_aiWriter, indent, " * index. This class extends IMgmtListBase");
            println(m_aiWriter, indent, " */");
            println(m_aiWriter, indent, "public interface IAbstract" + type + " extends IMgmtListBase");

            if(m_uiWriter != null)
            {
                println(m_uiWriter, indent, "/**");
                println(m_uiWriter, indent, " * Inner class representing " + type);
                println(m_uiWriter, indent, " * This inner class contains inner data items that have a zero based");
                println(m_uiWriter, indent, " * index. This class extends IAbstract" + type);
                println(m_uiWriter, indent, " */");
                println(m_uiWriter, indent, "public interface I" + type + " extends IAbstract" + type);
            }
        }
        else
        {
            println(m_aiWriter, indent, "/**");
            println(m_aiWriter, indent, " * Inner class representing " + type);
            println(m_aiWriter, indent, " * This class extends IMgmtListBase");
            println(m_aiWriter, indent, " */");
            println(m_aiWriter, indent, "public interface IAbstract" + type + " extends IMgmtSubBeanBase");

            if(m_uiWriter != null)
            {
                println(m_uiWriter, indent, "/**");
                println(m_uiWriter, indent, " * Inner class representing " + type);
                println(m_uiWriter, indent, " * This class extends IAbstract" + type);
                println(m_uiWriter, indent, " */");
                println(m_uiWriter, indent, "public interface I" + type + " extends IAbstract" + type);
            }
        }
        println(m_aiWriter, indent, "{");

        if(m_uiWriter != null)
        {
            println(m_uiWriter, indent, "{");
            println(m_uiWriter, indent, "}");
        }

        // Generate the header for the class file.
        //
        println(m_acWriter);
        println(m_acWriter, indent, "public static class Abstract" + type);
        if(unenumMap)
        {
            println(m_acWriter, indent, "extends MgmtMapBase implements IAbstract" + type);
        }
        else if(list)
        {
            println(m_acWriter, indent, "extends MgmtListBase implements IAbstract" + type);
        }
        else
        {
            println(m_acWriter, indent, "extends MgmtSubBeanBase implements IAbstract" + type);
        }
        println(m_acWriter, indent, "{");
        println(m_acWriter, indent, "    public Abstract" + type + "(IMgmtBase parent, String name, boolean isNew)");
        println(m_acWriter, indent, "    throws MgmtException");
        println(m_acWriter, indent, "    {");
        println(m_acWriter, indent, "        super(parent, name, isNew);");
        println(m_acWriter, indent, "    }");

        if(m_ucWriter != null)
        {
            println(m_ucWriter);
            println(m_ucWriter, indent, "public static class " + type);
            if(unenumMap)
            {
                println(m_ucWriter, indent, "extends Abstract" + type + " implements I" + type);
            }
            else if(list)
            {
                println(m_ucWriter, indent, "extends Abstract" + type + " implements I" + type);
            }
            else
            {
                println(m_ucWriter, indent, "extends Abstract" + type + " implements I" + type);
            }
            println(m_ucWriter, indent, "{");
            println(m_ucWriter, indent, "    public " + type + "(IMgmtBase parent, String name, boolean isNew)");
            println(m_ucWriter, indent, "    throws MgmtException");
            println(m_ucWriter, indent, "    {");
            println(m_ucWriter, indent, "        super(parent, name, isNew);");
            println(m_ucWriter, indent, "    }");
            println(m_ucWriter, indent, "}");
        }

        Set names = desc.getAttributeDescriptionNames();
        for(Iterator iter = names.iterator(); iter.hasNext(); )
        {
            String key = (String)iter.next();
            IAttributeDescription val = desc.getAttributeDescription(key);

            if(unenumMap)
            {
                generateUnenumMapAttribute(outer, indent + 1, key, val);
            }
            else if(list)
            {
                generateListAttribute(outer, indent + 1, key, val);
            }
            else
            {
                generateAttribute(outer, indent + 1, key, val);
            }
        }

        println(m_aiWriter, indent, "}");
        println(m_acWriter, indent, "}");
    }

    private void generateAttributeNameConstant(String key, IAttributeDescription desc)
    throws Exception
    {
        if(key.startsWith("_GUI"))
        {
            return;
        }

        String constantDefinition = "public static final String " + key.toUpperCase() + "_ATTR = " + '"' + key + "\";";
        String defaultDefinition  = generateDefaultDefinition(key, desc);
        String fixedDefinition    = generateFixedDefinition(key, desc);

        if (m_writtenConstants.contains(constantDefinition))
        {
            println(m_ciWriter, 1, "// " + constantDefinition);
            if(defaultDefinition != null)
            {
                println(m_ciWriter, 1, "// " + defaultDefinition);
            }
            if(fixedDefinition != null)
            {
                println(m_ciWriter, 1, "// " + fixedDefinition);
            }
        }
        else
        {
            println(m_ciWriter, 1, constantDefinition);
            if(defaultDefinition != null)
            {
                println(m_ciWriter, 1, defaultDefinition);
            }
            if(fixedDefinition != null)
            {
                println(m_ciWriter, 1, fixedDefinition);
            }
            m_writtenConstants.add(constantDefinition);

            generateEnumDefinition(1, key, desc);
        }

        if(desc.getType() == IAttributeMap.class || desc.getType() == IAttributeList.class)
        {
            String schemaName = (String)desc.getProperty(IAttributeDescription.SCHEMA_TYPE_NAME);

            if(schemaName == null)
            {
                throw new Exception("SCHEMA_TYPE_NAME not available for " + key);
            }

            String type = ConfigHelper.generateName(schemaName);

            if(type == null)
            {
                return;
            }

            LocalInnerClass innerClass = new LocalInnerClass(key, type, desc);
            generateInnerClassNameConstants(innerClass.m_name, innerClass.m_type, innerClass.m_desc);
        }
    }

    private void generateAttribute(String outer, int indent, String key, IAttributeDescription desc)
    throws Exception
    {
        String useName   = ConfigHelper.generateName(key);
        String classDesc = classToDescription(desc.getType());
        String classType = classToType(desc.getType());

        if(key.startsWith("_GUI") || isHidden(desc))
        {
            return;
        }

        if(desc.getType() == IAttributeMap.class)
        {
            generateMapAttribute(outer, indent, key, useName, desc);
        }
        else if(desc.getType() == IAttributeList.class)
        {
            generateMapAttribute(outer, indent, key, useName, desc);
        }
        else if(classDesc != null && classType != null)
        {
            generateSimpleAttribute(indent, key, useName, desc, classDesc, classType);
        }
        else
        {
            throw new Exception("Failed to generate code for Attribute " + key + " of type " + desc.getType().getName());
        }
    }

    private void generateSimpleAttribute(int indent, String key, String useName, IAttributeDescription desc, String method, String type)
    {
        boolean optional = isOptional(desc);
        boolean settable = isSettable(desc);

        // Writer the interface
        println(m_aiWriter);
        println(m_aiWriter, indent, "/**");
        println(m_aiWriter, indent, " * Get the attribute for " + key + " as a " + type + " value.");
        String annotation = (String)desc.getProperty(IAttributeDescription.SCHEMA_ANNOTATION);
        if (annotation != null)
        {
            printAnnotation(indent, annotation);
        }
        if(optional)
        {
            println(m_aiWriter, indent, " * This is an optional attribute and a MgmtException will be");
            println(m_aiWriter, indent, " * thrown if the attribute does not exist.");
        }
        println(m_aiWriter, indent, " * @return The " + type + " value for this attribute");
        println(m_aiWriter, indent, " * @throws MgmtException Thrown when this attribute doesn't exist or");
        println(m_aiWriter, indent, " * if there is some problem obtaining this attribute");
        println(m_aiWriter, indent, " */");
        println(m_aiWriter, indent, "public " + type + " get" + useName + "() throws MgmtException;");
        if(optional)
        {
            println(m_aiWriter);
            println(m_aiWriter, indent, "/**");
            println(m_aiWriter, indent, " * Get the attribute for " + key + " as a " + type + " value.");
            if (annotation != null)
            {
                printAnnotation(indent, annotation);
            }
            println(m_aiWriter, indent, " * If the attribute doesn't exist then return the specified default value.");
            println(m_aiWriter, indent, " * @param defValue The default value to return if this attribute doesn't exist");
            println(m_aiWriter, indent, " * @return The " + type + " value for this attribute");
            println(m_aiWriter, indent, " * @throws MgmtException Thrown when this attribute doesn't exist or");
            println(m_aiWriter, indent, " * if there is some problem obtaining this attribute");
            println(m_aiWriter, indent, " */");
            println(m_aiWriter, indent, "public " + type + " get" + useName + "(" + type + " defValue) throws MgmtException;");
        }
        if(settable)
        {
            println(m_aiWriter);
            println(m_aiWriter, indent, "/**");
            println(m_aiWriter, indent, " * Set the attribute for " + key + " to the specified " + type + " value.");
            if (annotation != null)
            {
                printAnnotation(indent, annotation);
            }
            println(m_aiWriter, indent, " * @param value The " + type + " value for this attribute");
            println(m_aiWriter, indent, " * @throws MgmtException Thrown if there is some problem setting this attribute");
            println(m_aiWriter, indent, " */");
            println(m_aiWriter, indent, "public void set" + useName + "(" + type + " value) throws MgmtException;");
        }

        println(m_aiWriter);
        println(m_aiWriter, indent, "/**");
        println(m_aiWriter, indent, " * Get the attribute metadata for " + key + ".");
        if (annotation != null)
        {
            printAnnotation(indent, annotation);
        }
        println(m_aiWriter, indent, " * @return The IMgmtAttributeMetaData value for this attribute");
        println(m_aiWriter, indent, " * @throws MgmtException Thrown when this attribute doesn't exist or");
        println(m_aiWriter, indent, " * if there is some problem obtaining this attribute's metadata");
        println(m_aiWriter, indent, " */");
        println(m_aiWriter, indent, "public IMgmtAttributeMetaData get" + useName + "MetaData() throws MgmtException;");

        // Writer the implementation
        println(m_acWriter);
        println(m_acWriter, indent, "public " + type + " get" + useName + "() throws MgmtException");
        println(m_acWriter, indent, "{");
        println(m_acWriter, indent, "    return get" + method + "Attribute(" + key.toUpperCase() + "_ATTR);");
        println(m_acWriter, indent, "}");
        if(optional)
        {
            println(m_acWriter);
            println(m_acWriter, indent, "public " + type + " get" + useName + "(" + type + " defValue) throws MgmtException");
            println(m_acWriter, indent, "{");
            println(m_acWriter, indent, "    try");
            println(m_acWriter, indent, "    {");
            println(m_acWriter, indent, "        return get" + method + "Attribute(" + key.toUpperCase() + "_ATTR);");
            println(m_acWriter, indent, "    }");
            println(m_acWriter, indent, "    catch(AttributeNotFoundException e)");
            println(m_acWriter, indent, "    {");
            println(m_acWriter, indent, "        return defValue;");
            println(m_acWriter, indent, "    }");
            println(m_acWriter, indent, "}");
        }
        if(settable)
        {
            println(m_acWriter);
            println(m_acWriter, indent, "public void set" + useName + "(" + type + " value) throws MgmtException");
            println(m_acWriter, indent, "{");
            println(m_acWriter, indent, "    set" + method + "Attribute(" + key.toUpperCase() + "_ATTR, value);");
            println(m_acWriter, indent, "}");
        }

        println(m_acWriter);
        println(m_acWriter, indent, "public IMgmtAttributeMetaData get" + useName + "MetaData() throws MgmtException");
        println(m_acWriter, indent, "{");
        println(m_acWriter, indent, "    return getAttributeMetaData(" + key.toUpperCase() + "_ATTR);");
        println(m_acWriter, indent, "}");
    }

    private void generateMapAttribute(String outer, int indent, String key, String useName, IAttributeDescription desc)
    throws Exception
    {
        String schemaName = (String)desc.getProperty(IAttributeDescription.SCHEMA_TYPE_NAME);

        if(schemaName == null)
        {
            throw new Exception("SCHEMA_TYPE_NAME not available for " + key);
        }

        String type = ConfigHelper.generateName(schemaName);

        // Writer the interface
        println(m_aiWriter);
        println(m_aiWriter, indent, "/**");
        println(m_aiWriter, indent, " * Get the attribute for " + key + ".");

        String annotation = (String)desc.getProperty(IAttributeDescription.SCHEMA_ANNOTATION);
        if (annotation != null)
        {
            printAnnotation(indent, annotation);
        }

        println(m_aiWriter, indent, " * @return The I" + type + " value for this attribute");
        println(m_aiWriter, indent, " * @throws MgmtException Thrown when this attribute doesn't exist or");
        println(m_aiWriter, indent, " * if there is some problem obtaining this attribute");
        println(m_aiWriter, indent, " */");
        println(m_aiWriter, indent, "public I" + outer + "Bean.I" + type + " get" + useName + "() throws MgmtException;");
        if(isSettable(desc))
        {
            println(m_aiWriter);
            println(m_aiWriter, indent, "/**");
            println(m_aiWriter, indent, " * Set the attribute for " + key + " to the specified I" + type + " value.");

            annotation = (String)desc.getProperty(IAttributeDescription.SCHEMA_ANNOTATION);
            if (annotation != null)
            {
                printAnnotation(indent, annotation);
            }

            println(m_aiWriter, indent, " * @param value The I" + type + " value for this attribute");
            println(m_aiWriter, indent, " * @throws MgmtException Thrown if there is some problem setting this attribute");
            println(m_aiWriter, indent, " */");
            println(m_aiWriter, indent, "public void set" + useName + "(I" + outer + "Bean.I" + type + " value) throws MgmtException;");
            println(m_aiWriter);
            println(m_aiWriter, indent, "/**");
            println(m_aiWriter, indent, " * Create an instance of an I" + type + " attribute. This new instance");
            println(m_aiWriter, indent, " * is not currently part of the configuration. Once it has been configured");
            println(m_aiWriter, indent, " * it should be set into the configuration using the set" + useName + " method.");
            println(m_aiWriter, indent, " * @return The new I" + type + " bean");
            println(m_aiWriter, indent, " * @throws MgmtException Thrown if there is some problem creating this object");
            println(m_aiWriter, indent, " */");
            println(m_aiWriter, indent, "public I" + outer + "Bean.I" + type + " create" + useName + "() throws MgmtException;");
        }

        println(m_aiWriter);
        println(m_aiWriter, indent, "/**");
        println(m_aiWriter, indent, " * Get the attribute metadata for " + key + ".");
        println(m_aiWriter, indent, " * @return The IMgmtAttributeMetaData value for this attribute");
        println(m_aiWriter, indent, " * @throws MgmtException Thrown when this attribute metadata doesn't exist or");
        println(m_aiWriter, indent, " * if there is some problem obtaining this attribute metadata");
        println(m_aiWriter, indent, " */");
        println(m_aiWriter, indent, "public IMgmtAttributeMetaData get" + useName + "MetaData() throws MgmtException;");

        // Writer the implementation
        println(m_acWriter);
        println(m_acWriter, indent, "public I" + outer + "Bean.I" + type + " get" + useName + "() throws MgmtException");
        println(m_acWriter, indent, "{");
        println(m_acWriter, indent, "    return new " + outer + "Bean." + type + "(this, " + key.toUpperCase() + "_ATTR, false);");
        println(m_acWriter, indent, "}");
        if(isSettable(desc))
        {
            println(m_acWriter);
            println(m_acWriter, indent, "public void set" + useName + "(I" + outer + "Bean.I" + type + " value) throws MgmtException");
            println(m_acWriter, indent, "{");
            println(m_acWriter, indent, "    super.setAttribute(" + key.toUpperCase() + "_ATTR, value);");
            println(m_acWriter, indent, "}");
            println(m_acWriter);
            println(m_acWriter, indent, "public I" + outer + "Bean.I" + type + " create" + useName + "() throws MgmtException");
            println(m_acWriter, indent, "{");
            println(m_acWriter, indent, "    return new " + outer + "Bean." + type + "(this, " + key.toUpperCase() + "_ATTR, true);");
            println(m_acWriter, indent, "}");
        }
        println(m_acWriter);
        println(m_acWriter, indent, "public IMgmtAttributeMetaData get" + useName + "MetaData() throws MgmtException");
        println(m_acWriter, indent, "{");
        println(m_acWriter, indent, "    return getAttributeMetaData(" + key.toUpperCase() + "_ATTR);");
        println(m_acWriter, indent, "}");

        addInnerClass(key, type, desc);
    }

    private void generateListAttribute(String outer, int indent, String key, IAttributeDescription desc)
    throws Exception
    {
        String schemaName = (String)desc.getProperty(IAttributeDescription.SCHEMA_TYPE_NAME);

        if(schemaName == null)
        {
            throw new Exception("SCHEMA_TYPE_NAME not available for " + key);
        }

        String internalType = null;
        String type = null;

        if((String)desc.getProperty(IAttributeDescription.SIMPLE_TYPE_NAME) != null)
        {
            type = null;
            internalType = classToType(desc.getType());
        }
        else
        {
            type = ConfigHelper.generateName(schemaName);
            internalType = "I" + outer + "Bean.I" + type;
        }

        String useName = ConfigHelper.generateName(key);

        // Writer the interface
        println(m_aiWriter);
        println(m_aiWriter, indent, "/**");
        println(m_aiWriter, indent, " * Get the " + internalType + " atribute from the list at the");
        println(m_aiWriter, indent, " * specified index");
        println(m_aiWriter, indent, " * @param index The index of the " + useName + " in the list.");
        println(m_aiWriter, indent, " * @return The " + internalType + " value for this attribute");
        println(m_aiWriter, indent, " * @throws MgmtException Thrown when this attribute doesn't exist or");
        println(m_aiWriter, indent, " * if there is some problem obtaining this attribute");
        println(m_aiWriter, indent, " */");
        println(m_aiWriter, indent, "public " + internalType + " get" + useName + "(int index) throws MgmtException;");
        if(type != null)
        {
            println(m_aiWriter);
            println(m_aiWriter, indent, "/**");
            println(m_aiWriter, indent, " * Create an instance of an " + internalType + " attribute. This new instance");
            println(m_aiWriter, indent, " * is not currently part of the configuration. Once it has been configured");
            println(m_aiWriter, indent, " * it should be set into the configuration using either the add" + useName);
            println(m_aiWriter, indent, " * or insert" + useName + " methods");
            println(m_aiWriter, indent, " * @return The new " + internalType + " bean");
            println(m_aiWriter, indent, " * @throws MgmtException Thrown if there is some problem creating this object");
            println(m_aiWriter, indent, " */");
            println(m_aiWriter, indent, "public " + internalType + " create" + useName + "() throws MgmtException;");
        }
        println(m_aiWriter);
        println(m_aiWriter, indent, "/**");
        println(m_aiWriter, indent, " * Add the new " + useName + " of type " + internalType + " into this list at");
        println(m_aiWriter, indent, " * the end of the list");
        println(m_aiWriter, indent, " * @param value The " + internalType + " value to add to the list");
        println(m_aiWriter, indent, " * @throws MgmtException Thrown if there is some problem adding this item to the list");
        println(m_aiWriter, indent, " */");
        println(m_aiWriter, indent, "public void add" + useName + "(" + internalType + " value) throws MgmtException;");
        println(m_aiWriter);
        println(m_aiWriter, indent, "/**");
        println(m_aiWriter, indent, " * Insert the new " + useName + " of type " + internalType + " into this list at");
        println(m_aiWriter, indent, " * the specified location");
        println(m_aiWriter, indent, " * @param index The location in the list to add the new item");
        println(m_aiWriter, indent, " * @param value The " + internalType + " value to add to the list");
        println(m_aiWriter, indent, " * @throws MgmtException Thrown if there is some problem inserting this item into the list");
        println(m_aiWriter, indent, " */");
        println(m_aiWriter, indent, "public void insert" + useName + "(int index, " + internalType + " value) throws MgmtException;");
        println(m_aiWriter);
        println(m_aiWriter, indent, "/**");
        println(m_aiWriter, indent, " * Delete the item at the specified index from the list");
        println(m_aiWriter, indent, " * @param index The index of the item to delete");
        println(m_aiWriter, indent, " * @throws MgmtException Thrown if there is some problem deleting this item from the list");
        println(m_aiWriter, indent, " */");
        println(m_aiWriter, indent, "public void delete" + useName + "(int index) throws MgmtException;");
        println(m_aiWriter);
        println(m_aiWriter, indent, "/**");
        println(m_aiWriter, indent, " * Get the attribute metadata for " + key + ".");
        println(m_aiWriter, indent, " * @return The IMgmtAttributeMetaData value for this attribute");
        println(m_aiWriter, indent, " * @throws MgmtException Thrown when this attribute metadata doesn't exist or");
        println(m_aiWriter, indent, " * if there is some problem obtaining this attribute metadata");
        println(m_aiWriter, indent, " */");
        println(m_aiWriter, indent, "public IMgmtAttributeMetaData get" + useName + "MetaData() throws MgmtException;");

        // Writer the implementation
        println(m_acWriter);
        println(m_acWriter, indent, "public " + internalType + " get" + useName + "(int index) throws MgmtException");
        println(m_acWriter, indent, "{");
        if(type == null)
        {
            println(m_acWriter, indent, "    return (" + internalType + ")super.getAttribute(Integer.toString(index));");
        }
        else
        {
            println(m_acWriter, indent, "    return new " + outer + "Bean." + type + "(this, Integer.toString(index), false);");
        }
        println(m_acWriter, indent, "}");
        if(type != null)
        {
            println(m_acWriter);
            println(m_acWriter, indent, "public " + internalType + " create" + useName + "() throws MgmtException");
            println(m_acWriter, indent, "{");
            println(m_acWriter, indent, "    return new " + outer + "Bean." + type + "(this, " + key.toUpperCase() + "_ATTR, true);");
            println(m_acWriter, indent, "}");
        }
        println(m_acWriter);
        println(m_acWriter, indent, "public void add" + useName + "(" + internalType +" value) throws MgmtException");
        println(m_acWriter, indent, "{");
        println(m_acWriter, indent, "    super.add(value);");
        println(m_acWriter, indent, "}");
        println(m_acWriter);
        println(m_acWriter, indent, "public void insert" + useName + "(int index, " + internalType +" value) throws MgmtException");
        println(m_acWriter, indent, "{");
        println(m_acWriter, indent, "    super.insert(index, value);");
        println(m_acWriter, indent, "}");
        println(m_acWriter);
        println(m_acWriter, indent, "public void delete" + useName + "(int index) throws MgmtException");
        println(m_acWriter, indent, "{");
        println(m_acWriter, indent, "    super.delete(index);");
        println(m_acWriter, indent, "}");
        println(m_acWriter);
        println(m_acWriter, indent, "public IMgmtAttributeMetaData get" + useName + "MetaData() throws MgmtException");
        println(m_acWriter, indent, "{");
        println(m_acWriter, indent, "    return getAttributeMetaData(" + key.toUpperCase() + "_ATTR);");
        println(m_acWriter, indent, "}");

        addInnerClass(key, type, desc);
    }

    private void generateUnenumMapAttribute(String outer, int indent, String key, IAttributeDescription desc)
    throws Exception
    {
        String schemaName = (String)desc.getProperty(IAttributeDescription.SCHEMA_TYPE_NAME);

        if(schemaName == null)
        {
            throw new Exception("SCHEMA_TYPE_NAME not available for " + key);
        }

        String internalType = null;
        String type = null;

        if((String)desc.getProperty(IAttributeDescription.SIMPLE_TYPE_NAME) != null)
        {
            type = null;
            internalType = classToType(desc.getType());
        }
        else
        {
            type = ConfigHelper.generateName(schemaName);
            internalType = "I" + outer + "Bean.I" + type;
        }

        String useName = ConfigHelper.generateName(key);

        // Writer the interface
        println(m_aiWriter);
        println(m_aiWriter, indent, "/**");
        println(m_aiWriter, indent, " * Get the " + internalType + " atribute from the map with the specified key name");
        println(m_aiWriter, indent, " * @param name The key name for this " + useName + " within the Map");
        println(m_aiWriter, indent, " * @return The " + internalType + " value for this attribute");
        println(m_aiWriter, indent, " * @throws MgmtException Thrown when this attribute doesn't exist or");
        println(m_aiWriter, indent, " * if there is some problem obtaining this attribute");
        println(m_aiWriter, indent, " */");
        println(m_aiWriter, indent, "public " + internalType + " get" + useName + "(String name) throws MgmtException;");
        if(type != null)
        {
            println(m_aiWriter);
            println(m_aiWriter, indent, "/**");
            println(m_aiWriter, indent, " * Create an instance of an " + internalType + " attribute. This new instance");
            println(m_aiWriter, indent, " * is not currently part of the configuration. Once it has been configured");
            println(m_aiWriter, indent, " * it should be set into the configuration using either the add" + useName);
            println(m_aiWriter, indent, " * or insert" + useName + " methods");
            println(m_aiWriter, indent, " * @return The new " + internalType + " bean");
            println(m_aiWriter, indent, " * @throws MgmtException Thrown if there is some problem creating this object");
            println(m_aiWriter, indent, " */");
            println(m_aiWriter, indent, "public " + internalType + " create" + useName + "() throws MgmtException;");
        }
        println(m_aiWriter);
        println(m_aiWriter, indent, "/**");
        println(m_aiWriter, indent, " * Add the new " + useName + " of type " + internalType + " into this map keyed");
        println(m_aiWriter, indent, " * off the specified key name");
        println(m_aiWriter, indent, " * @param name The key name for this " + useName + " in the map");
        println(m_aiWriter, indent, " * @param value The " + internalType + " value to add to the map");
        println(m_aiWriter, indent, " * @throws MgmtException Thrown if there is some problem adding this item to the map");
        println(m_aiWriter, indent, " */");
        println(m_aiWriter, indent, "public void add" + useName + "(String name, " + internalType + " value) throws MgmtException;");
        println(m_aiWriter);
        println(m_aiWriter, indent, "/**");
        println(m_aiWriter, indent, " * Delete the item with the specified name from this map");
        println(m_aiWriter, indent, " * @param name The key name for the item in the map");
        println(m_aiWriter, indent, " * @throws MgmtException Thrown if there is some problem deleting this item from the map");
        println(m_aiWriter, indent, " */");
        println(m_aiWriter, indent, "public void delete" + useName + "(String name) throws MgmtException;");

        println(m_aiWriter);
        println(m_aiWriter, indent, "/**");
        println(m_aiWriter, indent, " * Get the attribute metadata for " + key + ".");
        println(m_aiWriter, indent, " * @return The IMgmtAttributeMetaData value for this attribute");
        println(m_aiWriter, indent, " * @throws MgmtException Thrown when this attribute metadata doesn't exist or");
        println(m_aiWriter, indent, " * if there is some problem obtaining this attribute metadata");
        println(m_aiWriter, indent, " */");
        println(m_aiWriter, indent, "public IMgmtAttributeMetaData get" + useName + "MetaData() throws MgmtException;");

        // Writer the implementation
        println(m_acWriter);
        println(m_acWriter, indent, "public " + internalType + " get" + useName + "(String name) throws MgmtException");
        println(m_acWriter, indent, "{");
        if(type == null)
        {
            println(m_acWriter, indent, "    return (" + internalType + ")super.getAttribute(name);");
        }
        else
        {
            println(m_acWriter, indent, "    return new " + outer + "Bean." + type + "(this, name, false);");
        }
        println(m_acWriter, indent, "}");
        if(type != null)
        {
            println(m_acWriter);
            println(m_acWriter, indent, "public " + internalType + " create" + useName + "() throws MgmtException");
            println(m_acWriter, indent, "{");
            println(m_acWriter, indent, "    return new " + outer + "Bean." + type + "(this, " + key.toUpperCase() + "_ATTR, true);");
            println(m_acWriter, indent, "}");
        }
        println(m_acWriter);
        println(m_acWriter, indent, "public void add" + useName + "(String name, " + internalType +" value) throws MgmtException");
        println(m_acWriter, indent, "{");
        println(m_acWriter, indent, "    super.add(name, value);");
        println(m_acWriter, indent, "}");
        println(m_acWriter);
        println(m_acWriter, indent, "public void delete" + useName + "(String name) throws MgmtException");
        println(m_acWriter, indent, "{");
        println(m_acWriter, indent, "    super.delete(name);");
        println(m_acWriter, indent, "}");
        println(m_acWriter);
        println(m_acWriter, indent, "public IMgmtAttributeMetaData get" + useName + "MetaData() throws MgmtException");
        println(m_acWriter, indent, "{");
        println(m_acWriter, indent, "    return getAttributeMetaData(" + key.toUpperCase() + "_ATTR);");
        println(m_acWriter, indent, "}");

        addInnerClass(key, type, desc);
    }

    private String generateDefaultDefinition(String key, IAttributeDescription desc)
    throws Exception
    {
        Object def = getDefaultValue(desc);

        if(def == null)
        {
            return null;
        }

        StringBuffer buffer = new StringBuffer("public static final ");
        buffer.append(classToType(def.getClass()));
        buffer.append(" ");
        buffer.append(key.toUpperCase());
        buffer.append("_DEFAULT = ");
        if(def instanceof String)
        {
            buffer.append('"');
            buffer.append(def.toString());
            buffer.append('"');
        }
        else if(def instanceof java.math.BigDecimal)
        {
            buffer.append("new java.math.BigDecimal(");
            buffer.append(def.toString());
            buffer.append(")");
        }
        else
        {
            buffer.append(def.toString());
        }
        buffer.append(';');

        return buffer.toString();
    }

    private String generateFixedDefinition(String key, IAttributeDescription desc)
    throws Exception
    {
        Object fixed = getFixedValue(desc);

        if(fixed == null)
        {
            return null;
        }

        StringBuffer buffer = new StringBuffer("public static final ");
        buffer.append(classToType(fixed.getClass()));
        buffer.append(" ");
        buffer.append(key.toUpperCase());
        buffer.append("_FIXED = ");
        if(fixed instanceof String)
        {
            buffer.append('"');
            buffer.append(fixed.toString());
            buffer.append('"');
        }
        else if(fixed instanceof java.math.BigDecimal)
        {
            buffer.append("new java.math.BigDecimal(");
            buffer.append(fixed.toString());
            buffer.append(")");
        }
        else
        {
            buffer.append(fixed.toString());
        }
        buffer.append(';');

        return buffer.toString();
    }

    private void generateEnumDefinition(int indent, String key, IAttributeDescription desc)
    throws Exception
    {
        Object obj = desc.getProperty(IAttributeDescription.EMUMERATION_FACET);

        if(obj != null && (obj instanceof IAttributeList))
        {
            IAttributeList list = (IAttributeList)obj;

            for (int i = 0; i < list.size(); i++)
            {
                String facet = (String)list.getAttribute(i);
                String value = "public static final String " + key.toUpperCase() + "_" + convertSpaces(facet) + " = \"" + facet + "\";";

                println(m_ciWriter, indent, value);
            }
        }
    }

    private String convertSpaces(String val)
    {
        String res = val;
        
        res = res.replace('.', '_');
        res = res.replace(' ', '_');
        
        return res;
    }

    private void generateFactoryInterface()
    throws Exception
    {
        printCopywrite(m_fiWriter);
        println(m_fiWriter, "package " + m_package + ".gen;");
        println(m_fiWriter);
        println(m_fiWriter, "import java.util.*;");
        println(m_fiWriter, "import com.sonicsw.ma.mgmtapi.config.*;");
        println(m_fiWriter, "import " + m_package + ".*;");
        println(m_fiWriter, "import " + m_package + ".impl.*;");
        println(m_fiWriter);
        println(m_fiWriter, "public class Abstract" + m_facPrefix + "MgmtBeanFactory extends MgmtBeanFactory");
        println(m_fiWriter, "{");
        println(m_fiWriter, "    public Abstract" + m_facPrefix + "MgmtBeanFactory()");
        println(m_fiWriter, "    {");
        println(m_fiWriter, "        super();");
        println(m_fiWriter);

        for (int i = 0; i < m_argList.size(); i++)
        {
            GenerateArg arg = (GenerateArg)m_argList.get(i);

            println(m_fiWriter, "        register" + arg.getClassName() + "Bean();");
        }

        println(m_fiWriter, "    }");
        println(m_fiWriter);
    }

    private void generateFactoryClass(PrintWriter fcWriter)
    throws Exception
    {
        printCopywrite(fcWriter);
        println(fcWriter, "package " + m_package + ";");
        println(fcWriter);
        println(fcWriter, "import com.sonicsw.ma.mgmtapi.config.*;");
        println(fcWriter, "import " + m_package + ".gen.*;");
        println(fcWriter);
        println(fcWriter, "public class " + m_facPrefix + "MgmtBeanFactory extends Abstract" + m_facPrefix + "MgmtBeanFactory");
        println(fcWriter, "{");
        println(fcWriter, "    public " + m_facPrefix + "MgmtBeanFactory()");
        println(fcWriter, "    {");
        println(fcWriter, "        super();");
        println(fcWriter, "    }");
        println(fcWriter, "}");
    }

    private String classToDescription(Class cls) throws Exception
    {
        if(cls == String.class)
        {
            return "String";
        }

        if(cls == Integer.class)
        {
            return "Integer";
        }

        if(cls == Boolean.class)
        {
            return "Boolean";
        }

        if(cls == Long.class)
        {
            return "Long";
        }

        if(cls == BigDecimal.class)
        {
            return "BigDecimal";
        }

        if(cls == byte[].class)
        {
            return "ByteArray";
        }

        if(cls == IConfigElement.class)
        {
            return "Reference";
        }

        return null;
    }

    private String classToType(Class cls) throws Exception
    {
        if(cls == String.class)
        {
            return "String";
        }

        if(cls == Integer.class)
        {
            return "int";
        }

        if(cls == Boolean.class)
        {
            return "boolean";
        }

        if(cls == Long.class)
        {
            return "long";
        }

        if(cls == BigDecimal.class)
        {
            return "java.math.BigDecimal";
        }

        if(cls == byte[].class)
        {
            return "byte[]";
        }

        if(cls == IConfigElement.class)
        {
            return "IMgmtBeanBase";
        }

        return null;
    }

    private void addInnerClass(String key, String type, IAttributeDescription desc)
    {
        if(type == null)
        {
            return;
        }

        for(int i = 0; i < m_innerClasses.size(); i++)
        {
            LocalInnerClass temp = (LocalInnerClass)m_innerClasses.get(i);

            if(temp.m_type.equals(type))
            {
                return;
            }
        }

        m_innerClasses.add(new LocalInnerClass(key, type, desc));
    }

    private void println(PrintWriter writer)
    {
        writer.println();
    }

    private void println(PrintWriter writer, String text)
    {
        writer.println(text);
    }

    private void println(PrintWriter writer, int indent, String text)
    {
        for(int i = 0; i < indent; i++)
        {
            writer.print("    ");
        }

        writer.println(text);
    }

    private void printCopywrite(PrintWriter writer)
    {
        writer.println("/**");
        writer.println(" * Copyright (c) 2003 Sonic Software Corporation. All Rights Reserved.");
        writer.println(" *");
        writer.println(" * This software is the confidential and proprietary information of Sonic");
        writer.println(" * Software Corpoation. (Confidential Information).  You shall not");
        writer.println(" * disclose such Confidential Information and shall use it only in");
        writer.println(" * accordance with the terms of the license agreement you entered into");
        writer.println(" * with Sonic.");
        writer.println(" *");
        writer.println(" * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE");
        writer.println(" * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE");
        writer.println(" * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR");
        writer.println(" * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES");
        writer.println(" * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING");
        writer.println(" * THIS SOFTWARE OR ITS DERIVATIVES.");
        writer.println(" *");
        writer.println(" * CopyrightVersion 1.0");
        writer.println(" */");
        writer.println();
    }

    private Object getDefaultValue(IAttributeDescription desc)
    {
        return desc.getProperty(IAttributeDescription.DEFAULT_PROPERTY);
    }

    private Object getFixedValue(IAttributeDescription desc)
    {
        return desc.getProperty(IAttributeDescription.FIXED_PROPERTY);
    }

    private boolean isSettable(IAttributeDescription desc)
    {
        String val = (String)desc.getProperty(IAttributeDescription.FIXED_PROPERTY);

        if(val != null)
        {
            return Boolean.valueOf(val).booleanValue();
        }

        Long min = (Long)desc.getProperty(IAttributeDescription.MIN_OCCURS_PROPERTY);
        Long max = (Long)desc.getProperty(IAttributeDescription.MAX_OCCURS_PROPERTY);

        if(desc.getType() == IAttributeMap.class || desc.getType() == IAttributeList.class)
        {
            if(min != null && max != null && min.longValue() == 1 && max.longValue() == 1)
            {
                return false;
            }
        }
        return true;
    }

    private boolean isOptional(IAttributeDescription desc)
    {
        Long min = (Long)desc.getProperty(IAttributeDescription.MIN_OCCURS_PROPERTY);

        if(min != null && min.longValue() == 0)
        {
            return true;
        }

        return false;
    }

    private boolean isHidden(IAttributeDescription desc)
    {
        Boolean val = (Boolean)desc.getProperty("mx.config.hidden");

        if(val != null)
        {
            return val.booleanValue();
        }

        return false;
    }

    private String booleanObjectString(boolean value)
    {
        return value ? "Boolean.TRUE" : "Boolean.FALSE";
    }

    private void parseArgs(String[] args)
    throws Exception
    {
        GenerateArg arg = new GenerateArg();
        int         i   = 0;

        while(i < args.length)
        {
            if(args[i].trim().length() == 0)
            {
                i++;
                continue;
            }

            if(args[i].equals("-url"))
            {
                m_url = args[++i];
            }
            else if(args[i].equals("-user"))
            {
                m_user = args[++i];
            }
            else if(args[i].equals("-password"))
            {
                m_password = args[++i];
            }
            else if(args[i].equals("-domain"))
            {
                m_domain = args[++i];
            }
            else if(args[i].equals("-factory"))
            {
                m_factory = new Boolean(args[++i]).booleanValue();
            }
            else if(args[i].equals("-factory-prefix"))
            {
                m_facPrefix = args[++i];
            }
            else if(args[i].equals("-package"))
            {
                m_package = args[++i];
            }
            else if(args[i].equals("-output"))
            {
                m_output = args[++i];
            }
            else if(args[i].equals("-type"))
            {
                arg.m_type = args[++i];
            }
            else if(args[i].equals("-cVersion"))
            {
                arg.m_cVersion = args[++i];
            }
            else if(args[i].equals("-pVersion"))
            {
                arg.m_pVersion = args[++i];
            }
            else if(args[i].equals("-name"))
            {
                arg.setName(args[++i]);
            }
            else if(args[i].equals("-implementation") || args[i].equals("-impl"))
            {
                m_impl = new Boolean(args[++i]).booleanValue();
            }
            else if(args[i].equals("-private"))
            {
                arg.m_private = new Boolean(args[++i]).booleanValue();
            }
            else if(args[i].equals("-template"))
            {
                arg.m_template = new Boolean(args[++i]).booleanValue();
            }
            else if(args[i].equals("-property"))
            {
                arg = null;   // Not getting settings from command line

                String pFile = args[++i];

                try
                {
                    XProperties prop = new XProperties();
                    InputStream is   = new FileInputStream(pFile);
                    prop.load(is);

                    m_url       = prop.getProperty("url", m_url);
                    m_user      = prop.getProperty("user", m_user);
                    m_password  = prop.getProperty("password", m_password);
                    m_domain    = prop.getProperty("domain", m_domain);
                    m_facPrefix = handleFactoryPrefixCase(prop.getProperty("factory-prefix", m_facPrefix));
                    m_factory   = prop.getBooleanProperty("factory", m_factory);
                    m_package   = prop.getProperty("package", m_package);
                    m_output    = prop.getProperty("output",  m_output);
                    m_impl      = prop.getBooleanProperty("implementation", m_impl);

                    String[] list = prop.getStringArrayProperty("generate");

                    for (int j = 0; j < list.length; j++)
                    {
                        parseProperties(prop, list[j]);
                    }
                }
                catch(Exception e)
                {
                    throw new Exception("Failed to load properties " + pFile + " - " + e.getMessage());
                }
            }
            else
            {
                throw new Exception("Unsupported argument " + args[i]);
            }

            i++;
        }

        if (m_output == null)
        {
            throw new Exception("'output' is not specified");
        }

        if (arg != null)
        {
            arg.validate();

            m_argList.add(arg);
        }
    }

    /**
     * Supports the fact that the factory prefix enters as a lower case string (not enforced)
     * and that we want ma>MA, mf>MF, mq>MQ, esb>ESB but logger>Logger.
     */
    private String handleFactoryPrefixCase(String prefix)
    {
        if (prefix.length() <= 3)
        {
            // This catches basics such as ma, mf, mq and esb
            return prefix.toUpperCase();
        }

        // Deals with longer prefixes by capitalizing first character
        String subString = prefix.substring(0, 1);
        
        return subString.toUpperCase() + prefix.substring(1).toLowerCase();
    }

    private void parseProperties(XProperties prop, String type)
    throws Exception
    {
       GenerateArg arg = new GenerateArg();

       arg.m_type     = type;
       arg.setName(prop.getProperty(type + "." + "name"));
       arg.m_cVersion = prop.getProperty(type + "." + "cVersion");
       arg.m_pVersion = prop.getProperty(type + "." + "pVersion");
       arg.m_private  = prop.getBooleanProperty(type + "." + "private");
       arg.m_template = prop.getBooleanProperty(type + "." + "template");

       arg.validate();

       m_argList.add(arg);
    }


    private void printAnnotation(int indent, String annotation)
    {
        println(m_aiWriter, indent, " * ");
        StringTokenizer token = new StringTokenizer(annotation, "\n");
        while(token.hasMoreTokens())
        {
            println(m_aiWriter, indent, " * " + token.nextToken());
        }
        println(m_aiWriter, indent, " * ");
    }

    private void printUsage()
    {
        System.out.println("com.sonicsw.mf.gui.config.api.GenerateBean");
        System.out.println("    [-url <DS URL>]                     (Default = " + DEFAULT_URL + ")");
        System.out.println("    [-user <DS user name>]              (Default = " + DEFAULT_USER + ")");
        System.out.println("    [-password <DS password>]           (Default = " + DEFAULT_PASSWORD + ")");
        System.out.println("    [-domain <DS domain>]               (Default = " + DEFAULT_DOMAIN + ")");

        System.out.println("    [-type <config type name>]          (Default = None)");
        System.out.println("    [-cVersion <config version>]        (Default = None)");
        System.out.println("    [-pVersion <product version>]       (Default = None)");
        System.out.println("    [-package <generated package>]      (Default = " + DEFAULT_PACKAGE + ")");
        System.out.println("    [-name <generated name>]            (Default = None)");
        System.out.println("    [-impl <true or false>]             (Default = false)");
        System.out.println("    [-private <true or false>]          (Default = false)");
        System.out.println("    [-template <true or false>]         (Default = true)");
        System.out.println("    [-factoryPrefix <factory prefix>]   (Default = MF)");
        System.out.println("    [-factory <true or false>]          (Default = false)");
        System.out.println("    [-output <generated file location>] (Default = None)");
        System.out.println("    [-property <property file location>] (Default = None)");
    }

    private void closeWriter(Writer writer)
    throws IOException
    {
       if (writer != null)
       {
           writer.flush();
           writer.close();
       }
    }

    //-------------------------------------------------------------------------
    //
    // Inner classes
    //
    //-------------------------------------------------------------------------

    private class GenerateArg
    {
        String  m_type      = null;
        String  m_cVersion  = null;
        String  m_pVersion  = null;
        private String m_name = null;
        private String m_className = null;
        boolean m_private   = false;
        boolean m_template  = false;

        public void validate()
        throws Exception
        {
            if(m_type == null)
            {
                throw new Exception("'type' is not specified");
            }

            if(m_cVersion == null)
            {
                throw new Exception("'cVersion' is not specified");
            }

            if(m_pVersion == null)
            {
                throw new Exception("'pVersion' is not specified");
            }

            if (m_name == null)
            {
                throw new Exception("'name' is not specified");
            }

            if (m_className == null)
            {
                throw new Exception("'className' is not specified");
            }
        }
        
        public void setName(String name)
        {
            m_name = name;
            
            StringBuffer sb = new StringBuffer();
            StringTokenizer st = new StringTokenizer(name, " _");
            while (st.hasMoreTokens())
            {
                String token = (String)st.nextToken();
                
                if (Character.isUpperCase(token.charAt(0)))
                {
                    sb.append(token);
                }
                else
                {
                    sb.append(Character.toUpperCase(token.charAt(0)));
                    sb.append(token.substring(1));
                }
            }
            
            m_className = sb.toString();
        }
        
        public String getName() { return m_name; }
        public String getClassName() { return m_className; }
    }

    private class LocalInnerClass
    {
        public String                m_name;
        public String                m_type;
        public IAttributeDescription m_desc;

        public LocalInnerClass(String name, String type, IAttributeDescription desc)
        {
            m_name = name;
            m_type = type;
            m_desc = desc;
        }
    }

    //-------------------------------------------------------------------------

    public static void main(String[] args)
    {
        try
        {
            GenerateBean bean = new GenerateBean(args);

            bean.generate();

            bean.disconnect();
        }
        catch(Exception e)
        {
            System.out.println(e.getMessage());

            while(e != null && e instanceof ConfigServiceException)
            {
                e = ((ConfigServiceException)e).getLinkedException();

                if(e != null)
                {
                    System.out.println(e.getMessage());
                }
            }

            System.exit(-1);
        }
        System.exit(0);
    }
}