package com.sonicsw.mx.config.tools;

import java.util.ArrayList;
import java.util.Properties;
import java.util.StringTokenizer;

/**
 * A subclass of Properties that allows recursive
 * references for property values. For example,
 *
 * <pre><code>
 * A=12345678
 * B={A}90
 * C={B} plus more
 * </code></pre>
 *
 * will result in <code>getProperty("C")</code>
 * returning the value "1234567890 plus more".
 *
 * @author: Chris Mair
 * Java Developers Journal  Volume 06 Issue 12
 * "Enabling Constant Substitution in Property Values - A Useful Extension To
 *  Properties"
 */
public class XProperties extends Properties
{
    // The prefix and suffix for constant names within property values
    private static final String START_CONST = "{";
    private static final String END_CONST = "}";


    // The maximum depth for recursive substitution of constants within property values
    // (e.g., A={B} .. B={C} .. C={D} .. etc.)
    private static final int MAX_SUBST_DEPTH = 5;


    /**
     * Creates an empty property list with no default values.
     */
    public XProperties()
    {
        super();
    }

    /**
     * Creates an empty property list with the
     * specified defaults.
     * @param defaults java.util.Properties
     */
    public XProperties(Properties defaults)
    {
        super(defaults);
    }

    /**
     * Searches for the property with the specified
     * key in this property list. If the key is not
     * found in this property list, the default
     * property list, and its defaults, recursively,
     * are then checked. The method returns
     * <code>null</code> if the property is not found.
     *
     * @param key the property key.
     * @return the value in this property list with
     * the specified key value.
     */
    @Override
    public String getProperty(String key)
    {
        // Return the property value starting at level 0
        return getProperty(key, 0);
    }

    /**
     * Searches for the property with the specified
     * key in this property list. If the key is not
     * found in this property list, the default
     * property list, and its defaults, recursively,
     * are then checked. The method returns
     * <code>null</code> if the property is not found.
     *
     * <p>The level parameter specifies the current
     * level of recursive constant substitution. If
     * the requested property value includes a
     * constant, its value is substituted in place
     * through a recursive call to this method,
     * incrementing the level. Once level exceeds
     * MAX_SUBST_DEPTH, no further constant
     * substitutions are performed within the
     * current requested value.
     *
     * @param key the property key.
     * @param level the level of recursion so far
     * @return the value in this property list with
     * the specified key value.
     */
    private String getProperty(String key, int level)
    {
        String value = super.getProperty(key);

        if (value != null)
        {
            // Get the index of the first constant, if any
            int beginIndex = 0;
            int startName = value.indexOf(START_CONST, beginIndex);

            while (startName != -1)
            {
                // Exceeded MAX_SUBST_DEPTH  - return the value as is
                //
                if (level+1 > MAX_SUBST_DEPTH)
                {
                    return value;
                }

                int endName = value.indexOf(END_CONST, startName);

                // Terminating symbol not found - return the value as is
                //
                if (endName == -1)
                {
                    return value;
                }

                String constName = value.substring(startName+1, endName);
                String constValue = getProperty(constName, level+1);

                // Property name not found - return the value as is
                //
                if (constValue == null)
                {
                    return value;
                }


                // Insert the constant value into the original property value
                String newValue = (startName > 0) ? value.substring(0, startName) : "";
                newValue += constValue;


                // Start checking for constants at this index
                beginIndex = newValue.length();


                // Append the remainder of the value
                newValue += value.substring(endName + 1);

                value = newValue;

                // Look for the next constant
                startName = value.indexOf(START_CONST, beginIndex);
            }
        }

        return value;
    }

    public String[] getStringArrayProperty(String key)
    {
        return getStringArrayProperty(key, null);
    }

    public String[] getStringArrayProperty(String key, String[] defaultValue)
    {
        StringTokenizer st   = new StringTokenizer(getProperty(key), ",");
        ArrayList       list = new ArrayList(st.countTokens());

        while (st.hasMoreElements())
        {
            list.add(st.nextToken().trim());
        }

        return (list.isEmpty()) ? defaultValue : (String[])list.toArray(new String[list.size()]);
    }

    public boolean getBooleanProperty(String key)
    {
        return getBooleanProperty(key, false);
    }

    public boolean getBooleanProperty(String key, boolean defaultValue)
    {
        boolean res   = defaultValue;
        String  value = getProperty(key);

        if (value != null)
        {
            value = value.trim().toLowerCase();

            if (value.equals("true"))
            {
                res = true;
            }
            else
            if (value.equals("false"))
            {
                res = false;
            }
        }

        return res;
    }

}
