/*
 * Copyright (c) 1999-2000 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Sonic Software Corporation. ("Confidential Information"). You
 * shall not disclose such Confidential Information and shall use it
 * only in accordance with the terms of the license agreement you
 * entered into with Sonic Software.
 *
 * SONIC SOFTWARE MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY
 * OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. PROGRESS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 *
 */

package com.sonicsw.mx.config.tools.admin;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Hashtable;
import java.util.StringTokenizer;

import javax.management.MBeanException;
import javax.management.ReflectionException;
import javax.management.RuntimeErrorException;
import javax.management.RuntimeMBeanException;
import javax.management.RuntimeOperationsException;

import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigType;

import com.sonicsw.mf.common.MFException;
import com.sonicsw.mf.common.MFProxyException;
import com.sonicsw.mf.common.MFProxyRuntimeException;
import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.Version;
import com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException;

/**
 * ConfigCommandLine is a commandline processor which processes XML configuration
 * files.
 */
public final class ConfigCommandLine
{
    private static final String OVERRIDE         = "override";
    private static final String DEFAULT_URL      = "localhost";
    private static final String DEFAULT_USER     = "Administrator";
    private static final String DEFAULT_PASSWORD = "Administrator";     //NOSONAR field change is not required.
    private static final String DEFAULT_DOMAIN   = "Domain1";
    private static final String HELP_STRING      = "help";
    private static final String PROMPT_WORD      = "ConfigAdmin> ";

    private static volatile String[] m_arguments;

    private static boolean m_echo  = false;
    private static boolean m_debug = false;

    private static volatile ConfigServerUtility m_csu = null;

    private Method    m_commandMethod;
    private String    m_inputCommand;
    private int       m_argumentCount;
    private boolean   m_bye  = false;
    private boolean   m_help = false;
    private Method    m_newCommandMethod;
    private Hashtable m_commands = new Hashtable();
    private Class     m_clClass = ConfigCommandLine.class;

    public ConfigCommandLine()
    {
        try
        {
            m_commands.put("exit",         m_clClass.getDeclaredMethod("exit", new Class[0]));
            m_commands.put("bye",          m_clClass.getDeclaredMethod("exit", new Class[0]));
            m_commands.put("help",         m_clClass.getDeclaredMethod("help", new Class[0]));
            m_commands.put("?",            m_clClass.getDeclaredMethod("help", new Class[0]));
            m_commands.put("connect",      m_clClass.getDeclaredMethod("connect",new Class[0]));
            m_commands.put("direct-connect",      m_clClass.getDeclaredMethod("directConnect",new Class[0]));
            m_commands.put("disconnect",   m_clClass.getDeclaredMethod("disconnect",new Class[0]));
            m_commands.put("import-bean",  m_clClass.getDeclaredMethod("importBean",new Class[0]));
            m_commands.put("import-beans", m_clClass.getDeclaredMethod("importBeans",new Class[0]));
            m_commands.put("export-bean",  m_clClass.getDeclaredMethod("exportBean",new Class[0]));
            m_commands.put("export-beans", m_clClass.getDeclaredMethod("exportBeans",new Class[0]));
            m_commands.put("delete-bean",  m_clClass.getDeclaredMethod("deleteBean",new Class[0]));
            m_commands.put("delete-beans", m_clClass.getDeclaredMethod("deleteBeans",new Class[0]));
            m_commands.put("list",         m_clClass.getDeclaredMethod("list",new Class[0]));
            m_commands.put("import-type",  m_clClass.getDeclaredMethod("importType",new Class[0]));
            m_commands.put("import-type-properties",  m_clClass.getDeclaredMethod("importTypeProperties",new Class[0]));
            m_commands.put("import-ivalues", m_clClass.getDeclaredMethod("importInitialValues",new Class[0]));
            m_commands.put("delete-type",  m_clClass.getDeclaredMethod("deleteType",new Class[0]));
            m_commands.put("dump-ds",      m_clClass.getDeclaredMethod("dumpDS",new Class[0]));
            m_commands.put("seed-ds",      m_clClass.getDeclaredMethod("seedDS",new Class[0]));
            m_commands.put("set-workspace",m_clClass.getDeclaredMethod("setWorkspace",new Class[0]));
            m_commands.put("import-file",  m_clClass.getDeclaredMethod("importFile",new Class[0]));
            m_commands.put("copy-files",   m_clClass.getDeclaredMethod("copyFiles",new Class[0]));
            m_commands.put("import-files", m_clClass.getDeclaredMethod("importFiles",new Class[0]));
            m_commands.put("create-folder",m_clClass.getDeclaredMethod("createFolder",new Class[0]));
            m_commands.put("delete-files", m_clClass.getDeclaredMethod("deleteFiles",new Class[0]));
            m_commands.put("echo",         m_clClass.getDeclaredMethod("echo", new Class[0]));
            m_commands.put("debug",        m_clClass.getDeclaredMethod("debug", new Class[0]));
            m_commands.put("rename-folder",m_clClass.getDeclaredMethod("renameFolder",new Class[0]));
            m_commands.put("rename-file",  m_clClass.getDeclaredMethod("renameFile",new Class[0]));
            m_commands.put("copy-ds-files-to-container-caches",  m_clClass.getDeclaredMethod("copyDSFilesToContainerCaches",new Class[0]));
            m_commands.put("modify-archive-search-path",  m_clClass.getDeclaredMethod("modifyArchivesSearchPath",new Class[0]));


        }
        catch(NoSuchMethodException e)
        {
            System.err.println(e.getMessage());
        }
    }

    public void setArguments(String[] arguments)
    {
        m_arguments = arguments;
    }

    public void getCommand()
    {
        getCommand(new java.io.BufferedReader(new java.io.InputStreamReader(System.in)));
    }

    public void getCommand(java.io.BufferedReader reader)
    {
        StringTokenizer inputTokenizer;
        m_bye = false;

        while (!m_bye )
        {
            cleanInput();
            prompt();

            try
            {
                String line = reader.readLine();

                if (line == null || line.equals("exit") || line.equals("bye"))
                {
                    m_bye = true;
                    exit();
                }

                if (m_echo)
                {
                    System.out.println(line);
                    System.out.flush();
                }

                String spaceDelims = " \n\t\r";
                inputTokenizer = new StringTokenizer(line, spaceDelims, true);
                boolean unbalancedQuote = false;

                while ((inputTokenizer.hasMoreElements()) && !m_help && !unbalancedQuote)
                {
                    String token;
                    token = inputTokenizer.nextToken(spaceDelims);

                    // Ignore the tokens themselves.
                    if (spaceDelims.indexOf(token)==-1)
                    {
                        // If there are double quotes in this token, keep the double
                        // quotes paired (or try!!) Read some more of the input string
                        // to get to the next double quotes.
                        int lastIndex;

                        if ((lastIndex = token.indexOf("\"")) != -1)
                        {
                            int countDoubleQuotes = 1;

                            while ((lastIndex = token.indexOf("\"", lastIndex + 1)) != -1)
                            {
                                countDoubleQuotes++;
                            }

                            if ((countDoubleQuotes % 2) != 0)
                            {
                                // We've now got to move into the next tokens
                                // looking for the closing quote That means we
                                // require at least two more tokens (1. the
                                // rest of the quoted string and 2. the closing quote)
                                if (inputTokenizer.hasMoreTokens())
                                {
                                    token = token + inputTokenizer.nextToken("\""); // up to quote

                                    if (inputTokenizer.hasMoreTokens() && inputTokenizer.nextToken().equals("\""))
                                    {
                                        token = token + "\"";
                                    }
                                    else
                                    {
                                        unbalancedQuote = true;
                                    }
                                }
                                else
                                {
                                    unbalancedQuote = true;
                                }
                            }
                        }
                        try
                        {
                            if (!unbalancedQuote)
                            {
                                if(!dealWithInput(token))
                                {
                                    break;
                                }
                            }
                        }
                        catch (NoSuchMethodException e)
                        {
                            System.err.println(e.getMessage());
                        }
                    }
                }

                if (unbalancedQuote)
                {
                    System.err.println("UNBALANCED_QUOTES");
                }
                else if (m_commandMethod != null)
                {
                    try
                    {
                        m_commandMethod.invoke(m_clClass);
                    }
                    catch (InvocationTargetException e)
                    {
                        showError(e.getTargetException());
                    }
                    catch (Exception e)
                    {
                        showError(e);
                    }
                }
            }
            catch (IOException e)
            {
                System.err.println("cannot process the command " + e.getMessage());
                exit();
            }
        }
    }

    /**
     * Takes one word from the input and adds it to the command, or to
     * the argument vector.  We assume that if there is a recognizable
     * command already parsed, and some unknown word comes along, the
     * unknown word and everything to the end of the line is an argument
     * @param token single word read from the input
     */
    private boolean dealWithInput(String token) throws NoSuchMethodException
    {
        boolean ret = true;

        // Don't add a space at the beginning of the command
        if (m_inputCommand == null || m_inputCommand.equals(""))
        {
            m_inputCommand = token;
        }
        else
        {
            m_inputCommand = m_inputCommand + " " + token;
        }

        // if we haven't started adding tokens to the arguments
        if (m_argumentCount == 0)
        {
            m_newCommandMethod = (Method)(m_commands.get(m_inputCommand.toLowerCase()));

            if (m_newCommandMethod == null)
            {
                if (m_commandMethod != null)
                {
                    parseArgument(token);
                }
                else
                {
                    System.err.println("Unknown command : " + m_inputCommand);
                    System.err.println("  Type 'help' to view the list of available commands");
                    ret = false;
                }
            }
            else
            {
                m_commandMethod = m_newCommandMethod;
            }
        }
        else
        {
            parseArgument(token);
        }

        // Here, we either have a command or a partial phrase.  If it's
        // a partial phrase (including an empty string for the top level)
        // and this last token was a ?, invoke the generic
        // help which shows information about commands at a particular level
        // Strip off the question mark that has just been added to the input buffer,
        // so the remaining string in m_inputCommand will index
        // into the help array

        if ((m_commandMethod == null) && (token.compareTo(HELP_STRING)== 0))
        {
            m_commandMethod = (Method)(m_clClass.getDeclaredMethod("help", new Class[0]));

            if (m_inputCommand.compareTo(HELP_STRING) == 0)
            {
                m_inputCommand = "";
            }
            else
            {
                m_inputCommand = m_inputCommand.substring(0, m_inputCommand.length() - 2);
            }

            m_help = true;
        }
        return ret;
    }

    /**
     *  Strip off a possible commas in this token
     *  There are no spaces in this token, because they were already
     * striped off by the main input tokenizer.
     * Commas need to be left in the argument if the phrase is
     * encased in double quotes.
     */
    private  void parseArgument(String token)
    {
        String subToken;

        StringTokenizer firstArgTokenizer = new StringTokenizer(token, ",\"\t ", true);

        while (firstArgTokenizer.hasMoreTokens())
        {
            String arg = "";
            boolean inDoubleQuotes = false;
            boolean completeToken = false;

            while (!completeToken)
            {
                subToken = firstArgTokenizer.nextToken();

                if (subToken.compareTo(",") == 0)
                {
                    if (inDoubleQuotes)
                    {
                        arg=arg + subToken;
                    }
                    else
                    {
                        completeToken=true;
                    }
                }
                else
                {
                    if (subToken.compareTo("\"")== 0)
                    {
                        inDoubleQuotes = !inDoubleQuotes;
                        arg = arg + subToken;
                    }
                    else
                    {
                        arg = arg + subToken;
                    }
                }

                if (!firstArgTokenizer.hasMoreTokens())
                {
                    completeToken=true;
                }
            }

            // Double quotes must remain when embedded in a command argument, like
            // java.naming.provider.url="ldap://diablo:389/ou=TopicConnectionFactories,
            // ou=Message Service, ou=Apptivity,dc=progress,dc=com".
            // When the double quotes surround the whole argument, as in
            // del topic durables davidg "" ta,tb,tc
            // it is because the argument should be treated as a string with no parsing
            // and passed to the command.  To be passed to the command, we should strip
            // the double quotes.  That's what we're doing here.
            if (arg.length() > 0)
            {
                if ((arg.charAt(0) == '\"') && (arg.charAt(arg.length() - 1) == '\"'))
                {
                    if (arg.length() == 2)
                    {
                        arg = "";
                    }
                    else
                    {
                        arg = arg.substring(1, arg.length() - 1);
                    }
                }
                m_arguments[m_argumentCount++] = arg;
            }
        }
    }

    /**
     * Initialize the buffers which help parse the input.  This is called
     * after we're done with each line of input
     */
    private void cleanInput()
    {
        m_arguments        = new String[75];
        m_argumentCount    = 0;
        m_commandMethod    = null;
        m_newCommandMethod = null;
        m_inputCommand     = "";
    }

    public static void exit()
    {
        System.err.println("Shutting down...");
        System.exit(0);
    }

    private void showError(Throwable e)
    {
        String msg = null;

        if(e instanceof ProxyRuntimeException)
        {
            msg = ((ProxyRuntimeException)e).getTargetException().getMessage();
        }
        else
        {
            msg = e.getMessage();
        }

        if(m_debug)
        {
            printExceptionInfo(e);
        }

        System.err.println("Error: " + msg);
    }

    ///////////////////////////////////////////////////////////////////////////////////////////////
    //public methods
    ///////////////////////////////////////////////////////////////////////////////////////////////

    private static void checkNotConnected() throws Exception
    {
        if(m_csu != null)
        {
            throw new Exception("Already connected to a domain.");
        }
    }

    private static void checkConnected() throws Exception
    {
        if(m_csu == null)
        {
            throw new Exception("Not connected to a domain.");
        }
    }

    public static void directConnect() throws Exception
    {
        checkNotConnected();


        String url      = m_arguments[0];
        String domain   = m_arguments[1];
        String user     = m_arguments[2];
        String password = m_arguments[3];
        String timeoutStr = m_arguments[4];


        if (url == null)
        {
            url = "ds.xml";
        }

        if (domain == null)
        {
            domain = DEFAULT_DOMAIN;
        }


        System.out.println("Connecting to domain " + domain + " on " + url + " ...");

        try
        {
            m_csu = new ConfigServerUtility();

            if (timeoutStr != null)
            {
                m_csu.setRequestTimeout(new Integer(timeoutStr).intValue());
            }

            m_csu.connect(domain, url, user, password, true);

            System.out.println("    Connected to Configuration Server for domain " +
                               domain + " on " + url);
        }
        catch(Exception e)
        {
            m_csu = null;

            throw e;
        }
    }

    public static void connect() throws Exception
    {
        checkNotConnected();

        String url      = m_arguments[0];
        String user     = m_arguments[1];
        String password = m_arguments[2];
        String domain   = m_arguments[3];

        if (url == null)
        {
            url = DEFAULT_URL;
        }

        if (user == null)
        {
            user = DEFAULT_USER;
        }

        if (password == null)
        {
            password = DEFAULT_PASSWORD;
        }

        if (domain == null)
        {
            domain = DEFAULT_DOMAIN;
        }

        System.out.println("Connecting to domain " + domain + " on " + url + " ...");

        try
        {
            m_csu = new ConfigServerUtility();

            m_csu.connect(domain, url, user, password, true);

            System.out.println("    Connected to Configuration Server for domain " +
                               domain + " on " + url);
        }
        catch(Exception e)
        {
            m_csu = null;

            throw e;
        }
    }

    public static void disconnect() throws Exception
    {
        checkConnected();

        System.out.println("Disconnecting from domain ...");

        m_csu.disconnect();
        m_csu = null;

        System.out.println("    Done");
    }

    public static void importBean() throws Exception
    {
        checkConnected();

        String filename = m_arguments[0];

        if (filename == null || filename.length() == 0)
        {
            throw new Exception ("Filename is not specified");
        }

        boolean overwrite = false;

        if(m_arguments[1] != null && m_arguments[1].equalsIgnoreCase(OVERRIDE))
        {
            overwrite = true;
        }

        IConfigBean bean = m_csu.importBean(filename, overwrite);

        System.out.println("Imported Bean ...");
        System.out.println("    name    = " + bean.getName());
        System.out.println("    type    = " + bean.getConfigType().getName());
        System.out.println("    version = " + bean.getConfigType().getVersion());
    }

    public static void importBeans() throws Exception
    {
        checkConnected();

        String fileRoot = m_arguments[0];

        if (fileRoot == null || fileRoot.length() == 0)
        {
            throw new Exception ("File root is not specified");
        }

        boolean overwrite = false;

        if(m_arguments[1] != null && m_arguments[1].equalsIgnoreCase(OVERRIDE))
        {
            overwrite = true;
        }

        m_csu.importBeanPath(fileRoot, overwrite);

        System.out.println("Checking and fixing any broken bean references...");

        m_csu.repairDanglingBeanReferences();

        System.out.println("Imported Beans...");
        System.out.println("    file root = " + fileRoot);
    }

    public static void exportBean() throws Exception
    {
        checkConnected();

        String beanName = m_arguments[0];

        if (beanName == null || beanName.length() == 0)
        {
            throw new Exception("Bean name is not specified");
        }

        String filename = m_arguments[1];

        if (filename == null || filename.length() == 0)
        {
            throw new Exception("Filename is not specified");
        }

        String xml = m_csu.exportBean(beanName, filename);

        System.out.println("Exported Bean ...");
        System.out.println("    bean name = " + beanName);
        System.out.println("    filename  = " + filename);
    }

    public static void exportBeans() throws Exception
    {
        checkConnected();

        String beanPath = m_arguments[0];

        if (beanPath == null || beanPath.length() == 0)
        {
            throw new Exception("Bean path is not specified");
        }

        String fileRoot = m_arguments[1];

        if (fileRoot == null || fileRoot.length() == 0)
        {
            throw new Exception("File root is not specified");
        }

        m_csu.exportBeanPath(beanPath, fileRoot);

        System.out.println("Exported Beans...");
        System.out.println("    bean path = " + beanPath);
        System.out.println("    file root = " + fileRoot);
    }

    public static void deleteBean() throws Exception
    {
        checkConnected();

        String beanName = m_arguments[0];

        if (beanName == null || beanName.length() == 0)
        {
            throw new Exception("Bean name is not specified");
        }

        m_csu.deleteBean(beanName);

        System.out.println("Deleted Bean ...");
        System.out.println("    name = " + beanName);
    }

    public static void deleteBeans() throws Exception
    {
        checkConnected();

        String beanRoot = m_arguments[0];

        if (beanRoot == null || beanRoot.length() == 0)
        {
            throw new Exception("Bean root is not specified");
        }

        m_csu.deleteBeanPath(beanRoot);

        System.out.println("Deleted Beans...");
        System.out.println("    bean root = " + beanRoot);
    }

    public static void importType() throws Exception
    {
        checkConnected();

        String documentURL  = m_arguments[0];
        String namespaceURI = m_arguments[1];

        if (documentURL == null || documentURL.length() == 0)
        {
            throw new Exception("Document URL is not specified");
        }

        if (namespaceURI == null || namespaceURI.length() == 0)
        {
            throw new Exception("Namespace URI is not specified");
        }

        boolean overwrite = false;

        if(m_arguments[2] != null && m_arguments[2].equalsIgnoreCase(OVERRIDE))
        {
            overwrite = true;
        }

        IConfigType[] types = m_csu.importType(documentURL, namespaceURI, overwrite);

        System.out.println("Imported Type ...");

        for(int i = 0; i < types.length; i++)
        {
            System.out.println("    type = " + types[i].getName() + " version = " + types[i].getVersion());
        }
    }

    public static void importTypeProperties() throws Exception
    {
        checkConnected();

        String documentURL  = m_arguments[0];
        String namespaceURI = m_arguments[1];
        String path = m_arguments[2];

        if (documentURL == null || documentURL.length() == 0)
        {
            throw new Exception("Document URL is not specified");
        }

        if (namespaceURI == null || namespaceURI.length() == 0)
        {
            throw new Exception("Namespace URI is not specified");
        }

        if (path == null || path.length() == 0)
        {
            throw new Exception("Path is not specified");
        }

        m_csu.importTypeProperties(documentURL, namespaceURI, path);
    }

    public static void importInitialValues() throws Exception
    {
        checkConnected();

        String documentURL  = m_arguments[0];

        if (documentURL == null || documentURL.length() == 0)
        {
            throw new Exception("Document URL is not specified");
        }

        IConfigType type = m_csu.importInitialValues(documentURL);

        System.out.println("Imported initial values for '" + type.getName() + "', v" + type.getVersion());
    }

    public static void list() throws Exception
    {
        checkConnected();

        String path = m_arguments[0];

        if (path == null || path.length() == 0)
        {
            path = "/";
        }

        String[] ret = m_csu.listConfigElements(path);

        System.out.println("Listing for path " + path + " ...");

        for(int i = 0; i < ret.length; i++)
        {
            System.out.println("    " + ret[i]);
        }
    }

    public static void deleteType() throws Exception
    {
        checkConnected();

        String name = m_arguments[0];
        String version = m_arguments[1];

        if (name == null || name.length() == 0)
        {
            throw new Exception("Name is not specified");
        }

        if (version == null || version.length() == 0)
        {
            throw new Exception("Version is not specified");
        }

        m_csu.deleteType(name, version);

        System.out.println("Deleted Type ...");
        System.out.println("    type    = " + name);
        System.out.println("    version = " + version);
    }

    public static void dumpDS() throws Exception
    {
        checkConnected();

        String path = m_arguments[0];
        if (path == null || path.length() == 0)
        {
            throw new Exception("Path is not specified");
        }

        String output = m_arguments[1];
        if (output == null || output.length() == 0)
        {
            throw new Exception("Output location is not specified");
        }

        m_csu.dumpDirectoryService(path, output);

        System.out.println("Dump DS complete...");
        System.out.println("    path   = " + path);
        System.out.println("    output = " + output);
    }

    public static void seedDS() throws Exception
    {
        checkConnected();

        String file = m_arguments[0];
        if (file == null || file.length() == 0)
        {
            throw new Exception("File is not specified");
        }

        m_csu.seedDirectoryService(file);

        System.out.println("Seed DS complete...");
        System.out.println("    file = " + file);
    }

    public static void setWorkspace() throws Exception
    {
        checkConnected();

        String path = m_arguments[0];
        if (path == null || path.length() == 0)
        {
            throw new Exception("Path is not specified");
        }

        m_csu.setDSHandlerWorkspace(path);

        System.out.println("Workspace location set to: " + path);
    }

    /*
    public static void addPlugin() throws Exception
    {
        checkConnected();

        String name = m_arguments[0];

        if (name == null || name.length() == 0)
            throw new Exception("Name is not specified");

        String config    = null;
        String runtime   = null;
        String classpath = null;

        int i = 1;
        while(m_arguments[i] != null)
        {
            if(m_arguments[i].equalsIgnoreCase("-config"))
                config = m_arguments[++i];
            else if(m_arguments[i].equalsIgnoreCase("-runtime"))
                runtime = m_arguments[++i];
            else if(m_arguments[i].equalsIgnoreCase("-classpath"))
                classpath = m_arguments[++i];
            else
                throw new Exception(m_arguments[i] + " is not a valid argument");

            i++;
        }

        if(config == null && runtime == null)
            throw new Exception("No classname specified for config and runtime");

        m_csu.addPlugin(name, config, runtime, classpath);
    }
    */

    public static void deletePlugin() throws Exception
    {
        checkConnected();

        String name = m_arguments[0];

        if (name == null || name.length() == 0)
        {
            throw new Exception("Name is not specified");
        }

        m_csu.deletePlugin(name);
    }

    public static void importFile() throws Exception
    {
        checkConnected();

        String path = m_arguments[0];
        if (path == null || path.length() == 0)
        {
            throw new Exception("Path is not specified");
        }

        String filename = m_arguments[1];
        if (filename == null || filename.length() == 0)
        {
            throw new Exception("Filename is not specified");
        }

        m_csu.importFile(path, filename, true);

        System.out.println("Import file complete...");
        System.out.println("    path     = " + path);
        System.out.println("    filename = " + filename);
    }

    public static void renameFolder() throws Exception
    {
        checkConnected();

        String oldPath = m_arguments[0];
        String newPath = m_arguments[1];

        m_csu.renameFolder(oldPath, newPath);

        System.out.println("Rename-folder complete...");
        System.out.println("    Old path     = " + oldPath);
        System.out.println("    New  path = " + newPath);
    }


    public static void copyDSFilesToContainerCaches() throws Exception
    {
        checkConnected();

        String dsPath = m_arguments[0];
        String allContainersOrListFile = m_arguments[1];
        String exclude = m_arguments[2];

        if (dsPath == null || dsPath.length() == 0)
        {
            throw new Exception("Directory Service path was not specified");
        }

        if (allContainersOrListFile == null || allContainersOrListFile.length() == 0)
        {
            throw new Exception("A container list file or the 'all-containers' phrase were not specified");
        }

        boolean allContainers = allContainersOrListFile.equalsIgnoreCase("all-containers");

        boolean excludeContainers = false;
        if (exclude != null && exclude.equalsIgnoreCase("exclude"))
        {
            excludeContainers = true;
        }


        m_csu.copyDSFilesToContainerCaches(dsPath, allContainers, excludeContainers, allContainers ? "" : allContainersOrListFile);

        System.out.println("Copy ds files to container caches complete...");
        System.out.println("    Directory Service path     = " + dsPath);
        System.out.println("    All the containers     = " + allContainers);
        System.out.println("    Exclude containers     = " + excludeContainers);
        if (!allContainers)
        {
            System.out.println("    Containers list file     = " + allContainersOrListFile);
        }

    }

    public static void modifyArchivesSearchPath() throws Exception
    {
        checkConnected();

        String newSearchPath = m_arguments[0];
        String allContainersOrListFile = m_arguments[1];
        String exclude = m_arguments[2];

        if (newSearchPath == null || newSearchPath.length() == 0)
        {
            throw new Exception("A new search path was not specified");
        }

        if (allContainersOrListFile == null || allContainersOrListFile.length() == 0)
        {
            throw new Exception("A container list file or the 'all-containers' phrase were not specified");
        }

        boolean allContainers = allContainersOrListFile.equalsIgnoreCase("all-containers");

        boolean excludeContainers = false;
        if (exclude != null && exclude.equalsIgnoreCase("exclude"))
        {
            excludeContainers = true;
        }


        m_csu.modifyArchivesSearchPath(newSearchPath, allContainers, excludeContainers, allContainers ? "" : allContainersOrListFile);

        System.out.println("Archives search path modification complete...");
        System.out.println("    New search path     = " + newSearchPath);
        System.out.println("    All the containers     = " + allContainers);
        System.out.println("    Exclude containers     = " + excludeContainers);
        if (!allContainers)
        {
            System.out.println("    Containers list file     = " + allContainersOrListFile);
        }
    }


    public static void renameFile() throws Exception
    {
        checkConnected();

        String oldPath = m_arguments[0];
        String newPath = m_arguments[1];

        m_csu.renameFile(oldPath, newPath);

        System.out.println("Rename-file complete...");
        System.out.println("    Old path     = " + oldPath);
        System.out.println("    New  path = " + newPath);
    }

    public static void createFolder() throws Exception
    {
        checkConnected();

        String folderPath = m_arguments[0];
        if (folderPath == null || folderPath.length() == 0)
        {
            throw new Exception("Folder path is not specified");
        }

        m_csu.createFolder(folderPath);

        System.out.println("Folder creation complete...");
        System.out.println("    Folder path     = " + folderPath);
    }


    public static void deleteFiles() throws Exception
    {
        checkConnected();

        String path = m_arguments[0];
        if (path == null || path.length() == 0)
        {
            throw new Exception("Path is not specified");
        }

        m_csu.deleteFiles(path);

        System.out.println("Files deletion complete...");
        System.out.println("    Path     = " + path);
    }

    public static void copyFiles() throws Exception
    {
        checkConnected();

        String fromPath = m_arguments[0];
        if (fromPath == null || fromPath.length() == 0)
        {
            throw new Exception("Source path is not specified");
        }

        String toPath = m_arguments[1];
        if (toPath == null || toPath.length() == 0)
        {
            throw new Exception("Destination path is not specified");
        }

        m_csu.copyFiles(fromPath, toPath);

        System.out.println("Files copy complete...");
        System.out.println("    Source path     = " + fromPath);
        System.out.println("    Destination path = " + toPath);
    }

    public static void importFiles() throws Exception
    {
        checkConnected();

        String fromFSPath = m_arguments[0];
        if (fromFSPath == null || fromFSPath.length() == 0)
        {
            throw new Exception("File system source path is not specified");
        }

        String toDSPath = m_arguments[1];
        if (toDSPath == null || toDSPath.length() == 0)
        {
            throw new Exception("Directory Service destination path is not specified");
        }

        m_csu.importFiles(fromFSPath, toDSPath);

        System.out.println("Files import complete...");
        System.out.println("    File System Source path     = " + fromFSPath);
        System.out.println("    Directory Service destination path = " + toDSPath);
    }



    private static void prompt()
    {
        System.out.print(PROMPT_WORD);
        System.out.flush();
    }

    public static void help()
    {
        System.out.println("ConfigAdmin (" + Version.getVersionText() + ")");
        System.out.println("Available commands:");

        System.out.println("Connect/disconnect commands:");
        System.out.println("  connect <url> [username] [password] [domain] [request-timeout-seconds]");
        System.out.println("  disconnect");
        System.out.println("");

        System.out.println("Configuration commands:");
        System.out.println("  list <path>");
        System.out.println("  import-bean  <filename> [override]");
        System.out.println("  import-beans <fileRoot> [override]");
        System.out.println("  export-bean  <beanName> <filename>");
        System.out.println("  export-beans <beanPath> <fileRoot>");
        System.out.println("  delete-bean  <beanName>");
        System.out.println("  delete-beans <beanPath>");
        System.out.println("  import-type <filename> <namespaceURI> [override]");
        System.out.println("  import-type-properties <schema-filename> <namespaceURI> <path>");
        System.out.println("  import-ivalues <path>");
        System.out.println("  delete-type <typeName> <typeVersion>");
        System.out.println("  dump-ds <path> <filename>");
        System.out.println("  seed-ds <filename>");
        System.out.println("  set-workspace <path>");
        System.out.println("");

        System.out.println("Centralized Install and file manipulation commands:");
        System.out.println("  import-files <fromFSPath> <toDSPath>");
        System.out.println("  copy-files <fromDSPath> <toDSPath>");
        System.out.println("  create-folder <folderPath>");
        System.out.println("  delete-files <dsPath>");
        System.out.println("  rename-folder <oldPath> <newPath>");
        System.out.println("  rename-file <oldPath> <newPath>");
        System.out.println("");
        System.out.println("  copy-ds-files-to-container-caches <ds-path> <container-list-file>|all-containers [exclude]");
        System.out.println("  modify-archive-search-path <new-path> <container-list-file>|all-containers [exclude]");
        System.out.println("  Note: If the 'exclude' phrase is specified then the list of containers is used as an exclusion list");
        System.out.println("  Note: Each line of the container-list-file should contain a single container path");
        System.out.println("");

        System.out.println("Misc commands:");
        System.out.println("  debug <ON | OFF>");
        System.out.println("  echo  <ON | OFF>");
        System.out.println("  help");
        System.out.println("  exit | bye");


        System.out.println("");
        System.out.println("Deprecated commands:");
        System.out.println("  import-file <dsPath> <fsPath>");

        System.out.println();
        System.out.println("NOTE:");
        System.out.println("To use the import-type command ensure that the current working");
        System.out.println("directory is the schema root directory (e.g. <sonic install>/schema)");
    }

    public static void echo()
    {
        m_echo = !m_echo;

        System.out.println(m_echo ? "echo ON" : "echo OFF");
    }

    public static void debug()
    {
        m_debug = !m_debug;

        System.out.println(m_debug ? "debug ON" : "debug OFF");
    }

    public static void main(String argv[])
    {
        ConfigCommandLine command = new ConfigCommandLine();
        int argc = argv.length;

        while (argc-- > 0)
        {
            if (argv[argc].startsWith("-s"))
            {
                try
                {
                    command.getCommand(new BufferedReader(new InputStreamReader(new FileInputStream(argv[argc+1]))));
                }
                catch (Exception e)
                {
                    e.printStackTrace(System.err);
                }
            }
        }
        command.getCommand();
    }

    private static void printExceptionInfo(Throwable e)
    {
        PrintWriter pw = new PrintWriter(System.err);

        printExceptionInfo(e, pw, false);
    }

    private static void printExceptionInfo(Throwable e, PrintWriter pw, boolean isCause)
    {
        // don't trace wrappers
        if (e instanceof MFProxyException && ((MFProxyException)e).getActualException() != null)
        {
            e = ((MFProxyException)e).getActualException();
        }
        else
        if (e instanceof MFProxyRuntimeException && ((MFProxyRuntimeException)e).getActualException() != null)
        {
            e = ((MFProxyRuntimeException)e).getActualException();
        }
        else
        if (e instanceof ProxyRuntimeException && ((ProxyRuntimeException)e).getTargetException() != null)
        {
            e = ((ProxyRuntimeException)e).getTargetException();
        }

        // Output the stacktrace information ourselves to avoid having the
        // java.lang.Throwable output the JDK 1.4 cause information too.
        // We are going to handle the cause ourselves.
        synchronized (pw)
        {
            if (isCause)
            {
                pw.print("Caused by: ");
            }

            pw.println(e);
            StackTraceElement[] trace = e.getStackTrace();
            for (int i = 0; i < trace.length; i++)
            {
                pw.println("\tat " + trace[i]);
            }
            pw.flush();
        }

        if (e instanceof ConfigServiceException && ((ConfigServiceException)e).getLinkedException() != null)
        {
            printExceptionInfo(((ConfigServiceException)e).getLinkedException(), pw, true);
        }
        else
        if (e instanceof MFProxyException && ((MFProxyException)e).getActualException() != null)
        {
            printExceptionInfo(((MFProxyException)e).getActualException(), pw, true);
        }
        else
        if (e instanceof MFException && ((MFException)e).getLinkedException() != null)
        {
            printExceptionInfo(((MFException)e).getLinkedException(), pw, true);
        }
        else
        if (e instanceof MFRuntimeException && ((MFRuntimeException)e).getLinkedException() != null)
        {
            printExceptionInfo(((MFRuntimeException)e).getLinkedException(), pw, true);
        }
        else
        if (e instanceof ReflectionException)
        {
            printExceptionInfo(((ReflectionException)e).getTargetException(), pw, true);
        }
        else
        if (e instanceof MBeanException)
        {
            printExceptionInfo(((MBeanException)e).getTargetException(), pw, true);
        }
        else
        if (e instanceof InvocationTargetException)
        {
            printExceptionInfo(((InvocationTargetException)e).getTargetException(), pw, true);
        }
        else
        if (e instanceof RuntimeOperationsException)
        {
            printExceptionInfo(((RuntimeOperationsException)e).getTargetException(), pw, true);
        }
        else
        if (e instanceof RuntimeMBeanException)
        {
            printExceptionInfo(((RuntimeMBeanException)e).getTargetException(), pw, true);
        }
        else
        if (e instanceof RuntimeErrorException)
        {
            printExceptionInfo(((RuntimeErrorException)e).getTargetError(), pw, true);
        }
        else
        if (e.getCause() != null)
        {
            printExceptionInfo(e.getCause(), pw, true);
        }
    }
}
