package com.sonicsw.mx.config.tools.admin;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import com.sonicsw.mx.config.impl.ConfigServer;
import com.sonicsw.mx.config.impl.ConfigTypeDocument;
import com.sonicsw.mx.config.impl.ConfigTypeImpl;
import com.sonicsw.mx.config.impl.InitialValuesImpl;
import com.sonicsw.mx.config.util.ConfigHelper;

import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;
import com.sonicsw.mf.framework.directory.DirectoryServiceFactory;
import com.sonicsw.mf.framework.directory.IDirectoryService;
import com.sonicsw.mf.framework.directory.IFSStorage;
import com.sonicsw.mf.mgmtapi.config.constants.IBackupDirectoryServiceConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IDirectoryServiceConstants;

public final class ConfigTypesSeeder
{
    private static final String DIRECTORY_PREFIX = "/mx/configTypes/";
    private static final String DOMAIN_NAME      = "@DOMAIN_NAME@";
    private IDirectoryService m_ds;
    private String m_domainName;

    public static void main(String[] args)  throws Exception
    {
        if (args.length < 1 || args[0].equals("?"))
        {
            printUsage();
        }

        String file   = args[0];
        String dsFile = null;

        if(args.length > 1)
        {
            dsFile = args[1];
        }

        try
        {
            new ConfigTypesSeeder(file, dsFile);
        }
        catch(Throwable e)
        {
            ConfigHelper.printExceptionInfo(e);
        }
    }

    private ConfigTypesSeeder(String file, String dsFile)
    throws Exception
    {
        File dsDir = null;

        if(dsFile == null)
        {
            // we are going to create the DS under the system temp directory
            String tempDir = System.getProperty("java.io.tmpdir");

            if(tempDir == null || tempDir.trim().length() == 0)
            {
                throw new Exception("System temporary directory is not defined");
            }

            dsDir = new File(tempDir, "ds" + System.currentTimeMillis());

            // Create the directory
            dsDir.mkdir();

            // create a temporary in-process instance of the DS with the
            // storage at the specified temporary location
            m_ds = createDSInstance(dsDir);
        }
        else
        {
             String dsXML = getXMLStringFromFile(dsFile);

            // create an in-process instance of the DS with the storage
            // determined by the information in dsXML
            m_ds = createDSInstance(dsXML);
        }

        ConfigServer server = new ConfigServer();

        // create the /mx/configTypes directory in the DS
        try
        {
            m_ds.createDirectory("/mx");
            m_ds.createDirectory(DIRECTORY_PREFIX);
        }
        catch(Throwable e) { }

        try
        {
            String importParent = new File(file).getAbsoluteFile().getParent() + File.separator;

            // get the list of schemas we want to import
            String[] importEntries = getImportListFromFile(file);

            for(int i = 0; i < importEntries.length; i++)
            {
                String entry = importEntries[i];
                int    index = entry.indexOf(' ');

                if (index == -1)
                {
                    throw new Exception("Namespace is not defined for " + entry);
                }

                String xsdName = entry.substring(0, index);
                String ivName  = generateIVName(xsdName);
                String nSpace  = entry.substring(index + 1);

                if (xsdName.trim().length() == 0)
                {
                    throw new Exception("Name is not defined for " + entry);
                }

                if (nSpace.trim().length() == 0)
                {
                    throw new Exception("Namespace is not defined for " + entry);
                }

                String importName = importParent + xsdName;

                System.out.println("Importing Config Type from " + importName);

                Collection types = ConfigTypeDocument.parse(nSpace, importName, server, null);
                ConfigTypeImpl configType = null;
                
                for(Iterator iter = types.iterator(); iter.hasNext(); )
                {
                    configType = (ConfigTypeImpl)iter.next();

                    // then create the directory - this will include the version
                    // number part
                    try
                    {
                        m_ds.createDirectory(configType.getDSDirectory());
                    }
                    catch(Exception e)
                    {
                    }

                    // first delete any existing config type from the DS
                    m_ds.deleteElement(configType.getDSName(), null);

                    // Big assumption is that the last item walked in the iterator is the
                    // actual type that was imported - earlier items are super-types on
                    // which this type is based.
                    // We want to set initial values on this ConfigType.
                    if (!iter.hasNext())
                    {
                        // Import the Initial Values file if it exists...
                        File ivFile = new File(importParent + ivName);
                        if (ivFile.exists())
                        {
                            System.out.println("  with Initial Values from " + ivFile.toString());
                                    
                            InitialValuesImpl.attach(configType, ivFile.toString());
                        }
                    }
                    
                    // add the new config type into the DS
                    m_ds.setElement(configType.doneUpdate(), null);
                }
            }

            // delete the default view and back reference type list before we export
            deleteDirectory(IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_SCHEMA_DIR);

            // export from the root so that the XML includes required parent directories
            String generatedXML = m_ds.exportDirectoryToXML("/");

            if(dsFile == null)
            {
                // We are going to export to a file based on the original
                // import file location

                int index = file.lastIndexOf('.') ;

                String exportFile = null;

                if(index == -1)
                {
                    exportFile = file + ".seed";
                }
                else
                {
                    exportFile = file.substring(0, index) + ".seed";
                }

                BufferedWriter writer = new BufferedWriter(new FileWriter(exportFile));
                writer.write(generatedXML);
                writer.flush();
                writer.close();

                System.out.println("Successfully generated Config Types to " + exportFile);
            }
            else
            {
                // now place in the DS
                System.out.println("Seeding directory store ...");

                m_ds.importFromXML(generatedXML);

                System.out.println("Successfully seeded directory store");
            }
        }
        finally
        {
            if(m_ds != null)
            {
                m_ds.close();
            }

            if(dsDir != null)
            {
                System.out.println("Deleting temporary DS file system");
                deleteDir(dsDir);
            }
        }

        System.exit(0);
    }

    private void deleteDir(File dir) throws IOException
    {
        File[] files = dir.listFiles();

        for (int x = 0; x < files.length; x++)
        {
            if (files[x].isDirectory())
            {
                deleteDir(files[x]);
            }
            else
            {
                files[x].delete();
            }
        }
        dir.delete();
    }

    void deleteDirectory(String path)
    throws DirectoryServiceException
    {
        try
        {

            // Determine if directory exists
            try
            {
                m_ds.listAll(path);
            }
            catch (DirectoryServiceException e)
            {   // Directory does not exist: nothing to delete so return.
                return;
            }

            IElementIdentity[] ids = m_ds.listElements(path);
            for (int i = 0; i < ids.length; i++)
            {
                m_ds.deleteElement(ids[i].getName(), null);
            }

            IDirIdentity[] dirIds = m_ds.listDirectories(path);
            for (int i = 0; i < dirIds.length; i++)
            {
                deleteDirectory(dirIds[i].getName());
            }

            m_ds.deleteDirectory(path);

        }
        catch (Exception e)
        {
            DirectoryServiceException ex = new DirectoryServiceException("Failed to delete directory '" + path + "'");
            ex.setLinkedException(e);
            throw ex;
        }
    }

    private static void printUsage()
    {
        System.out.println();
        System.out.println("Usage: com.sonicsw.mx.config.tools.admin.ConfigTypesSeeder <import.lst> [ds.xml]");
        System.out.println();
        System.out.println("Where: <import.lst>   File containing names of type schemas to import.");
        System.out.println("           <ds.xml>   A DS boot file. If this is specified the seeder will");
        System.out.println("                      seed the data directly into the DS.");

        System.exit(1);
    }

    private String[] getImportListFromFile(String file)
    throws IOException
    {
        BufferedReader bReader = new BufferedReader(new FileReader(file));
        List list = new ArrayList();
        String line;

        while ((line = bReader.readLine()) != null)
        {
            list.add(line);
        }

        return (String[])list.toArray(new String[0]);
    }

    private String getXMLStringFromFile(String xmlFilename)
    throws Exception
    {
        FileInputStream fis = new FileInputStream(xmlFilename);
        byte[] bytes = new byte[fis.available()];
        fis.read(bytes);
        return new String(bytes);
    }

    private IDirectoryService createDSInstance(File hostDir)
    throws Exception
    {
        System.out.println("Creating temporary DS in " + hostDir.toString());

        m_domainName = DOMAIN_NAME;

        Hashtable directoryEnv = new Hashtable();
        directoryEnv.put(IDirectoryService.STORAGE_TYPE_ATTRIBUTE, IDirectoryService.PSE_STORAGE);
        directoryEnv.put(IFSStorage.HOST_DIRECTORY_ATTRIBUTE, hostDir.toString());
        DirectoryServiceFactory factory = new DirectoryServiceFactory(directoryEnv);

        return factory.createDirectoryService(m_domainName);
    }

    private IDirectoryService createDSInstance(String dsXML)
    throws Exception
    {
        System.out.println("Seeding directly to specified DS location");

        IDirElement dsConfig = ElementFactory.importElementFromXML(dsXML, null, IDirectoryServiceConstants.DS_TYPE);
        if (dsConfig == null)
        {
            dsConfig = ElementFactory.importElementFromXML(dsXML, null, IBackupDirectoryServiceConstants.DS_TYPE);
        }

        IAttributeSet dsAttributes = dsConfig.getAttributes();
        String hostDir    = (String)dsAttributes.getAttribute("HOST_DIRECTORY");
        String domainName = (String)dsAttributes.getAttribute("DOMAIN_NAME");
        Object tmp        = dsAttributes.getAttribute("FILE_SYSTEM_STORAGE");

        if (domainName == null || tmp == null)
        {
            throw new MFRuntimeException("Bad Directory Service Configuration - must contain DOMAIN_NAME and FILE_SYSTEM_STORAGE.");
        }

        if (!(tmp instanceof IAttributeSet))
        {
            throw new MFRuntimeException("Bad Directory Service Configuration - FILE_SYSTEM_STORAGE must be an attribute set.");
        }

        m_domainName = domainName;

        IAttributeSet fsStorage = (IAttributeSet)tmp;

        String hostDirDepricated = (String)fsStorage.getAttribute("HOST_DIRECTORY");
        String password          = (String)fsStorage.getAttribute("PASSWORD");

        // Creates the directory service object
        Hashtable directoryEnv = new Hashtable();
        directoryEnv.put(IDirectoryService.STORAGE_TYPE_ATTRIBUTE, IDirectoryService.PSE_STORAGE);
        if (hostDirDepricated != null)
        {
            directoryEnv.put(IFSStorage.FS_HOST_DIRECTORY_ATTRIBUTE, hostDirDepricated);
        }
        if (hostDir != null)
        {
            directoryEnv.put(IFSStorage.HOST_DIRECTORY_ATTRIBUTE, hostDir);
        }
        if ((password != null) && (password.length() != 0))
        {
            directoryEnv.put(IFSStorage.PASSWORD_ATTRIBUTE, password);
        }
        DirectoryServiceFactory factory = new DirectoryServiceFactory(directoryEnv);
        return factory.createDirectoryService(domainName);
    }
    
    private String generateIVName(String xsdName)
    {
        int lastIndex = xsdName.lastIndexOf('.');
        String prefix = xsdName.substring(0, Math.max(0, lastIndex));
        
        return prefix + ".iv";
    }
}
