package com.sonicsw.mx.config.tools.admin;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.StringTokenizer;

import com.sonicsw.mx.config.impl.ConfigServer;

import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;
import com.sonicsw.mf.framework.directory.DirectoryServiceFactory;
import com.sonicsw.mf.framework.directory.IDirectoryService;
import com.sonicsw.mf.framework.directory.IFSStorage;
import com.sonicsw.mf.mgmtapi.config.constants.IBackupDirectoryServiceConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IDirectoryServiceConstants;

public final class StorageHintsSeeder
{
    private static final String DOMAIN_NAME      = "@DOMAIN_NAME@";
    private IDirectoryService m_ds;
    private String m_domainName;

    public static void main(String[] args)  throws Exception
    {
        if (args.length < 1 || args[0].equals("?"))
        {
            printUsage();
        }

        String file   = args[0];
        String dsFile = null;

        if(args.length > 1)
        {
            dsFile = args[1];
        }

        try
        {
            new StorageHintsSeeder(file, dsFile);
        }
        catch(Exception e)
        {
            e.printStackTrace();
        }
    }

    private StorageHintsSeeder(String file, String dsFile)
    throws Exception
    {
        File dsDir = null;

        if(dsFile == null)
        {
            // we are going to create the DS under the system temp directory
            String tempDir = System.getProperty("java.io.tmpdir");

            if(tempDir == null || tempDir.trim().length() == 0)
            {
                throw new Exception("System temporary directory is not defined");
            }

            dsDir = new File(tempDir + "ds");

            // Create the directory
            dsDir.mkdir();

            // create a temporary in-process instance of the DS with the
            // storage at the specified temporary location
            m_ds = createDSInstance(dsDir);
        }
        else
        {
             String dsXML = getXMLStringFromFile(dsFile);

            // create an in-process instance of the DS with the storage
            // determined by the information in dsXML
            m_ds = createDSInstance(dsXML);
        }

        ConfigServer server = new ConfigServer();

        try
        {
            String importParent = new File(file).getAbsoluteFile().getParent() + File.separator;

            // get the list of schemas we want to import
            List list = getImportListFromFile(file);

            for(int i = 0; i < list.size(); i++)
            {
                String entry = (String)list.get(i);
                StringTokenizer tokenizer = new StringTokenizer(entry);
                if (tokenizer.countTokens() < 3)
                {
                    throw new Exception(entry + " must specify at least <type_name> <is_complex_type> <storage_directory>");
                }

                String typeName = tokenizer.nextToken();
                boolean isComplexType = new Boolean(tokenizer.nextToken()).booleanValue();
                String storageDir = tokenizer.nextToken();
                String postfix = null;
                if (tokenizer.hasMoreTokens())
                {
                    postfix = tokenizer.nextToken();
                }

                try
                {
                    ((IDirectoryAdminService)m_ds).createDirectory(storageDir);
                }
                catch (DirectoryServiceException e) { /* ignore */ }

                if (isComplexType)
                {
                    m_ds.setComplexStorageHint(typeName, storageDir);
                }
                else
                {
                    if (postfix == null)
                    {
                        m_ds.setStorageHint(typeName, storageDir);
                    }
                    else
                    {
                        m_ds.setStorageHint(typeName, postfix, storageDir);
                    }
                }
            }

            // delete all config elements in _MFSchema directory except storage_hints element before we export
            clearSchemaDir();

            // export from the root so that the XML includes required parent directories
            String generatedXML = m_ds.exportDirectoryToXML("/");

            if(dsFile == null)
            {
                // We are going to export to a file based on the original
                // import file location

                int index = file.lastIndexOf('.') ;

                String exportFile = null;

                if(index == -1)
                {
                    exportFile = file + ".seed";
                }
                else
                {
                    exportFile = file.substring(0, index) + ".seed";
                }

                BufferedWriter writer = new BufferedWriter(new FileWriter(exportFile));
                writer.write(generatedXML);
                writer.flush();
                writer.close();

                System.out.println("Successfully generated Config Types to " + exportFile);
            }
            else
            {
                // now place in the DS
                System.out.println("Seeding directory store ...");

                m_ds.importFromXML(generatedXML);

                System.out.println("Successfully seeded directory store");
            }
        }
        finally
        {
            if(m_ds != null)
            {
                m_ds.close();
            }

            if(dsDir != null)
            {
                System.out.println("Deleting temporary DS file system");
                deleteDir(dsDir);
            }
        }

        System.exit(0);
    }

    private void deleteDir(File dir) throws IOException
    {
        File[] files = dir.listFiles();

        for (int x = 0; x < files.length; x++)
        {
            if (files[x].isDirectory())
            {
                deleteDir(files[x]);
            }
            else
            {
                files[x].delete();
            }
        }
        dir.delete();
    }

    void clearSchemaDir()
    throws DirectoryServiceException
    {
        try
        {
            IElementIdentity[] ids = m_ds.listElements("/_MFSchema");
            for (int i = 0; i < ids.length; i++)
            {
                if (ids[i].getName().equals("/_MFSchema/storage_hints"))
                {
                    continue;
                }
                m_ds.deleteElement(ids[i].getName(), null);
            }

            IDirIdentity[] dirIds = m_ds.listDirectories("/_MFSchema");
            for (int i = 0; i < dirIds.length; i++)
            {
                deleteDirectory(dirIds[i].getName());
            }
        }
        catch (Exception e)
        {
            DirectoryServiceException ex = new DirectoryServiceException("Failed to delete directory '_MFSchema'");
            ex.setLinkedException(e);
            throw ex;
        }
    }

    void deleteDirectory(String path)
    throws DirectoryServiceException
    {
        try
        {

            // Determine if directory exists
            try
            {
                m_ds.listAll(path);
            }
            catch (DirectoryServiceException e)
            {   // Directory does not exist: nothing to delete so return.
                return;
            }

            IElementIdentity[] ids = m_ds.listElements(path);
            for (int i = 0; i < ids.length; i++)
            {
                m_ds.deleteElement(ids[i].getName(), null);
            }

            IDirIdentity[] dirIds = m_ds.listDirectories(path);
            for (int i = 0; i < dirIds.length; i++)
            {
                deleteDirectory(dirIds[i].getName());
            }

            m_ds.deleteDirectory(path);

        }
        catch (Exception e)
        {
            DirectoryServiceException ex = new DirectoryServiceException("Failed to delete directory '" + path + "'");
            ex.setLinkedException(e);
            throw ex;
        }
    }

    private static void printUsage()
    {
        System.out.println();
        System.out.println("Usage: com.sonicsw.mx.config.tools.admin.StorageHintsSeeder <import.lst> [ds.xml]");
        System.out.println();
        System.out.println("Where: <import.lst>   File containing storage hints to import.");
        System.out.println("           <ds.xml>   A DS boot file. If this is specified the seeder will");
        System.out.println("                      seed the data directly into the DS.");

        System.exit(1);
    }

    private List getImportListFromFile(String file)
    throws IOException
    {
        BufferedReader bReader = new BufferedReader(new FileReader(file));

        ArrayList list = new ArrayList();

        String line;

        while((line = bReader.readLine()) != null)
        {
            list.add(line);
        }

        return list;
    }

    private String getXMLStringFromFile(String xmlFilename)
    throws Exception
    {
        FileInputStream fis = new FileInputStream(xmlFilename);
        byte[] bytes = new byte[fis.available()];
        fis.read(bytes);
        return new String(bytes);
    }

    private IDirectoryService createDSInstance(File hostDir)
    throws Exception
    {
        System.out.println("Creating temporary DS in " + hostDir.toString());

        m_domainName = DOMAIN_NAME;

        Hashtable directoryEnv = new Hashtable();
        directoryEnv.put(IDirectoryService.STORAGE_TYPE_ATTRIBUTE, IDirectoryService.PSE_STORAGE);
        directoryEnv.put(IFSStorage.HOST_DIRECTORY_ATTRIBUTE, hostDir.toString());
        DirectoryServiceFactory factory = new DirectoryServiceFactory(directoryEnv);

        return factory.createDirectoryService(m_domainName);
    }

    private IDirectoryService createDSInstance(String dsXML)
    throws Exception
    {
        System.out.println("Seeding directly to specified DS location");

        IDirElement dsConfig = ElementFactory.importElementFromXML(dsXML, null, IDirectoryServiceConstants.DS_TYPE);
        if (dsConfig == null)
        {
            dsConfig = ElementFactory.importElementFromXML(dsXML, null, IBackupDirectoryServiceConstants.DS_TYPE);
        }

        IAttributeSet dsAttributes = dsConfig.getAttributes();
        String hostDir    = (String)dsAttributes.getAttribute("HOST_DIRECTORY");
        String domainName = (String)dsAttributes.getAttribute("DOMAIN_NAME");
        Object tmp        = dsAttributes.getAttribute("FILE_SYSTEM_STORAGE");

        if (domainName == null || tmp == null)
        {
            throw new MFRuntimeException("Bad Directory Service Configuration - must contain DOMAIN_NAME and FILE_SYSTEM_STORAGE.");
        }

        if (!(tmp instanceof IAttributeSet))
        {
            throw new MFRuntimeException("Bad Directory Service Configuration - FILE_SYSTEM_STORAGE must be an attribute set.");
        }

        m_domainName = domainName;

        IAttributeSet fsStorage = (IAttributeSet)tmp;

        String hostDirDepricated = (String)fsStorage.getAttribute("HOST_DIRECTORY");
        String password          = (String)fsStorage.getAttribute("PASSWORD");

        // Creates the directory service object
        Hashtable directoryEnv = new Hashtable();
        directoryEnv.put(IDirectoryService.STORAGE_TYPE_ATTRIBUTE, IDirectoryService.PSE_STORAGE);
        if (hostDirDepricated != null)
        {
            directoryEnv.put(IFSStorage.FS_HOST_DIRECTORY_ATTRIBUTE, hostDirDepricated);
        }
        if (hostDir != null)
        {
            directoryEnv.put(IFSStorage.HOST_DIRECTORY_ATTRIBUTE, hostDir);
        }
        if ((password != null) && (password.length() != 0))
        {
            directoryEnv.put(IFSStorage.PASSWORD_ATTRIBUTE, password);
        }
        DirectoryServiceFactory factory = new DirectoryServiceFactory(directoryEnv);
        return factory.createDirectoryService(domainName);
    }
}
