/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.mx.config.util;

import java.io.FileInputStream;
import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import javax.management.MBeanException;
import javax.management.ReflectionException;
import javax.management.RuntimeErrorException;
import javax.management.RuntimeMBeanException;
import javax.management.RuntimeOperationsException;

import com.sonicsw.ma.mgmtapi.config.MgmtException;
import com.sonicsw.mx.config.ConfigFactory;
import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.ConfigServiceRuntimeException;
import com.sonicsw.mx.config.IAttributeDescription;
import com.sonicsw.mx.config.IAttributeList;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigBeanFile;
import com.sonicsw.mx.config.IConfigElement;
import com.sonicsw.mx.config.IConfigPath;
import com.sonicsw.mx.config.IConfigPrototype;
import com.sonicsw.mx.config.IConfigServer;
import com.sonicsw.mx.config.impl.ConfigBeanFile;
import com.sonicsw.mx.config.impl.Util;

import com.sonicsw.mf.common.MFException;
import com.sonicsw.mf.common.MFProxyException;
import com.sonicsw.mf.common.MFProxyRuntimeException;
import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.config.query.BooleanExpression;
import com.sonicsw.mf.common.config.query.EqualExpression;
import com.sonicsw.mf.common.config.query.FromFolder;
import com.sonicsw.mf.common.config.query.Query;
import com.sonicsw.mf.common.config.query.Where;
import com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException;

/**
 * Helper class full of static methods that initialize and debug the
 * config layer.
 */
public class ConfigHelper
{
    public static final String MA_SYSTEM = "SYSTEM";

    public static final String FOLDER_SYSTEM = "" + com.sonicsw.mf.common.config.IMFDirectories.MF_DIR_SEPARATOR + "System";



    /**
     * @deprecated
     */
    public static IConfigBean createBean(IConfigServer server,
                                         String        newName,
                                         String        type,
                                         String        configVersion,
                                         String        productVersion)
        throws ConfigServiceException
    {
        return createBean(server, newName, type, configVersion, productVersion, false);
    }

    public static IConfigBean createBean(IConfigServer server,
                                         String        newName,
                                         String        type,
                                         String        configVersion,
                                         String        productVersion,
                                         boolean       isTemplate)
        throws ConfigServiceException
    {
        IConfigBean bean    = server.createConfigBean(newName, type, configVersion, isTemplate);
        HashMap     metaMap = new HashMap();

        metaMap.put(ConfigServerUtility.TYPE,            type);
        metaMap.put(ConfigServerUtility.PRODUCT_VERSION, productVersion);
        metaMap.put(ConfigServerUtility.CONFIG_VERSION,  configVersion);

        if (isTemplate)
        {
            metaMap.put(ConfigServerUtility.TEMPLATE_TYPE, ConfigServerUtility.TEMPLATE_TYPE_PROTOTYPE);
        }

        bean.setMetaAttributes(metaMap);

        return bean;
    }

    /**
     * Creates a template configuration bean from an existing regular bean.
     *
     * @param regularBean
     * @param newName
     * @return
     * @throws ConfigServiceException
     */
    public static IConfigBean createPrototypeBean(IConfigBean regularBean,
                                                  String      newName)
        throws ConfigServiceException
    {
        IConfigBean prototypeBean = (IConfigBean)regularBean.createPrototype(newName);
        Map         metaMap       = regularBean.getMetaAttributes();

        metaMap.put(ConfigServerUtility.TEMPLATE_TYPE, ConfigServerUtility.TEMPLATE_TYPE_PROTOTYPE);

        prototypeBean.setMetaAttributes(Util.mapToHashMap(metaMap));

        return prototypeBean;
    }

    public static IConfigBean createInstanceBean(IConfigPrototype prototypeBean,
                                                 String           newName)
        throws ConfigServiceException
    {
        IConfigBean instanceBean = (IConfigBean)prototypeBean.newInstance(newName);
        Map         metaMap      = ((IConfigBean)prototypeBean).getMetaAttributes();

        metaMap.put(ConfigServerUtility.TEMPLATE_TYPE, ConfigServerUtility.TEMPLATE_TYPE_INSTANCE);

        instanceBean.setMetaAttributes(Util.mapToHashMap(metaMap));

        return instanceBean;
    }

    public static IConfigBean copyBean(IConfigBean inBean, String newName)
        throws ConfigServiceException
    {
        IConfigBean outBean = null;
        
        if (inBean.isPrototypeInstance())
        {
            outBean = (IConfigBean)inBean.clonePrototypeInstance(newName);
        }
        else
        {
            outBean = (IConfigBean)inBean.clone();
            
            outBean.setName(newName);
        }
        
        Map metaMap = inBean.getMetaAttributes();

        outBean.setMetaAttributes(Util.mapToHashMap(metaMap));

        return outBean;
    }

    public static String buildPath(String base, String name)
    {
        StringBuffer sb = new StringBuffer(base);

        if (!base.endsWith("/"))
        {
            sb.append("/");
        }

        return sb.append(name).toString();
    }

    public static Object createChild(IAttributeMap top, IConfigPath childPath)
    {
        Object child  = null;

        try
        {
            IAttributeDescription ad = top.getAttributeDescription().getAttributeDescription(childPath);

            IConfigPath parentPath = ConfigFactory.createConfigPath(childPath);
            parentPath.removeComponent(childPath.size()-1);

            IAttributeMap parent = (parentPath.size() == 0) ? top
                   : (IAttributeMap)top.getAttribute(parentPath);

            if (ad.getType().equals(IAttributeMap.class))
            {
                child = parent.createAttributeMap(childPath.getLastComponent());
            }
            else
            if (ad.getType().equals(IAttributeList.class))
            {
                child = parent.createAttributeList(childPath.getLastComponent());
            }
        }
        catch (ConfigServiceException e)
        {
           e.printStackTrace();
        }

        return child;
    }

    public static void throwConfigServiceException(String message, Exception e)
        throws ConfigServiceException
    {
        if (e instanceof ConfigServiceException)
        {
            throw (ConfigServiceException)e;
        }
        else
        {
            throw new ConfigServiceException(message, e);
        }
    }

    public static String generateName(String name)
    {
        if(!name.equals(name.toUpperCase()))
        {
            // Name is mixed case, so leave as is but make first character
            // uppercase
            StringBuffer token = new StringBuffer(name);
            token.setCharAt(0, Character.toUpperCase(token.charAt(0)));
            return token.toString();
        }

        // Take a name such as CONTAINER_NAME and convert it to
        // ContainerName

        StringBuffer buffer = new StringBuffer();

        StringTokenizer st = new StringTokenizer(name, "_");

        while(st.hasMoreTokens())
        {
            StringBuffer token = new StringBuffer(st.nextToken().toLowerCase());
            token.setCharAt(0, Character.toUpperCase(token.charAt(0)));

            buffer.append(token);
        }
        return buffer.toString();
    }

    public static boolean isEqual(IAttributeMap data1, IAttributeMap data2)
    {
        for (Iterator i = data1.keySet().iterator(); i.hasNext(); )
        {
            String attributeName = (String)i.next();

            if (attributeName.startsWith("_MF"))
            {
                continue;
            }

            Object obj1 = null;
            Object obj2 = null;

            // Can't assume that get (IAttributeMap) is going to return null if
            // it can't find anything...the ConfigLayer throws exceptions if
            // referenced entries can't be found...it shouldn't do this really!
            //
            try { obj1 = data1.get(attributeName); } catch (ConfigServiceRuntimeException e) {}
            try { obj2 = data2.get(attributeName); } catch (ConfigServiceRuntimeException e) {}

            if (!doIsEqual(obj1, obj2))
            {
                return false;
            }
        }

        return true;
    }

    public static boolean isEqual(IAttributeList data1, IAttributeList data2)
    {
        for (int i = 0; i < data1.size(); i++)
        {
            Object obj1 = null;
            Object obj2 = null;

            // Can't assume that get (IAttributeMap) is going to return null if
            // it can't find anything...the ConfigLayer throws exceptions if
            // referenced entries can't be found...it shouldn't do this really!
            //
            try { obj1 = data1.get(i); } catch (ConfigServiceRuntimeException e) {}
            try { obj2 = data2.get(i); } catch (ConfigServiceRuntimeException e) {}

            if (!doIsEqual(obj1, obj2))
            {
                return false;
            }
        }

        return true;
    }

    protected static boolean doIsEqual(Object obj1, Object obj2)
    {
        // Check to see if both values are null - if so they are the same!
        //
        if ((obj1 == null) && (obj2 == null))
        {
            return true;
        }

        // Otherwise one of the values is null and therefore they are not the
        // same...
        //
        if ((obj1 == null) || (obj2 == null))
        {
            return false;
        }

        // We do a couple of instanceof checks here to catch that we have the
        // same type of object (config element/prototype) before doing a direct
        // equals on the class
        //
        if ((obj1 instanceof IConfigElement) &&
            (obj2 instanceof IConfigElement))
        {
            ;
        }
        else
        if ((obj1 instanceof IConfigPrototype) &&
            (obj2 instanceof IConfigPrototype))
        {
            ;
        }
        else
        if (!obj1.getClass().equals(obj2.getClass()))
        {
            return false;
        }

        if (obj1 instanceof IAttributeMap)
        {
            if (!isEqual((IAttributeMap)obj1, (IAttributeMap)obj2))
            {
                return false;
            }
        }
        else if(obj1 instanceof IAttributeList)
        {
            if (!isEqual((IAttributeList)obj1, (IAttributeList)obj2))
            {
                return false;
            }
        }
        else
        if (!obj1.equals(obj2))
        {
            return false;
        }

        return true;
    }

    public static boolean isInstanceOf(IConfigBean instance, IConfigBean prototype)
    {
        if (!instance.isPrototypeInstance())
        {
            return false;
        }

        return (instance.getPrototype() == prototype);
    }

    private static final List getInstancesList(IConfigPrototype prototype)
        throws ConfigServiceException
    {
        ArrayList list = new ArrayList();

        Iterator i = prototype.getPrototypeInstances().iterator();

        while (i.hasNext())
        {
            Object instance = i.next();

            if (instance != null)
            {
                list.add(instance);
            }
        }

        return list;
    }

    /**
     * Given a config prototype (template) this method returns an array of all
     * the instances of the prototype.
     *
     * @param prototype  The prototype config element
     * @return  An array of IConfigElement's representing the prototype's instances.
     * @throws ConfigServiceException
     */
    public static final IConfigElement[] getInstancesOf(IConfigPrototype prototype)
        throws ConfigServiceException
    {
        List iList = getInstancesList(prototype);

        return (IConfigElement[])iList.toArray(new IConfigElement[iList.size()]);
    }

    /**
     * Given a config prototype (template) this method returns an array of all
     * the instances of the prototype.
     *
     * @param prototype  The prototype config element
     * @return  An array of IConfigBean's representing the prototype's instances.
     * @throws ConfigServiceException
     */
    public static final IConfigBean[] getBeanInstancesOf(IConfigPrototype prototype)
        throws ConfigServiceException
    {
        List iList = getInstancesList(prototype);

        return (IConfigBean[])iList.toArray(new IConfigBean[iList.size()]);
    }

    /**
     * Helper method for easily (re)loading a bean in a different server.
     * Beans from different servers can NOT be mixed so this method provides
     * an easy way to switch a bean ref from one server to another.
     *
     * @param server The server underwhich we want to load the bean
     * @param bean  The bean we want referenced through the server
     * @return  The bean (relative to the server)
     * @throws ConfigServiceException
     */
    public static final IConfigBean loadConfigBean(IConfigServer server,
                                                   IConfigBean   bean)
        throws ConfigServiceException
    {
        return (IConfigBean)server.loadConfigElement(bean.getName());
    }

    /**
     * Helper method for easily (re)loading beans in a different server.
     * Beans from different servers can NOT be mixed so this method provides
     * an easy way to switch bean references from one server to another.
     *
     * @param server The server underwhich we want to load the bean
     * @param bean  The beans we want referenced through the server
     * @return  The beans (relative to the server)
     * @throws ConfigServiceException
     */
    public static final IConfigBean[] loadConfigBean(IConfigServer server,
                                                     IConfigBean[] bean)
        throws ConfigServiceException
    {
        IConfigBean[] res = new IConfigBean[bean.length];

        for (int i = 0; i < bean.length; i++)
        {
            res[i] = (IConfigBean)server.loadConfigElement(bean[i].getName());
        }

        return res;
    }

    /**
     * Makes an iterator on the attribute structure that allows modifications.
     *
     * @param attribute  Either an IAttributeList or IAttributeMap structure
     * @return           An iterator
     */
    public static Iterator getAttributeIterator(Object attribute)
    {
        if (attribute instanceof IAttributeList)
        {
            return ((IAttributeList)attribute).listIterator();
        }

        if (attribute instanceof IAttributeMap)
        {
            return ((IAttributeMap)attribute).keySet().iterator();
        }

        return new EmptyIterator();
    }

    public static Object getNextAttribute(Object attribute, Iterator i)
    {
        Object nextAttribute = i.next();

        if (attribute instanceof IAttributeList)
        {
            return nextAttribute;
        }

        if (attribute instanceof IAttributeMap)
        {
            return ((IAttributeMap)attribute).getAttribute((String)nextAttribute);
        }

        return null;
    }

    public static Object deleteAttribute(Object attribute, Object delObject)
    {
        Object res = null;

        if (attribute instanceof IAttributeList)
        {
            if (((IAttributeList)attribute).remove(delObject))
            {
                res = delObject;
            }
        }
        else
        if (attribute instanceof IAttributeMap)
        {
            Iterator i = ((IAttributeMap)attribute).keySet().iterator();

            while (i.hasNext())
            {
                String key   = (String)i.next();
                Object entry = ((IAttributeMap)attribute).getAttribute(key);

                if (entry == delObject)
                {
                    res = ((IAttributeMap)attribute).remove(key);
                    break;
                }
            }
        }

        return res;
    }

    /**
     * Adds the given object into the complex attribute (map or list).
     *
     * If the complex attribute is a list then the new object is appended to
     * the end of the list.
     * If the complex attribute is a map then the new object is set using a
     * new 'unique' key.
     *
     * @param attribute  The complex attribute (map or list)
     * @param addObject  The object being added into the complex attribute
     * @throws ConfigServiceException  Thrown if a failure occurs
     */
    public static void addAttribute(Object attribute, Object addObject)
        throws ConfigServiceException
    {
        if (attribute instanceof IAttributeList)
        {
            ((IAttributeList)attribute).addAttribute(addObject);
        }
        else
        if (attribute instanceof IAttributeMap)
        {
            ((IAttributeMap)attribute).setAttribute(ConfigFactory.createGUID(), addObject);
        }
    }

    /**
     * Set method that sets a value into the underlying complex attribute (list
     * or map) using the key.
     *
     * If the complex attribute is an attribute list then the key must be an
     * Integer indicating the set index or null if the value just needs to be
     * appended.
     * If the complex attribute is an attribute map then the key must be a
     * String (or effectively implement the toString() method) or null if
     * the value is new and needs to be added using a unique key.
     *
     * @param parent     The parent complex attribute map or list.
     * @param key        The key (String or Integer) that is used to set the
     *                   attribute value.
     * @param value      The value that is to be set into the complex attribute.
     * @throws ConfigServiceException  If the underlying set fails then an
     *                   exception is thrown.
     */
    public static void setAttribute(Object parent, Object key, Object value)
        throws ConfigServiceException
    {
        if (parent instanceof IAttributeList)
        {
            if (key != null)
            {
                if (!(key instanceof Integer))
                {
                    throw new ConfigServiceException("setAttribute on an AttributeList needs integer key: key was " + key.getClass().getName());
                }

                Integer index = (Integer)key;

                if (index.intValue() >= 0)
                {
                    ((IAttributeList)parent).setAttribute(index.intValue(), value);
                }
                else
                {
                    ((IAttributeList)parent).addAttribute(value);
                }
            }
            else
            {
                ((IAttributeList)parent).addAttribute(value);
            }
        }
        else
        if (parent instanceof IAttributeMap)
        {
            if (key != null)
            {
                ((IAttributeMap)parent).setAttribute(key.toString(), value);
            }
            else
            {
                ((IAttributeMap)parent).setAttribute(ConfigFactory.createGUID(), value);
            }
        }
    }

    /**
     * Finds the key (or index) of a value in a complex attribute.
     *
     * @param parent  Either an IAttributeMap or IAttributeList instance
     * @param child   The child attribute to find in the parent complex attribute
     * @return        Returns null if no match was found otherwise
     *                a String if the complex attribute is an IAttributeMap
     *                or an Integer if the complex attribute is an IAttributeList.
     */
    public static Object getAttributeKey(Object parent, Object child)
    {
        if (parent instanceof IAttributeMap)
        {
            IAttributeMap map = (IAttributeMap)parent;
            Iterator i = map.keySet().iterator();

            while (i.hasNext())
            {
                String mapKey = (String)i.next();
                Object mapValue = map.getAttribute(mapKey);

                if ((mapValue != null) && (mapValue == child))
                {
                    return mapKey;
                }
            }
        }
        else
        if (parent instanceof IAttributeList)
        {
            IAttributeList list = (IAttributeList)parent;

            for (int i = 0; i < list.size(); i++)
            {
                Object listValue  = list.getAttribute(i);

                if ((listValue != null) && (listValue == child))
                {
                    return new Integer(i);
                }
            }
        }

        return null;
    }

    static class EmptyIterator implements Iterator
    {
        EmptyIterator()
        {
        }

        @Override
        public boolean hasNext()
        {
            return false;
        }

        @Override
        public Object next()
        {
            return null;
        }

        @Override
        public void remove()
        {
        }
    }

    //-------------------------------------------------------------------------

    public static void createFile(IConfigServer configServer,
                                  String        dsPath,
                                  String        user)
        throws ConfigServiceException
    {

        try
        {
            // first check to see if the file already exists
            IConfigElement element = configServer.loadConfigElement(dsPath);

            if (element != null)
            {
                // Something already exists in the DS with this name - what is it ?
                if (!(element instanceof IConfigBeanFile))
                {
                    throw new ConfigServiceException("csu-import-file-exists", new Object[] { dsPath });
                }
            }
            else
            {
                String parent = dsPath.substring(0, dsPath.lastIndexOf('/'));

                // create the parent folders for this file
                if ((parent != null) && (parent.length() > 0))
                {
                    configServer.createPath(parent, true);
                    //configServer.createFolder(parent, null, true);

                }

                element = configServer.createConfigBeanFile(dsPath, user);

                HashMap meta = new HashMap();
                meta.put(ConfigServerUtility.TYPE, ConfigBeanFile.CONFIG_TYPE);
                element.setMetaAttributes(meta);
            }

            // ...and store the bean file
            //
            configServer.storeConfigElement(element);
        }
        catch(ConfigServiceException e)
        {
            throw e;
        }
        catch(Exception e)
        {
            throw new ConfigServiceException("csu-import-file-failed", new Object[] { "na" }, e);
        }
    }

    public static void importFile(IConfigServer configServer,
                                  String        dsPath,
                                  String        fsPath,
                                  String        user,
                                  boolean       overwrite)
        throws ConfigServiceException
    {
        FileInputStream stream = null;


        try
        {
            stream = new FileInputStream(fsPath);

            // first check to see if the file already exists
            IConfigElement element = configServer.loadConfigElement(dsPath);

            if (element != null)
            {
                // Something already exists in the DS with this name - what is it ?
                if (!(element instanceof IConfigBeanFile) || !overwrite)
                {
                    throw new ConfigServiceException("csu-import-file-exists", new Object[] { dsPath });
                }
            }
            else
            {
                String parent = dsPath.substring(0, dsPath.lastIndexOf('/'));

                // create the parent folders for this file
                if ((parent != null) && (parent.length() > 0))
                {
                    configServer.createPath(parent, true);
                    //configServer.createFolder(parent, null, true);

                }

                element = configServer.createConfigBeanFile(dsPath, user);

                HashMap meta = new HashMap();
                meta.put(ConfigServerUtility.TYPE, ConfigBeanFile.CONFIG_TYPE);
                element.setMetaAttributes(meta);
            }

            // Attach the input stream as the new contents...
            //
            ((IConfigBeanFile)element).setContents(stream);

            // ...and store the bean file
            //
            configServer.storeConfigElement(element);
        }
        catch(ConfigServiceException e)
        {
            throw e;
        }
        catch(Exception e)
        {
            throw new ConfigServiceException("csu-import-file-failed", new Object[] { fsPath }, e);
        }
        finally
        {
            if(stream != null)
            {
                try { stream.close(); } catch(Exception e) { }
            }
        }
    }

    public static String[] getOrderedFolders(String path)
    {
        ArrayList       res = new ArrayList();
        StringTokenizer st  = new StringTokenizer(path, "" + com.sonicsw.mf.common.config.IMFDirectories.MF_DIR_SEPARATOR, false);
        StringBuffer    sb  = new StringBuffer();

        while (st.hasMoreTokens())
        {
            sb.append(com.sonicsw.mf.common.config.IMFDirectories.MF_DIR_SEPARATOR);
            sb.append(st.nextToken());

            res.add(sb.toString());
        }

        return (String[])res.toArray(new String[res.size()]);
    }

    public static void setMetaAttribute(IConfigServer server, String path, String key, String value)
        throws ConfigServiceException
    {
        Map map = null;

        try
        {
            map = server.getMetaAttributes(path);
        }
        catch (Exception e)
        {
            e.printStackTrace();

            map = new HashMap();
        }

        map.put(key, value);

        server.setMetaAttributes(path, map);
    }

    /**
     * Checks to find a match on an element which contains a attribute key/value
     * under a specific logical path.
     * This is a quick and efficient way of finding a particular element.
     *
     * NOTE: if checking for uniqueness and the logical path is the same as
     *       an element you are trying to ensure uniqueness for, then this
     *       method will not produce the desired results. When using a transacted
     *       config server, the local element cache is also searched...so if you
     *       create a new Broker, and before committing query all other brokers,
     *       then the broker you just created will be returned in the results
     *       set and this method will return false!
     *
     * @param configServer  The Config Server connection to use when making
     *                      query. Note that for a transacted connection the
     *                      result set may contain (any new) elements just
     *                      created but NOT comitted.
     * @param logicalPath   The logical location underwhich the search is being
     *                      performed.
     * @param attrKey       The attribute key to match.
     * @param attrValue     The attribute value to find.
     * @return              true if the attribute key/value is unique, otherwise
     *                      false.
     * @throws ConfigServiceException  If the operation failed for whatever reason.
     */
    public static boolean isAttrValueUniqueWithinFolder(IConfigServer configServer,
                                                        String        logicalPath,
                                                        IConfigPath   attrKey,
                                                        String        attrValue)
        throws ConfigServiceException
    {
        return isAttrValueUniqueWithinFolder(configServer, logicalPath, attrKey.toString(), attrValue);
    }

    /**
     * Checks to find a match on an element which contains a attribute key/value
     * under a specific logical path.
     * This is a quick and efficient way of finding a particular element.
     *
     * NOTE: if checking for uniqueness and the logical path is the same as
     *       an element you are trying to ensure uniqueness for, then this
     *       method will not produce the desired results. When using a transacted
     *       config server, the local element cache is also searched...so if you
     *       create a new Broker, and before committing query all other brokers,
     *       then the broker you just created will be returned in the results
     *       set and this method will return false!
     *
     * @param configServer  The Config Server connection to use when making
     *                      query. Note that for a transacted connection the
     *                      result set may contain (any new) elements just
     *                      created but NOT comitted.
     * @param logicalPath   The logical location underwhich the search is being
     *                      performed.
     * @param attrKey       The attribute key to match.
     * @param attrValue     The attribute value to find.
     * @return              true if the attribute key/value is unique, otherwise
     *                      false.
     * @throws ConfigServiceException  If the operation failed for whatever reason.
     */
    public static boolean isAttrValueUniqueWithinFolder(IConfigServer configServer,
                                                        String        logicalPath,
                                                        String        attrKey,
                                                        String        attrValue)
        throws ConfigServiceException
    {
        // Build and execute query to find list of elements that have the
        // attribute set.
        //
        try
        {
            EqualExpression ee = new EqualExpression(new AttributeName(attrKey), attrValue);
            Where w = new Where(new BooleanExpression[]
                                {ee});
            Query query = (new Query()).setFrom(new FromFolder(logicalPath)).setWhere(w);
            Set set = configServer.listConfigElements(query);

            return set.isEmpty();
        }
        catch (Exception e)
        {
            ConfigServiceException ee = new ConfigServiceException("isAttrValueUniqueWithinFolder(" + logicalPath +
                                             ", " + attrKey + " = " + attrValue + ")");
            ee.setLinkedException(e);

            throw ee;
        }
    }

    public static boolean isAttrsValuesUniqueWithinFolder(IConfigServer configServer,
                                                          String        logicalPath,
                                                          String        beanName,
                                                          String[]      attrKeys,
                                                          String[]      attrValues)
        throws ConfigServiceException
    {
        // Build and execute query to find list of elements that have the
        // specified attributes set.
        //
        try
        {
            ArrayList booleanExpression = new ArrayList(attrKeys.length);

            for (int i = 0; i < attrKeys.length; i++)
            {
                booleanExpression.add(new EqualExpression(new AttributeName(attrKeys[i]), attrValues[i]));
            }

            Where w = new Where((BooleanExpression[])booleanExpression.toArray(new BooleanExpression[attrKeys.length]));
            Query query = (new Query()).setFrom(new FromFolder(logicalPath)).setWhere(w);
            Set set = configServer.listConfigElements(query);

            if (beanName != null && set.contains(beanName))
            {
                //filtered it from a cache
                set.remove(beanName);
            }

            return set.isEmpty();
        }
        catch (Exception e)
        {
            ConfigServiceException ee = new ConfigServiceException("isAttrsValuesUniqueWithinFolder(" + logicalPath  + ")");
            ee.setLinkedException(e);
            throw ee;
        }
    }

    public static String[] buildGroupNamesList(IConfigServer server, String path, String initialValue)
    {
        ArrayList list = new ArrayList();
        Query query = (new Query()).setFrom(new FromFolder(path));
        try
        {
            Set set = server.listConfigElements(query);

            if(initialValue != null)
            {
                list.add("<None>");
            }

            for(Iterator i = set.iterator(); i.hasNext();)
            {
                String name = (String)i.next();
                list.add(name.substring(name.lastIndexOf('/') + 1));
            }
        }
        catch(ConfigServiceException ce) {ce.printStackTrace();}

        String[] names = (String[])list.toArray(new String[list.size()]);

        return names;

    }

    public static void printExceptionInfo(Throwable e)
    {
        PrintWriter pw = new PrintWriter(System.err);

        printExceptionInfo(e, pw, false);
    }

    public static void printExceptionInfo(Throwable e, PrintWriter pw)
    {
        printExceptionInfo(e, pw, false);
    }

    public static void printExceptionInfo(Throwable e, PrintWriter pw, boolean isCause)
    {
        // don't trace wrappers
        if (e instanceof MFProxyException && ((MFProxyException)e).getActualException() != null)
        {
            e = ((MFProxyException)e).getActualException();
        }
        else
        if (e instanceof MFProxyRuntimeException && ((MFProxyRuntimeException)e).getActualException() != null)
        {
            e = ((MFProxyRuntimeException)e).getActualException();
        }
        else
        if (e instanceof ProxyRuntimeException && ((ProxyRuntimeException)e).getTargetException() != null)
        {
            e = ((ProxyRuntimeException)e).getTargetException();
        }
        
        synchronized (pw)
        {
            if (isCause)
            {
                pw.print("Caused by: ");
            }

            pw.println(e);
            StackTraceElement[] trace = e.getStackTrace();
            for (int i = 0; i < trace.length; i++)
            {
                pw.println("\tat " + trace[i]);
            }
            pw.flush();
        }

        if (e instanceof MgmtException && ((MgmtException)e).getLinkedException() != null)
        {
            printExceptionInfo(((MgmtException)e).getLinkedException(), pw, true);
        }
        else
        if (e instanceof ConfigServiceException && ((ConfigServiceException)e).getLinkedException() != null)
        {
            printExceptionInfo(((ConfigServiceException)e).getLinkedException(), pw, true);
        }
        else
        if (e instanceof MFException && ((MFException)e).getLinkedException() != null)
        {
            printExceptionInfo(((MFException)e).getLinkedException(), pw, true);
        }
        else
        if (e instanceof MFRuntimeException && ((MFRuntimeException)e).getLinkedException() != null)
        {
            printExceptionInfo(((MFRuntimeException)e).getLinkedException(), pw, true);
        }
        else
        if (e instanceof ReflectionException)
        {
            printExceptionInfo(((ReflectionException)e).getTargetException(), pw, true);
        }
        else
        if (e instanceof MBeanException)
        {
            printExceptionInfo(((MBeanException)e).getTargetException(), pw, true);
        }
        else
        if (e instanceof InvocationTargetException)
        {
            printExceptionInfo(((InvocationTargetException)e).getTargetException(), pw, true);
        }
        else
        if (e instanceof RuntimeOperationsException)
        {
            printExceptionInfo(((RuntimeOperationsException)e).getTargetException(), pw, true);
        }
        else
        if (e instanceof RuntimeMBeanException)
        {
            printExceptionInfo(((RuntimeMBeanException)e).getTargetException(), pw, true);
        }
        else
        if (e instanceof RuntimeErrorException)
        {
            printExceptionInfo(((RuntimeErrorException)e).getTargetError(), pw, true);
        }
        else
        if (e.getCause() != null)
        {
            printExceptionInfo(e.getCause(), pw, true);
        }
    }
}
