/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.mx.config.util;

import java.util.Objects;
import java.util.StringTokenizer;

import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.dirconfig.IDirElement;

/**
 * This class is a wrapped around a file or directory returned my the
 * SonicFSFileSystem. This class encapsultates information about the file,
 * it's name, size, createdBy and lastModified
 */
public class SonicFSFile
{
    protected long    m_creationTimestamp;
    protected long    m_lastModified;
    protected int     m_size;
    protected String  m_createdBy;
    protected String  m_fullname;
    protected boolean m_isDir = false;

    /**
     * Copy constructor to create a new SonicFSFile based on the specified
     * SonicFSFile
     */
    protected SonicFSFile(SonicFSFile copy)
    {
        m_creationTimestamp = copy.m_creationTimestamp;
        m_lastModified      = copy.m_lastModified;
        m_size              = copy.m_size;
        m_createdBy         = copy.m_createdBy;
        m_fullname          = copy.m_fullname;
        m_isDir             = copy.m_isDir;
    }

    /**
     * Create a SonicFSFile object for the specified filename
     */
    protected SonicFSFile(String name)
    {
        m_fullname = name;
    }

    /**
     * Helper method to create a SonicFSFile object with the specified filename
     */
    static SonicFSFile createFile(String name)
    {
        SonicFSFile info = new SonicFSFile(name);
        info.m_isDir = false;

        return info;
    }

    /**
     * Helper method to create a SonicFSFile object for a directory with the
     * specified name
     */
    static SonicFSFile createDirectory(String name)
    {
        SonicFSFile info = new SonicFSFile(name);
        info.m_isDir = true;

        return info;
    }

    /**
     * Helper method to create a SonicFSFile object for a file where the
     * information for the file is passed as a IDirElement. This IDirElement
     * is obtained from the DS and this method extracts information from the
     * IDirElement (e.g. size, lastModified) to populate the SonicFSFile
     */
    static SonicFSFile createFile(IDirElement element)
    {
        SonicFSFile info = createFile(element.getIdentity().getName());

        IAttributeSet set = element.getAttributes();

        info.m_creationTimestamp = ((Long)set.getAttribute(SonicFSFileSystem.CREATION_TIME)).longValue();
        info.m_lastModified      = ((Long)set.getAttribute(SonicFSFileSystem.LAST_MODIFIED_TIME)).longValue();
        info.m_size              = ((Integer)set.getAttribute(SonicFSFileSystem.SIZE)).intValue();
        info.m_createdBy         = ((String)set.getAttribute(SonicFSFileSystem.CREATED_BY));

        return info;
    }

    /**
     * Given a full path to a file, return the filename (the last component in
     * the path).
     */
    public static String toName(String name)
    {
        return name.substring(name.lastIndexOf('/')+1);
    }

    /**
     * Returns the creationTimestamp.
     * @return long
     */
    public long getCreationTimestamp()
    {
        return m_creationTimestamp;
    }

    /**
     * Returns the lastModified.
     * @return long
     */
    public long getLastModified()
    {
        return m_lastModified;
    }

    /**
     * Returns the name.
     * @return String
     */
    public String getName()
    {
        return toName(getFullName());
    }

    /**
     * Returns the full name.
     * @return String
     */
    public String getFullName()
    {
        return m_fullname;
    }

    /**
     * Returns the size.
     * @return int
     */
    public int getSize()
    {
        return m_size;
    }

    /**
     * Returns the creator
     * @return String
     */
    public String getCreatedBy()
    {
        return m_createdBy;
    }

    /**
     * Tests whether the file denoted by this resource info is a normal file.
     * A file is normal if it is not a directory and, in addition,
     * satisfies other system-dependent criteria.
     * @return true if and only if the file exists and is a normal file; false otherwise
     */
    public boolean isFile()
    {
        return !m_isDir;
    }

    /**
     * Tests whether the file denoted by this resource info is a directory.
     * @return true if and only if the directory exists; false otherwise
     */
    public boolean isDirectory()
    {
        return m_isDir;
    }

    /**
     * Tests whether this abstract pathname is absolute.
     *
     * A pathname is absolute if its prefix is "/".
     *
     * @return  true if this abstract pathname is absolute, false otherwise
     */
    public boolean isAbsolute()
    {
        return getFullName().startsWith("/");
    }

    /**
     * Tests whether this SonicFSFile is the parent of the specified file
     */
    public boolean isParent(SonicFSFile file)
    {
        if (file != null)
        {
            SonicFSFile parent = file.getParentFile();

            if ((parent != null) && parent.equals(this))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * Returns the pathname string of this abstract pathname's parent, or null
     * if this pathname does not name a parent directory.
     * The parent of an abstract pathname consists of the pathname's prefix,
     * if any, and each name in the pathname's name sequence except for the last.
     * If the name sequence is empty then the pathname does not name a parent
     * directory.
     *
     * @return  The pathname string of the parent directory named by this
     *          abstract pathname, or null if this pathname does not name a parent
     */
    public String getParent()
    {
        return toParent(getFullName());
    }

    /**
     * Returns the parent String for the specified path
     */
    public static String toParent(String name)
    {
        String[] c = split(name);

        return (c.length >= 1) ? combine(c, 0, c.length-1) : null;
    }

    /**
     * Returns the abstract pathname of this abstract pathname's parent, or null
     * if this pathname does not name a parent directory.
     * The parent of an abstract pathname consists of the pathname's prefix,
     * if any, and each name in the pathname's name sequence except for the last.
     * If the name sequence is empty then the pathname does not name a parent
     * directory.
     *
     * @return  The abstract pathname of the parent directory named by this
     *          abstract pathname, or null if this pathname does not name a parent
     */
    public SonicFSFile getParentFile()
    {
        String res = getParent();

        return (res != null) ? createDirectory(res) : null;
    }

    protected static String[] split(String name)
    {
        StringTokenizer st = new StringTokenizer(name, "/");
        String[] res = new String[st.countTokens()];
        int i = 0;
        while (st.hasMoreTokens())
        {
            res[i++] = st.nextToken();
        }

        return res;
    }

    protected static String combine(String[] s, int start, int stop)
    {
        StringBuffer sb = new StringBuffer("/");

        for (int i = start; i < stop; i++)
        {
            if (i != start)
            {
                sb.append('/');
            }

            sb.append(s[i]);
        }

        return sb.toString();
    }

    /**
     * Convert this SonicFSFile into a fully qualified URL e.g.sonicfs:///<path>
     * @return
     */
    public String toURL()
    {
        return SonicFSFileSystem.SCHEME + "://" + getFullName();
    }

    /**
     * Return the string representation of this SonicFSFile.
     * @return
     */
    @Override
    public String toString()
    {
        return getFullName();
    }

    /**
     * Test if this SonicFSFile is equal to the specified Object
     * @param o
     * @return
     */
    @Override
    public boolean equals(Object o)
    {
        if (o == null)
        {
            return false;
        }

        if (this.getClass() != o.getClass())
        {
            return false;
        }

        if (((SonicFSFile)o).isDirectory() != isDirectory())
        {
            return false;
        }

        return ((SonicFSFile)o).getFullName().equals(getFullName());
    }
    
    @Override
    public int hashCode() {
        return Objects.hash(isDirectory(), getFullName());
    }

    static String getCanonicalDirPath(String path) throws SonicFSException
    {
        if (!path.startsWith("/"))
        {
            throw new SonicFSException("\"" + path + "\" does not start with /");
        }

        StringBuffer sb = new StringBuffer();

        int pathLength = path.length();
        boolean prevWasSeparator = false;
        for (int i = 0; i < pathLength; i++)
        {
            char c = path.charAt(i);
            if (prevWasSeparator && c == '/')
            {
                continue;
            }

            sb.append(c);

            prevWasSeparator = c == '/';

        }

        String canonical = sb.toString();

        int canonicalLength = canonical.length();

        //Should end with  '/' if exists and is not the root directory
        if (canonical.charAt(canonicalLength - 1) != '/')
        {
            canonical += "/";
        }

        return canonical;


    }
}
