/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.mx.config.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import javax.management.ObjectName;

import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.IConfigServer;
import com.sonicsw.mx.config.impl.ConfigBeanFile;
import com.sonicsw.mx.config.impl.Util;

import com.sonicsw.mf.common.IDSTransaction;
import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDeltaDirElement;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.jmx.client.DirectoryServiceProxy;
import com.sonicsw.mf.jmx.client.JMSConnectorClient;

/**
 * This class provides an API for accessing files stored in the DS using
 * sonicfs://
 */
public class SonicFSFileSystem
{
    public static final String     SCHEME     = "sonicfs";
    public static final Comparator<SonicFSFile> COMPARATOR = new AlphaComparator();

    protected static final String CREATION_TIME      = "CREATION_TIME";
    protected static final String LAST_MODIFIED_TIME = "LAST_MODIFIED_TIME";
    protected static final String SIZE               = "SIZE";
    protected static final String CREATED_BY         = "CREATED_BY";
    protected static final String PERMISSIONS        = "PERMISSIONS";
    protected static final String HIDDEN             = "HIDDEN";
    protected static final String FILE_TYPE          = "MF_FILE";
    protected static final String TOOL_ATTRIBUTES    = "TOOL_ATTRIBUTES";

    protected IDirectoryFileSystemService m_dfs;
    protected String m_user;

    public static final char separatorChar = '/';
    public static final String separator = "/";

    /**
     * Note: This constructor is for internal use only.
     *
     * Constructor for the SonicFSFileSystem. This object will provide access
     * to the specified directory service
     * @param dfs The Directory Service being accessed
     * @param user The name of the user (will be written into sonicfs elements)
     */
    public SonicFSFileSystem(IDirectoryFileSystemService dfs, String user)
    {
        m_dfs  = dfs;
        m_user = user;
    }

    /**
     * Constructor for the SonicFSFileSystem. This object will provide access
     * to the specified directory service
     * @param connector The JMS Connection to use for the DS connection
     * @param domain The name of the DS domain
     * @param user The name of the user (will be written into sonicfs elements)
     * @throws SonicFSException
     */
    public SonicFSFileSystem(JMSConnectorClient connector, String domain, String user)
        throws SonicFSException
    {
        try
        {
            m_dfs = new DirectoryServiceProxy(connector,
                new ObjectName(domain + ".DIRECTORY SERVICE:ID=DIRECTORY SERVICE"));

            m_dfs.getDirectoryServiceVersion();

            m_user = user;
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to connect to Directory Service", e);
        }
    }

    /**
     * Close the DS if local to this JVM
     *
     */
    public void close() throws SonicFSException
    {
        try
        {
            m_dfs.close();
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to close the DIrectory Service " + e.getMessage(), e);
        }
    }


    /**
     * Get the directory service being used for this SonicFSFileSystem
     * @return the Directory Service reference
     */
    public IDirectoryFileSystemService getDirectoryService()
    {
        return m_dfs;
    }

    /**
     * Get the user name associated with this SonicFSFileSystem
     * @return the user as a String
     */
    public String getUser()
    {
        return m_user;
    }

    /**
     * Get the root folder as a String. For SonicFSFileSystem this is always '/'
     * @return the root folder name
     */
    public String getRoot()
    {
        return "/";
    }

    /**
     * Get the root folder as a SonicFSFile object.
     * @return the root folder
     */
    public SonicFSFile getRootFile()
    {
        return SonicFSFile.createDirectory(getRoot());
    }

    /**
     * List the contents of the DS at the specified path. The listing will
     * return all files and subfolders starting at path. If listSubFolders == true,
     * the method will recursively traverse subfolders. Only folders which can be read
     * by the user will be traversed.
     * @param path the DS path to list
     * @param listSubFolders if 'true', lists subfolders recursively
     * @return A String array of the contents of the path.
     * @throws SonicFSException
     */

    public String[] list(String path, boolean listSubFolders)
    throws SonicFSException
    {
        return list(path, listSubFolders, true, null);
    }

    /**
     * List the contents of the DS at the specified path. The listing will
     * return all files at the specified path. If recursive is true, the
     * method recurses through subfolders returning file names. If listFolders
     * is true, the method also returns folder names. If extension is non-null
     * only the files ending with the extension are returned. The extension filter
     * does not apply to folder names. Only folders that the user can read will be
     * traversed.
     * @param path the DS path to list
     * @param recursive if 'true', lists subfolders recursively
     * @param listFolders if true, the method will recurse through subfolders
     * @param extension If non-null, files will only be included in the return
     * array if their names end with the extension.
     * @return A String array of the contents of the path.
     * @throws SonicFSException
     */
    public String[] list(String path, boolean recursive, boolean listFolders, String extension)
    throws SonicFSException
    {
        if (isFile(path))
        {
            return new String[]{path};
        }

        path = SonicFSFile.getCanonicalDirPath(path);
        if (!recursive)
        {
            return list(path, extension, listFolders);
        }

        ArrayList<HashMap> childrenMaps = new ArrayList<HashMap>();

        ArrayList<String> result = new ArrayList<String>(childrenMaps.size());

        // in the spirit of the pre-existing functionality, add the
        // root of the call if we think it's a folder. This perpetuates
        // a difference between the recursive call and the non recursive call.
        // In 8.0, the non recursive call would not have returned the root, the recursive one
        // did, with the following test. See SonicFSFileSystem.recursiveList

        boolean isFile = isFile(path);
        if (listFolders || isFile)
        {
            result.add(path);
        }

        try
        {
            childrenMaps = m_dfs.recursiveList(path, listFolders, true, extension);
        }
        catch (DirectoryServiceException dirE)
        {
            throw new SonicFSException("Unable to list the contents of folder " + path, dirE);
        }

        for (HashMap map: childrenMaps)
        {
            String folderName = (String)map.get(IConfigServer.FOLDER_NAME);
            IElementIdentity ID = (IElementIdentity)map.get(IConfigServer.ELEMENT_IDENTITY);
            if (folderName != null && listFolders && isRealFolder(map))
            {
                result.add(folderName + IMFDirectories.MF_DIR_SEPARATOR);
            }
            else
            {
                String name = getValidFileName(map);
                if (name != null)
                {
                    result.add(ID.getName());
                }
            }
        }
        Collections.sort(result);
        return result.toArray(new String[result.size()]);
    }

    /**
     * List the contents of the DS at the specified path. The listing will
     * return all files at the specified path and any first level subfolders (returned
     * as <folder>/
     * @param path the DS path to list
     * @return A String array of the contents of the path.
     * @throws SonicFSException
     */
    public String[] list(String path) throws SonicFSException
    {
        return list(path, null, true);
    }

    /**
     * List the contents of the DS at the specified path. The listing will
     * return all files at the specified path and any subfolders (returned
     * as <folder>/. Elements returned are filtered by the file extension
     * given.
     * @param path the DS path to list
     * @param fileExtension The extension of filenames desired. A null value means return all elements.
     * @return A String array of the contents of the path.
     * @throws SonicFSException
     */
    private String[] list(String path, String fileExtension, boolean listFolders)
    throws SonicFSException
    {
        String usePath = path;

        // first check to see if the specified path exists and is a folder
        if(!isDirectory(usePath))
        {
            // Fix for Sonic00035516:
            // Hacky fix to handle the case where we are listing the contents of /workspace/
            // and the workspace is empty and we are using the 7.0 Eclipse DS handler.
            // In this case /workspace/ will return false - ie not a directory
            // However, if you pass in /workspace then this works
            // In the 7.5 handler this bug has been fixed

            if(usePath.equals("/workspace/"))
            {
                usePath = "/workspace";
            }

            if(!isDirectory(usePath))
            {
                throw new SonicFSException("'" + path + "' is not a valid directory");
            }
        }

        try
        {
            String name = null;
            List<String> result = new ArrayList<String>();

            Map[] maps = m_dfs.listFSAll(usePath, listFolders, fileExtension);

            // First add all of the folders into the returned data
            for (int i = 0; i < maps.length; i++)
            {
                // Each folder is of the format Folder1/
                name = (String)maps[i].get(IConfigServer.FOLDER_NAME);

                if (name != null)
                {
                    // Check that we have a 'real' file folder

                    if (isRealFolder(maps[i]))
                    {
                        result.add(name.substring(name.lastIndexOf('/') + 1) + '/');
                    }
                }
                else
                {
                    name = getValidFileName(maps[i]);

                    if (name != null)
                    {
                        result.add(name.substring(name.lastIndexOf('/') + 1));
                    }
                }
            }

            Collections.sort(result);

            return result.toArray(new String[result.size()]);
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to list '" + path + "' - " + e.getMessage(), e);
        }
    }

    /**
     * Obtain a listing of the DS contents at the specified location returning
     * the information as an array of SonicFSFile objects.
     * @param path the DS path to list
     * @return An array of SonicFSFile objects for the contents of the path.
     * @throws SonicFSException
     */
    public SonicFSFile[] listDetails(String path)
    throws SonicFSException
    {
        // first check to see if the specified path exists and is a folder
        if(!isDirectory(path))
        {
            throw new SonicFSException("'" + path + "' is not a valid directory");
        }

        try
        {
            List<SonicFSFile> result = new ArrayList<SonicFSFile>();

            Map[] maps = m_dfs.listFolders(path);

            for(int i = 0; i < maps.length; i++)
            {
                if (isRealFolder(maps[i]))
                {
                    result.add(SonicFSFile.createDirectory((String)maps[i].get(IConfigServer.FOLDER_NAME)));
                }
            }

            // We are going to get all elements in a particular folder in the DS
            IDirElement[] elements = m_dfs.getFSElements(path, false);

            for(int i = 0; i < elements.length; i++)
            {
                if(elements[i].getIdentity().getType().equals(FILE_TYPE))
                {
                    result.add(SonicFSFile.createFile(elements[i]));
                }
            }

            Collections.sort(result, COMPARATOR);

            return result.toArray(new SonicFSFile[result.size()]);
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to listDetails '" + path + "' - " + e.getMessage(), e);
        }
    }

    /**
     * List the contents of the DS at the specified path. The listing will
     * return all the immediate subfolders (returned as <folder>/)
     * @param path
     * @return
     * @throws SonicFSException
     */
    public String[] listFolders(String path)
    throws SonicFSException
    {
        // first check to see if the specified path exists and is a folder
        if(!isDirectory(path))
        {
            throw new SonicFSException("'" + path + "' is not a valid directory");
        }

        try
        {
            // First add all of the folders into the returned data
            Map[] maps = m_dfs.listFolders(path);
            List<String>  res  = new ArrayList<String>(maps.length);

            // Each folder is of the format Folder1/
            for (int i = 0; i < maps.length; i++)
            {
                if (isRealFolder(maps[i]))
                {
                    res.add((String)maps[i].get(IConfigServer.FOLDER_NAME) + '/');
                }
            }

            Collections.sort(res);

            return res.toArray(new String[res.size()]);
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to list '" + path + "' - " + e.getMessage(), e);
        }
    }

    /**
     * Obtain a listing of the DS contents at the specified location returning
     * the information as an arrange of SonicFSFile objects.
     * @param path
     * @return
     * @throws SonicFSException
     */
    public SonicFSFile[] listFolderDetails(String path)
    throws SonicFSException
    {
        // first check to see if the specified path exists and is a folder
        if(!isDirectory(path))
        {
            throw new SonicFSException("'" + path + "' is not a valid directory");
        }

        try
        {
            Map[] maps = m_dfs.listFolders(path);
            List<SonicFSFile>  res  = new ArrayList<SonicFSFile>(maps.length);

            for(int i = 0; i < maps.length; i++)
            {
                if (isRealFolder(maps[i]))
                {
                    res.add(SonicFSFile.createDirectory((String)maps[i].get(IConfigServer.FOLDER_NAME)));
                }
            }

            Collections.sort(res, COMPARATOR);

            return res.toArray(new SonicFSFile[res.size()]);
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to listFolderDetails '" + path + "' - " + e.getMessage(), e);
        }
    }

    /**
     * Obtain a listing of the DS contents at the specified location returning
     * the information as an arrange of SonicFSFile objects. It includes details
     * for the root folder.
     *
     * @return
     * @throws SonicFSException
     */
    public SonicFSFile[] listAllFolderDetails()
    throws SonicFSException
    {
        try
        {
            Map[] maps = m_dfs.listAllFolders();
            List<SonicFSFile>  res  = new ArrayList<SonicFSFile>(maps.length);
            List<String>  remove = new ArrayList<String>();

            for(int i = 0; i < maps.length; i++)
            {
                String name = (String)maps[i].get(IConfigServer.FOLDER_NAME);

                if (isRealFolder(maps[i]))
                {
                    res.add(SonicFSFile.createDirectory(name));
                }
                else
                {
                    remove.add(name);
                }
            }

            removeDanglingSubFolders(res, remove);

            Collections.sort(res, COMPARATOR);

            return res.toArray(new SonicFSFile[res.size()]);
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to listAllFolderDetails - " + e.getMessage(), e);
        }
    }

    /**
     * Obtain a recursive listing of all folders within the DS
     * @return
     * @throws SonicFSException
     */
    public String[] listAllFolders()
    throws SonicFSException
    {
        try
        {
            Map[] maps = m_dfs.listAllFolders();
            List<String>  res  = new ArrayList<String>();
            List<String>  remove = new ArrayList<String>();

            // Each folder is of the format Folder1/
            for (int i = 0; i < maps.length; i++)
            {
                String name = (String)maps[i].get(IConfigServer.FOLDER_NAME);

                if (isRealFolder(maps[i]))
                {
                    if (!name.endsWith("/"))
                    {
                        name += '/';
                    }

                    res.add(name);
                }
                else
                {
                    remove.add(name);
                }
            }

            removeDanglingSubFolders(res, remove);

            Collections.sort(res);

            return res.toArray(new String[res.size()]);
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to listAllFolders - " + e.getMessage(), e);
        }
    }

    /**
     * Get detailed information about the specified file or folder
     * @param path
     * @return
     * @throws SonicFSException
     */
    public SonicFSFile getDetails(String path)
    throws SonicFSException
    {
        if(path.length() == 0 || (path.length() == 1 && path.charAt(0) == '/'))
        {
            return SonicFSFile.createDirectory("/");
        }

        try
        {
            Map meta = m_dfs.getMetaAttributes(path);

            if (meta == null)
            {
                throw new Exception("'" + path + "' does not exist");
            }

            // Try as folder...
            if (meta.isEmpty() ||     // Complex type sub-folder
                (meta.get(IConfigServer.FOLDER_NAME) != null))
            {
                return SonicFSFile.createDirectory(path);
            }

            // Try as file...
            if (meta.get(IConfigServer.ELEMENT_IDENTITY) == null)
            {
                throw new Exception("'" + path + "' is not a valid file");
            }

            IDirElement element = m_dfs.getFSElement(path, false);

            if ((element == null) ||
                !element.getIdentity().getType().equals(FILE_TYPE))
            {
                throw new Exception("'" + path + "' is not a valid file");
            }

            return SonicFSFile.createFile(element);
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to get details for '" + path + "' - " + e.getMessage(), e);
        }
    }

    /**
     * Get detailed information about the specified file. If the path doesn't
     * reference a valid file then an exception is thrown.
     * @param path
     * @return
     * @throws SonicFSException
     */
    public SonicFSFile getFileDetails(String path)
    throws SonicFSException
    {
        SonicFSFile file = getDetails(path);

        if(file.isDirectory())
        {
            throw new SonicFSException("'" + path + "' is a directory");
        }

        return file;
    }

    /**
     * Get detailed information for a set of specified DS locations
     * @param path
     * @return
     * @throws SonicFSException
     */
    public SonicFSFile[] getDetails(String[] path)
    throws SonicFSException
    {
        List<SonicFSFile> result = new ArrayList<SonicFSFile>();

        for(int i = 0; i < path.length; i++)
        {
            result.add(getDetails(path[i]));
        }

        return result.toArray(new SonicFSFile[result.size()]);
    }

    /**
     * Check to see if a path is valid in the DS
     * @param path
     * @return
     */
    public boolean exists(String path)
    {
        try
        {
            getDetails(path);

            return true;
        }
        catch(Exception e)
        {
        }
        return false;
    }

    /**
     * Check to see if a path is a valid file in the DS
     * @param path
     * @return
     */
    public boolean isFile(String path)
    {
        try
        {
            SonicFSFile file = getDetails(path);

            return file.isFile();
        }
        catch(Exception e)
        {
        }
        return false;
    }

    /**
     * Check to see if a path is a valid directory in the DS
     * @param path
     * @return
     */
    public boolean isDirectory(String path)
    {
        try
        {
            SonicFSFile file = getDetails(path);

            return file.isDirectory();
        }
        catch(Exception e)
        {
        }
        return false;
    }

    /**
     * Create the specified directory. The parent directory must already exist
     * @param path
     * @throws SonicFSException
     */
    public SonicFSFile createDirectory(String path)
    throws SonicFSException
    {
        try
        {
            m_dfs.createFolder(path, true);

            return SonicFSFile.createDirectory(path);
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to create directory '" + path + "' - " + e.getMessage(), e);
        }
    }

    /**
     * Create a directory under the specified subfolder
     * @param baseFolder
     * @param newSubFolderName
     * @return
     * @throws SonicFSException
     */
    public SonicFSFile createDirectory(SonicFSFile baseFolder, String newSubFolderName)
    throws SonicFSException
    {
        if (!baseFolder.isDirectory())
        {
            throw new SonicFSException("Failed to create folder : baseFolder is not a folder");
        }

        SonicFSFile newFolder = SonicFSFile.createDirectory(baseFolder.getFullName() + "/" + newSubFolderName);

        try
        {
            m_dfs.createFolder(newFolder.getFullName(), true);
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to create directory '" + newFolder + "' - " + e.getMessage(), e);
        }

        return newFolder;
    }

    /**
     * Create the specified directory. This method will create all directories
     * in the path.
     * @param path
     * @throws SonicFSException
     */
    public void createDirectoryPath(String path)
    throws SonicFSException
    {
        try
        {
            StringTokenizer st = new StringTokenizer(path, "/");
            StringBuffer buffer = new StringBuffer();

            while (st.hasMoreTokens())
            {
                buffer.append("/").append(st.nextToken());
                m_dfs.createFolder(buffer.toString(), true);
            }
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to create directory path '" + path + "' - " + e.getMessage(), e);
        }
    }

    /**
     * Delete the specified directory. This method will fail if the directory
     * isn't empty.
     * @param path
     * @throws SonicFSException
     */
    public void deleteDirectory(String path)
    throws SonicFSException
    {
        try
        {
            m_dfs.deleteFolder(path);
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to delete directory '" + path + "' - " + e.getMessage(), e);
        }
    }

    /**
     * Recursively delete all files and folders under the specified path
     * including the path folder itself.
     * Note this method will fail if a folder contains a configuration.
     * @param path
     * @throws SonicFSException
     */
    public void deleteDirectoryPath(String path)
    throws SonicFSException
    {
        try
        {
            String name;

            Map[] maps = m_dfs.listFSElements(path);

            // Use a transaction to delete multiple elements at the same time
            IDSTransaction txn = m_dfs.createTransaction();

            for(int i = 0; i < maps.length; i++)
            {
                name = getValidFileName(maps[i]);
                if(name != null)
                {
                    txn.addDeleteElement(name);
                }
            }

            m_dfs.executeTransaction(txn);

            maps = m_dfs.listFolders(path);

            for(int i = 0; i < maps.length; i++)
            {
                name = (String)maps[i].get(IConfigServer.FOLDER_NAME);
                deleteDirectoryPath(name);
            }
            m_dfs.deleteFolder(path);
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to delete directory path '" + path + "' - " + e.getMessage(), e);
        }
    }

    /**
     * Rename either a folder or a file from oldPath to newPath
     * @param oldPath
     * @param newPath
     * @throws SonicFSException
     */
    public void rename(String oldPath, String newPath)
    throws SonicFSException
    {
        if (oldPath == null)
        {
            throw new SonicFSException("Cannot rename( >>> null <<<, " + newPath + ")");
        }

        if (newPath == null)
        {
            throw new SonicFSException("Cannot rename(" + oldPath + ", >>> null <<<)");
        }

        if (oldPath.equals(getRoot()))
        {
            throw new SonicFSException("Cannot rename the root folder");
        }

        if (oldPath.equals(newPath))
        {
            throw new SonicFSException("Cannot rename : oldPath and newPath are identical");
        }

        try
        {
            m_dfs.rename(oldPath, newPath);
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to rename '" + oldPath + "' to '" + newPath + "' - " + e.getMessage(), e);
        }
    }

    /**
     * Rename either a folder or a file where the original object is identied
     * by the specified SonicFSFile object and the new name is specified as
     * a String. This method then returns the new SonicFSFile object for
     * the renamed object
     * @param oldFile
     * @param newName
     * @return
     * @throws SonicFSException
     */
    public SonicFSFile rename(SonicFSFile oldFile, String newName)
    throws SonicFSException
    {
        if (oldFile == null)
        {
            throw new SonicFSException("Cannot rename( >>> null <<<, " + newName + ")");
        }

        if (newName == null)
        {
            throw new SonicFSException("Cannot rename(" + oldFile + ", >>> null <<<)");
        }

        if (oldFile.equals(getRootFile()))
        {
            throw new SonicFSException("Cannot rename the root folder");
        }

        StringBuilder newFullPath = new StringBuilder(oldFile.getParent());
        if (!newName.startsWith(separator))
        {
             newFullPath.append(separator);
        }
        newFullPath.append(newName);
        SonicFSFile newFile = SonicFSFile.createDirectory(newFullPath.toString());

        rename(oldFile.getFullName(), newFile.getFullName());

        return newFile;
    }

    /**
     * Get the content of the specified file returning the content as an InputStream.
     * @param path the file to read
     * @return the input stream or null
     * @throws SonicFSException if we cannot read the file
     */
    public InputStream getInputStream(String path)
    throws SonicFSException
    {
      // First check to see if this is a valid file
      SonicFSFile file = getFileDetails(path);
      
      // The get the blob attached to this file
      IBlob blob;
      try {
        blob = m_dfs.getFSBlob(path, false);
      } catch (DirectoryServiceException e) {
        throw new SonicFSException("Failed to read input stream '" + path + "' - " + e.getMessage(), e);
      }

      if (blob == null)
    {
        return null;
    }

      return blob.getBlobStream();
    }
    
    /**
     * Get the content of the specified file returning the content as a String
     * @param path
     * @return
     * @throws SonicFSException
     */
    public String getContent(String path)
    throws SonicFSException
    {
        // First check to see if this is a valid file
        SonicFSFile file = getFileDetails(path);

        ByteArrayOutputStream os = getBlobOutputStream(path);

        if(os == null)
        {
            return null;
        }

        String content = os.toString();

        try { os.close(); } catch(Exception e) { }

        return content;
    }

    /**
     * Get the content of the specified file returning the content as a byte[]
     * @param path
     * @return
     * @throws SonicFSException
     */
    public byte[] getContentBytes(String path)
    throws SonicFSException
    {
        // First check to see if this is a valid file
        SonicFSFile file = getFileDetails(path);

        ByteArrayOutputStream os = getBlobOutputStream(path);

        if(os == null)
        {
            return null;
        }

        byte[] content = os.toByteArray();

        try { os.close(); } catch(Exception e) { }

        return content;
    }

    /**
     * Delete the specified file. An exception is thrown if this path specifies
     * a folder or a non-file configuration in the DS.
     * @param path
     * @throws SonicFSException
     */
    public void deleteFile(String path)
    throws SonicFSException
    {
        try
        {
            // first see if the element already exists in the DS.
            IDirElement element = m_dfs.getFSElement(path, true);

            if(element == null)
            {
                throw new Exception("The file '" + path + "' does not exist");
            }

            // if it does then check its type
            if (!element.getIdentity().getType().equals(FILE_TYPE))
            {
                throw new Exception("'" + path + "' is not a valid file");
            }

            m_dfs.detachFSBlob((IDeltaDirElement)element.doneUpdate());
            m_dfs.deleteFSElement(path);
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to delete file '" + path + "' - " + e.getMessage(), e);
        }
    }

    /**
     * Create a file in sonicfs with the specified file name and the specified
     * content. If the file already exists, or a folder or other configuration
     * already exists then an exception is thrown.
     * @param path The logical name of the file in the DS
     * @param file The disk file with the contents for the DS file. An InputStream
     * will be created from this file and the contents read to create the new
     * DS file.
     * @throws SonicFSException If there is a problem reading the file on disk
     */
    public void createFile(String path, File file)
    throws SonicFSException
    {
    	FileInputStream stream = null;
    	try
    	{
    		stream = new FileInputStream(file);
    	}
    	catch (Exception ex)
    	{
    		throw new SonicFSException("Unable to create file stream for " + file + ": " + ex.toString(), ex);
    	}
    	try
    	{
            createFile(path, stream, (int)file.length());
    	}
    	finally
    	{
    		try
    		{
    		    stream.close();
    		}
    		catch (IOException ioE)
    		{
    			// Ignore Exceptions during resource cleanup.
    		}
    	}
    }

    /**
     * Create a file in sonicfs with the specified file name and the specified
     * content. If the file already exists, or a folder or other configuration
     * already exists then an exception is thrown.
     * @param path
     * @param content
     * @throws SonicFSException
     */
    public void createFile(String path, String content)
    throws SonicFSException
    {
        createFile(path, content.getBytes());
    }

    /**
     * Create a file in sonicfs with the specified file name and the specified
     * content. If the file already exists, or a folder or other configuration
     * already exists then an exception is thrown.
     * @param path The logical name of the file in the DS
     * @param content The contents as a byte array
     * @throws SonicFSException
     */
    public void createFile(String path, byte[]content)
        throws SonicFSException
    {
    	if (content != null)
        {
            createFile(path, new ByteArrayInputStream(content), content.length);
        }
        else
        {
            createFile(path, null, 0);
        }
    }

    /**
     * Create a file in sonicfs with the specified file name and the specified
     * content. If the file already exists, or a folder or other configuration
     * already exists then an exception is thrown.
     * @param path The logical name of the file element in the DS
     * @param stream The input stream for the contents of the file
     * @param size The size of the entire contents of the file.
     * @throws SonicFSException
     */
    public void createFile(String path, InputStream stream, int size)
    throws SonicFSException
    {
        try
        {
            // first check to see if this file exists
            HashMap meta = m_dfs.getMetaAttributes(path);

            if(meta != null)
            {
                if (meta.get(IConfigServer.FOLDER_NAME) != null)
                {
                    throw new Exception("A folder called '" + path + "' already exists");
                }

                throw new Exception("'" + path + "' already exists");
            }

            IDirElement element = ElementFactory.createElement(path,
                ConfigBeanFile.CONFIG_TYPE, ConfigBeanFile.CONFIG_VERSION);

            String user = (m_user != null) ? m_user : "";
            Long timeNow = new Long(System.currentTimeMillis());

            IAttributeSet set = element.getAttributes();

            set.setLongAttribute(SonicFSFileSystem.CREATION_TIME, timeNow);
            set.setStringAttribute(SonicFSFileSystem.CREATED_BY, user);
            set.setLongAttribute(SonicFSFileSystem.LAST_MODIFIED_TIME, timeNow);
            set.setIntegerAttribute(SonicFSFileSystem.SIZE, new Integer(size));
            set.setStringAttribute(SonicFSFileSystem.PERMISSIONS, "");
            set.setBooleanAttribute(SonicFSFileSystem.HIDDEN, Boolean.FALSE);

            IDSTransaction txn = m_dfs.createTransaction();

            if(stream != null)
            {
                //m_dfs.attachFSBlob(element, content);
                txn.addAttachBlob(element.doneUpdate(), stream);
            }
            else
            {
                //m_dfs.createFSElement(element);
                txn.addCreateElement(element);
            }

            meta = new HashMap();
            meta.put(TOOL_ATTRIBUTES, "TYPE=" + FILE_TYPE);
            // m_dfs.setMetaAttributes(path, meta);
            txn.addSetAttributes(path, meta);

            m_dfs.executeTransaction(txn);
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to create file - " + e.getMessage(), e);
        }
    }

    /**
     * Update the contents of the specified file where the content is stored
     * in the specified file.
     * If the file doesn't already exist in the DS and create == true
     * then the file will be created.
     * If the file doesn't already exist in the DS and create == false
     * then an exception is thrown.
     * @param path The logical name of the file in the DS
     * @param file The disk file where the contents of the updated DS file are read from
     * @param create true if the file should be created if it's not already in the DS
     * @throws SonicFSException
     */
    public void updateFile(String path, File file, boolean create)
    throws SonicFSException
    {
    	FileInputStream stream = null;
    	try
    	{
    		stream = new FileInputStream(file);
    	}
    	catch (Exception ex)
    	{
    		throw new SonicFSException("Unable to create a atream for the new file from " + file + ": " + ex.toString(), ex);
    	}
    	try
    	{
            updateFile(path, stream, create, (int)file.length());
    	}
    	finally
    	{
    		try
    		{
    		stream.close();
    		}
    		catch (IOException ioE)
    		{
                // Ignore Exceptions during resource cleanup.

    		}
    	}
    }

    /**
     * Update the contents of the specified file using the given string
     * contents.
     * If the file doesn't already exist in the DS and create == true
     * then the file will be created.
     * If the file doesn't already exist in the DS and create == false
     * then an exception is thrown.
     * @param path
     * @param content
     * @param create
     * @throws SonicFSException
     */
    public void updateFile(String path, String content, boolean create)
    throws SonicFSException
    {
        updateFile(path, content.getBytes(), create);
    }

    public void updateFile(String path, byte[]content, boolean create)
        throws SonicFSException
    {
    	if (content == null)
        {
            updateFile(path, null, create, 0);
        }
        else
        {
            updateFile(path, new ByteArrayInputStream(content), create, content.length);
        }
    }

    /**
     * Update the contents of the specified file using the given stream
     * contents.
     * If the file doesn't already exist in the DS and create == true
     * then the file will be created.
     * If the file doesn't already exist in the DS and create == false
     * then an exception is thrown.
     * @param path the file name
     * @param stream the content
     * @param create if to create the file
     * @param size the size of the input stream
     * @throws SonicFSException
     */
    public void updateFile(String path, InputStream stream, boolean create, int size)
    throws SonicFSException
    {
        try
        {
            // first see if the element already exists in the DS.
            IDirElement element = m_dfs.getFSElement(path, true);

            // if it does and we don't want the update to create the element
            // then throw exception
            if(element != null)
            {
                // Element already exists - don't allow an update if it isn't a sonicfs file.
                if (!element.getIdentity().getType().equals(FILE_TYPE))
                {
                    throw new Exception("'" + path + "' is not a valid file");
                }
            }
            else
            {
                // element doesn't exist - error if we don't want to create it
                if(!create)
                {
                    throw new Exception("'" + path + "' doesn't exist");
                }
            }

            if(element == null)
            {
                // if it doesn't already exist and we want to create it then
                // use the createFile() code
            	if (create) {
            		createFile(path, stream, size);
            	}
            }
            else
            {
                // Element already exists and we want to update its content
                // to the new content
                Long timeNow = new Long(System.currentTimeMillis());

                IAttributeSet set = element.getAttributes();

                set.setLongAttribute(SonicFSFileSystem.LAST_MODIFIED_TIME, timeNow);
                set.setIntegerAttribute(SonicFSFileSystem.SIZE, new Integer(size));

                // m_dfs.attachFSBlob(element, content);

                IDSTransaction txn = m_dfs.createTransaction();

                txn.addAttachBlob(element.doneUpdate(), stream);

                m_dfs.executeTransaction(txn);
            }
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to update file - " + e.getMessage(), e);
        }
    }

    public void importFiles(String fsPath, String dsPath) throws SonicFSException
    {
        try
        {
            if (fsPath == null || fsPath.length() < 1 || dsPath == null || dsPath.length() < 1)
            {
                throw new SonicFSException("importFiles bad parameter(s) - fsPath and dsPath must be passed");
            }

            //Very unlikely, but this cases is handled for completness
            if (fsPath.equals("/"))
            {
                String[] subList = new File(fsPath).list();
                for (int i = 0; i < subList.length; i++)
                {
                    importFiles("/" + subList[i], dsPath);
                }
            }

            File fsFile = new File(fsPath);
            if (!fsFile.exists())
            {
                throw new SonicFSException("\"" + fsFile.getAbsolutePath() + "\" does not exist");
            }

            if (fsFile.isFile())
            {
                 createPath(dsPath);
                 createFile(dsPath + (dsPath.equals("/") ? "" : "/") + fsFile.getName(), fsPath);
                 return;
            }

            ArrayList<String> files = listFiles(fsFile.getAbsolutePath());
            for (int i = 0; i < files.size(); i++)
            {
                String dsFilePath =  SonicFSFile.getCanonicalDirPath(dsPath) + files.get(i);
                File fsFilePath = new File(fsFile.getParent(), files.get(i));

                //Find out whether the file already exists
                SonicFSFile dsFile = null;
                try
                {
                    dsFile = getDetails(dsFilePath);
                }
                catch (Exception e){}

                if (dsFile != null)
                {
                    if (dsFile.isFile())
                    {
                        deleteFile(dsFilePath);
                    }
                    else
                    {
                        throw new SonicFSException("Cannot import \"" + fsFilePath.getAbsolutePath() + "\" since \"" + dsFilePath + "\" is a folder");
                    }
                }

                createPath(new SonicFSFile(dsFilePath).getParent());
                createFile(dsFilePath, fsFilePath);
            }
        }
        catch (Exception e)
        {
            if (e instanceof SonicFSException)
            {
                throw (SonicFSException)e;
            }
            else
            {
                throw new SonicFSException(e.toString(), e);
            }
        }
    }

    public void deleteFiles(String path) throws SonicFSException
    {
        String canonicalPath = SonicFSFile.getCanonicalDirPath(path);
        SonicFSFile file = getDetails(canonicalPath);
        if (file.isFile())
        {
            deleteFile(canonicalPath);
            return;
        }

        SonicFSFile[] list = listDetails(canonicalPath);

        for (int i = 0; i < list.length; i++)
        {
            deleteFiles(list[i].getFullName());
        }

        deleteDirectory(canonicalPath);
    }

    /**
     * Return an InputStream for a zip file containing the files named in filenames
     * Each fileName is tested for read management permissions and only those which the
     * user making the request can read will be returned.
     * @param fileNames A list of DS file names
     * @return An InputStream to the zip file created to contain all the files in fileNames
     * that the user is allowed to read. If no files are zipped, null is returned
     * @throws DirectoryServiceException if there is a problem accessing the files
     */

    public InputStream getFiles(String[] fileNames) throws SonicFSException
    {
        try
        {
            IBlob zipBlob = m_dfs.getFiles(fileNames);
            if (zipBlob != null)
            {
                return zipBlob.getBlobStream();
            }
            else
            {
                return null;
            }
        }
        catch (DirectoryServiceException dirE)
        {
            throw new SonicFSException("SonicFSFileSystem.getFiles error ", dirE);
        }
    }

    /**
     * Return an InputStream for a zip file containing the files contained in path
     * and its subfolders, if recurse if set to true.
     * Each fileName is tested for read management permissions and only those which the
     * user making the request can read will be returned.
     * @param path The folder to start searching files from
     * @param recurse If set to true, traverse subfolders
     * @param extension If set, return only those elements with a name that ends
     * with the extension.
     * @param fileNames A list of DS file names
     * @return An InputStream to the zip file created to contain all the files in fileNames
     * that the user is allowed to read. If no files are zipped, null is returned.
     * @throws DirectoryServiceException if there is a problem accessing the files
     */

    public InputStream getFiles(String path, boolean recurse, String extension) throws SonicFSException
    {
        try
        {
            IBlob zipBlob = m_dfs.getFiles(path, recurse, extension);
            if (zipBlob != null)
            {
                return zipBlob.getBlobStream();
            }
            else
            {
                return null;
            }
        }
        catch (DirectoryServiceException dirE)
        {
            throw new SonicFSException("SonicFSFileSystem.getFiles error ", dirE);
        }
    }


    private void createPath (String dsPath) throws Exception
    {
        SonicFSFile dsDir = null;
        try
        {
            dsDir = getDetails(dsPath);
        }
        catch (Exception e){}

        if (dsDir != null)
        {
            if (dsDir.isDirectory())
            {
                return;
            }
            else
            {
                throw new Exception("Cannot create the \"" + dsPath + "\" folder");
            }
        }
        else
        {
            createPath(new SonicFSFile(dsPath).getParent());
            createDirectory(dsPath);
        }

    }

    private static ArrayList<String> listFiles(String dirPath) throws Exception
    {
        File directory = new File(dirPath);
        if (!directory.exists() || !directory.isDirectory())
        {
            throw new Exception("\"" + dirPath + "\" is not a directory");
        }
        ArrayList<String> resultList = new ArrayList<String>();
        listFiles(directory.getParent(), directory.getName(), resultList);
        return resultList;
    }

    private static void listFiles(String rootPath, String currentPath, ArrayList<String> fileList)
    {
        File currentFile = new File(new File(rootPath), currentPath);
        if (currentFile.isDirectory())
        {
            String[] list = currentFile.list();
            for (int i = 0; i < list.length; i++)
            {
                listFiles(rootPath, currentPath + "/" + list[i], fileList);
            }
        }
        else
        {
            fileList.add(currentPath);
        }
    }

    private ByteArrayOutputStream getBlobOutputStream(String path)
    throws SonicFSException
    {
        try
        {
            // The get the blob attached to this file
            IBlob blob = m_dfs.getFSBlob(path, false);

            if (blob == null)
            {
                return null;
            }

            ByteArrayOutputStream os = new ByteArrayOutputStream();
            InputStream is = blob.getBlobStream();
            if (is == null)
             {
                return os; // return an empty stream
            }
            byte[] buffer = new byte[1024];

            while (true)
            {
                int avail = is.read(buffer);

                if (avail == -1)
                {
                    break;
                }

                os.write(buffer, 0, avail);
            }
            is.close();

            return os;
        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to read input stream '" + path + "' - " + e.getMessage(), e);
        }
    }

    /**
     * Determines whether a folder is a real file folder, rather than a special
     * "Configuration" folder, e.g. complex type or Broker-style type.
     *
     * @param map  The map of meta-attributes representing the file entry.
     * @return     true if the folder is valid, otherwise false.
     */
    private boolean isRealFolder(Map map)
    {
        try
        {
            Map res = Util.splitToolMetaAttributes(map);

            return (!(res.containsKey(IConfigServer.FOLDER_NAME) &&
                      res.containsKey(ConfigServerUtility.TYPE)));
        }
        catch (Exception e)
        {
        }

        return false;
    }

    /**
     * Removes any folders from the folders list that start with the folders
     * listed in the toRemove list.
     *
     * Use this method to filter out unwanted sub-folders, e.g. complex
     * configuration sub-folders.
     *
     * @param folders   The list of folders to process
     * @param toRemove  The list of folders marked for removal
     * @return          The list of folders post-processing
     */
    private <T> List<T> removeDanglingSubFolders(List<T> folders, List<String> toRemove)
    {
        Iterator<String> i = toRemove.iterator();
        while (i.hasNext())
        {
            String delFolder = i.next();

            Iterator<T> j = folders.iterator();
            while (j.hasNext())
            {
                T aFolder = j.next();

                if (aFolder instanceof String)
                {
                    if (((String)aFolder).startsWith(delFolder))
                    {
                        j.remove();
                    }
                }
                else
                if (aFolder instanceof SonicFSFile)
                {
                    if (((SonicFSFile)aFolder).getFullName().startsWith(delFolder))
                    {
                        j.remove();
                    }
                }
            }
        }

        return folders;
    }

    private String getValidFileName(Map map)
    {
        String ret = null;

        // Check to see if we have an element identity
        IElementIdentity id = (IElementIdentity)map.get(IConfigServer.ELEMENT_IDENTITY);

        if (id != null)
        {
            // Check to see if this is a MF_FILE type...
            // ... This is all stored in a TOOLS_INFO attribute
            String tool = (String)map.get(TOOL_ATTRIBUTES);

            if (tool != null && tool.indexOf("TYPE=" + FILE_TYPE) != -1)
            {
                ret = id.getName();
            }
        }
        return ret;
    }

    private static class AlphaComparator implements Comparator<SonicFSFile>
    {
        @Override
        public int compare(SonicFSFile o1, SonicFSFile o2)
        {
            if ((o1 instanceof SonicFSFile) && (o2 instanceof SonicFSFile))
            {
                String name1 = ((SonicFSFile)o1).getFullName();
                String name2 = ((SonicFSFile)o2).getFullName();

                if ((name1 != null) && (name2 != null))
                {
                    return name1.compareTo(name2);
                }
            }

            return -1;
        }
    }



    //Unit test
    public static void main(String[]args) throws Exception
    {
        java.util.Hashtable<String, String> env = new java.util.Hashtable<String, String>();
        env.put("ConnectionURLs", "tcp://localhost:2506");
        com.sonicsw.mf.jmx.client.JMSConnectorAddress address = new com.sonicsw.mf.jmx.client.JMSConnectorAddress(env);
        com.sonicsw.mf.jmx.client.JMSConnectorClient connector = new com.sonicsw.mf.jmx.client.JMSConnectorClient();
        connector.connect(address);
        new SonicFSFileSystem(connector, "Domain1", null).importFiles("C:/test/Archives", "/Archives8.0");

    }


}
