// Copyright (c) 2009 Progress Software Corporation.  All Rights Reserved.

package com.sonicsw.mx.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

// one per VM
public abstract class CLI
{
    private static Map m_helpMap = Collections.synchronizedMap(new HashMap());
    private static Vector m_keywords = new Vector();
    private static Map m_rootNode = Collections.synchronizedMap(new HashMap());
    private static Map m_currentNode = null;

    private static volatile Method m_commandMethod;
    private static volatile Method m_helpMethod;
    private static List<String> m_cmdLine;
    private static List<String> m_params;

    protected static final String DEFAULT_PROMPT = "> ";
    private static volatile String m_promptString = DEFAULT_PROMPT;

    private static boolean m_prompt;
    private static final String HELP_REQUEST_CHAR = "?";
    private static final String COMMENT_TOKEN = "//";
    private static boolean m_commandFinished = true;
    private static volatile Thread m_readerThread;
    private static BufferedReader m_inputReader;
    private static boolean m_paused = false;

    protected static Class m_thisClassObject = com.sonicsw.mx.util.CLI.class;
    protected static Class m_cliClassObject = com.sonicsw.mx.util.CLI.class;

    private static final String CMD_TOKEN_DELIMITERS = " \n\t\r";
    public static final String ECHO_CMD_PROP = "sonicsw.mx.cli.echo";
    public static final String SHOW_PROMPT_PROP = "sonicsw.mx.cli.prompt";
    public static final String TEXT_WIDTH_PROP = "sonicsw.mx.cli.wrap";

    protected static final boolean ECHO_CMD = new Boolean(System.getProperty(ECHO_CMD_PROP, "false")).booleanValue();
    protected static boolean SHOW_PROMPT = new Boolean(System.getProperty(SHOW_PROMPT_PROP, "true")).booleanValue();
    protected static int TEXT_WIDTH = Integer.parseInt(System.getProperty(TEXT_WIDTH_PROP, "79"));

    private static final Object METHOD_KEY = null;
    private static final Object HELP_KEY = "";

    static
    {
        try
        {
            addCommand("show help",
                       m_thisClassObject.getDeclaredMethod("showHelp", IEmptyArray.EMPTY_CLASS_ARRAY),
                       m_thisClassObject.getDeclaredMethod("showHelpHelp", IEmptyArray.EMPTY_CLASS_ARRAY));
            addCommand("show keywords",
                       m_thisClassObject.getDeclaredMethod("showKeywords", IEmptyArray.EMPTY_CLASS_ARRAY),
                       m_thisClassObject.getDeclaredMethod("showKeywordsHelp", IEmptyArray.EMPTY_CLASS_ARRAY));
            addCommand("?",
                       m_thisClassObject.getDeclaredMethod("help", new Class[] {String.class,String.class,String.class,String.class,String.class}),
                       m_thisClassObject.getDeclaredMethod("helpHelp", IEmptyArray.EMPTY_CLASS_ARRAY));
            addCommand(COMMENT_TOKEN,
                       m_thisClassObject.getDeclaredMethod("comment", IEmptyArray.EMPTY_CLASS_ARRAY),
                       m_thisClassObject.getDeclaredMethod("commentHelp", IEmptyArray.EMPTY_CLASS_ARRAY));
        }
        catch(NoSuchMethodException e) { e.printStackTrace(); } // should not happen !
    }

    protected static void addCommand(String command, Method implementation, Method help)
    {
        // build the command tree
        Map parentNode = m_rootNode;
        StringTokenizer st = new StringTokenizer(command, " ");
        while (st.hasMoreTokens())
        {
            String token = st.nextToken();
            if (!m_keywords.contains(token))
            {
                m_keywords.add(token);
            }

            HashMap childNode = (HashMap)parentNode.get(token);
            if (childNode == null)
            {
                childNode = new HashMap();
                parentNode.put(token, childNode);
            }
            // if this is the final token then store the method (using null key and help using "" key)
            if (!st.hasMoreTokens())
            {
                childNode.put(METHOD_KEY, implementation);
                childNode.put(HELP_KEY, help);
            }
            else
            {
                parentNode = childNode;
            }
        }

        // update the help map
        m_helpMap.put(command, help);
    }

    protected static ArrayList getCmdLine()
    {
        return CLI.m_cmdLine == null ? null : (ArrayList<String>)((ArrayList)CLI.m_cmdLine).clone();
    }

    protected static ArrayList getParams()
    {
        return CLI.m_params == null ? null : (ArrayList<String>)((ArrayList<String>)CLI.m_params).clone();
    }

    public static void setPrompt(String prompt) { m_promptString = prompt; }

    public static void cancelReadParseExecute()
    {
        if (m_readerThread == null)
        {
            return;
        }

        OStream.println();
        m_paused = true;
    }

    public static synchronized void readParseExecute()
    {
        m_paused = false;

        if (m_readerThread != null)
        {
            if (m_prompt)
            {
                prompt();
            }
            return;
        }

        m_readerThread = new Thread("Command Line Reader")
        {
            @Override
            public void run()
            {
                BufferedReader inputReader = null;
                String enc = System.getProperty("inputEncoding");
                if (enc == null)
                {
                    inputReader = new BufferedReader(new InputStreamReader(System.in));
                }
                else
                {
                    try
                    {
                        inputReader = new BufferedReader(new InputStreamReader(System.in, enc));

                    }
                    catch (UnsupportedEncodingException e)
                    {
                        OStream.println("Unsupported encoding:" + enc);
                        inputReader = new BufferedReader(new InputStreamReader(System.in));
                    }
                }

                boolean bye = false;
                CLI.m_prompt = true;
                cmdLoop:
                while (!(bye || m_paused))
                {
                    if (m_prompt)
                    {
                        prompt();
                    }

                    String cmd = null;
                    try
                    {
                        cmd = inputReader.readLine();
                        if (m_paused)
                        {
                            m_readerThread = null;
                            return;
                        }
                    }
                    catch(IOException e)
                    {
                        if (!isInterrupted())
                        {
                            OStream.printStackTrace(e);
                        }
                        return;
                    }

                    if (cmd == null) // happens when someone hits ctrl-break
                    {
                        m_paused = true;
                        m_readerThread = null;
                        return;
                    }
                    if (cmd.length() == 0)
                    {
                        continue;
                    }

                    // get rid of leading and trailing white space
                    cmd = cmd.trim();

                    if (ECHO_CMD)
                    {
                        more(cmd);
                    }

                    // ignore comments lines
                    if (cmd.startsWith(COMMENT_TOKEN))
                    {
                        continue;
                    }

                    // reset the command node in the command tree and working variables
                    m_currentNode = Collections.synchronizedMap(m_rootNode);
                    m_commandMethod = null;
                    m_helpMethod = null;
                    m_cmdLine = Collections.synchronizedList(new ArrayList<String>());
                    m_params = Collections.synchronizedList(new ArrayList<String>());

                    boolean unbalancedQuotes = false;
                    StringBuffer cmdToken = new StringBuffer();
                    for (int i = 0; i < cmd.length(); i++)
                    {
                        char c = cmd.charAt(i);
                        switch (c)
                        {
                            case '\\':
                            {
                                // escape char to blindly append the char that follows
                                if (++i < cmd.length())
                                {
                                    cmdToken.append(cmd.charAt(i));
                                }
                                break;
                            }
                            case '"':
                            {
                                // blindly go through the rest of the command until we hit another quote
                                while (++i < cmd.length() && cmd.charAt(i) != '"')
                                {
                                    char blindChar = cmd.charAt(i);
                                    switch (blindChar)
                                    {
                                        case '\\':
                                        {
                                            // escape char to blindly append the char that follows
                                            if (++i < cmd.length())
                                            {
                                                cmdToken.append(cmd.charAt(i));
                                            }
                                            break;
                                        }
                                        default:
                                        {
                                            cmdToken.append(blindChar);
                                            break;
                                        }
                                    }
                                }
                                if (i == cmd.length())
                                {
                                    unbalancedQuotes = true;
                                }
                                break;
                            }
                            case ' ':
                            {
                                if (!handleNextToken(cmdToken.toString(), (m_commandMethod == null)))
                                {
                                    continue cmdLoop;
                                }
                                cmdToken = new StringBuffer();
                                break;
                            }
                            default:
                            {
                                cmdToken.append(c);
                                break;
                            }
                        }
                    }
                    if (cmdToken.length() > 0) // last token
                    {
                        if (!handleNextToken(cmdToken.toString(), (m_commandMethod == null)))
                        {
                            continue cmdLoop;
                        }
                    }

                    if (unbalancedQuotes)
                    {
                        OStream.println("Unbalanced quotes");
                    }
                    else
                    if (m_commandMethod != null)
                    {
                        invokeCommand();
                    }
                    else
                    {
                        displayGenericHelp();
                    }
                }
            }
        };
        m_readerThread.setDaemon(true);
        m_readerThread.start();
    }

    /**
     * Determine if we should handle the next token
     *
     * @param token
     * @param cmdRequired
     * @return True if we should handle the next token, false if help was displayed
     */
    private static boolean handleNextToken(String token, boolean cmdRequired)
    {
        // is there an entry for the token in the current node ?
        HashMap node = (HashMap)m_currentNode.get(token);
        if (node == null) // then its either a help request or an arg
        {
            if (token.equals(HELP_REQUEST_CHAR) || cmdRequired)
            {
                if (m_helpMethod != null)
                {
                    displayCommandHelp();
                }
                else
                {
                    displayGenericHelp();
                }
                return false;
            }
            else
            {
                m_params.add(token);
            }
        }
        else // then its a valid command word
        {
            m_commandMethod = (Method)node.get(METHOD_KEY);
            m_helpMethod = (Method)node.get(HELP_KEY);
            m_currentNode = Collections.synchronizedMap(node);
        }
        m_cmdLine.add(token);
        return true;
    }

    // A special method that prompts only when commands aren't being executed.
    // This way, you don't get prompts while command execution uses the container's
    // logMessage mechanism.
    public static void redoPrompt()
    {
        if (m_prompt && m_commandFinished)
        {
            prompt();
        }
    }

    // This gets called from AgentCLI, which gets called from the Agent,
    // to print a CR before the containers logMessage. The CR will only be
    // printed if a command is not in the middle of executing.
    public static void crAfterPrompt()
    {
        if (SHOW_PROMPT && m_prompt && m_commandFinished)
        {
            OStream.println();
        }
    }

    private static void more(String toPrint)
    {
        OStream.println(toPrint);
    }

    private static void displayGenericHelp()
    {
        OStream.println("\nCommands are: \n");
        Vector helpStrings = new Vector(m_currentNode.keySet());
        more(helpStrings, true);
        OStream.println();
    }

    private static void displayCommandHelp()
    {
        String commandOutput = null;
        m_commandFinished = false;

        try
        {
            commandOutput = (String)m_helpMethod.invoke(m_cliClassObject, IEmptyArray.EMPTY_OBJECT_ARRAY);
        }
        catch (InvocationTargetException e)
        {
            // Any exception which happens from the invoke call gets wrapped
            // in a java.lang.reflect.InvocationtargetException and
            // e.printStackTrace alone is not useful.  The exception has to
            // be unwrapped before we can get something useful.
            Throwable ex = e.getTargetException();
            OStream.printStackTrace(ex);
        }
        catch (IllegalAccessException iae)
        {
            OStream.printStackTrace(iae);
        }

        OStream.println();
        OStream.println(commandOutput);
        OStream.println();

        m_commandFinished = true;
    }

    private static void invokeCommand()
    {
        String commandOutput = null;
        m_commandFinished = false;
        try
        {
            if (m_params.size() > m_commandMethod.getParameterTypes().length)
            {
                // invoke the help for the command
                if (m_helpMethod != null)
                {
                    displayCommandHelp();
                }
            }
            else
            {
                String[] params = new String[m_commandMethod.getParameterTypes().length];
                System.arraycopy(m_params.toArray(IEmptyArray.EMPTY_STRING_ARRAY), 0, params, 0, m_params.size());
                commandOutput = (String)m_commandMethod.invoke(m_cliClassObject, (Object[])params);
            }

            if (commandOutput != null)
            {
                OStream.println();
                OStream.println(commandOutput);
                OStream.println();
            }

        }
        // Any exception which happens from the invoke call gets wrapped
        // in a java.lang.reflect.InvocationtargetException and
        // e.printStackTrace alone is not useful.  The exception has to
        // be unwrapped before we can get something useful.
        catch (InvocationTargetException e)
        {
            Throwable ex = e.getTargetException();
            OStream.printStackTrace(ex);
        }
        catch (Exception e)
        {
            OStream.printStackTrace(e);
        }
        m_commandFinished = true;
    }

    private static void prompt()
    {
        if (SHOW_PROMPT)
        {
            OStream.print(m_promptString);
            OStream.flush();
        }
    }

    /**
     * Print several lines of input, and allows the user to specify that the
     * output should be printed as columns of items.
     */
    private static void more(Vector toPrint, boolean columns)
    {
        if (columns)
        {
            String []alpha_ordered = Sorter.sort(toPrint);
            String []columnized = putInColumns(alpha_ordered, 0);
            moreNoAlpha(columnized);
        }
        else
        {
            more(toPrint);
        }
    }

    /**
     * Takes an array of String items and makes lines of output
     * with the items placed in alphabetized columns.
     */
    protected static String[]putInColumns(String []items, int cols)
    {
        int columns;
        int columnLength;
        if (cols != 0)
        {
            columns = cols;
            columnLength = 79 / cols;
        }
        else
        {
            columns = TEXT_WIDTH / 15;
            columnLength = 15;
        }
        String []itemsInColumns = new String[items.length];
        String currentLine = "";
        String currentItem;
        int lineIndex = 0;
        for (int itemIndex = 0; itemIndex < items.length; itemIndex++)
        {
            currentItem=items[itemIndex];
            if ((itemIndex % columns) == 0)
            {
                // If there are five items on this line already,
                // start a new line
                if (itemIndex > 0)
                {
                    itemsInColumns[lineIndex] = currentLine;
                    lineIndex++;
                }
                currentLine = "";
            }
            currentLine = currentLine + currentItem;
            // Pad the line with spaces to get to the next column
            for (int padIndex = currentLine.length(); (padIndex % columnLength) != 0; padIndex++)
            {
                currentLine = currentLine + " ";
            }
        }
        // The last line still has to be added

        itemsInColumns[lineIndex] = currentLine;
        return itemsInColumns;
    }

    protected static void moreNoAlpha(String[] toPrint)
    {
        int counter;
        for (counter=0; counter < toPrint.length; counter++)
        {
            if (toPrint[counter] != null)
            {
                OStream.println(toPrint[counter]);
            }
        }
    }

    private static void more(Vector toPrint)
    {
        String []alpha_ordered = Sorter.sort(toPrint);
        for (int index = 0; index < alpha_ordered.length; index++)
        {
            OStream.println(alpha_ordered[index]);
        }
    }

    public static String showHelp()
    {
        String[] commands = (String[])Sorter.sort((String[])m_helpMap.keySet().toArray(IEmptyArray.EMPTY_STRING_ARRAY));
        StringBuffer buffer = new StringBuffer();

        // now loop through displaying the help
        try
        {
            for (int i = 0; i < commands.length; i++)
            {
                if (i > 0)
                {
                    buffer.append("\n\n\n");
                }

                Method helpMethod = (Method)m_helpMap.get(commands[i]);
                if (helpMethod != null)
                {
                    buffer.append((String)helpMethod.invoke(m_cliClassObject, IEmptyArray.EMPTY_OBJECT_ARRAY));
                }
                else
                {
                    buffer.append("Help not implemented for " + commands[i]);
                }
            }
        }
        // Any exception which happens from the invoke call gets wrapped
        // in a java.lang.reflect.InvocationtargetException and
        // e.printStackTrace alone is not useful.  The exception has to
        // be unwrapped before we can get something useful.
        catch (InvocationTargetException e)
        {
            Throwable ex = e.getTargetException();
            ex.printStackTrace();
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }

        return buffer.toString();
    }

    public static String showKeywords()
    {
        OStream.println("\nKeywords are: \n");
        more(m_keywords, true);
        OStream.println();

        return null;
    }

    // allow for up to five command tokens
    public static String help(String token1, String token2, String token3, String token4, String token5)
    {
        HashMap node = null;
        if (token1 != null && token1.length() > 0 && m_rootNode.containsKey(token1))
        {
            node = (HashMap)m_rootNode.get(token1);
        }
        if (token2 != null && token2.length() > 0 && node.containsKey(token2))
        {
            node = (HashMap)node.get(token2);
        }
        if (token3 != null && token3.length() > 0 && node.containsKey(token3))
        {
            node = (HashMap)node.get(token3);
        }
        if (token4 != null && token4.length() > 0 && node.containsKey(token4))
        {
            node = (HashMap)node.get(token4);
        }
        if (token5 != null && token5.length() > 0 && node.containsKey(token5))
        {
            node = (HashMap)node.get(token5);
        }

        // if no recognized command the generic help
        if (node == null)
        {
            m_currentNode = Collections.synchronizedMap(m_rootNode);
            displayGenericHelp();
            return null;
        }

        m_helpMethod = (Method)node.get(HELP_KEY);

        // show generic help for this node if we dont have a full command
        if (m_helpMethod == null)
        {
            m_currentNode = Collections.synchronizedMap(node);
            displayGenericHelp();
            return null;
        }

        displayCommandHelp();

        return null;
    }

    public static String comment() { return null; }

    public static String showHelpHelp()
    {
        return "Usage: show help" +
               "\n\n" +
               "Displays help for all commands.";
    }

    public static String showKeywordsHelp()
    {
        return "Usage: show keywords" +
               "\n\n" +
               "Displays all the known command keywords.";
    }

    public static String helpHelp()
    {
        return "Usage: {? <command> | <command> ?}" +
               "\n\n" +
               "Displays help for a command.";
    }

    public static String commentHelp()
    {
        return "Usage: //" +
               "\n\n" +
               "Treats the text to the end of the line as a comment.";
    }

}
