
package com.sonicsw.mx.util;

import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectStreamClass;
import java.lang.reflect.Array;

/**
 * The class deserializes an object in the context of a specific class loader.
 */
public class LoaderInputStream
extends ObjectInputStream
{
    private ClassLoader loader;

    public LoaderInputStream(InputStream in, ClassLoader loader)
    throws IOException
    {
        super(in);
        this.loader = loader;
    }

    @Override
    protected Class resolveClass(ObjectStreamClass osClass)
    throws IOException, ClassNotFoundException
    {
        Class cls  = null; 
        try {
            if (loader == null)
            {
                cls = super.resolveClass(osClass);
            } 
            else 
            {
                String classname = osClass.getName();
                int i = 0;
                if (classname.charAt(i++) == '[') // its an array
                {
                    while (classname.charAt(i) == '[')
                    {
                        // array or arrays
                        i++;
                    }
                    Class clazz;
                    if (classname.charAt(i) == 'L')
                    {
                        clazz = loader.loadClass(classname.substring(i + 1, classname.length() - 1));
                    }
                    else // expecting a base type
                    {
                        if (classname.length() != i + 1)
                        {
                            throw new ClassNotFoundException(classname);
                        }
                        clazz = primitiveType(classname.charAt(i));
                    }

                    int dimensions[] = new int[i];
                    for (int j = 0; j < i; j++)
                    {
                        dimensions[j] = 0;
                    }

                    cls = Array.newInstance(clazz, dimensions).getClass();
                }
                else
                {
                    cls = loader.loadClass(classname);
                }
            }
        /*
         * Fix for MQ-34703
         * The ObjectInputStream class ignores ClassNotFoundException and continues to re-construct the Object even if the Class is not
         * available in the target ClassLoader. This may cause OutOfMemoryError.
         * 
         * Wrapping ClassNotFoundException in an IOException breaks ObjectInputStream.readObject() and ContainerUtil.transferObject()
         * returns the original object that was passed to it.
         * 
         * */    
        } catch (ClassNotFoundException cnfe) {
            throw new IOException("Unable to resolve class " + osClass.getName(), cnfe);
        }

        return cls;
    }

    private Class primitiveType(char c)
    {
        switch(c)
        {
            case 66: /* 'B' */
                return Byte.TYPE;

            case 67: /* 'C' */
                return Character.TYPE;

            case 68: /* 'D' */
                return Double.TYPE;

            case 70: /* 'F' */
                return Float.TYPE;

            case 73: /* 'I' */
                return Integer.TYPE;

            case 74: /* 'J' */
                return Long.TYPE;

            case 83: /* 'S' */
                return Short.TYPE;

            case 90: /* 'Z' */
                return Boolean.TYPE;
        }
        return null;
    }
}