// Copyright (c) 2009 Progress Software Corporation.  All Rights Reserved.

package com.sonicsw.mx.util;

import java.io.BufferedWriter;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;


/**
 * This class implements print and println using the system outputEncoding
 * property if it has been specified.
 *
 *
 * @version 1.0  18-jul-00
 * @author Steve Watt
 * @version 2.0 1-Jan-02
 * @author Mari Davila
 */
public class OStream
{
    private static class PrintStreamHolder {
        private static final PrintStream DELEGATE = System.out;    
    }
    
    private static class PrintWriterHolder {
        
        private static final PrintWriter OUT = getPrintWriter();
        private static PrintWriter getPrintWriter() {
            PrintWriter out = null;
            String enc = System.getProperty("outputEncoding");
            if (enc != null) {
                try {
                    out = new PrintWriter(new BufferedWriter(new OutputStreamWriter(PrintStreamHolder.DELEGATE, enc)),true);
                } catch (UnsupportedEncodingException ex) {
                    // Ignore the encoding and try again.
                }
            }

            if (out == null) {
                out = new PrintWriter(new BufferedWriter(new OutputStreamWriter(PrintStreamHolder.DELEGATE)),true);
            }
            return out;
        }
    }
    

    /**
     * Prints a line, with newline, using outputEncoding property if specified.
     */
    public static void println(String str)
    {
        synchronized (PrintStreamHolder.DELEGATE)
        {
            PrintWriterHolder.OUT.println(str);
        }
    }

    /**
     * Prints a line, with newline.
     */
    public static void println()
    {
        synchronized (PrintStreamHolder.DELEGATE)
        {
            PrintWriterHolder.OUT.println();
        }
    }

    /**
     * Prints a line, without newline, using outputEncoding property if specified.
     */
    public static void print(String str)
    {
        synchronized (PrintStreamHolder.DELEGATE)
        {
            PrintWriterHolder.OUT.print(str);
        }
    }

    /**
     * Prints a line, using outputEncoding property if specified.
     */
    public static void print(char [] array)
    {
        synchronized (PrintStreamHolder.DELEGATE)
        {
            PrintWriterHolder.OUT.print(array);
        }
    }

    /**
     * flushes the stream.
     */
    public static void flush()
    {
        synchronized (PrintStreamHolder.DELEGATE)
        {
            PrintWriterHolder.OUT.flush();
        }
    }

    /**
     * Prints the specified throwable and its backtrace.
     * @param throwable the throwable
     */
    public static void printStackTrace(Throwable throwable)
    {
        synchronized (PrintStreamHolder.DELEGATE)
        {
            throwable.printStackTrace(PrintWriterHolder.OUT);
        }
    }
}