package com.sonicsw.mx.util;


// Monitors and maintains a service by periodically asking it perform a task. If the service is not accessible
// then the ServiceMaintainer changes its state to non-accessible. Change in accessibility can trigger some
// activity. Other threads can call waitToBeAccessible() to wait for the service to be accessible. If we know the service
// is accessible (because the service initiated communication, for example), we can call setAccesssibility(true)
// to change its state to accessible even before it detects it itself.
public class ServiceMaintainer extends Thread
{
    private boolean m_isAccessible;
    private long m_frequency;
    private ServiceMaintenance m_maintenance;
    private boolean m_open;
    private boolean m_dontSleepInitially;
    private String m_contextID;
    private Object m_lock = new Object();


    public ServiceMaintainer(ServiceMaintenance maintenance, long frequency, boolean initialAccessibility)
    {
        super("Service Maintainer");
        init(maintenance, frequency, initialAccessibility, null);
    }

    public ServiceMaintainer(String purpose, ServiceMaintenance maintenance, long frequency, boolean initialAccessibility)
    {
        super("Service Maintainer: " + purpose);
        init(maintenance, frequency, initialAccessibility, null);
    }

    public ServiceMaintainer(ServiceMaintenance maintenance, long frequency, boolean initialAccessibility, String contextID)
    {
        super("mx.util.ServiceMaintainer for " + contextID);
        init(maintenance, frequency, initialAccessibility, contextID);
    }


    private void init(ServiceMaintenance maintenance, long frequency, boolean initialAccessibility, String contextID)
    {
        super.setDaemon(true);
        m_contextID = contextID;
        m_open = true;
        m_frequency = frequency;
        m_maintenance = maintenance;
        m_isAccessible = initialAccessibility;
        if (!m_isAccessible)
        {
            m_dontSleepInitially = true;
        }
        else
        {
            m_dontSleepInitially = false;
        }
        start();
    }
    
    public Object getLock()
    {
        return m_lock;
    }


    // Modify accessibility state and excute the onAccessibilityChange trigger
    // Returns true if accesibility status changed
    public boolean setAccesssibility(boolean isAccessible)
    {
        synchronized (m_lock)
        {
            if (!m_open)
            {
                return false;
            }
    
            if (isAccessible == m_isAccessible)
            {
                return false;
            }
    
            m_maintenance.onAccessibilityChange(isAccessible);
    
            m_isAccessible = isAccessible;
    
            if (isAccessible)
            {
                m_lock.notifyAll();
            }
    
            return true;
        }
    }

    // Wait for the service to be accessible. Will return false if ServiceMaintainer
    // is closed before the service became accessible
    public boolean waitToBeAccessible()
    {
        synchronized (m_lock)
        {
            if (!m_open)
            {
                return false;
            }
    
            if (m_isAccessible)
            {
                return true;
            }
            else
            {
                try
                {
                        while (!m_isAccessible)
                        {
                            m_lock.wait();
                        }
                } catch (InterruptedException e){}
                return m_isAccessible;
            }
        }
    }

    // Will cause threads that called waitToBeAccessible to return false
    public void close()
    {
        synchronized (m_lock)
        {
            if (m_open)
            {
                interrupt();
                m_open = false;
            }
        }
    }

    public boolean isAccessible()
    {
        if (!m_open)
        {
            return false;
        }

        return m_isAccessible;
    }

    @Override
    public void run()
    {
        try
        {
            while(m_open)
            {
                if (!m_dontSleepInitially)
                {
                    synchronized (m_lock)
                    {
                        m_lock.wait(m_frequency);
                    }
                }

                m_dontSleepInitially = false;
                Exception e = doMaintenance();
                m_maintenance.setException(e);
                setAccesssibility(e == null);
            }
        }
        catch (InterruptedException e){}
    }

    private Exception doMaintenance()
    {
        synchronized (m_lock)
        {
            if (m_contextID == null)
            {
                return m_maintenance.doMaintenance();
            }
            else
            {
                return m_maintenance.doMaintenance(m_contextID);
            }
        }
    }

}
