package com.sonicsw.mx.util;


import java.text.CollationKey;
import java.text.Collator;
import java.util.Vector;

/**
 * Provides specialized methods for sorting the content of certain object/array
 * types using collators.
 */
// This code used to reside in BrokerManager, but since this is commercialized
// in 3.0 we need to remove such public utility functional we did not want to
// expose.
public final class Sorter
{
    public static final Collator COLLATOR = Collator.getInstance();

    /**
     * Sort a vector of strings. Assumes there are no duplicates in orderThese.
     * Returns an array of strings that contain the strings in the original vector in alphabetical order.
     * @param orderThese - Vector of strings.
     */
    public static String[] sort(Vector orderThese)
    {
        String[] ordered = new String[orderThese.size()];
        orderThese.copyInto(ordered);
        try
        {
            return (String[])sort(ordered, ordered.length);
        }
        catch(Exception e)
        {
            e.printStackTrace(); // can we really fail the sort ?
        }
        return ordered;
    }

    /**
     * Sort an array of objects. Assumes there are no duplicates in orderThese.
     * If each array element holds an array of objects, then the first element of that array is
     * assumed to be a string, otherwise the main array is assumed to be an array of  strings.
     *
     * This sort handles array elements that are either arrays of strings, arrays
     * of arrays where the first element of the inner array is a string or a
     * combination of both.
     *
     * Returns a sorted array. Use casting on the return value.
     */
    public static Object[] sort(Object[] orderThese)
    {
        return sort(orderThese, orderThese.length);
    }

    /**
     * Sort a subset of an array of objects specified by a count from the first item. Assumes there are no duplicates in orderThese. If each
     * array element holds an array of objects, then the first element of that array is
     * assumed to be a string, otherwise the main array is assumed to be an array of
     * strings.
     *
     * This sort handles array elements that are either arrays of strings, arrays
     * of arrays where the first element of the inner array is a string or a
     * combination of both.
     *
     * Returns a sorted array. Use casting on the return value.
     *
     * @param orderThese The array of items to be ordered.
     * @param count Sorts up to the specified number of elements in the array
     */
    public static Object[] sort(Object[] orderThese, int count)
    {
        try
        {
            CollationKey[] keys = new CollationKey[count];
            sort(orderThese, keys, count);
        }
        catch(Exception e)
        {
            e.printStackTrace(); // can we really fail the sort ?
        }
        return orderThese;
    }

    /**
     * Sort a subset of an array of objects and collation keys specified by a count from the first item.
     * Assumes there are no duplicates in orderThese. If each
     * array element holds an array of objects, then the first element of that array is
     * assumed to be a string, otherwise the main array is assumed to be an array of
     * strings.
     *
     * This sort handles array elements that are either arrays of strings, arrays
     * of arrays where the first element of the inner array is a string or a
     * combination of both.
     *
     * Returns a sorted array. Use casting on the return value.
     *
     * @param orderThese The array of items to be ordered.
     * @param keys A corresponding array of collation keys (the key array is also sorted).
     *             If count > 0 and keys[0] equals null, the keys array will be populated.
     * @param count Sorts up to the specified number of elements in the array.
     */
    public static Object[] sort(Object[] orderThese, CollationKey[] keys, int count)
    {
        try
        {
            if (count > 0)
            {
                if (keys[0] == null)
                {
                    for (int i = 0; i < count; i++)
                    {
                        String keyString = null;
                        if (orderThese[i] instanceof String)
                        {
                            keyString = (String)orderThese[i];
                        }
                        else
                        if (orderThese[i] instanceof Object[])
                        {
                            keyString = (String)((Object[])orderThese[i])[0];
                        }
                        else
                        {
                            keyString = orderThese[i].toString();
                        }
                        keys[i] = COLLATOR.getCollationKey(keyString);
                    }
                }
                internalSort(orderThese, keys, count);
            }
        }
        catch(Exception e)
        {
            e.printStackTrace(); // can we really fail the sort ?
        }
        return orderThese;
    }

    /**
     * Sort a subset of an array of objects specified by a count from the first item.
     * Assumes there are no duplicates in orderThese. If each
     * array element holds an array of objects, then the first element of that array is
     * assumed to be a string, otherwise the main array is assumed to be an array of
     * strings.
     *
     * This sort handles array elements that are either arrays of strings, arrays
     * of arrays where the first element of the inner array is a string or a
     * combination of both.
     *
     * Returns a sorted array. Use casting on the return value.
     *
     * @param orderThese The array of items to be ordered.
     * @param keyProvider A collation key provider for the objects being sorted.
     * @param count Sorts up to the specified number of elements in the array.
     */
    public static Object[] sort(Object[] orderThese, ICollationKeyProvider keyProvider, int count)
    {
        try
        {
            if (count > 0)
            {
                CollationKey[] keys = new CollationKey[count];
                if (keys[0] == null)
                {
                    for (int i = 0; i < count; i++)
                    {
                        keys[i] = keyProvider.getCollationKey(orderThese[i]);
                    }
                }
                internalSort(orderThese, keys, count);
            }
        }
        catch(Exception e)
        {
            e.printStackTrace(); // can we really fail the sort ?
        }
        return orderThese;
    }

    /**
     * This is a generic version of C.A.R Hoare's Quick Sort
     * algorithm.  This will handle arrays that are already
     * sorted, and arrays with duplicate keys.<BR>
     *
     * If you think of a one dimensional array as going from
     * the lowest index on the left to the highest index on the right
     * then the parameters to this function are lowest index or
     * left and highest index or right.  The first time you call
     * this function it will be with the parameters 0, a.length - 1.
     *
     * @param a       an array
     * @param keys    an array of collation keys
     * @param lo0     left boundary of array partition
     * @param hi0     right boundary of array partition
     */
    private static void quickSort(Object[] a, CollationKey[] keys, int l, int r)
    throws Exception
    {
        int M = 4;
        int i;
        int j;
        String v = null;

        if ((r - l) > M)
        {
            i = (r + l) >> 1;

            if (keys[l].compareTo(keys[i]) > 0)
            {
                swap(a, keys, l, i);
            }
            if (keys[l].compareTo(keys[r]) > 0)
            {
                swap(a, keys, l, r);
            }
            if (keys[i].compareTo(keys[r]) > 0)
            {
                swap(a, keys, i, r);
            }

            j = r - 1;
            swap(a, keys, i, j);
            i = l;

            CollationKey vKey = keys[j];
            for(;;)
            {
                while(keys[++i].compareTo(vKey) < 0)
                {
                    ;
                }
                while(keys[--j].compareTo(vKey) > 0)
                {
                    ;
                }
                if (j < i)
                {
                    break;
                }
                swap(a, keys, i, j);
            }
            swap(a, keys, i, r - 1);
            quickSort(a, keys, l, j);
            quickSort(a, keys, i + 1, r);
        }
    }

    private static void swap(Object[] a, CollationKey[] keys, int i, int j)
    {
        Object temp;
        temp = a[i];
        a[i] = a[j];
        a[j] = temp;
        temp = keys[i];
        keys[i] = keys[j];
        keys[j] = (CollationKey)temp;
    }

    private static void insertionSort(Object[] a, CollationKey[] keys, int lo0, int hi0)
    throws Exception
    {
        int i;
        int j;

        for (i = lo0 + 1; i <= hi0; i++)
        {
            Object o = a[i];
            CollationKey vKey = keys[i];
            j = i;
            while ((j > lo0) && (keys[j - 1].compareTo(vKey) > 0))
            {
                a[j] = a[j-1];
                keys[j] = keys[j-1];
                j--;
            }
            a[j] = o;
            keys[j] = vKey;
        }
    }

    /**
     * The sort handles null entries and places those at the tail of the array.
     *
     * This sort handles array elements that are either arrays of strings, arrays
     * of arrays where the first element of the inner array is a string or a
     * combination of both.
     */
    private static void internalSort(Object[] a, CollationKey[] keys, int length)
    throws Exception
    {
        length--;

        quickSort(a, keys, 0, length);
        insertionSort(a, keys, 0, length);
    }
}

