package com.sonicsw.mx.util;

/**
 * This class provides basic utility methods for building URL strings
 * where IPv6 hostnames are wrapped in square brackets as per RFC2732.
 * 
 * @author pmeadows
 */
public class URLUtil
{
    /**
     * Build a broker connection URL from component parts - protocol, host, port
     * 
     * @param protocol Protocol part of URL, e.g. "tcp", "ssl"
     * @param host Hostname/IP address - IPv6 addresses can be either wrapped or unwrapped
     * @param port Port - use '-1' if port should be omitted from url
     * @return String representing URL built from the above components
     */
    public static String buildUrlString(String protocol, String host, int port)
    {
        String portAsString = null;
        
        if (port > 0)
        {
            portAsString = Integer.toString(port);
        }
        
        return buildUrlString(protocol, host, portAsString);
    }
    
    /**
     * Build a broker connection URL from component parts - protocol, host, port
     * 
     * @param protocol Protocol part of URL, e.g. "tcp", "ssl"
     * @param host Hostname/IP address - IPv6 addresses can be either wrapped or unwrapped
     * @param port Port - use <code>null</code> if port should be omitted from url
     * @return String representing URL built from the above components
     */
    public static String buildUrlString(String protocol, String host, String port)
    {
        StringBuilder url = new StringBuilder();
        
        if (protocol != null && !protocol.equals(""))
        {
            url.append(protocol).append("://");
        }
        
        url.append(wrapIPv6Address(host));
        
        if (port != null & port.length() > 0)
        {
            url.append(":").append(port);
        }
        
        return url.toString();
    }
    
    /**
     * Returns a version of the hostname suitable for use in a URL.
     * If the hostname is a literal IPv6 address and it's not already
     * wrapped in square brackets then the brackets are added.
     * 
     * @param host Hostname/IP address - IPv6 addresses can be either wrapped or unwrapped
     * @return Wrapped hostname, or original if no additional wrapping is needed
     */
    public static String wrapIPv6Address(String host)
    {
        if (host == null)
        {
            return null;
        }
        
        if (!host.startsWith("[") && host.contains(":"))
        {
            host = "[" + host + "]";
        }
        
        return host;
    }
    
    /**
     * Strips square brackets, if present, from an IPv6 address
     * 
     * @param host Hostname/IP address - IPv6 addresses can be either wrapped or unwrapped
     * @return Unwrapped hostname
     */
    public static String unwrapIPv6Address(String host)
    {
        if (host == null)
        {
            return null;
        }
        
        if (host.startsWith("[") && host.contains(":") && host.endsWith("]"))
        {
            host = host.substring(1, host.length() - 1);
        }
        
        return host;
    }    
    
}
