package com.sonicsw.sdf;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

public abstract class AbstractDiagnosticsProvider implements IDiagnosticsProvider, IDiagnosticsConstants
{
    public static final String NON_TRACE_LEVEL = "no_trace";
    public static final String BASIC_TRACE_LEVEL = "basic_trace";
    public static final String VERBOSE_TRACE_LEVEL = "verbose_trace";

    public static final String OP_PARAM_OVERWRITE = "overwrite";
    public static final String OP_PARAM_OUTPUT_LOCATION = "output";
    public static final String OP_PARAM_FILE_ACCESS = "output_access";

    public static String OP_PARAM_OVERWRITE_DESCRIPTION;
    public static String OP_PARAM_OUTPUT_LOCATION_DESCRIPTION;
    public static String OP_PARAM_FILE_ACCESS_DESCRIPTION;

    private static String[] OPERATIONS;
    private static String TRACE_LEVEL_PARAM_DESCRIPTION;

    private static HashMap UPDATE_TRACE_LEVEL_PARAM_DESCIPTOR = new HashMap();
    private static HashMap DUMP_STATE_PARAM_DESCIPTOR = new HashMap();
    private static HashMap DESCRIBE_PARAM_DESCIPTOR = new HashMap();
    private static HashMap LIST_DIAGNOSTICS_INSTANCES_PARAM_DESCIPTOR = new HashMap();

    private static HashMap PARAM_DESCRIPTOR = new HashMap();

    static
    {
        StringBuilder tmp = new StringBuilder();
        tmp.append("Values: ").append(NON_TRACE_LEVEL).append(" | ").append(BASIC_TRACE_LEVEL).append(" | ").append(VERBOSE_TRACE_LEVEL);
        TRACE_LEVEL_PARAM_DESCRIPTION = tmp.toString();

        tmp = new StringBuilder();
        tmp.append("Optional, Values: ").append("true").append(" | ").append("false")
        .append("\n\tEstablishes if the dump file should be overwritten or not.")
        .append("\n\texample: overwrite=true");
        OP_PARAM_OVERWRITE_DESCRIPTION = tmp.toString();

        tmp = new StringBuilder();
        tmp.append("Optional, home folder for the output files, it should have write access")
        .append("\n\tchild folders will be created if necessary, default to")
        .append("\n\tcontainer's home folder")
        .append("\n\texample: output=/home/user/dumps ");

        OP_PARAM_OUTPUT_LOCATION_DESCRIPTION = tmp.toString();

        tmp = new StringBuilder();
        tmp.append("Optional, specify the dump file access permissions in the form")
        .append("\n\tof 9 characters within 3 groups of 3 chars each for user-group-all")
        .append("\n\taccess being 'r' read, 'w' write and 'x' execute, if no particular")
        .append("\n\tpermission need to be set then use the character '-'")
        .append("\n\texample: output_access=rw-r--r-- to set read to everyone and write")
        .append("\n\tonly to the owner.\n\t");

        OP_PARAM_FILE_ACCESS_DESCRIPTION = tmp.toString();

    }

    static
    {
        UPDATE_TRACE_LEVEL_PARAM_DESCIPTOR.put(STRING_TRACE_LEVEL_PARAM, TRACE_LEVEL_PARAM_DESCRIPTION);

        PARAM_DESCRIPTOR.put(DUMP_STATE_OP, DUMP_STATE_PARAM_DESCIPTOR);
        PARAM_DESCRIPTOR.put(DESCRIBE_OP, DESCRIBE_PARAM_DESCIPTOR);
        PARAM_DESCRIPTOR.put(UPDATE_TRACE_LEVEL_OP, UPDATE_TRACE_LEVEL_PARAM_DESCIPTOR);
        PARAM_DESCRIPTOR.put(LIST_DIAGNOSTICS_INSTANCES_OP, LIST_DIAGNOSTICS_INSTANCES_PARAM_DESCIPTOR);

        OPERATIONS = toOpnameArray(PARAM_DESCRIPTOR);
    }



    protected String m_traceLevel;
    protected String m_stateDescription;
    protected String m_subsystemName;
    protected String m_subsystemID;
    protected IDiagnosticsContext m_diagnosticsContext;
    protected ITracer m_tracer;

    public AbstractDiagnosticsProvider(String subsystemName)
    {
        m_subsystemName = subsystemName;
        m_traceLevel = NON_TRACE_LEVEL;
        m_diagnosticsContext = null;
        m_tracer = null;
        m_stateDescription = "initialState";
        m_subsystemID = null;
    }

    public IDiagnosticsContext register()
    {
        IDiagnosticsManager manager = DiagnosticsManagerAccess.getManager();

        if (manager == null)
        {
            return DiagnosticsManagerAccess.createContext(this);
        }

        m_diagnosticsContext = manager.register(this);
        return m_diagnosticsContext;
    }

    //Demonstrates the update of the trace level and the use of SDF trace files
    @Override
    public void updateTraceLevel(String doiID, HashMap parameters, StringBuffer buffer)
    {
        m_traceLevel = (String)parameters.get(STRING_TRACE_LEVEL_PARAM);

        if (m_traceLevel == null)
        {
            return;
        }

        if (m_traceLevel.equalsIgnoreCase(NON_TRACE_LEVEL))
        {
            if (m_tracer != null)
            {
                m_tracer.close();
            }
            return;
        }

        try
        {
            if (m_tracer == null)
            {
                m_tracer = m_diagnosticsContext.getTracer();
            }
            buffer.append("Tracer file for '").append(m_subsystemName).append("' is '").append(m_tracer.getFilePath()).append("'");

            m_tracer.trace("Start tracing with tracing level '" + m_traceLevel + "'", true);

        }
        catch (Exception e)
        {
            e.printStackTrace();
            buffer.append(e.toString());
        }
    }
    @Override
    public void showTraceLevel(String doiID, HashMap parameters, StringBuffer buffer)
    {
        buffer.append("Trace level is: " + m_traceLevel);
    }

    //Demonstrates the dumping of the state to an SDF file
    @Override
    public void appendStateDump(String doiID, HashMap parameters, StringBuffer buffer)
    {
        IStateWriter writer = null;
        try
        {
            writer = m_diagnosticsContext.getStateWriter();
            
            String instruction = WriterUtil.rebuildInstruction(this, DUMP_STATE_OP, parameters);
            WriterUtil.writeHeader(writer, instruction, null);
        	
            // main body of state dump goes here
            writer.write(m_stateDescription);
        	
            WriterUtil.writeFooter(writer, null);
        }
        catch (Exception e)
        {
            buffer.append(m_subsystemName).append(" failed to write state dump:");
            buffer.append(NEWLINE).append(WriterUtil.getExceptionAsString(e));
            
            return;
        }
        finally
        {
            if (writer != null)
            {
                writer.close();
            }
        }


        buffer.append("State written to file ").append(writer.getFilePath());
        writer.close();
    }

    @Override
    public void test(HashMap parameters, StringBuffer buffer)
    {
        buffer.append(this.getClass().getName()).append(" does not implement test()");
    }

    @Override
    public String getSubsystemName()
    {
        return m_subsystemName;
    }

    @Override
    public String getSubsystemID()
    {
        return m_subsystemID;
    }


    @Override
    public String describe()
    {
        return "Replace this with a text decription of the subsystem";
    }

    @Override
    public String[] getTags()
    {
        return null;
    }

    @Override
    public String[] getOperations()
    {
        return OPERATIONS;
    }

    @Override
    public String[] getDOInstances()
    {
        return null;
    }

    @Override
    public HashMap describeParameters(String operationName)
    {
        return (HashMap)PARAM_DESCRIPTOR.get(operationName);
    }

    @Override
    public void executeOperation(String doiID, String operation, HashMap parameters, StringBuffer buffer)
    {
        buffer.append(this.getClass().getName()).append(" does not implement '").append(operation).append("'");
    }

    public static String[] toOpnameArray(HashMap paramDescriptor)
    {
        ArrayList list = new ArrayList();
        Iterator iter = paramDescriptor.keySet().iterator();
        while (iter.hasNext())
        {
            list.add(iter.next());
        }

        String[] result = new String[list.size()];
        list.toArray(result);
        return result;
    }

	/**
     * Simply dumps the content of the given file
     * into the given buffer, this function is useful
     * for example in the "help" text functions to
     * externalise this text from the source code.
     *
     * @param file text file to be dumped
     * @param buffer to where text will be dumped
     */
    protected void printFromFileToBuffer(String file, StringBuffer buffer){
        BufferedReader reader = new BufferedReader(
                new InputStreamReader(
                        this.getClass().getResourceAsStream(file)
                )
        );

        String line;
        try {
            while((line = reader.readLine()) != null){
                buffer.append(line)
                        .append(NEWLINE);
            }
        }catch(IOException e){
            throw new RuntimeException(e);
        }
    }
}
