package com.sonicsw.sdf;

import java.io.IOException;
import java.util.Map;

/**
 * IDiagnosticsContext is obtained by the diagnostics subsystem when it registers.
 */
public interface IDiagnosticsContext
{
 /**
  * Gets a trace file to be used by diagnostics subsystems.
  *
  * @param tracerFilePath the path of the trace file
  * @param maxTracerFileLength the combined maximum size in bytes of the trace file and the overflow trace file.
  *
  * @return tracing interface
  */
  ITracer getTracer(String tracerFilePath, long maxTracerFileLength) throws IOException;

 /**
  * Gets a trace file to be used by diagnostics subsystems. There is no limit on the trace file size.
  *
  * @param tracerFilePath the path of the trace file
  *
  * @exception thrown if write() failed
  *
  * @return tracing interface
  */
  ITracer getTracer(String tracerFilePath) throws IOException;

 /**
  * Gets a trace file to be used by diagnostics subsystems. The file name is generated from the subsystem name and ID.
  *
  * @param maxTracerFileLength the combined maximum size in bytes of the trace file and the overflow trace file.
  *
  * @return tracing interface
  *
  * @exception thrown if write() failed
  */
  ITracer getTracer(long maxTracerFileLength) throws IOException;

 /**
  * Gets a trace file to be used by diagnostics subsystems. The file name is generated from the subsystem name and ID.
  * There is no limit on the trace file size.
  *
  * @return tracing interface

  * @exception thrown if write() failed
  *
  */
  ITracer getTracer() throws IOException;

 /**
  * Gets a state dump file to be used by the dumpState operation of diagnostics subsystems.
  *
  * @param stateFilePath the path of the state dump file
  *
  * @exception thrown if write() failed
  *
  * @return state writing interface
  */
  IStateWriter getStateWriter(String stateFilePath) throws IOException;

 /**
  * Gets a state dump file to be used by the dumpState operation of diagnostics subsystems. The file name is generated from the subsystem name and ID.
  *
  * @return state writing interface
  *
  * @exception thrown if write() failed
  *
  */
  IStateWriter getStateWriter() throws IOException;

 /**
  * Gets a state writer with certain tweaks depending
  * on the given parameters like if the file should
  * be overwritten or the home folder is different
  * from the default one.
  *
  * @param parameters hashmap with the parameters
  * @return state writing interface
  * @throws IOException
     */
  IStateWriter getStateWriter(Map<String, String> parameters) throws IOException;

 /**
  * Generates the core component of a filename for diagnostics output.  The
  * filename is derived from the container name (if available), a timestamp,
  * and the subsystem name/id.  For example:
  * <blockquote>
  *   [&lt;containername&gt;.]&lt;timestamp&gt;.&lt;subsystemname/id&gt;
  * </blockquote>
  * The caller would normally add a suitable extension to the generated
  * name, e.g. '.state', '.trace', '.xml'.
  * <p>
  * Where possible the getTracer() and getStateWriter() methods should be
  * used - these provide a tracer/writer whose filename is generated by
  * this method.  However, there may be cases where an SDF component needs
  * to perform its own direct file access in which case generateFileNameBase()
  * is useful.
  * <p>
  * The 'makeUnique' flag indicates whether generateFileNameBase() should
  * try to generate a unique filename (it doesn't guarantee it's unique).
  * If the flag is false the timestamp will be today's date (no time) -
  * appropriate when several dumps/traces can be appended to the same
  * file, e.g. thread dumps.  If the flag is true then greater resolution
  * is included in the timestamp to try and generate a unique filename -
  * e.g. for heap dumps, xml output etc...
  *
  * @param makeUnique flag indicating whether the caller would like the
  * returned filename to be unique (though no guarantee that it actually is unique)
  * @return the generated filename component
  */
  String generateFileNameBase(boolean makeUnique);

 /**
  * Returns a IDiagnosticsHistoryTracker object. Used by diagnostics subsystems to trace their recent history in memory.
  * The diagnostics subsystem can obtain multiple instances of IDiagnosticsHistoryTracker.
  *
  * @param label describes the type of information tracked by this IDiagnosticsHistoryTracker.
  * @param providerObject the array is generated by the diagnostics subsystems. It should contain the number of history
  *        items that should be cached in memory at any time. The providerObject array is created once to avoid the performance penalty
  *        of creating the objects repeatadly.
  *
  * @return the IDiagnosticsHistoryTracker tracking object
  */
  IDiagnosticsHistoryTracker createHistoryTrackerObject(String label, Object[] providerObject);
}