package com.sonicsw.sdf;

import java.util.HashMap;

/**
 * IDiagnosticsProvider must be implemented by a diagnostics subsystem. It is registered with the diagnostics framework using IDiagnosticsManager.register().
 * IDiagnosticsProvider typically extends AbstractDiagnosticsProvider rather than implementing IDiagnosticsProvider from scratch.
 * Each IDiagnosticsProvider implementation can support multiple instances - it must than return a non null value from getSubsystemID.
 * The IDiagnosticsProvider implementation could control multiple DOI (Diagnostics Object Instance). In that case it should return the IDs for these objects
 * from getDOInstances() and should be able to diagnose these objects individually as their IDs are passed to IDiagnosticsProvider methods. If any of
 * IDiagnosticsProvider thorows and exception, the framework prints out the exception's message to framework's status output text.
 */
public interface IDiagnosticsProvider {
	/**
	 * Updates the trace level of this IDiagnosticsProvider.
	 * The new trace level will be passed to the IDiagnosticsProvider as one of the parameters (Integer or String). The IDiagnosticsProvider implementation
	 * is expeced to use one of the IDiagnosticsContext.getTracer() methods to obtain a trace file (but it could also use its own implementation).
	 *
	 * @param doiID      an optional parameter - if updateTraceLevel is called to trace a specific DOI
	 * @param parameters contains the new trace level and optionally other parameters
	 * @param buffer     could be used by IDiagnosticsProvider to provide status/error output
	 */

	void updateTraceLevel(String doiID, HashMap parameters, StringBuffer buffer);

	/**
	 * Shows the trace level of this IDiagnosticsProvider.
	 *
	 * @param doiID      an optional parameter - when showTraceLevel is called to show the trace level a specific DOI
	 * @param parameters optinal parameters
	 * @param buffer     append requested information in this buffer
	 */
	void showTraceLevel(String doiID, HashMap parameters, StringBuffer buffer);

	/**
	 * Dumps the state of this IDiagnosticsProvider instance. The state should be typically written to a IStateWriter file obtained from one of the
	 * IDiagnosticsContext.getStateWriter() methods. But if the state output is small it could be appended to the StringBuffer buffer.
	 *
	 * @param doiID      an optional parameter - if appendStateDump is called to dump the state of a specific DOI
	 * @param Parameters optional parameters
	 * @param buffer     could be used by IDiagnosticsProvider to provide status/error output
	 */
	void appendStateDump(String doiID, HashMap Parameters, StringBuffer buffer);

	/**
	 * Test the system (used by QA). Calling this method has no effect if the system property ENABLE_DIAGNOSTICS_TESTS_SYSPROP is not set to 'true'
	 *
	 * @param parameters could be used to indicate what to test
	 * @param buffer     could be used by IDiagnosticsProvider to provide status/error output
	 */
	void test(HashMap parameters, StringBuffer buffer);

	/**
	 * Returns the unique name of this IDiagnosticsProvider implementation - required.
	 *
	 * @return the unique name of this IDiagnosticsProvider implementation. Required, must not return null.
	 */
	String getSubsystemName();

	/**
	 * Returns the unique ID of this IDiagnosticsProvider instance - optional.
	 *
	 * @return the unique ID of this IDiagnosticsProvider instance. Optional - can return null.
	 */
	String getSubsystemID();

	/**
	 * Returns text description of this IDiagnosticsProvider implementation - required.
	 *
	 * @return text that describes this IDiagnosticsProvider implementation. Required, must not return null.
	 */
	String describe();

	/**
	 * Returns a list of tags that associates this IDiagnosticsProvider implementation with specific topics of diagnostics. Optional - can return null.
	 *
	 * @return a list of tags. For instance: {SSL,COMM,SECURITY}. Optional - can return null.
	 */
	String[] getTags();

	/**
	 * Returns a list of operations supported by this IDiagnosticsProvider implementation. Used for description purposes. Optional - can return null.
	 *
	 * @return a list of operations supported by this IDiagnosticsProvider implementation. For instance: {dumpState,updateTraceLevel}. Optional - can return null.
	 */
	String[] getOperations();

	/**
	 * Returns a list of DOIs controlled by this IDiagnosticsProvider instance. Optional - can return null.
	 *
	 * @return a list of DOIs controlled by this IDiagnosticsProvider instance.Optional - can return null.
	 */
	String[] getDOInstances();

	/**
	 * Returns a HashMap table desribing the parameters of one specific operation. Used for description purposes. Optional - can return null.
	 *
	 * @param operationName the name of the described operation
	 * @return HashMap table describing the parameters of one specific operation. The key in the HashMap table is a name of a parameter; the value is
	 * text that describes that parameter.
	 */
	HashMap describeParameters(String operationName);

	/**
	 * Executes an operation in this instance. The operation is typically not the 'standard' dumpState or updateTraceLevel operation.
	 *
	 * @param doiID      an optional parameter - if executeOperation is called to execute an operation for a specific DOI.
	 * @param operation  the name of the operation.
	 * @param parameters an optional table of parameters
	 * @param buffer     could be used by IDiagnosticsProvider to provide status/error output
	 */
	void executeOperation(String doiID, String operation, HashMap parameters, StringBuffer buffer);

}
