//Copyright (c) 2010 Progress Software Corporation.  All Rights Reserved.

package com.sonicsw.sdf;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.management.ManagementFactory;
import java.lang.management.OperatingSystemMXBean;
import java.lang.management.RuntimeMXBean;
import java.lang.reflect.Method;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.text.DecimalFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

/**
 * Utilities to help provide consistency in dumps generated by SDF component.
 * For example, provides methods for writing a common header and footer that
 * include timestamps and some environment info'.
 */
public class WriterUtil {

    private static final String FILE_PERMISSIONS_REGEX = "([r-][w-][x-]){3}";
    private static final String NO_FILE_PERMISSIONS_REGEX = "(---){3}";

    private static final String TEXT_DUMP_SEPARATOR =    "========================================================================";
    private static final String TEXT_SECTION_SEPARATOR = "------------------------------------------------------------------------";

    private static final RuntimeMXBean s_runtimeMXBean = ManagementFactory.getRuntimeMXBean();
    private static final OperatingSystemMXBean s_opsysMXBean = ManagementFactory.getOperatingSystemMXBean();

    private static DecimalFormat s_doubleFormat = new DecimalFormat("###0.00");

    private static Method getSystemLoadAverage_METHOD = null;  // OperatingSystemMXBean

    // Get Java 1.6 classes/methods (if available)
    static
    {
        try
        {
            getSystemLoadAverage_METHOD = OperatingSystemMXBean.class.getMethod("getSystemLoadAverage");
        }
        catch (Throwable t)
        {
        }
    }
    
    
    /**
     * Write a standard header at the start of an SDF dump.  The
     * IDiagnosticsProvider that calls this can supply additional
     * headers to be included.
     * 
     * @param writer to write header to
     * @param instruction SDF instruction that triggered the dump ({@link #rebuildInstruction(IDiagnosticsProvider, String, Map)})
     * @param additionalHeaders a map of additional header name/value properties, or null if no additional headers are needed
     * @throws IOException if there's a problem writing to the writer
     */
    public static void writeHeader(IStateWriter writer, String instruction, Map<String, String> additionalHeaders) throws IOException
    {
        writer.writeln(TEXT_DUMP_SEPARATOR + IDiagnosticsConstants.NEWLINE);
        writeProperty(writer, IDiagnosticsConstants.HEADERNAME_INSTRUCTION, instruction);

        Map<String, String> headers = getEnvironmentProperties();
        if (additionalHeaders != null)
        {
            headers.putAll(additionalHeaders);
        }
        
        for (Map.Entry<String, String> envProp : headers.entrySet()) {
            writeProperty(writer, envProp.getKey(), envProp.getValue());
        }

        writer.writeln();
        writeProperty(writer, IDiagnosticsConstants.HEADERNAME_STARTTIME, new Date().toString());       
        
        writer.writeln(IDiagnosticsConstants.NEWLINE + TEXT_SECTION_SEPARATOR + IDiagnosticsConstants.NEWLINE);     
    }
    
    /**
     * Write a standard footer at the end of an SDF dump.  The
     * IDiagnosticsProvider that calls this can supply additional
     * footers to be included.
     * 
     * @param writer to write footer to
     * @param additionalFooters a map of additional header name/value properties, or null if no additional headers are needed
     * @throws IOException if there's a problem writing to the writer
     */
    public static void writeFooter(IStateWriter writer, Map<String, String> additionalFooters) throws IOException
    {
        writer.writeln(IDiagnosticsConstants.NEWLINE + TEXT_SECTION_SEPARATOR + IDiagnosticsConstants.NEWLINE); 

        writeProperty(writer, IDiagnosticsConstants.HEADERNAME_ENDTIME, new Date().toString());     
        
        if (additionalFooters != null)
        {
            for (Map.Entry<String, String> envProp : additionalFooters.entrySet())
            {
                writeProperty(writer, envProp.getKey(), envProp.getValue());
            }
        }
        
        writer.writeln(IDiagnosticsConstants.NEWLINE);      
    }
    
    /**
     * Write individual header/footer property
     */
    private static void writeProperty(IStateWriter writer, String name, String value) throws IOException
    {
        writer.writeln(name + ": " + value);
    }
    
    
    /**
     * This is a fudge so we can include the SDF instruction in the dump.
     * It would be better if we had direct access to the original instruction
     * string.  (Maybe IDiagnosticsProvider.dumpState() and the like should
     * take the instruction (ref' InstructionsFileParser.Instruction) rather
     * than just the instruction's parameters?)
     */
    public static String rebuildInstruction(IDiagnosticsProvider provider, String operation, Map<?, ?> parameters)
    {
        StringBuilder sb = new StringBuilder();
        sb.append(provider.getSubsystemName()).append(' ').append(operation);
        
        for (Map.Entry<?, ?> param : parameters.entrySet()) {
            sb.append(' ').append(param.getKey()).append('=').append(param.getValue());
        }
        
        return sb.toString();
    }

    /**
     * Returns a basic set of environment properties for inclusion in the header
     */
    private static Map<String, String> getEnvironmentProperties()
    {
        String containerName = DiagnosticsManagerAccess.getManager().getContainerName();
        String hostname = "<unknown>";
        String opsys = "<unknown>";
        String jvmVersion = "<unknown>";
        String jvmStartTime = "<unknown>";
        String processorCount = "<unknown>";
        String systemLoad = null;
        
        try
        {
            hostname = InetAddress.getLocalHost().getHostName();
        }
        catch(UnknownHostException e) { }
        
        jvmVersion = System.getProperty("java.vm.name") + " (" + System.getProperty("java.vendor") + ", " + System.getProperty("java.version") + ")";
        
        if (s_runtimeMXBean != null)
        {
            long startTime = s_runtimeMXBean.getStartTime();
            if (startTime > 0)
            {
                jvmStartTime = new Date(startTime).toString();
            }
        }
        
        if (s_opsysMXBean != null)
        {
            opsys = s_opsysMXBean.getName() + ", " + s_opsysMXBean.getVersion() + " (" + s_opsysMXBean.getArch() + ")";
            processorCount = Integer.toString(s_opsysMXBean.getAvailableProcessors());
            
            if (getSystemLoadAverage_METHOD != null)
            {
                try {
                    double load = ((Double)getSystemLoadAverage_METHOD.invoke(s_opsysMXBean)).doubleValue();
                    if (load >= 0.0)
                    {
                        systemLoad = s_doubleFormat.format(load) + "%";
                    }
                } catch (Exception e) {
                    System.err.println("Error getting value for system load:");
                    e.printStackTrace();
                }
            }
        }       
        
        Map<String, String> result = new LinkedHashMap<String, String>();
        if (containerName != null)
        {
            // Only write container name header if running in MF Container 
            result.put(IDiagnosticsConstants.HEADERNAME_CONTAINER, containerName);
        }
        result.put(IDiagnosticsConstants.HEADERNAME_HOST, hostname);
        result.put(IDiagnosticsConstants.HEADERNAME_OPERATING_SYS, opsys);
        result.put(IDiagnosticsConstants.HEADERNAME_JVM, jvmVersion);
        result.put(IDiagnosticsConstants.HEADERNAME_START_TIME, jvmStartTime);
        result.put(IDiagnosticsConstants.HEADERNAME_PROCESSOR_COUNT, processorCount);
        if (systemLoad != null)
        {
            result.put(IDiagnosticsConstants.HEADERNAME_AVERAGE_LOAD, systemLoad);
        }
        
        return result;
    }
    
    
    /**
     * Returns result of Throwable.printStackTrace() as a String
     * (e.g. for writing to the SDF result buffer) 
     */
    public static String getExceptionAsString(Throwable t)
    {
        StringWriter writer = new StringWriter();
        t.printStackTrace(new PrintWriter(writer));
        return writer.toString();
    }


    /**
     * Ensures that the parent folders of the given path
     * exists, if is not the case then tries to create them
     * or will raise IOException otherwise
     * @param path
     */
    public static void ensureParentFoldersExists(Path path)
    throws IOException
    {
        Path parent = path.getParent();
        Iterator<Path> it = parent.iterator();
        Path base = path.getRoot();
        while(it.hasNext())
        {
            base = base.resolve(it.next());
            if(!Files.exists(base))
            {
                Files.createDirectory(base);
            }
        }
    }

    /**
     * Establishes the file permissions according to
     * the given permission string.
     *
     * For the permission string format see:
     *
     * java.nio.file.attribute.PosixFilePermissions
     *
     * @param filePath path to establish the permissions
     * @param permissions string
     * @throws IOException
     */
    public static void setFileAccessPermissions(Path filePath, String permissions)
    throws IOException
    {

        //check the received permission string pattern
        if(!permissions.matches(FILE_PERMISSIONS_REGEX) ||
                permissions.matches(NO_FILE_PERMISSIONS_REGEX)){
            throw new IllegalArgumentException("The file permission mask is not valid, please check and/or use the 'describe' operation for help.");
        }
        if(FileSystems.getDefault().supportedFileAttributeViews().contains("posix"))
        {
            Set<PosixFilePermission> permissionSet = PosixFilePermissions.fromString(permissions);
            Files.setPosixFilePermissions(filePath, permissionSet);
        }
    }
}
