package com.sonicsw.sdf.impl;


import static com.sonicsw.sdf.AbstractDiagnosticsProvider.OP_PARAM_OUTPUT_LOCATION;
import static com.sonicsw.sdf.AbstractDiagnosticsProvider.OP_PARAM_OVERWRITE;

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;

import com.sonicsw.sdf.*;

final public class DiagnosticsContext implements IDiagnosticsContext, IDiagnosticsConstants
{
    private static final String PSEUDO_CONTAINER_NAME = "sonic.diagnostics";  // Used in filename when an (MF) container name is not available 
    
    private final SimpleDateFormat m_dateFormatterShort = new SimpleDateFormat("yyyyMMdd");
    private final SimpleDateFormat m_dateFormatterLong = new SimpleDateFormat("yyyyMMdd-HHmmss-SSS");
    
    String m_subsystemName;
    String m_subsystemID;
    String m_name;

    public DiagnosticsContext (String subsystemName, String subsystemID)
    {
        m_subsystemName = subsystemName;
        m_subsystemID = subsystemID;
        m_name = m_subsystemName;
        if (subsystemID != null)
        {
            m_name += "." + m_subsystemID;
        }
    }

    @Override
    public ITracer getTracer(String tracerFilePath, long maxTracerFileLength) throws IOException
    {
        return new RolloverLogger(tracerFilePath, maxTracerFileLength);
    }

    @Override
    public ITracer getTracer(long maxTracerFileLength) throws IOException
    {
        String filename = generateFileNameBase(false) + TRACE_FILE_POSTFIX;
        return new RolloverLogger(filename, maxTracerFileLength);
    }

    @Override
    public ITracer getTracer(String tracerFilePath) throws IOException
    {
        return new RolloverLogger(tracerFilePath, NO_LIMIT_TRACE_FILE_SIZE);
    }

    @Override
    public ITracer getTracer() throws IOException
    {
        String filename = generateFileNameBase(false) + TRACE_FILE_POSTFIX;
        return new RolloverLogger(filename, NO_LIMIT_TRACE_FILE_SIZE);
    }

    @Override
    public IDiagnosticsHistoryTracker createHistoryTrackerObject(String label, Object[] providerObjects)
    {
        return new DiagnosticsHistoryTracker(label, providerObjects);
    }

    @Override
    public IStateWriter getStateWriter(String stateFilePath) throws IOException
    {
        return new RolloverLogger(stateFilePath, NO_LIMIT_TRACE_FILE_SIZE);
    }

    @Override
    public IStateWriter getStateWriter() throws IOException
    {
        String filename = generateFileNameBase(false) + STATE_FILE_POSTFIX;
        return new RolloverLogger(filename, NO_LIMIT_TRACE_FILE_SIZE);
    }

    @Override
    public IStateWriter getStateWriter(Map<String, String> parameters) throws IOException
    {
        boolean overwrite = true;
        if(parameters.containsKey(OP_PARAM_OVERWRITE))
        {
            overwrite = Boolean.parseBoolean(parameters.get(OP_PARAM_OVERWRITE));
        }

        String filePath = parameters.get(OP_PARAM_OUTPUT_LOCATION);

        String filename = generateFileNameBase(!overwrite) + STATE_FILE_POSTFIX;

        Path outputFile = Paths.get(filename);
        if(filePath != null)
        {
            outputFile = Paths.get(filePath).resolve(outputFile);
            WriterUtil.ensureParentFoldersExists(outputFile);
        }

        return new RolloverLogger(outputFile.toString(), NO_LIMIT_TRACE_FILE_SIZE);
    }

    @Override
    public String generateFileNameBase(boolean makeUnique)
    {
        StringBuilder filename = new StringBuilder();
        
        String containerName = DiagnosticsManagerAccess.getManager().getContainerName();
        if (containerName == null)
        {
            containerName = PSEUDO_CONTAINER_NAME;
        }
        
        filename.append(containerName).append('.');
        
        SimpleDateFormat dateFormat = makeUnique ? m_dateFormatterLong : m_dateFormatterShort;
        filename.append(dateFormat.format(new Date())).append('.');
        
        filename.append(m_name);
        
        for (int i = 0; i < filename.length(); i++)
        {
            char c = filename.charAt(i);
            if (!Character.isLetterOrDigit(c) && c != '-' && c != '.' && c != '$')
            {
                filename.setCharAt(i, '_');
            }
        }
        
        return filename.toString();
    }
}