package com.sonicsw.sdf.impl;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;

import com.sonicsw.sdf.IDiagnosticsConstants;
import com.sonicsw.sdf.IDiagnosticsHistoryTracker;

final public class DiagnosticsHistoryTracker implements IDiagnosticsHistoryTracker, IDiagnosticsConstants
{
    private static final ThreadLocal<SimpleDateFormat> SIMPLE_DATE_FORMAT = new ThreadLocal<SimpleDateFormat>() {
        @Override
        protected SimpleDateFormat initialValue() {
            return new SimpleDateFormat("yy/MM/dd HH:mm:ss");
        }
    };

    private HistoryItem[] m_history;
    private int m_currentPointer;
    private String m_label;
    private long m_itemCounter;

    //Unit test
    public static void main(String[] args) throws Exception
    {
        TestItem[] items = new TestItem[20];
        for (int i = 0; i < 20; i++)
        {
            items[i] = new TestItem();
        }

        DiagnosticsHistoryTracker tracker = new DiagnosticsHistoryTracker("My Tracker", items);

        for (int i = 0; i < 59; i++)
        {
            TestItem item = (TestItem)tracker.getHistoryItem();
            item.m_value = new Integer(i).toString();
        }

        StringBuffer buffer = new StringBuffer();
        tracker.appendHistory(buffer, true);
        System.out.println(buffer);

    }

    DiagnosticsHistoryTracker(String label, Object[] providerObjects)
    {
        m_label = label;
        m_itemCounter = 0;
        m_currentPointer = 0;
        int numItems = providerObjects.length;
        m_history = new HistoryItem[numItems];
        for (int i = 0; i < numItems; i++)
        {
            m_history[i] = new HistoryItem(providerObjects[i]);
        }
    }

    @Override
    public Object getHistoryItem()
    {
        Object providerObject = m_history[m_currentPointer].getProviderObject();
        m_history[m_currentPointer].setTimestamp();
        m_currentPointer = (m_currentPointer + 1) % m_history.length;
        m_itemCounter++;
        return providerObject;
    }

    @Override
    public synchronized void appendHistory(StringBuffer buffer)
    {
        appendHistory(buffer, true);
    }

    public synchronized void appendHistory(StringBuffer buffer, boolean doFormatTime)
    {
        buffer.append(NEWLINE + m_label);
        if (m_itemCounter == 0)
        {
            buffer.append(" *** No items ***" + NEWLINE);
            return;
        }
        else
        {
            buffer.append(NEWLINE);
        }

        int crntItem;
        int numItems;

        if (m_itemCounter <= m_history.length)
        {
            crntItem = 0;
            numItems = (int)m_itemCounter;
        }
        else
        {
            crntItem = m_currentPointer;
            numItems = m_history.length;
        }

        for (int i = 0; i < numItems; i++)
        {
            long time = m_history[crntItem].getTimestamp();
            String timeString = doFormatTime ? formatTime(time) + " " : time + "UTC ";
            buffer.append(timeString + m_history[crntItem].getProviderObject() + NEWLINE);
            crntItem = (crntItem + 1) % m_history.length;
        }
    }


    private static String formatTime(long timestamp)
    {
        return SIMPLE_DATE_FORMAT.get().format(new Date(timestamp));
    }

    private class HistoryItem
    {
        private Object m_providerObject;
        private long m_timestamp;

        HistoryItem(Object providerObject)
        {
            m_providerObject = providerObject;
        }

        Object getProviderObject()
        {
            return m_providerObject;
        }

        long getTimestamp()
        {
            return m_timestamp;
        }

        void setTimestamp()
        {
            m_timestamp = System.currentTimeMillis();
        }
    }




    /*** Utlity Methods ***/

    public void appendMap(HashMap map, StringBuffer buffer)
    {
        Iterator it = map.keySet().iterator();
        while (it.hasNext())
        {
            Object key = it.next();
            Object value = map.get(key);
            buffer.append(key).append(" ").append(value).append(NEWLINE);
        }
        buffer.append("------------------------------------------" + NEWLINE);
    }

    static class TestItem
    {
        String m_value;

        @Override
        public String toString()
        {
            return m_value;
        }
    }

}

