package com.sonicsw.sdf.impl;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.StringTokenizer;

import com.sonicsw.sdf.AbstractDiagnosticsProvider;
import com.sonicsw.sdf.AbstractMFComponentTracing;
import com.sonicsw.sdf.DiagnosticsException;
import com.sonicsw.sdf.IDiagnosticsConstants;
import com.sonicsw.sdf.IDiagnosticsContext;
import com.sonicsw.sdf.IDiagnosticsManager;
import com.sonicsw.sdf.IDiagnosticsProvider;
import com.sonicsw.sdf.IShutdown;
import com.sonicsw.sdf.impl.InstructionsFileParser.Instruction;


final public class DiagnosticsManager implements IDiagnosticsManager, IDiagnosticsConstants
{

    public static final String HELP_TEXT_PATH = "/com/sonicsw/sdf/text/sdf-help";
    private static final String NO_INSTRUCTIONS_TEXT = "    There are no instructions to execute";

    private static String[] OPERATIONS;
    private static HashMap DUMP_STATE_PARAM_DESCIPTOR = new HashMap();
    private static HashMap DESCRIBE_PARAM_DESCIPTOR = new HashMap();
    private static HashMap DESCRIBE_ALL_PARAM_DESCIPTOR = new HashMap();
    private static HashMap SETUP_PARAM_DESCIPTOR = new HashMap();
    private static HashMap LISTALL_PARAM_DESCIPTOR = new HashMap();
    private static HashMap HELP_PARAM_DESCIPTOR = new HashMap();

    private static HashMap PARAM_DESCRIPTOR = new HashMap();
    private static String DESCRIPTION_TEXT;
    private static String SYS_PROP_DESCRIPTION_TEXT;

    //IDiagnosticsProvider Description
    static
    {
        SETUP_PARAM_DESCIPTOR.put(PROCESS_COUNT_PARAM, "The number of times the instructions file should get executed. Default is 1 (ignored in remote diagnostics)");
        SETUP_PARAM_DESCIPTOR.put(STATUS_FILE_DEFAULT_PARAM, "Status file path. Default is '" + DIAGNOSTICS_STATUS_FILE_NAME + "'" + " (ignored in remote diagnostics)");

        PARAM_DESCRIPTOR.put(DUMP_STATE_OP, DUMP_STATE_PARAM_DESCIPTOR);
        PARAM_DESCRIPTOR.put(DESCRIBE_OP, DESCRIBE_PARAM_DESCIPTOR);
        PARAM_DESCRIPTOR.put(DESCRIBE_ALL_OP, DESCRIBE_ALL_PARAM_DESCIPTOR);
        PARAM_DESCRIPTOR.put(SETUP_OP, SETUP_PARAM_DESCIPTOR);
        PARAM_DESCRIPTOR.put(LISTALL_OP, LISTALL_PARAM_DESCIPTOR);
        PARAM_DESCRIPTOR.put(HELP_OP, HELP_PARAM_DESCIPTOR);
        OPERATIONS = AbstractDiagnosticsProvider.toOpnameArray(PARAM_DESCRIPTOR);
    }

    static
    {
        StringBuffer tmp = new StringBuffer();
        tmp.append("The ").append(INSTRUCTIONS_FILE_SEARCH_FREQUENCY_SYSPROP).append(" system property can be used to change the default of ").append(INSTRUCTIONS_FILE_SEARCH_FREQUENCY_DFLT).append(" seconds frequency of processing the instructions file (minimum is ").append(INSTRUCTIONS_FILE_SEARCH_FREQUENCY_MIN).append(" seconds)").append(NEWLINE);
        tmp.append("The ").append(DIAGNOSTICS_INSTRUCTIONS_FILE_PATH_SYSPROP).append(" system property can be used to change the '").append(DIAGNOSTICS_INSTRUCTIONS_FILE_NAME_DFLT).append("' instructions file name default");
        SYS_PROP_DESCRIPTION_TEXT = tmp.toString();

        tmp = new StringBuffer();
        tmp.append("Provides setup and shows the state of the diagnostics system").append(NEWLINE);
        tmp.append(SYS_PROP_DESCRIPTION_TEXT);
        DESCRIPTION_TEXT = tmp.toString();
    }


    private HashMap m_subSystems;
    private File m_instructionsFile;
    private File m_shutdownFile;
    private DiagnosticsSubsystem m_diagnosticsSys;
    private HashMap m_delayedTraceRequests;
    private boolean m_enableTests;
    private IShutdown m_shutdownInterface = null;
    private String m_containerName = null;


    //For unit tests
    public static void main (String[] args)
    {
        com.sonicsw.sdf.DiagnosticsManagerAccess.createManager();
/*
        new TestMFComponent("sonic.ds.component", "16=startup,32=updates,64=pass,128=container access,256=configuration notifications,512=validation triggers,1024=all directory service access,2048=fault detection").register();
        new TestSubsystem("mf.ds.subsys1").register();

        new TestSubsystem("mf.ds.subsys2").register();
        new TestSubsystem("subsys3").register();

        String sysID = "i1$abc&a^";

        new TestMultiInstSubsystem("mf.am.multiSubsys1", sysID).register();
        new TestMultiInstSubsystem("mf.am.multiSubsys2", "i2").register();
        new TestMultiInstSubsystem("mf.util", "i3").register();
        new TestMultiInstSubsystem("multiSubsys2", "i4").register();

        new TestMultiInstSubsystem("multiSubsys2", "i5").register();

        try
        {
            Thread.sleep(100);
        }
        catch (Exception e)
        {
        }
        new TestSubsystem("sonic.mq.debug").register();

*/
        doSleep(1000000);

    }


    public DiagnosticsManager()
    {
        m_shutdownFile = new File(CONTAINER_SHUTDOWN_FILE_NAME);

        m_enableTests = Boolean.getBoolean(ENABLE_DIAGNOSTICS_TESTS_SYSPROP);

        m_delayedTraceRequests = new HashMap();
        
        // SNC00081641 - Allow the instruction file watcher thread to be disabled by
        // setting system property 'SonicDiagnosticsFileFrequency' to zero.
        // Deprecated an earlier name for this property which was missing a 'Sonic'
        // prefix. The new name takes precedence over the old name if both are set.
        int tmp = Integer.getInteger(INSTRUCTIONS_FILE_SEARCH_FREQUENCY_SYSPROP_DEPRECATED, INSTRUCTIONS_FILE_SEARCH_FREQUENCY_DFLT).intValue();
        tmp = Integer.getInteger(INSTRUCTIONS_FILE_SEARCH_FREQUENCY_SYSPROP, tmp).intValue();
        if (tmp > 0 && tmp < INSTRUCTIONS_FILE_SEARCH_FREQUENCY_MIN)
        {
            tmp = INSTRUCTIONS_FILE_SEARCH_FREQUENCY_MIN;
        }
        if (tmp > INSTRUCTIONS_FILE_SEARCH_FREQUENCY_MAX)
        {
            tmp = INSTRUCTIONS_FILE_SEARCH_FREQUENCY_MAX;
        }

        final int waitTimeBetweenFileSearch = tmp;

        boolean fileSystemPropertySet = false;
        String instructionsFilePath = System.getProperty(DIAGNOSTICS_INSTRUCTIONS_FILE_PATH_SYSPROP);
        if (instructionsFilePath != null)
        {
            fileSystemPropertySet = true;
        }
        else
        {
            instructionsFilePath = DIAGNOSTICS_INSTRUCTIONS_FILE_NAME_DFLT;
        }

        m_instructionsFile = new File(instructionsFilePath);

        if (fileSystemPropertySet)
        {
            UnblockingPrintln("System property '" + DIAGNOSTICS_INSTRUCTIONS_FILE_PATH_SYSPROP + "' was set to '" + m_instructionsFile.getAbsolutePath() + "' ");
        }

        m_subSystems = new HashMap();

        //Create and register DiagnosticsSubsystem that represents DiagnosticsManager
        m_diagnosticsSys = new DiagnosticsSubsystem(SONIC_DIAGNOSTICS_SUBSYS);
        IDiagnosticsContext diagContext = register(m_diagnosticsSys);
        m_diagnosticsSys.setContext(diagContext);

        //Create and register JVM_THREADS_SUBSYS
        JVMThreadDiagnostics jvmThreadDiag = new JVMThreadDiagnostics();
        IDiagnosticsContext jvmThreadContext = register(jvmThreadDiag);
        jvmThreadDiag.setContext(jvmThreadContext);

        //Create and register JVM_HEAP_SUBSYS
        JVMHeapDiagnostics jvmHeapDiag = new JVMHeapDiagnostics();
        IDiagnosticsContext jvmHeapContext = register(jvmHeapDiag);
        jvmHeapDiag.setContext(jvmHeapContext);


        if (waitTimeBetweenFileSearch > 0)  // File watcher thread disabled if wait time is zero
        {
            Thread diagnosticsThread = new Thread("DiagnosticsManager: Executes diagnostics requests submitted through an instructions file")
            {
                @Override
                public void run()
                {

                   long prevLastModified = 0;
                   int numExecutions = 0;
                   while (true)
                   {
                       if (m_shutdownFile.exists() && m_shutdownInterface != null)
                    {
                        startShutdown(m_shutdownInterface);
                    }

                       if (m_instructionsFile.exists())
                       {
                           long lastModified = m_instructionsFile.lastModified();
                           if (lastModified != prevLastModified)
                           {
                               m_diagnosticsSys.init();
                               prevLastModified = lastModified;
                               processInstructionsFile(m_instructionsFile);
                               numExecutions = m_diagnosticsSys.getNumExecutions();
                           }
                           else if (numExecutions > 1)
                           {
                               processInstructionsFile(m_instructionsFile);
                               numExecutions--;
                           }
                       }
                       doSleep(waitTimeBetweenFileSearch);
                       }
                }
            };

            diagnosticsThread.setDaemon(true);
            diagnosticsThread.start();
        }
    }

    @Override
    public void setShutdownInterface(IShutdown shutdownInterface)
    {
        m_shutdownInterface = shutdownInterface;
    }

    @Override
    public void setContainerName(String containerName) {
        m_containerName = containerName;
    }

    @Override
    public String getContainerName() {
        return m_containerName;
    }

    @Override
    public synchronized IDiagnosticsContext register(IDiagnosticsProvider diagnosedSystem) throws DiagnosticsException
    {
        String subsystemName = diagnosedSystem.getSubsystemName();
        String subsystemID = diagnosedSystem.getSubsystemID();


        if (subsystemName == null)
        {
            throw new DiagnosticsException("Cannot register IDiagnosticsProvider object " + diagnosedSystem + " its getName() method returns null ");
        }


        Object allreadyRegistered = m_subSystems.get(subsystemName);

        if (allreadyRegistered != null && subsystemID == null && allreadyRegistered instanceof HashMap)
        {
            throw new DiagnosticsException("Cannot register IDiagnosticsProvider object " + subsystemName + " getSubsystemID() is null ");
        }

        if (allreadyRegistered != null && subsystemID != null && allreadyRegistered instanceof IDiagnosticsProvider)
        {
            throw new DiagnosticsException("Cannot register IDiagnosticsProvider object " + subsystemName + " previous getSubsystemID() was null ");
        }

        if (allreadyRegistered == null)
        {
            if (subsystemID == null)
            {
                m_subSystems.put(subsystemName, diagnosedSystem);
            }
            else
            {
                HashMap instances = new HashMap();
                instances.put(subsystemID, diagnosedSystem);
                m_subSystems.put(subsystemName, instances);
            }
        }
        else // llreadyRegistered != null
        {
            if (subsystemID == null)
            {
                m_subSystems.put(subsystemName, diagnosedSystem); //Replaces the registered system
            }
            else
            {
                ((HashMap)allreadyRegistered).put(subsystemID, diagnosedSystem);
            }
        }

        //Executes requests that were delayed since the subsystem didn't yet register
        String sysInstanceName = generateFullName(null, subsystemName, subsystemID);
        if (m_delayedTraceRequests.containsKey(sysInstanceName))
        {
            ArrayList delayedRequests = (ArrayList)m_delayedTraceRequests.remove(sysInstanceName);

            int numRequests = delayedRequests.size();
            for (int i = 0; i < numRequests; i++)
            {
                DelayedRequest request = (DelayedRequest)delayedRequests.remove(0);
                request.executeDelayedInProvider(diagnosedSystem);
            }
        }

        return new DiagnosticsContext(subsystemName, subsystemID);
    }

    public static IDiagnosticsContext createContext(IDiagnosticsProvider diagnosedSystem)
    {
        String subsystemName = diagnosedSystem.getSubsystemName();
        String subsystemID = diagnosedSystem.getSubsystemID();
        return new DiagnosticsContext(subsystemName, subsystemID);
    }

    @Override
    public void executeOperation (String opName, String subsystemName, String[] tags, String subsystemID, String doiID, StringBuffer buffer, HashMap parameters)
    {
        String namePrefix = null;
        try
        {
            namePrefix = getWildcardPrefix(subsystemName);
        }
        catch (Exception e)
        {
            buffer.append(e.getMessage());
            return;
        }

        if (namePrefix == null)
        {
            executeOperationInternal(doiID, subsystemName, subsystemID, opName, buffer, parameters);
            return;
        }

        if (doiID != null || subsystemID != null)
        {
            throw getUniquenessException(subsystemName);
        }

        IDiagnosticsProvider[] systems = findRegisteredSystems(namePrefix, opName.equalsIgnoreCase(DESCRIBE_OP), tags);
        executeOperationInternal(systems, opName, buffer, parameters);

    }


    @Override
    public void executeOperation (String opName, String subsystemName, StringBuffer buffer, HashMap parameters)
    {

        String namePrefix = null;
        try
        {
            namePrefix = getWildcardPrefix(subsystemName);
        }
        catch (Exception e)
        {
            buffer.append(e.getMessage());
            return;
        }

        if (namePrefix == null)
        {
            executeOperationInternal(null, subsystemName, null, opName, buffer, parameters);
            return;
        }

        IDiagnosticsProvider[] systems = findRegisteredSystems(namePrefix, opName.equalsIgnoreCase(DESCRIBE_OP), null);
        executeOperationInternal(systems, opName, buffer, parameters);

    }


    @Override
    public synchronized HashMap getRegistrationList()

    {
        HashMap result = new HashMap();
        Iterator iter = m_subSystems.keySet().iterator();
        while (iter.hasNext())
        {
            String sysName = (String)iter.next();
            Object tmp = m_subSystems.get(sysName);
            String[] instArray = new String[0];
            if (tmp instanceof HashMap)
            {

                HashMap instaces = (HashMap)tmp;
                ArrayList intList = new ArrayList();
                Iterator instIter = instaces.keySet().iterator();
                while (instIter.hasNext())
                {
                    intList.add(instIter.next());
                }
                instArray = new String[intList.size()];
                intList.toArray(instArray);
            }

            result.put(sysName, instArray);
        }
        return result;
    }

    @Override
    public String executeDiagnosticsInstructions(String diagnosticsInstructionsString)
    {
        return processInstructionsString(diagnosticsInstructionsString);
    }

    private void listInstances(String systemName, String subsystemID, IDiagnosticsProvider diagnosticsSystem, StringBuffer buffer)
    {
        String sysName = generateFullName(null, systemName, subsystemID);
        String[] diagObjects = diagnosticsSystem.getDOInstances();
        boolean hasObjects = diagObjects != null && diagObjects.length > 0;
        if (hasObjects)
        {
            buffer.append(NEWLINE).append("Diagnostics objects of '" + generateFullName(null, systemName, subsystemID)).append("':").append(NEWLINE);
        }
        else
        {
            buffer.append("'").append(generateFullName(null, systemName, subsystemID)).append("' does not contain diagnostics object instances").append(NEWLINE);
            return;
        }

        for (int i = 0; i < diagObjects.length; i++)
        {
            buffer.append("  ").append(diagObjects[i]).append(NEWLINE);
        }
    }

    private void describe(String systemName, IDiagnosticsProvider subsys, StringBuffer buffer)
    {
        if (subsys == null)
        {
            subsys = findRegisteredSystem(systemName, null, true);
        }
        buffer.append(NEWLINE).append(systemName).append(": ").append(subsys.describe()).append(NEWLINE);

        String[] operations = subsys.getOperations();

        for (int i = 0; i < operations.length; i++)
        {
            buffer.append("  operation '").append(operations[i]);
            HashMap parameters = subsys.describeParameters(operations[i]);
            if (parameters.size() > 0)
            {
                buffer.append("' parameters: ").append(NEWLINE);
            }
            else
            {
                buffer.append("' no parameters ").append(NEWLINE);
            }

            Iterator iter = parameters.keySet().iterator();
            while (iter.hasNext())
            {
                String parameter = (String)iter.next();
                buffer.append("    '").append(parameter).append("' ").append(parameters.get(parameter)).append(NEWLINE);
            }
        }

        String[] tagArray = subsys.getTags();
        if (tagArray != null && tagArray.length > 0)
        {
            buffer.append("  Tags: ").append(tagArrayToList(tagArray)).append(NEWLINE);
        }


    }

    private synchronized void listAll(StringBuffer buffer)
    {
        buffer.append(NEWLINE).append(" * Registered subsystems:").append(NEWLINE).append(NEWLINE);
        HashMap regList = getRegistrationList();

        Iterator iter = regList.keySet().iterator();
        while (iter.hasNext())
        {
            String sysName = (String)iter.next();

            String[] instances = (String[])regList.get(sysName);
            if (instances.length == 0)
            {
                buffer.append(sysName).append(NEWLINE);
                continue;
            }

            for (int i = 0; i < instances.length; i++)
            {
                buffer.append(sysName).append(" subsystemID=" + instances[i]).append(NEWLINE);
            }
        }
    }

    private synchronized void describeAll(StringBuffer buffer)
    {
        Iterator iter = m_subSystems.keySet().iterator();
        while (iter.hasNext())
        {
            describe((String)iter.next(), null,buffer);
        }
    }

    private void executeInProvider(IDiagnosticsProvider provider, String doiID, String operation, HashMap parameters, StringBuffer buffer)
    {
        int beforeLength = 0;
        if (buffer != null)
        {
            beforeLength = buffer.length();
        }

        try
        {
            if (operation.equalsIgnoreCase(UPDATE_TRACE_LEVEL_OP))
            {
                provider.updateTraceLevel(doiID, parameters, buffer);
            }
            else if (operation.equalsIgnoreCase(SHOW_TRACE_LEVEL_OP))
            {
                provider.showTraceLevel(doiID, parameters, buffer);
            }
            else if (operation.equalsIgnoreCase(DUMP_STATE_OP))
            {
                provider.appendStateDump(doiID, parameters, buffer);
            }
            else if (operation.equalsIgnoreCase(TEST_OP))
            {
                if (m_enableTests)
                {
                    provider.test(parameters, buffer);
                }
                else
                {
                    buffer.append("The '").append(ENABLE_DIAGNOSTICS_TESTS_SYSPROP).append("' system property must be set to 'true' to enable the execution of test() methods - ignored");
                }

            }
            else
            {
                provider.executeOperation(doiID, operation, parameters, buffer);
            }

            int afterLength = 0;
            if (buffer != null)
            {
                afterLength = buffer.length();
            }

            if (afterLength > beforeLength && buffer.charAt(afterLength - 1) != '\n')
            {
                buffer.append(NEWLINE);
            }

        }
        catch (Throwable t)
        {
            buffer.append("Execution of '").append(operation).append("' in '").append(provider.getSubsystemName()).append("' caused: ").append(t);
        }
    }


    private synchronized IDiagnosticsProvider findRegisteredSystem(String systemName, String subsystemID, boolean anyInstance)
    {
        Object instances = m_subSystems.get(systemName);
        if (instances == null)
        {
            return null;
        }
        else if (instances instanceof HashMap)
        {
            if (subsystemID == null)
            {
                if(!anyInstance)
                {
                    throw new DiagnosticsException("*** Cannot find subsystem " + systemName + " instance ID is null");
                }
                else
                {
                    Iterator it = ((HashMap)instances).keySet().iterator();
                    if (!it.hasNext())
                    {
                        return null;
                    }
                    else
                    {
                        String id = (String)it.next();
                        return (IDiagnosticsProvider)((HashMap)instances).get(id);
                    }
                }
            }
            else
            {
                return (IDiagnosticsProvider)((HashMap)instances).get(subsystemID);
            }
        }
        else
        {
            // Ignores subsystemID
            return (IDiagnosticsProvider)instances;
        }

    }
    private synchronized IDiagnosticsProvider[] findRegisteredSystems(String systemNamePrefix, boolean singleInstancePerSystem, String[] filterTags)
    {
        IDiagnosticsProvider[] result = null;
        ArrayList resultList = new ArrayList();
        Iterator it = m_subSystems.keySet().iterator();

        while (it.hasNext())
        {
            String subsysName = (String)it.next();
            if (!subsysName.startsWith(systemNamePrefix))
            {
                continue;
            }

            Object instancesObj = m_subSystems.get(subsysName);
            if (instancesObj instanceof HashMap)
            {
                HashMap instances = (HashMap)instancesObj;
                Iterator instIt = instances.keySet().iterator();
                while (instIt.hasNext())
                {
                    resultList.add(instances.get(instIt.next()));
                    if (singleInstancePerSystem)
                    {
                        break;
                    }
                }
            }
            else
            {
                resultList.add(instancesObj);
            }
        }

        // Filter out providers not marked with any of the requested tags
        if (filterTags != null && filterTags.length > 0)
        {
            for (int i = resultList.size() - 1; i>= 0; i--)
            {
                IDiagnosticsProvider provider = (IDiagnosticsProvider)resultList.get(i);
                if (!tagFilter(provider.getTags(), filterTags))
                {
                    resultList.remove(i);
                }
            }
        }

        result = new IDiagnosticsProvider[resultList.size()];
        resultList.toArray(result);
        return result;

    }

    // If subsysName is a.b.c.* then return a.b.c
    // If subsysName is a.b.c then return null
    // If more than a single '*' or '*' is not the last component then throw an exception
    private static String getWildcardPrefix(String subsysName)
    {
        int wilcardIndex = subsysName.indexOf(WILDE_CARD_STRING);

        if (wilcardIndex == -1)
        {
            return null;
        }

        if (subsysName.length() == 1 && wilcardIndex == 0)
         {
            return ""; //All sub systems
        }

        // Wild card is not the last character
        if (wilcardIndex != subsysName.length() - 1)
        {
            throw new DiagnosticsException(WILD_CARD_ERROR_MESSAGE);
        }

        if (subsysName.charAt(wilcardIndex - 1) != COMPONENT_SEPERATOR_CHAR)
        {
            throw new DiagnosticsException(WILD_CARD_ERROR_MESSAGE);
        }

        return subsysName.substring(0, wilcardIndex - 1);
    }

    //Return true if the subsystem tags match any of the filter tags passed in the Instructions File
    private static boolean tagFilter(String[] subsystemTags, String[] filterTags)
    {
        //No tags are required
        if (filterTags == null || filterTags.length == 0)
        {
            return true;
        }

        //Tags are required but the provider has non
        if (subsystemTags == null || subsystemTags.length == 0)
        {
            return false;
        }

        // Any matching tag cause returning 'true'
        for (int i = 0; i < filterTags.length; i++)
        {
            for (int j = 0; j < subsystemTags.length; j++)
            {
                if (filterTags[i].equalsIgnoreCase(subsystemTags[j]))
                {
                    return true;
                }
            }
        }

        return false;
    }

    //Convert a comma seperated list of tags to a String array
    private static String[] tagListToArray(String list)
    {
        if (list == null || list.length() == 0)
        {
            return null;
        }

        ArrayList tmpList = new ArrayList();

        StringTokenizer tokens = new StringTokenizer(list, COMMA_STRING);
        while (tokens.hasMoreElements())
        {
            String tag = tokens.nextToken();
            if (tag.length() > 0)
            {
                tmpList.add(tag);
            }
        }
        String[] result = new String[tmpList.size()];
        tmpList.toArray(result);
        return result;


    }

    private static String tagArrayToList(String[] tagArray)
    {
        StringBuffer tagList = new StringBuffer();
        for (int i = 0; i < tagArray.length; i++)
        {
            tagList.append(tagArray[i]);
            if (i + 1 < tagArray.length)
            {
                tagList.append(COMMA_STRING);
            }
        }

        return tagList.toString();
    }


    //Exceutes an operation in multiple susbsystems
    private void executeOperationInternal(IDiagnosticsProvider[] providers, String operation, StringBuffer buffer, HashMap parameters)
    {
        for (int i = 0; i < providers.length; i++)
        {
            buffer.append(NEWLINE).append("    - ").append(providers[i].getSubsystemName()).append(" -").append(NEWLINE);
            executeOperationInternal(null,
                                     providers[i].getSubsystemName(),
                                     providers[i].getSubsystemID(),
                                     operation,
                                     buffer,
                                     parameters);
        }
    }

    private void executeOperationInternal(String doiID, String systemName, String subsystemID, String operation, StringBuffer buffer, HashMap parameters)
    {
        boolean describeOp = operation.equalsIgnoreCase(DESCRIBE_OP);
        boolean listObjectsOp = operation.equalsIgnoreCase(LIST_DIAGNOSTICS_INSTANCES_OP);
        boolean updateTraceLevelOp = operation.equalsIgnoreCase(UPDATE_TRACE_LEVEL_OP);

        IDiagnosticsProvider diagnosticsSystem = null;
        try
        {
            diagnosticsSystem = findRegisteredSystem(systemName, subsystemID, describeOp);
        }
        catch (Exception e)
        {
            buffer.append(e.getMessage());
            return;
        }

        if (diagnosticsSystem == null)
        {
            String sysInstanceName = generateFullName(null, systemName, subsystemID);
            buffer.append("*** Subsystem '" + sysInstanceName + "' is not registered");
            if (updateTraceLevelOp)
            {

                buffer.append(NEWLINE + " The trace level will be updated when '" + sysInstanceName + "' registers");
                ArrayList delayedRequests = (ArrayList)m_delayedTraceRequests.get(sysInstanceName);
                if (delayedRequests == null)
                {
                    delayedRequests = new ArrayList();
                    m_delayedTraceRequests.put(sysInstanceName, delayedRequests);
                }
                delayedRequests.add(new DelayedRequest(doiID, systemName, subsystemID, operation, parameters));
            }
        }
        else if (listObjectsOp)
        {
            listInstances(systemName, subsystemID, diagnosticsSystem, buffer);
        }
        else if (describeOp)
        {
            describe(systemName, diagnosticsSystem, buffer);
        }
        else
        {
            executeInProvider(diagnosticsSystem, doiID, operation, parameters, buffer);
        }
    }


    private static String generateFullName(String doiID, String systemName, String subsystemID)
    {
        StringBuffer buffer = new StringBuffer();
        buffer.append(systemName);
        if (subsystemID != null)
        {
            buffer.append(".").append(subsystemID);
        }
        if (doiID != null)
        {
            buffer.append(".").append(doiID);
        }
        return buffer.toString();
    }

    private static void doSleep(int scnds)
    {
        try
        {
            Thread.sleep(scnds*1000);
        }
        catch (Exception e)
        {
        }
    }

    // Process from a file
    //First process all the sonic.diagnostics subsystem instructions and then all the other instructions
    private void processInstructionsFile(File instructionsFile)
    {
        PrintWriter writer = null;
        try
        {
            ArrayList<Instruction> instructions = InstructionsFileParser.parse(instructionsFile);
            StringBuffer sonicDiagBuffer = new StringBuffer();
            ArrayList<Instruction> sonicDiagInstructions = extractInstructions(instructions, true);
            ArrayList<Instruction> regularInstructions = extractInstructions(instructions, false);

            processInstructions(sonicDiagInstructions, null, sonicDiagBuffer);

            String dateString = RolloverLogger.formatTime(System.currentTimeMillis());
            logDiagnosticsFromFileProcessingMessage(dateString, instructionsFile);

            writer = new PrintWriter(new BufferedWriter(new FileWriter(m_diagnosticsSys.getOutFile(), true)), true);
            StringBuffer titleBuffer = new StringBuffer();
            titleBuffer.append("    --- ").append(dateString).append(" Executing instruction file '");
            titleBuffer.append(instructionsFile.getAbsolutePath()).append("' ---");
            writer.println(titleBuffer);
            String sonicDiagInfo = sonicDiagBuffer.toString();
            if (sonicDiagInfo != null && sonicDiagInfo.length() > 0)
            {
                writer.print(sonicDiagInfo);
            }
            writer.flush();

            processInstructions(regularInstructions, writer, null);
            if (regularInstructions.size() + sonicDiagInstructions.size() == 0)
            {
                writer.println(NO_INSTRUCTIONS_TEXT);
                writer.flush();
            }


        }
        catch (Exception e)
        {
            UnblockingPrintln("Failed to process the '" + instructionsFile.getAbsolutePath() + "' instructions file. Caused by...");
            UnblockingPrintln(getThrowableStack(e));
        }
        finally
        {
            if (writer != null)
            {
                try
                {
                    writer.close();
                }
                catch (Exception ce)
                {
                }
            }
        }


    }

    private void startShutdown(final IShutdown shutdownInterface)
    {
        long tmpID = 0;
        boolean forcedShutdown = false;
        BufferedReader reader = null;
        try
        {
            reader = new BufferedReader(new InputStreamReader(new FileInputStream(m_shutdownFile)));
            String line = reader.readLine();
            tmpID = new Long(line.trim()).longValue();

            try
            {
                forcedShutdown = new Boolean (reader.readLine().trim()).booleanValue();
            }
            catch (Exception e) {} //If there was no second line in the file then forcedShutdown is false
        }
        catch (Exception e)
        {
            //Could happen if the m_shutdownFile file was in the middle of being written
        }
        finally
        {
            if (reader != null)
            {
                try
                {
                    reader.close();
                }
                catch (Exception e){}
            }
        }

        //This could happen if m_shutdownFile was in the middle of being written - so we just ignore it and will read it at the next iteration
        if (tmpID == 0)
        {
            return;
        }

        m_shutdownFile.delete();

        final long callerID = tmpID;

        String forcedMessage = forcedShutdown ? " - the forced flag is on - calling Runtime.getRuntime().halt(0)" : "";
        UnblockingPrintln("Detected file \"" + m_shutdownFile.getAbsolutePath() + "\" with ID " + callerID + forcedMessage);

        if (forcedShutdown)
        {
            forcedShutdown(callerID);
        }


        Thread shutdownThread = new Thread("com.sonicsw.sdf.impl.DiagnosticsManager shutdown Thread")
        {
            @Override
            public void run()
            {
                if (shutdownInterface != null)
                {
                    shutdownInterface.shutdownCallback(callerID);
                }
            }
        };
        shutdownThread.setDaemon(true);
        shutdownThread.start();
    }

    private void forcedShutdown(long callerID)
    {
       long startupTimestamp = 0;
       BufferedReader reader = null;
       try
       {
           reader = new BufferedReader(new InputStreamReader(new FileInputStream(new File(START_TIMESTAMP_FILE))));
           startupTimestamp = new Long(reader.readLine().trim()).longValue();
       }
       catch (Exception e)
       {

       }
       finally
       {
           if (reader != null)
        {
            try
               {
                   reader.close();
               }
               catch (Exception e){}
        }
       }

       if (startupTimestamp != 0 && callerID == startupTimestamp)
    {
        Runtime.getRuntime().halt(0);
    }
    }

    private void logDiagnosticsFromFileProcessingMessage(String dateString, File instructionsFile)
    {
        StringBuffer logEntry = new StringBuffer();
        logEntry.append('[').append(dateString).append("] (info) ");
        logEntry.append("Executing diagnostics instructions from '").append(instructionsFile.getAbsolutePath()).append("'; ");
        logEntry.append("status output is written to '").append(m_diagnosticsSys.getOutFile().getAbsolutePath()).append('\'');
        UnblockingPrintln(logEntry.toString());
    }

    // Process from a String
    //First process all the sonic.diagnostics subsystem instructions and then all the other instructions
    private String processInstructionsString(String instructionsString)
    {
        StringBuffer statusBuffer = new StringBuffer();
        try
        {
            ArrayList<Instruction> instructions = InstructionsFileParser.parse(instructionsString);
            ArrayList<Instruction> sonicDiagInstructions = extractInstructions(instructions, true);
            ArrayList<Instruction> regularInstructions = extractInstructions(instructions, false);

            processInstructions(sonicDiagInstructions, null, statusBuffer);

            String dateString = RolloverLogger.formatTime(System.currentTimeMillis());
            logRemoteDiagnosticsProcessingMessage(dateString, instructions);

            statusBuffer.append("    --- ").append(dateString).append(" Executing remote diagnostics instructions ---").append(NEWLINE);
            warnAboutNoops(sonicDiagInstructions, statusBuffer);
            processInstructions(regularInstructions, null, statusBuffer);
            if (regularInstructions.size() + sonicDiagInstructions.size() == 0)
            {
                statusBuffer.append(NO_INSTRUCTIONS_TEXT + NEWLINE);
            }

        }
        catch (Throwable th)
        {
            statusBuffer.append("Failed to process remote diagnostics instructions. Caused by: ").append(th.toString()).append(NEWLINE);
            StringBuffer logEntry = new StringBuffer(statusBuffer);
            logEntry.append(getThrowableStack(th));
            UnblockingPrintln(logEntry.toString());
        }
        return statusBuffer.toString();
    }

    private void logRemoteDiagnosticsProcessingMessage(String dateString, ArrayList<Instruction> instructions)
    {
        StringBuffer logEntry = new StringBuffer();
        logEntry.append('[').append(dateString).append("] (info) ");
        logEntry.append("Executing remote diagnostics instructions: ");
        if (instructions.isEmpty())
        {
            logEntry.append(NO_INSTRUCTIONS_TEXT);
        }
        else if (instructions.size() == 1)
        {
            logEntry.append(instructions.get(0).getInstructionLine().trim());
        }
        else
        {
            for (Instruction instruction : instructions)
            {
                logEntry.append(NEWLINE).append("  ");
                logEntry.append(instruction.getInstructionLine().trim());
            }
        }
        UnblockingPrintln(logEntry.toString());
    }

    private void warnAboutNoops(ArrayList<Instruction> sonicDiagInstructions, StringBuffer statusBuffer)
    {
        for (Instruction instruction : sonicDiagInstructions)
        {
            if (instruction.getInstructionName().equals(SETUP_OP))
            {
                HashMap params = instruction.getParameters();
                if (params != null)
                {
                    if (params.containsKey(PROCESS_COUNT_PARAM))
                    {
                        statusBuffer.append("*** '").append(PROCESS_COUNT_PARAM).append("' is ignored in remote diagnostics").append(NEWLINE);
                    }
                    if (params.containsKey(STATUS_FILE_DEFAULT_PARAM))
                    {
                        statusBuffer.append("*** '").append(STATUS_FILE_DEFAULT_PARAM).append("' is ignored in remote diagnostics").append(NEWLINE);
                    }
                }
            }
        }
    }

    private ArrayList<Instruction> extractInstructions(ArrayList<Instruction> instructions, boolean sonicDiag)
    {
        ArrayList<Instruction> outList = new ArrayList<Instruction>();
        for (Instruction instruction : instructions)
        {
            if(!instruction.isSyntaxOK())
            {
                if (!sonicDiag)
                {
                    outList.add(instruction);
                }
                continue;
            }

            boolean sonicDiagInstruction = instruction.getSubsystemName().equals(SONIC_DIAGNOSTICS_SUBSYS);
            if (sonicDiag && sonicDiagInstruction || !sonicDiag && !sonicDiagInstruction)
            {
                outList.add(instruction);
            }
        }
        return outList;
    }

    private void processInstructions(ArrayList<Instruction> instructions, PrintWriter writer, StringBuffer buffer) throws Exception
    {
        for (Instruction instruction : instructions)
        {
            if (!instruction.isSyntaxOK())
            {
                if (writer != null)
                {
                    writer.println("*** Invalid syntax: " + instruction.getInstructionLine() + NEWLINE);
                }
                continue;
            }

            HashMap parameters = instruction.getParameters();

            String subsystemID = null;
            String doiID = null;
            String[] tags = null;

            if (parameters != null)
            {
                subsystemID = (String)parameters.remove(SUBSYSTEM_ID_PARAM);
                doiID = (String)parameters.remove(DIAGNOSTICS_OBJECT_ID_PARAM);
                tags = tagListToArray((String)parameters.remove(FILTER_TAGS_PARAM));
            }


            StringBuffer opBuffer = buffer != null ? buffer : new StringBuffer();
            appendExecStatement(instruction.getInstructionName(), doiID, instruction.getSubsystemName(), subsystemID, opBuffer);

            executeOperation (instruction.getInstructionName(),
                              instruction.getSubsystemName(),
                              tags,
                              subsystemID,
                              doiID,
                              opBuffer,
                              parameters);
            opBuffer.append(NEWLINE+NEWLINE);

            if (writer != null)
            {
                writer.print(opBuffer);
                writer.flush();
            }
        }

    }

    private static void appendExecStatement(String op, String doiID, String subsysName, String subsysID, StringBuffer buffer)
    {
        buffer.append(" - Executing ").append(op).append(" in ").append(generateFullName(doiID, subsysName, subsysID)).append(NEWLINE);
    }

    private DiagnosticsException getUniquenessException(String subsysName)
    {
        return new DiagnosticsException("A wild card is specified in the following subsystem '" + subsysName + "'; that requires that no " + DIAGNOSTICS_OBJECT_ID_PARAM + " parameter and no " + SUBSYSTEM_ID_PARAM + " parameter are specified");
    }

    private void UnblockingPrintln(final String s)
    {
        Thread t = new Thread()
        {
            @Override
            public void run()
            {
                System.out.println(s);
            }
        };

        t.setDaemon(true);
        t.start();

        try
        {
            Thread.sleep(1000);
        }
        catch (Exception e){}

    }

    private String getThrowableStack(Throwable t)
    {
        java.io.ByteArrayOutputStream bst = new java.io.ByteArrayOutputStream();
        java.io.PrintWriter pw = new java.io.PrintWriter(bst);
        t.printStackTrace(pw);
        pw.close();
        return bst.toString();
    }

    class DelayedRequest
    {
        private String m_doiID;
        private String m_systemName;
        private String m_subsystemID;
        private String m_operation;
        private HashMap m_parameters;

        DelayedRequest(String doiID, String systemName, String subsystemID, String operation,  HashMap parameters)
        {
            m_doiID = doiID;
            m_systemName = systemName;
            m_subsystemID = subsystemID;
            m_operation = operation;
            m_parameters = parameters;
        }

        void executeDelayedInProvider(IDiagnosticsProvider provider)
        {
            try
            {
                StringBuffer throwAwayBuffer = new StringBuffer();
                executeInProvider(provider, m_doiID, m_operation, m_parameters, throwAwayBuffer);
            }
            catch (Exception e)
            {
                throw new DiagnosticsException("Failed to execute " + m_operation + " in " + m_systemName + ": " + e.toString());
            }
        }

    }

    class DiagnosticsSubsystem extends AbstractDiagnosticsProvider
    {
        private int m_numExecutions;
        private File m_statusFile;

        DiagnosticsSubsystem(String subsystemName)
        {
            super(subsystemName);
            init();

        }

        final void init()
        {
            m_statusFile = statusFileFromInstructionsFile();
            m_numExecutions = 1;
        }

        File getOutFile()
        {
            return m_statusFile;

        }

        int getNumExecutions()
        {
            return m_numExecutions;
        }

        void setContext (IDiagnosticsContext diagContext)
        {
            m_diagnosticsContext = diagContext;
        }

        @Override
        public void executeOperation(String doiID, String operation, HashMap parameters, StringBuffer buffer)
        {
            if (operation.equalsIgnoreCase(SETUP_OP))
            {
                setup(parameters, buffer);
            }
            else if (operation.equalsIgnoreCase(DESCRIBE_ALL_OP))
            {
                describeAll(buffer);
            }
            else if (operation.equalsIgnoreCase(LISTALL_OP))
            {
                listAll(buffer);
            }
            else if (operation.equalsIgnoreCase(HELP_OP))
            {
                printFromFileToBuffer(HELP_TEXT_PATH, buffer);
            }
            else
            {
                buffer.append(operation).append(" not implemented").append(NEWLINE);
            }

        }

        @Override
        public String describe()
        {
             return DESCRIPTION_TEXT;
        }

        @Override
        public String[] getOperations()
        {
            return OPERATIONS;
        }

        @Override
        public HashMap describeParameters(String operationName)
        {
            return (HashMap)PARAM_DESCRIPTOR.get(operationName);
        }

        @Override
        public void appendStateDump(String doiIDNotUsed, HashMap Parameters, StringBuffer buffer)
        {
            listAll(buffer);
            buffer.append(NEWLINE).append(" * Diagnostics subsystem descriptions").append(NEWLINE);
            describeAll(buffer);
        }

        private void setup(HashMap processParameters, StringBuffer buffer)
        {
            String tmp = (String)processParameters.get(STATUS_FILE_DEFAULT_PARAM);
            if (tmp != null)
            {
                m_statusFile = new File(tmp);
            }

            tmp = (String)processParameters.get(PROCESS_COUNT_PARAM);
            if (tmp != null)
            {
                m_numExecutions = new Integer(tmp).intValue();
            }

            buffer.append(STATUS_FILE_DEFAULT_PARAM).append(" is set to ").append(m_statusFile.getAbsolutePath()).append(NEWLINE);
            buffer.append(PROCESS_COUNT_PARAM).append(" is set to ").append(new Integer(m_numExecutions).toString()).append(NEWLINE);
        }

        private File statusFileFromInstructionsFile()
        {
            String statFileDir = new File(m_instructionsFile.getAbsolutePath()).getParent();
            return new File(statFileDir, DIAGNOSTICS_STATUS_FILE_NAME);
        }


    }

    static class TestSubsystem extends AbstractDiagnosticsProvider
    {
         TestSubsystem(String subsystemName)
         {
             super(subsystemName);
         }

         @Override
        public void updateTraceLevel(String doiID, HashMap parameters, StringBuffer buffer)
         {
             System.out.println("updateTraceLevel doiID " + doiID + " param " + parameters);
         }

         @Override
        public String describe()
         {
             return "Tests single instance diagnostics providers";
         }

         @Override
        public String[] getTags()
         {
             return new String[]{"comm"};
         }

         @Override
        public String[] getDOInstances()
         {
             return new String[]{"listener-tcp://2370","listener-tcp://2371","sender-tcp://2370","sender-tcp://2371"};
         }

         @Override
        public void test(HashMap parameters, StringBuffer buffer)
         {
             buffer.append(m_subsystemName).append(" tested").append(parameters.toString());
         }
    }

    static class TestMultiInstSubsystem extends AbstractDiagnosticsProvider
    {
         TestMultiInstSubsystem(String subsystemName, String id)
         {
             super(subsystemName);
             m_subsystemID = id;
         }

         @Override
        public String describe()
         {
             return  "Tests single instance diagnostics providers";
         }

         @Override
        public String[] getTags()
         {
             return new String[]{"ssl","comm"};
         }

         @Override
        public void test(HashMap parameters, StringBuffer buffer)
         {
             buffer.append(m_subsystemName).append (" tested").append(parameters.toString());
         }
    }

    static class TestMFComponent extends AbstractMFComponentTracing
    {
         Integer m_traceLevel;

         TestMFComponent(String diagSubsysName, String maskNameToIntMap)
         {
             super(diagSubsysName, maskNameToIntMap);
             m_traceLevel = getMFComponentCurrentMask();
         }

         private Integer getMFComponentCurrentMask() {
             return getCurrentMask();
         }
         
         @Override
        public void updateTraceLevel(String doiIDNotUsed, HashMap parameters, StringBuffer buffer)
         {
             super.updateTraceLevel(doiIDNotUsed, parameters, buffer);
             m_traceLevel = getCurrentMask();
             System.out.println("The current mask is " + m_traceLevel);
         }
    }
    static class TestShutdown implements  IShutdown
    {
        @Override
        public void shutdownCallback(long callerID)
        {
            System.out.println("Shudown called with ID " + callerID);
        }

    }

}
