package com.sonicsw.sdf.impl;


import static com.sonicsw.sdf.AbstractDiagnosticsProvider.OP_PARAM_FILE_ACCESS;
import static com.sonicsw.sdf.AbstractDiagnosticsProvider.OP_PARAM_OUTPUT_LOCATION;
import static com.sonicsw.sdf.AbstractDiagnosticsProvider.OP_PARAM_OVERWRITE;

import java.lang.reflect.Method;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Map;

import com.sonicsw.sdf.IDiagnosticsContext;
import com.sonicsw.sdf.WriterUtil;


//Generates heap dump for HotSpot JVMs
public class HotSpotHeapDump implements com.sonicsw.sdf.IDiagnosticsConstants
{
    private static final String HOTSPOT_HEAP_FILE_POSTFIX = ".hprof";
    
    // This is the name of the HotSpot Diagnostic MBean
    private static final String HOTSPOT_BEAN_NAME =
         "com.sun.management:type=HotSpotDiagnostic";

    private static boolean m_heapDumpSupported = true;

    private static final Class[] EMPTY_CLASS_ARRAY = new Class[0];
    private static final Object[] EMPTY_OBJECT_ARRAY = new Object[0];

    private static Class HotSpotDiagnosticMXBean_CLASS;
    private static Class ManagementFactory_CLASS;
    private static Class MBeanServer_CLASS;
    private static Class MBeanServerConnection_CLASS;

    static
    {
        try
        {
            HotSpotDiagnosticMXBean_CLASS = Class.forName("com.sun.management.HotSpotDiagnosticMXBean");
            ManagementFactory_CLASS = Class.forName("java.lang.management.ManagementFactory");
            MBeanServer_CLASS = Class.forName("javax.management.MBeanServer");
            MBeanServerConnection_CLASS = Class.forName("javax.management.MBeanServerConnection");

        }
        catch (Throwable t)
        {
            m_heapDumpSupported = false;
        }

    }

    static Class[] NEW_PLATFORM_PROXY_SIGNATURE = new Class[]
    {
        MBeanServerConnection_CLASS,
        String.class,
        Class.class
    };
    private static Method newPlatformMXBeanProxy_METHOD;

    private static Method getPlatformMBeanServer_METHOD;

    static Class[] DUMP_HEAP_SIGNATURE = new Class[]
    {
        String.class,
        boolean.class
    };
    private static Method dumpHeap_METHOD;

    static
    {
        try
        {

            newPlatformMXBeanProxy_METHOD = ManagementFactory_CLASS.getMethod("newPlatformMXBeanProxy", NEW_PLATFORM_PROXY_SIGNATURE);
            getPlatformMBeanServer_METHOD = ManagementFactory_CLASS.getMethod("getPlatformMBeanServer", EMPTY_CLASS_ARRAY);
            dumpHeap_METHOD = HotSpotDiagnosticMXBean_CLASS.getMethod("dumpHeap", DUMP_HEAP_SIGNATURE);

        }
        catch (Throwable t)
        {
            m_heapDumpSupported = false;
        }

    }

    private static void doDumpHeap(String fileName) throws Exception
    {
        Object server = getPlatformMBeanServer_METHOD.invoke(null, EMPTY_OBJECT_ARRAY);
        Object hotspotMBean = newPlatformMXBeanProxy_METHOD.invoke(null, new Object[]{server,HOTSPOT_BEAN_NAME,HotSpotDiagnosticMXBean_CLASS});
        dumpHeap_METHOD.invoke(hotspotMBean, new Object[]{fileName,Boolean.TRUE});
    }

    static String describe()
    {
        StringBuffer tmp = new StringBuffer();
        tmp.append("This HotSpot JVM generates binary JVM heap dumps; 'jhat' or other third-party utilities can be used to browse the dump file");
        return tmp.toString();
    }

    static boolean heapDumpSupported()
    {
        return m_heapDumpSupported;
    }

    static void dumpHeap(StringBuffer buffer, IDiagnosticsContext context)
    {
        try
        {
            String dumpFileName = context.generateFileNameBase(true) + HOTSPOT_HEAP_FILE_POSTFIX;
            doDumpHeap(dumpFileName);

            buffer.append("Binary heap dump was written to file ").append(dumpFileName);
        }
        catch(Throwable t)
        {
            buffer.append(JVM_HEAP_SUBSYS).append(" failed to generate a heap dump for a HostSpot JVM:");
            buffer.append(NEWLINE).append(WriterUtil.getExceptionAsString(t));
        }

    }

    /**
     * This version do some tweaks to the dump file generation depending
     * on the given parameters passed as a java.util.Map, you can choose
     * to overwrite the output file or to dump it in a different location
     * than the default one.
     *
     * @param buffer
     * @param context
     * @param parameters
     */
    static void dumpHeap(StringBuffer buffer, IDiagnosticsContext context, Map<String, String> parameters)
    {
        try
        {
            boolean overwrite = Boolean.parseBoolean(parameters.get(OP_PARAM_OVERWRITE));
            String outputFolder = parameters.get(OP_PARAM_OUTPUT_LOCATION);
            String permissions = parameters.get(OP_PARAM_FILE_ACCESS);

            String dumpFileName = context.generateFileNameBase(!overwrite) + HOTSPOT_HEAP_FILE_POSTFIX;


            Path filePath = Paths.get(dumpFileName);
            if(outputFolder != null)
            {
                filePath = Paths.get(outputFolder).resolve(filePath);
                WriterUtil.ensureParentFoldersExists(filePath);
            }

            if(overwrite)
            {
                Files.deleteIfExists(filePath);
            }

            doDumpHeap(filePath.toString());

            //Set the permissions if were informed
            if(permissions != null){
                WriterUtil.setFileAccessPermissions(filePath, permissions);
            }

            buffer.append("Binary heap dump was written to file ").append(filePath.toString());
        }
        catch(IllegalArgumentException e)
        {
            buffer.append(JVM_HEAP_SUBSYS).append(" failed to generate a heap dump for a HostSpot JVM:");
            buffer.append(NEWLINE).append(e.getMessage());
        }
        catch(Throwable t)
        {
            buffer.append(JVM_HEAP_SUBSYS).append(" failed to generate a heap dump for a HostSpot JVM:");
            buffer.append(NEWLINE).append(WriterUtil.getExceptionAsString(t));
        }
    }


}


