package com.sonicsw.sdf.impl;

import java.nio.file.Paths;
import java.util.HashMap;

import com.sonicsw.sdf.AbstractDiagnosticsProvider;
import com.sonicsw.sdf.IDiagnosticsContext;
import com.sonicsw.sdf.IStateWriter;
import com.sonicsw.sdf.WriterUtil;
import com.sonicsw.sdf.threads.ThreadDumpHandler;

/**
 * Uses JVM 1.5/1.6 java.lang.management facilities to dump the state
 * of the JVM threads into a file
 */
public class JVMThreadDiagnostics extends AbstractDiagnosticsProvider
{
    private static String[] OPERATIONS;
    private static HashMap DUMP_STATE_PARAM_DESCIPTOR = new HashMap();
    private static HashMap DESCRIBE_PARAM_DESCIPTOR = new HashMap();
    private static HashMap PARAM_DESCRIPTOR = new HashMap();
    static
    {

        DUMP_STATE_PARAM_DESCIPTOR.put(OP_PARAM_OVERWRITE, OP_PARAM_OVERWRITE_DESCRIPTION + " Default to true");
        DUMP_STATE_PARAM_DESCIPTOR.put(OP_PARAM_OUTPUT_LOCATION, OP_PARAM_OUTPUT_LOCATION_DESCRIPTION);
        DUMP_STATE_PARAM_DESCIPTOR.put(OP_PARAM_FILE_ACCESS, OP_PARAM_FILE_ACCESS_DESCRIPTION + " Default to 'rw-r--r--'");

        PARAM_DESCRIPTOR.put(com.sonicsw.sdf.AbstractDiagnosticsProvider.DUMP_STATE_OP, DUMP_STATE_PARAM_DESCIPTOR);
        PARAM_DESCRIPTOR.put(com.sonicsw.sdf.AbstractDiagnosticsProvider.DESCRIBE_OP, DESCRIBE_PARAM_DESCIPTOR);

        OPERATIONS = com.sonicsw.sdf.AbstractDiagnosticsProvider.toOpnameArray(PARAM_DESCRIPTOR);
    }


    JVMThreadDiagnostics()
    {
        super(JVM_THREADS_SUBSYS);
    }

    void setContext (IDiagnosticsContext diagContext)
    {
        m_diagnosticsContext = diagContext;
    }

    @Override
    public String describe()
    {
        String desc = "Generates a JVM thread dump.";
        if (!ThreadDumpHandler.isLockInfoSupported())
        {
            desc += "  Use a 1.6 JVM to get full lock information in the thread dump.";
        }
        return desc;
    }

    @Override
    public void appendStateDump(String doiIDNotUsed, HashMap parameters, StringBuffer buffer)
    {
        IStateWriter writer = null;

        try
        {
            if(parameters.isEmpty())
            {
                writer = m_diagnosticsContext.getStateWriter();
            }
            else
            {
                writer = m_diagnosticsContext.getStateWriter(parameters);
            }
        }
        catch (Exception e)
        {
            buffer.append(JVM_THREADS_SUBSYS).append(" failed to use state file:");
            buffer.append(NEWLINE).append(WriterUtil.getExceptionAsString(e));
            if (writer != null)
            {
                writer.close();
            }
            return;
        }


        try
        {
            String instruction = WriterUtil.rebuildInstruction(this, DUMP_STATE_OP, parameters);
            WriterUtil.writeHeader(writer, instruction, null);
            
            ThreadDumpHandler.writeThreadDump(writer);
            
            WriterUtil.writeFooter(writer, null);

            //Set file permissions if informed
            String filePath = writer.getFilePath();
            if(parameters.containsKey(OP_PARAM_FILE_ACCESS))
            {
                String permissions = parameters.get(OP_PARAM_FILE_ACCESS).toString();
                WriterUtil.setFileAccessPermissions(Paths.get(filePath), permissions);
            }
            
            buffer.append("The thread dump is written to file ").append(filePath);
        }
        catch(IllegalArgumentException e)
        {
            buffer.append(JVM_HEAP_SUBSYS).append(" failed to generate a thread dump:");
            buffer.append(NEWLINE).append(e.getMessage());
        }
        catch(Throwable t)
        {
            buffer.append(JVM_THREADS_SUBSYS).append(" failed to generate a thread dump:");
            buffer.append(NEWLINE).append(WriterUtil.getExceptionAsString(t));
        }

        finally
        {
            if (writer != null)
            {
                writer.close();
            }
        }
    }

    @Override
    public HashMap describeParameters(String operationName)
    {
        return (HashMap)PARAM_DESCRIPTOR.get(operationName);
    }

    //NOOP
    @Override
    public void updateTraceLevel(String doiID, HashMap parameters, StringBuffer buffer)
    {
    }

    @Override
    public String[] getOperations()
    {
        return OPERATIONS;
    }

}
