package com.sonicsw.sdf.impl;

import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.text.SimpleDateFormat;
import java.util.Date;

import com.sonicsw.sdf.IDiagnosticsConstants;
import com.sonicsw.sdf.IStateWriter;
import com.sonicsw.sdf.ITracer;

//Implements a length bound log using roll-over files
final public class RolloverLogger implements IDiagnosticsConstants, ITracer, IStateWriter
{
    private static final ThreadLocal<SimpleDateFormat> SIMPLE_DATE_FORMAT = new ThreadLocal<SimpleDateFormat>() {
        @Override
        protected SimpleDateFormat initialValue() {
            return new SimpleDateFormat("yy/MM/dd HH:mm:ss");
        }
    };
    private static Class[] EMPTY_CLASS_ARRAY = new Class[0];
    private static Object[] EMPTY_OBJECT_ARRAY = new Object[0];

    private static final String OVRFLW_STRING = "_ovrflw";
    private static final boolean ON_WINDOWS = File.separatorChar == '\\';
    private static final int EOLN_LENGTH = ON_WINDOWS ? 2 : 1;

    private PrintWriter m_currentWriter;

    private String m_logPath;
    private String m_ovrflwLogPath;
    private File m_logFile;
    private File m_ovrflwLogFile;
    private File m_crntLogFile;
    private boolean m_crntOvrflw;
    private long m_totalLength;
    private long m_maxLength;

    //Unit test
    public static void main(String[] args) throws Exception
    {
        RolloverLogger logger = new RolloverLogger("c:/sonic/test/trace.txt", (long)10000);
        for (int i = 0; i < 7000; i++)
        {
            System.out.println("print " + i);
            Exception e1 = new Exception("Cause " + i);
            Exception e = new Exception(e1);
            logger.trace("xxxxxxxxx " + i, true);
        }
    }


    RolloverLogger(String logPath, long maxLength) throws IOException
    {
        m_maxLength = maxLength;
        m_logPath = logPath;
        m_ovrflwLogPath = getOvrflwName(logPath);
        m_logFile = new File(m_logPath);
        m_ovrflwLogFile = new File(m_ovrflwLogPath);

        if (!m_logFile.exists() && m_ovrflwLogFile.exists())
        {
            m_ovrflwLogFile.renameTo(m_logFile);
        }

        if (m_logFile.exists())
        {
            m_totalLength = m_logFile.length();
        }

        if (m_ovrflwLogFile.exists())
        {
            m_totalLength += m_ovrflwLogFile.length();
        }

        m_crntLogFile = m_ovrflwLogFile.exists() ? m_ovrflwLogFile : m_logFile;
        m_crntOvrflw = m_crntLogFile == m_ovrflwLogFile;
        openCurrent();


    }

    @Override
    public String getFilePath()
    {
        return m_logFile.getAbsolutePath();
    }

    @Override
    public void write(String message) throws IOException
    {
        logMessageInternal(message, false);
    }

    @Override
    public void writeln(String message) throws IOException
    {
        logMessageInternal(message, true);
    }
    
    @Override
    public void writeln() throws IOException
    {
        logMessageInternal("", true);
    }
    
    @Override
    public void trace(String message, Throwable exception, boolean logTimestamp) throws IOException
    {
        if (exception != null)
        {
            message = appendThrowableToMessage(message, exception);
        }
        logMessageInternal(createLogMessage(message, logTimestamp), true);

    }

    @Override
    public void trace(String message, boolean logTimestamp) throws IOException
    {
        logMessageInternal(createLogMessage(message, logTimestamp), true);
    }


    @Override
    public void close()
    {
        m_currentWriter.close();
    }

    static String formatTime(long time)
    {
        return SIMPLE_DATE_FORMAT.get().format(new Date(time));
    }


    private void logMessageInternal(String message, boolean newline) throws IOException
    {
        if (m_crntOvrflw)
        {
            if (m_maxLength != NO_LIMIT_TRACE_FILE_SIZE && m_totalLength + estimateBytes(message.length()) > m_maxLength)
            {
                rollover();
            }
        }
        else if (m_maxLength != NO_LIMIT_TRACE_FILE_SIZE && m_totalLength + estimateBytes(message.length()) > m_maxLength / 2)
        {
            startOverflow();
        }

        if (newline)
        {
            m_currentWriter.println(message);
        }
        else
        {
            m_currentWriter.print(message);
        }
        m_totalLength += estimateBytes(message.length());

    }

    private static int estimateBytes(int msgLength)
    {
        return msgLength + EOLN_LENGTH;
    }

    private void rollover() throws IOException
    {
        m_currentWriter.close();

        m_logFile.delete();
        if (!m_ovrflwLogFile.renameTo(m_logFile))
        {
            throw new IOException("Failed to rename '" + m_ovrflwLogPath + "' to '" + m_logPath + "'");
        }
        m_totalLength =  m_logFile.length();
        m_crntLogFile = m_ovrflwLogFile;
        openCurrent();
    }

    private void startOverflow() throws IOException
    {
        m_currentWriter.close();
        m_crntOvrflw = true;
        m_totalLength =  m_logFile.length();
        m_crntLogFile = m_ovrflwLogFile;
        openCurrent();
    }

    private void openCurrent() throws IOException
    {
        m_currentWriter = new PrintWriter(new BufferedWriter(new FileWriter(m_crntLogFile, true)), true);
    }


    private static String getOvrflwName(String logName)
    {
        int postfixIndex = logName.lastIndexOf(".");
        if (postfixIndex == -1)
        {
            return logName + OVRFLW_STRING;
        }

        String name = logName.substring(0, postfixIndex);
        return name + OVRFLW_STRING + logName.substring(postfixIndex);
    }

    private String createLogMessage(String message, boolean logTimestamp)
    {
        StringBuffer timestampedMessage = new StringBuffer();

        if (logTimestamp)
        {
            timestampedMessage.append('[');
            synchronized(SIMPLE_DATE_FORMAT)
            {
                timestampedMessage.append(formatTime(System.currentTimeMillis()));
            }
            timestampedMessage.append("] ");
        }

        timestampedMessage.append(message);

        return timestampedMessage.toString();

    }


    private String appendThrowableToMessage(String message, Throwable exception)
    {
        StringBuffer buffer = new StringBuffer(message);
        buffer.append(NEWLINE);
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        PrintWriter writer = new PrintWriter(out, true);
        printStackTrace(exception, writer);
        byte[] data = out.toByteArray();
        buffer.append(new String(data));
        return buffer.toString();
    }

    private String printStackTrace(Throwable exception, String message)
    {
        StringBuffer buffer = new StringBuffer(message);
        buffer.append(NEWLINE);
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        PrintWriter writer = new PrintWriter(out, true);
        printStackTrace(exception, writer);
        byte[] data = out.toByteArray();

        try
        {
            out.close();
        }
        catch (IOException io) { }

        buffer.append(new String(data));
        return buffer.toString();
    }

    private void printStackTrace(Throwable exception, PrintWriter writer)
    {
        exception.printStackTrace(writer);

        Class exceptionClass = exception.getClass();

        try
        {
            Method getTargetExceptionMethod = exceptionClass.getMethod("getTargetException", EMPTY_CLASS_ARRAY);
            Throwable targetException = (Throwable)getTargetExceptionMethod.invoke(exception, EMPTY_OBJECT_ARRAY);
            if (targetException != null)
            {
                writer.println("Caused by...");
                printStackTrace(targetException, writer);
                return;
            }
        }
        catch (NoSuchMethodException nsme)
        {} // ignore
        catch (IllegalAccessException iae)
        {} // ignore
        catch (InvocationTargetException ite)
        {} // ignore

        try
        {
            Method getTargetErrorMethod = exceptionClass.getMethod("getTargetError", EMPTY_CLASS_ARRAY);
            Throwable targetError = (Throwable)getTargetErrorMethod.invoke(exception, EMPTY_OBJECT_ARRAY);
            if (targetError != null)
            {
                writer.println("Caused by...");
                printStackTrace(targetError, writer);
                return;
            }
        }
        catch (NoSuchMethodException nsme)
        {} // ignore
        catch (IllegalAccessException iae)
        {} // ignore
        catch (InvocationTargetException ite)
        {} // ignore

        try
        {
            Method getLinkedExceptionMethod = exceptionClass.getMethod("getLinkedException", EMPTY_CLASS_ARRAY);
            Throwable linkedException = (Throwable)getLinkedExceptionMethod.invoke(exception, EMPTY_OBJECT_ARRAY);
            if (linkedException != null)
            {
                writer.println("Caused by...");
                printStackTrace(linkedException, writer);
                return;
            }
        }
        catch (NoSuchMethodException nsme)
        {} // ignore
        catch (IllegalAccessException iae)
        {} // ignore
        catch (InvocationTargetException ite)
        {} // ignore


        // the classes below print out the actual exception so only do for other exceptions
        if (exceptionClass.getName().indexOf("MFProxyRuntimeException") == -1 && exceptionClass.getName().indexOf("MFProxyException") == -1)
        {
            try
            {
                Method getActualExceptionMethod = exceptionClass.getMethod("getActualException", EMPTY_CLASS_ARRAY);
                Throwable cause = (Throwable)getActualExceptionMethod.invoke(exception, EMPTY_OBJECT_ARRAY);
                if (cause != null)
                {
                    writer.println("Caused by...");
                    printStackTrace(cause, writer);
                    return;
                }
            }
            catch (NoSuchMethodException nsme)
            {} // ignore
            catch (IllegalAccessException iae)
            {} // ignore
            catch (InvocationTargetException ite)
            {} // ignore
        }

    }



}
