//Copyright (c) 2010 Progress Software Corporation.  All Rights Reserved.

/**
 * Used to collect per-thread CPU usage information associated
 * with a thread dump.
 */

package com.sonicsw.sdf.threads;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

public class ThreadCPUStats
{
    private Map<Long, StatEntry> m_stats;
    
    /**
     * Constructor - creates new ThreadCPUStats instance.  The statistics
     * (ThreadCPUStats.StatEntry) are stored in a HashMap, created with the
     * given initialCapacity.
     */
    public ThreadCPUStats(int initialCapacity)
    {
        m_stats = new HashMap<Long, StatEntry>(initialCapacity);
    }
    
    /**
     * Adds a new CPU usage stat for the given thread.  Total usage should be
     * a value in nanoseconds acquired from ThreadMXBean.getThreadCpuTime().
     * The time the stat is created is automatically recorded.
     */
    public void addStat(long threadId, long totalUsage)
    {
        m_stats.put(threadId, new StatEntry(threadId, totalUsage));
    }
    
    /**
     * Returns the StatEntry for the given thread.  Returns null if a stat
     * has not been created for this thread.
     */
    public StatEntry getStat(long threadId)
    {
        return m_stats.get(threadId);
    }
    
    /**
     * Returns a collection view of all the stats.
     */
    public Collection<StatEntry> getAllStats()
    {
        return m_stats.values();
    }
    
    
    public class StatEntry implements Comparable<StatEntry>
    {
        private long m_threadId;
        private long m_time;  // milliseconds
        private long m_totalUsage;  // nanoseconds
        private long m_usageLastPeriod = -1;  // nanoseconds
        private long m_percentLastPeriod = -1000000;  // *1000000
            
        /**
         * Creates a CPU stats entry given the total usage by the thread
         * to date.  A timestamp is automatically added to the stat. 
         */
        StatEntry(long threadId, long totalUsage)
        {
            m_time = System.currentTimeMillis();
            m_threadId = threadId;
            m_totalUsage = totalUsage;
        }
            
        /**
         * Computes usage (in clock ticks and percentage) during period between
         * the current stat and the given 'old' stat.
         */
        public void computeUsageInLastPeriod(StatEntry oldStat)
        {
            m_usageLastPeriod = m_totalUsage - oldStat.m_totalUsage;
            
            long elapsedTime = m_time - oldStat.m_time;
            m_percentLastPeriod = (elapsedTime <= 0) ? 0 : m_usageLastPeriod / elapsedTime;
            
            // Avoid -ve % value - possible if Java thread id gets reused by a new thread?
            m_percentLastPeriod = (m_percentLastPeriod < 0) ? 0 : m_percentLastPeriod;
        }
            
        /**
         * The id of the thread associated with this statistic
         */
        public long getThreadId()
        {
            return m_threadId;
        }
        
        /**
         * Returns the time at which this stat was recorded (milliseconds)
         */
        public long getTime()
        {
            return m_time;
        }

        /**
         * Returns the total CPU usage clocked up by the thread at the
         * associated time (nanoseconds)
         */
        public long getTotalUsage()
        {
            return m_totalUsage;
        }
        
        /**
         * Returns the CPU usage clocked up by the thread during the
         * in last collection interval (nanoseconds), as computed by
         * computeUsageInLastPeriod().  Returns negative value if 
         * computeUsageInLastPeriod() has not been called.
         */
        public long getUsageInLastPeriod()
        {
            return m_usageLastPeriod;
        }
        
        /**
         * Returns a percentage computed by dividing the CPU usage
         * clocked up during the last collection interval by the time
         * elapsed over that interval.  Returns a negative value if 
         * computeUsageInLastPeriod() has not been called.
         * 
         * Note that if a thread has used all the cycles available on
         * a given CPU this figure could (theoretically at least) be
         * 100%.  If there are multiple CPU's and all are fully occupied
         * then the sum of the percentages for the threads could be
         * 100 * number of CPU's.
         */
        public float getPercentInLastPeriod()
        {
            // divide by 1000000 (calculated from nanosecs), multiply by 100 (for percent)
            return (float)m_percentLastPeriod / 10000.0f;  
        }

        /**
         * Allows for sorting the threads/stats based on percent CPU usage
         */
        @Override
        public int compareTo(StatEntry compareStat) {
            if (m_percentLastPeriod == compareStat.m_percentLastPeriod)
            {
                return 0;
            }
            else if (m_percentLastPeriod > compareStat.m_percentLastPeriod)
            {
                return 1;
            }
            else
            {
                return -1;
            }
        }
        
    }
    
}
