//Copyright (c) 2010 Progress Software Corporation.  All Rights Reserved.

/**
 * Class that representing a Java thread dump - creates and writes the
 * thread dump.
 * 
 * The thread dump combines information gathered from the java.lang.management
 * classes (ThreadMXBean, ThreadInfo, and, in the case of a 1.6 JVM, LockInfo),
 * and java.lang.Thread.
 * 
 * The thread dump also includes CPU usage information where available
 * (see ThreadMXBean.isThreadCpuTimeEnabled()).  The per-thread CPU usage
 * from one thread dump is cached/saved so it's available the next time a
 * thread dump is taken.  The latter thread dump will then include
 * CPU usage info' for the intervening period.
 */

package com.sonicsw.sdf.threads;

import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.management.ThreadInfo;
import java.lang.management.ThreadMXBean;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.text.DecimalFormat;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import com.sonicsw.sdf.IStateWriter;

public class ThreadDump {
    
    // Display additional diagnostic info' - e.g. details of 'expected'
    // exceptions normally suppressed.
    private static final boolean DEBUG = Boolean.getBoolean("Sonic.sdf.ThreadDumpDebug");
    
    // Java 1.6 classes/methods that need to be accessed via reflection 
    static Class<?> LockInfo_CLASS = null;
    static Class<?> MonitorInfo_CLASS = null;

    static Method dumpAllThreads_METHOD = null;  // ThreadMXBean
    static Method findDeadlockedThreads_METHOD = null;  // ThreadMXBean

    static Method getLockInfo_METHOD = null;  // ThreadInfo
    static Method getLockedMonitors_METHOD = null;  // ThreadInfo
    static Method getLockedSynchronizers_METHOD = null;  // ThreadInfo
    
    static Method getLockClassName_METHOD = null;  // LockInfo
    static Method getLockIdentityHashCode_METHOD = null;  // LockInfo
    static Method getLockedStackDepth_METHOD = null;  // MonitorInfo

    // Indicates whether lock info' is available (i.e. whether this is a 1.6 JVM)
    private static boolean s_lockInfoSupported = true;
    private static final LockDetail[] EMPTY_LOCK_ARRAY = new LockDetail[0];
    
    // Root thread group used to get a collection of all the java.lang.Thread objects
    private static ThreadGroup s_rootThreadGroup = null;

    // Get ThreadMXBean for local JVM
    private static final ThreadMXBean s_threadMXBean = ManagementFactory.getThreadMXBean();

    // Formatting
    private static final String INDENT = "    ";
    private static final DecimalFormat s_cpuPercentFormat = new DecimalFormat("###0.00");
    
    // Minimum CPU usage % for a thread to show up in the top consumers 
    private static final Float CPU_USAGE_DISPLAY_THRESHOLD = 0.01f;
    
    
    static
    {
        // Recurse back through parents of current thread group to find root
        s_rootThreadGroup = Thread.currentThread().getThreadGroup();
        ThreadGroup parent = s_rootThreadGroup.getParent();
        while (parent != null) {
            s_rootThreadGroup = parent;
            parent = s_rootThreadGroup.getParent();
        }       
        
        // Get Java 1.6 classes/methods (if available)
        try
        {
            LockInfo_CLASS = Class.forName("java.lang.management.LockInfo");
            MonitorInfo_CLASS = Class.forName("java.lang.management.MonitorInfo");

            dumpAllThreads_METHOD = ThreadMXBean.class.getMethod("dumpAllThreads", boolean.class, boolean.class);
            findDeadlockedThreads_METHOD = ThreadMXBean.class.getMethod("findDeadlockedThreads");

            getLockInfo_METHOD = ThreadInfo.class.getMethod("getLockInfo");
            getLockedMonitors_METHOD = ThreadInfo.class.getMethod("getLockedMonitors");
            getLockedSynchronizers_METHOD = ThreadInfo.class.getMethod("getLockedSynchronizers");
            
            getLockClassName_METHOD = LockInfo_CLASS.getMethod("getClassName");
            getLockIdentityHashCode_METHOD = LockInfo_CLASS.getMethod("getIdentityHashCode");
            getLockedStackDepth_METHOD = MonitorInfo_CLASS.getMethod("getLockedStackDepth");
        }
        catch (Throwable t)
        {
            // If (any of) the above Java 1.6 look-ups fail we're
            // running on a 1.5 JVM so lock info' won't be available
            s_lockInfoSupported = false;
            
            if (DEBUG)
            {
                System.err.println("SDF thread dump: Lock info support disabled due to:");
                t.printStackTrace();
            }
        }
    }   

    private static long s_savedTime = -1;  // time of previous thread dump (for which stats may have been saved)
    private static ThreadCPUStats s_savedCpuStats = null;  // stats saved from previous thread dump
    static final long SAVED_CPU_STATS_TIMEOUT_MINS = 15;  // time-out saved CPU stats if older than 15 mins 
    
    // Member data
    private long m_startTime;  // start of data collection period
    private long m_endTime;    // end of data collection period
    private long m_prevTime;   // time the last thread dump was taken
    private Map<Long, ThreadInfo> m_mgmtThreads;
    private Map<Long, Thread> m_runtimeThreads;
    private long[] m_deadlockedThreads;
    private ThreadCPUStats m_cpuStats = null;
    private boolean m_cpuUsageAvailable = false;  // true if stats for this *and* previous thread dump available (allowing calculation of usage over last period)


    /**
     * Constructor - takes snapshot of current state of threads.
     */
    private ThreadDump() {
        m_startTime = System.currentTimeMillis();
        
        // get the raw thread info' up-front to minimize discrepancies
        // due to threads changing/disappearing
        ThreadInfo[] mgmtThreadArray = getThreadsFromMgmt();
        Thread[] runtimeThreadArray = getThreadsFromRuntime();
        m_deadlockedThreads = getDeadlockedThreads();

        m_endTime = System.currentTimeMillis();
        
        // collect CPU usage if available
        if (s_threadMXBean.isThreadCpuTimeSupported() && s_threadMXBean.isThreadCpuTimeEnabled())
        {
            m_cpuStats = new ThreadCPUStats(mgmtThreadArray.length);
            for (ThreadInfo ti : mgmtThreadArray)
            {
                if (ti != null)
                {
                    long tid = ti.getThreadId();
                    long cpuTime = s_threadMXBean.getThreadCpuTime(tid);
                    if (cpuTime >= 0)
                    {
                        m_cpuStats.addStat(tid, cpuTime);
                    }
                }
            }
        }
        
        // build map of id -> ThreadInfo
        // Use linked map to preserve original iteration order, in case there's any significance there
        m_mgmtThreads = new LinkedHashMap<Long, ThreadInfo>(mgmtThreadArray.length);
        for (ThreadInfo ti : mgmtThreadArray) {
            if (ti != null)
            {
                m_mgmtThreads.put(ti.getThreadId(), ti);
            }
        }       
        
        // build map of id -> Thread
        m_runtimeThreads = new HashMap<Long, Thread>(runtimeThreadArray.length);
        for (Thread t : runtimeThreadArray) {
            if (t != null)
            {
                m_runtimeThreads.put(t.getId(), t);
            }
        }

    }
    
    /**
     * takes snapshot of current state of threads.
     * @return ThreadDump
     */
    public static synchronized ThreadDump createThreadDump() {
        
        ThreadDump dump = new ThreadDump();
        
        // clear saved CPU usage stats if they're too old
        if (dump.m_startTime - s_savedTime > (SAVED_CPU_STATS_TIMEOUT_MINS * 60 * 1000))
        {
            s_savedCpuStats = null;
        }
        // compute CPU usage figures for interval since last thread dump
        // if required info' available
        if (dump.m_cpuStats != null && s_savedCpuStats != null)
        {
            for (long tid : dump.m_mgmtThreads.keySet())
            {
                ThreadCPUStats.StatEntry oldStat = s_savedCpuStats.getStat(tid);
                ThreadCPUStats.StatEntry newStat = dump.m_cpuStats.getStat(tid);
                if (oldStat != null && newStat != null)
                {
                    newStat.computeUsageInLastPeriod(oldStat);
                }
            }
            dump.m_cpuUsageAvailable = true;
        }
        // update saved time/cpu info' so it's available the
        // next time a thread dump is taken
        dump.m_prevTime = s_savedTime;
        s_savedTime = dump.m_startTime;
        s_savedCpuStats = dump.m_cpuStats;
        
        return dump;
    }
    
    /**
     * Returns the time at which the thread dump was taken (i.e.
     * when we started collecting the raw thread dump data)
     */
    public long getStartTime()
    {
        return m_startTime;
    }
    
    /**
     * Returns the time in milliseconds taken to collect the raw data
     * for the thread dump.  The time taken may give an indication of
     * the completeness/reliability of the dump.  
     */
    public long getElapsedTime()
    {
        return m_endTime - m_startTime;
    }
    
    /**
     * Returns the time the previous thread dump was taken.  IF CPU
     * stats are available then they're based on the interval between
     * the previous thread dump and this one.
     */
    public long getPreviousThreadDumpTime()
    {
        return m_prevTime;
    }
    
    /**
     * Returns the number of threads in the thread dump
     */
    public int getThreadCount()
    {
        return m_mgmtThreads.size();
    }
    
    /**
     * Returns the number of threads involved in deadlocks
     */
    public int getDeadlockedThreadCount()
    {
        return (m_deadlockedThreads == null) ? 0 : m_deadlockedThreads.length;
    }
    
    /**
     * Writes thread, stack, and lock info (if available) for all threads 
     * @param writer 
     * @throws IOException if an error occurs writing the output stream
     */
    public void showThreads(IStateWriter writer)
    throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, IOException
    {
        for (ThreadInfo ti : m_mgmtThreads.values())
        {
            printThreadInfo(writer, ti);
            writer.writeln();
        }
    }
    
    /**
     * Writes details of any deadlocks 
     * @throws IOException if an error occurs writing the output stream
     */
    public void showDeadlockedThreads(IStateWriter writer)
    throws IOException
    {
        if (m_deadlockedThreads == null)
        {
            return;
        }
        
        for (long tid : m_deadlockedThreads)
        {
            ThreadInfo ti = m_mgmtThreads.get(tid);
            if (ti != null)
            {
                writer.writeln("\"" + ti.getThreadName() + "\" (tid=0x" + Long.toHexString(tid) + ")");
                writer.writeln(INDENT + "waiting for " + ti.getLockName());
                writer.writeln(INDENT + "which is held by \"" + ti.getLockOwnerName() + "\" (tid=0x" + Long.toHexString(ti.getLockOwnerId()) + ")");
            }
            else
            {
                writer.writeln("Thread tid=0x" + Long.toHexString(tid) + " - no details available");                
            }
        }
    }
    
    /**
     * Writes a table of threads that have consumed CPU since the
     * previous thread dump, sorted by CPU usage.  maxEntries is the
     * maximum number of threads that are to be shown.
     * @throws IOException if an error occurs writing the output stream
     */
    public void showCpuUsage(IStateWriter writer, int maxEntries)
    throws IOException
    {
        if (!m_cpuUsageAvailable  || m_cpuStats == null)
        {
            writer.writeln("<CPU usage not available>");
            return;
        }
        
        List<ThreadCPUStats.StatEntry> stats = new LinkedList<ThreadCPUStats.StatEntry>();
        for (ThreadCPUStats.StatEntry statEntry : m_cpuStats.getAllStats()) {
            if (statEntry.getPercentInLastPeriod() >= CPU_USAGE_DISPLAY_THRESHOLD)
            {
                stats.add(statEntry);
            }
        }
        Collections.sort(stats, Collections.reverseOrder());
        
        if (stats.isEmpty())
        {
            writer.writeln("<no threads reporting significant CPU usage>");
        }
        else
        {
            int numThreadsToDisplay = stats.size() > maxEntries ? maxEntries : stats.size();
            for (int i = 0; i < numThreadsToDisplay; i++)
            {
                ThreadCPUStats.StatEntry stat = stats.get(i);
                ThreadInfo ti = m_mgmtThreads.get(stat.getThreadId());

                StringBuilder sb = new StringBuilder();
                sb.append('\"').append(ti.getThreadName()).append('\"');
                sb.append(" tid=0x").append(Long.toHexString(ti.getThreadId()));
                sb.append(" cpu=").append(stat.getUsageInLastPeriod() / 1000000);  // convert nanosecs to millisecs
                sb.append("(").append(s_cpuPercentFormat.format(stat.getPercentInLastPeriod())).append("%)");
                
                writer.writeln(sb.toString());
            }
        }
    }   
    
    /**
     * Indicates whether CPU usage is available for this thread dump.
     * This required CPU information to have been available for both the
     * current thread dump and the previous one.  The usage is based on
     * the intervening period.
     */
    public boolean isCpuUsageAvailable()
    {
        return m_cpuUsageAvailable;
    }
    
    /**
     * Indicates whether CPU usage monitoring is enabled.  If it's enabled,
     * but isCpuUsageAvailable() returns false, then CPU usage should be
     * available for the next thread dump provided it's taken within
     * SAVED_CPU_STATS_TIMEOUT_MINS minutes of the current thread dump.
     */
    public boolean isCpuUsageEnabled()
    {
        synchronized (ThreadDump.class)
        {
            return s_savedCpuStats != null;
        }
    }
    
    
    public static boolean isLockInfoSupported()
    {
        return s_lockInfoSupported;
    }

    
    /**
     * Return array of all java.lang.management.ThreadInfo objects
     * (obtained from ThreadMXBean)
     */
    private ThreadInfo[] getThreadsFromMgmt() {
        ThreadInfo[] result = null;
        
        // Use Java 1.6 methods if available (via reflection)
        if (dumpAllThreads_METHOD != null)
        {
            try {
                result = (ThreadInfo[])dumpAllThreads_METHOD.invoke(s_threadMXBean, true, true);
            } catch (Exception e) {
                System.err.println("Failed to get lock info' for thread dump:");
                e.printStackTrace();
            }
        }
        
        // Otherwise use 1.5 API
        if (result == null)
        {
            long[] tids = s_threadMXBean.getAllThreadIds();
            result = s_threadMXBean.getThreadInfo(tids, Integer.MAX_VALUE);
        }
        
        return result;
    }
    
    /**
     * Return array of all java.lang.Thread objects. Obtained by
     *  calling ThreadGroup.enumerate() on the root thread group.
     */
    private Thread[] getThreadsFromRuntime() {

        // We have to allocate the array for the ThreadGroup.enumerate()
        // result, but we can't be sure how large it needs to be. Use
        // ThreadMXBean.getThreadCount() as a starting value, but increase
        // if the resulting thread count matches the array size (since
        // some threads may have been lost)
        Thread[] threads;
        int arraySize = s_threadMXBean.getThreadCount();
        int resultSize;
        do {
            arraySize *= 1.5;
            threads = new Thread[arraySize];
            resultSize = s_rootThreadGroup.enumerate(threads, true);
        } while (resultSize >= arraySize);

        return threads;
    }
    
    /**
     * Returns an array of ids representing the threads involved
     * in any deadlocks.  Returns null if there are no deadlocks.
     */
    private long[] getDeadlockedThreads() {

        // Use Java 1.6 method if available (via reflection)
        if (findDeadlockedThreads_METHOD != null)
        {
            try {
                return (long[])findDeadlockedThreads_METHOD.invoke(s_threadMXBean);
            } catch (Exception e) {
                System.err.println("Failed to get deadlocks with synchronizers:");
                e.printStackTrace();
            }
        }
        
        // Otherwise use 1.5 API
        // Only finds deadlocks involving monitors, not synchronizers
        return s_threadMXBean.findMonitorDeadlockedThreads();
    }
    
    private void printThreadInfo(IStateWriter writer, ThreadInfo ti)
    throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, IOException {
        
        // print thread title/headline
        printThreadTitle(writer, ti);

        // print stack trace (with locks if available)
        StackTraceElement[] stacktrace = ti.getStackTrace();
        String blockedByName = null;
        LockDetail[] monitors = EMPTY_LOCK_ARRAY;
        if (s_lockInfoSupported)
        {
            // get blocking lock and owned monitors
            LockDetail blockedBy = getBlockedByLock(ti);
            blockedByName = (blockedBy == null) ? null : blockedBy.toString();
            monitors = getLockedMonitors(ti);
        }
        else
        {
            // get blocking lock (owned monitors not available in Java 1.5)
            blockedByName = ti.getLockName();
        }
        
        for (int i = 0; i < stacktrace.length; i++) {
            // display stack frame
            StackTraceElement ste = stacktrace[i];
            writer.writeln(INDENT + "at " + ste.toString());
            
            // add any lock info' associated with the stack frame
            if (blockedByName != null && i == 0)
            {
                if (ti.getThreadState() == Thread.State.BLOCKED)
                {
                    writer.write(INDENT + "  - waiting to lock " + blockedByName);
                }
                else
                {
                    writer.write(INDENT + "  - waiting on " + blockedByName);
                }
                
                if (ti.getLockOwnerName() != null)
                {
                    writer.writeln(" [owned by \"" + ti.getLockOwnerName() + "\" tid=0x"
                            + Long.toHexString(ti.getLockOwnerId()) + "]");
                }
                else
                {
                    writer.writeln();
                }
            }
            for (LockDetail monitor : monitors) {
                if (monitor.getLockedStackDepth() == i) {
                    writer.writeln(INDENT + "  - locked " + monitor);
                }
            }
        }
        
        // print owned synchronizers
        if (s_lockInfoSupported)
        {
            LockDetail[] synchronizers = getLockedSynchronizers(ti);
            if (synchronizers.length > 0)
            {
                writer.writeln(INDENT + "Locked synchronizers (" + synchronizers.length + ")");
                for (LockDetail synchronizer : synchronizers)
                {
                    writer.writeln(INDENT + "  - " + synchronizer);
                }
            }
        }
    }
    
    private void printThreadTitle(IStateWriter writer, ThreadInfo ti)
    throws IOException
    {
        Thread t = m_runtimeThreads.get(ti.getThreadId());
        
        StringBuilder sb = new StringBuilder();
        sb.append('\"').append(ti.getThreadName()).append('\"');

        if (t != null)
        {
            if (t.isDaemon())
            {
                sb.append(" daemon");
            }
            sb.append(" prio=").append(t.getPriority());
        }
        else
        {
            sb.append(" <daemon/prio unavailable>");
        }
        
        sb.append(" tid=0x").append(Long.toHexString(ti.getThreadId()));
        
        if (m_cpuUsageAvailable)
        {
            ThreadCPUStats.StatEntry stat = m_cpuStats.getStat(ti.getThreadId());
            if (stat != null && stat.getUsageInLastPeriod() >= 0)
            {
                sb.append(" cpu=").append(stat.getUsageInLastPeriod() / 1000000);  // convert nanosecs to millisecs
                sb.append("(").append(s_cpuPercentFormat.format(stat.getPercentInLastPeriod())).append("%)");
            }
            else
            {
                sb.append(" cpu=n/a");
            }
        }
        
        sb.append(" state=").append(ti.getThreadState());
        
        if (ti.isSuspended())
        {
            sb.append(" (suspended)");
        }
        if (ti.isInNative())
        {
            sb.append(" (running in native)");
        }
        
        writer.writeln(sb.toString());      
    }   

    private LockDetail getBlockedByLock(ThreadInfo ti)
    throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
        Object lock = getLockInfo_METHOD.invoke(ti);

        if (lock == null)
        {
            return null;
        }
        else
        {
            return new LockDetail(lock);
        }
    }

    private LockDetail[] getLockedMonitors(ThreadInfo ti)
    throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
        Object[] monitors = (Object[])getLockedMonitors_METHOD.invoke(ti);

        // Copy to LockDetail array
        LockDetail[] result = new LockDetail[monitors.length];
        for (int i = 0; i < monitors.length; i++)
        {
            result[i] = new LockDetail(monitors[i]);
        }
        
        return result;
    }
    
    private LockDetail[] getLockedSynchronizers(ThreadInfo ti)
    throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
        Object[] locks = (Object[])getLockedSynchronizers_METHOD.invoke(ti);

        // Copy to LockDetail array
        LockDetail[] result = new LockDetail[locks.length];
        for (int i = 0; i < locks.length; i++)
        {
            result[i] = new LockDetail(locks[i]);
        }
        
        return result;
    }

}


class LockDetail
{
    String className = null;
    int hashCode = 0;
    int stackDepth = -1;

    /**
     * Builds LockDetail instance from given LockInfo/MonitorInfo object
     * using reflection.  
     */
    public LockDetail(Object lock)
    throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
    {
            className = (String)ThreadDump.getLockClassName_METHOD.invoke(lock);
            hashCode = (Integer)ThreadDump.getLockIdentityHashCode_METHOD.invoke(lock);
            if (ThreadDump.MonitorInfo_CLASS.isInstance(lock))
            {
                stackDepth = (Integer)ThreadDump.getLockedStackDepth_METHOD.invoke(lock);
            }
    }
    
    public String getClassName()
    {
        return className;
    }
    
    public int getIdentityHashCode()
    {
        return hashCode;
    }
    
    /**
     * If this lock represents a locked monitor, the stack frame
     * that locked the object monitor is returned.
     * Otherwise -1 is returned.
     */
    public int getLockedStackDepth()
    {
        return stackDepth;
    }
    
    /**
     * Returns a string representation of the lock
     */
    @Override
    public String toString()
    {
        // format used in Ctrl-Break thread dumps:
        //  <0x173a10f> (a java.lang.String)
        //return "<0x" + Integer.toHexString(hashCode) + "> (a " + className + ")";
        
        // format used elsewhere in SDF thread dump (e.g. in thread title
        // and deadlock info') - this therefore provides better internal
        // consistency:
        //  java.lang.String@173a10f
        return className + '@' + Integer.toHexString(hashCode);
    }
}
