//Copyright (c) 2010 Progress Software Corporation.  All Rights Reserved.

/**
 * Entry-point for generating a container thread dump
 */

package com.sonicsw.sdf.threads;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import com.sonicsw.sdf.IDiagnosticsConstants;
import com.sonicsw.sdf.IStateWriter;

public class ThreadDumpHandler {
    
    /**
     * Generates thread dump information
     * @param writer 
     * @throws IOException 
     */
    public static synchronized void writeThreadDump(IStateWriter writer)
    throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, IOException {

        // create thread dump
        ThreadDump threadDump = ThreadDump.createThreadDump();
        
        // display threads/stacks
        writer.writeln("Java Thread Dump (thread details collected in " + (threadDump.getElapsedTime())+ "ms)");
        threadDump.showThreads(writer);
        writer.writeln("" + threadDump.getThreadCount() + " threads");
        
        writer.writeln();

        // display deadlock info
        if (threadDump.getDeadlockedThreadCount() == 0)
        {
            writer.writeln("No deadlocks found");
        }
        else
        {
            writer.writeln("Found " + threadDump.getDeadlockedThreadCount() + " deadlocked threads...");        
            threadDump.showDeadlockedThreads(writer);           
        }
        
        // display cpu usage info
        if (threadDump.isCpuUsageEnabled())
        {
            if (threadDump.isCpuUsageAvailable())
            {
                long period = (threadDump.getStartTime() - threadDump.getPreviousThreadDumpTime()) / 1000;
                writer.writeln(IDiagnosticsConstants.NEWLINE + "Threads with highest CPU usage over last " + period + " seconds");
                threadDump.showCpuUsage(writer, 10);
            }
            else
            {
                writer.writeln(IDiagnosticsConstants.NEWLINE + "Take another thread dump within "
                        + ThreadDump.SAVED_CPU_STATS_TIMEOUT_MINS + " minutes to see CPU usage information.");
            }
        }       
    }
    
    public static boolean isLockInfoSupported()
    {
        return ThreadDump.isLockInfoSupported();
    }
    
    
    
    // ------- The rest of this file is test code -------
    
    public static void main(String[] args)
    {
        System.out.println("Setting up threads for test.");
        System.out.println("\nThere are six deadlocked threads in total: 3 deadlock cycles");
        System.out.println("each involving 2 threads.  A 1.6 JVM should detect all three");
        System.out.println("cycles.  A 1.5 JVM will only detect 1 cycle (2 threads) since");
        System.out.println("two of the deadlocks involve synchronizers.");
        System.out.println("\nOne thread spins, consuming a large amount of CPU.  This should");
        System.out.println("be picked up and reported in the second thread dump if the JVM");
        System.out.println("has CPU monitoring enabled.");
        
        
        setupTestThreads();

        System.out.println();
        
        try
        {
            IStateWriter writer = new TestStdoutWriter();

            System.out.println("\nThread dump 1...");
            ThreadDumpHandler.writeThreadDump(writer);

            System.out.println("\nSleeping for 5 seconds...");
            try { Thread.sleep(5000); } catch(InterruptedException ie) { }

            System.out.println("\nThread dump 2 (may include CPU usage info)...");
            ThreadDumpHandler.writeThreadDump(writer);
        }
        catch (Exception e)
        {
            System.err.println("Failed to generate thread dump");
            e.printStackTrace();
        }
        finally 
        {
            shutdownRequested = true;
        }
    }

    static Object monitor1 = new String("monitor1");
    static Object monitor2 = new String("monitor2");
    static Object monitor3 = new String("monitor3");
    static Lock lock1 = new ReentrantLock();
    static Lock lock2 = new ReentrantLock();
    static Lock lock3 = new ReentrantLock();
    static volatile boolean shutdownRequested;
    
    public static void setupTestThreads() {

        class DeadlockThread extends Thread {

            Object lockA = null;
            Object lockB = null;

            public DeadlockThread(String name, Object lockA, Object lockB) {
                super(name);
                this.lockA = lockA;
                this.lockB = lockB;
            }

            private void doDeadlockA() {
                if (lockA instanceof Lock)
                {
                    ((Lock)lockA).lock();
                    System.out.println("Thread " + getName() + " got lock A (synchronizer).  Sleeping...");
                    try { sleep(1000); } catch (InterruptedException e) { }
                    doDeadlockB();
                    ((Lock)lockA).unlock();
                }
                else
                {
                    synchronized (lockA) {
                        System.out.println("Thread " + getName() + " got lock A (monitor).  Sleeping...");
                        try { sleep(1000); } catch (InterruptedException e) { }
                        doDeadlockB();
                    }                   
                }
            }
            
            private void doDeadlockB() {
                if (lockB instanceof Lock)
                {
                    ((Lock)lockB).lock();
                    System.out.println("Thread " + getName() + " got lock B (synchronizer).  Done.");
                    ((Lock)lockB).unlock();
                }
                else
                {
                    synchronized (lockB) {
                        System.out.println("Thread " + getName() + " got lock B (monitor).  Done.");
                    }                   
                }
            }           

            @Override
            public void run() {
                doDeadlockA();
            }
        }   
        
        class WaitThread extends Thread {
            
            Object waitObj;
            
            public WaitThread(String name, Object waitObj) {
                super(name);
                this.waitObj = waitObj;
            }
            
            private void doWait()
            {
                synchronized(waitObj)
                {
                    System.out.println("Thread " + getName() + " waiting...");
                    try 
                    { 
                        while (!shutdownRequested)
                        {
                            waitObj.wait();
                        } 
                    } 
                    catch (InterruptedException e) { }
                }
            }
            
            @Override
            public void run() {
                doWait();
            }           
        }
        
        class BusyThread extends Thread {
            
            public BusyThread(String name) {
                super(name);
            }
            
            private void getBusy()
            {
                int i = 0;
                while(true)
                {
                    i = (i > 100) ? 0 : i + 1;
                }
            }
            
            @Override
            public void run() {
                getBusy();
            }           
        }   
        
        class ShortLifeThread extends Thread {
            
            private int m_lifetime;  // seconds
            
            public ShortLifeThread(String name, int lifetime) {
                super(name);
                m_lifetime = lifetime;
            }
            
            private void sleepAndTerminate()
            {
                System.out.println("Thread " + getName() + " sleeping for " + m_lifetime + " seconds...");
                try { sleep(m_lifetime * 1000); } catch (InterruptedException e) { }
                System.out.println("Thread " + getName() + " done.");
            }
            
            @Override
            public void run() {
                sleepAndTerminate();
            }           
        }       
        
        
        Thread[] threads = new Thread[9];
        threads[0] = new DeadlockThread("Monitor Deadlock Thread 1", monitor1, monitor2);
        threads[1] = new DeadlockThread("Monitor Deadlock Thread 2", monitor2, monitor1);
        threads[2] = new DeadlockThread("Synchronizer Deadlock Thread 1", lock1, lock2);
        threads[3] = new DeadlockThread("Synchronizer Deadlock Thread 2", lock2, lock1);
        threads[4] = new DeadlockThread("Monitor-Synchronizer Deadlock Thread 1", monitor3, lock3);
        threads[5] = new DeadlockThread("Monitor-Synchronizer Deadlock Thread 2", lock3, monitor3);
        threads[6] = new WaitThread("Waiting Thread", new Integer(0));
        threads[7] = new BusyThread("Busy Thread");
        threads[8] = new ShortLifeThread("Short-Lived Thread", 8);

        for (int i = 0; i < threads.length; i++)
        {
            threads[i].setDaemon(true);
            threads[i].start();
        }

        try {
            Thread.sleep(2000);
        } catch (InterruptedException e) {
        }
    }

}

// Trivial implementation of IStateWriter for test - writes to stdout
class TestStdoutWriter implements IStateWriter
{
    @Override
    public void write(String state) throws IOException {
        System.out.print(state);
    }

    @Override
    public void writeln(String state) throws IOException {
        System.out.println(state);
    }

    @Override
    public void writeln() throws IOException {
        System.out.println();
    }
    
    @Override
    public String getFilePath() {
        return null;
    }
    
    @Override
    public void close() {
    }   
}
